/* Scheme implementation intended for JACAL.
   Copyright (C) 1989, 1990 Aubrey Jaffer.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

The author can be reached at jaffer@ai.mit.edu or
Aubrey Jaffer, 84 Pleasant St., Wakefield MA 01880
*/

#include <stdio.h>
#include <signal.h>
#include <setjmp.h>
#include "scm.h"

#ifdef unix
#include <sys/types.h>
#include <sys/times.h>
struct tms time_buffer;
#define TIME_IN_MSEC(x) (((x)*50)/3)
long myruntime()
{
	times(&time_buffer);
	return time_buffer.tms_utime;
}
#else
#ifdef vms
#include <time.h>
#define TIME_IN_MSEC(x) ((x)*10)
long myruntime()
{
	return clock();
}
#else
#ifdef MSDOS
#include <stdlib.h>
#include <sys\types.h>
#include <sys\timeb.h>
#include <time.h>
#define TIME_IN_MSEC(x) ((x))
struct timeb time_buffer;
long myruntime()
{
	ftime(&time_buffer);
	return (long) time_buffer.time*1000 + time_buffer.millitm;
}
#else
#include <stdlib.h>
#define TIME_IN_MSEC(x) ((x)*1000)
long myruntime()
{
	long x;
	long time();
	time(&x);
	return (long) x;
}
#endif
#endif
#endif

jmp_buf errjmp;
int errjmp_ok = 0, sig_disabled = 1,  sig_deferred = 0;
SCM sym_errobj = BOOL_F;
char s_load[]="load";
long line_num = 1;
char *load_name = NULL;
static char filextension[] = FILE_EXTENSION;

#ifndef verbose
int verbose = 0;
#endif
long cells_allocated = 0, gc_rt, gc_time_taken;
long gc_cells_collected, gc_malloc_collected, gc_ports_collected;

unsigned char upcase[char_code_limit];
unsigned char downcase[char_code_limit];
unsigned char lowers[]="abcdefghijklmnopqrstuvwxyz";
unsigned char uppers[]="ABCDEFGHIJKLMNOPQRSTUVWXYZ";
init_tables()
{
	int i;
	for(i=0;i<char_code_limit;i++) upcase[i]=downcase[i]=i;
	for(i=0;i<sizeof(lowers);i++) {
		upcase[lowers[i]]=uppers[i];
		downcase[uppers[i]]=lowers[i];
	}
}

main(argc,argv)
int argc; 
char **argv;
{
  if (argc <= 1) {
#ifndef verbose
    verbose=1;
#endif
    fputs("SCM version ",stdout);
    fputs(SCMVERSION,stdout);
    puts(", Copyright (C) 1989, 1990 Aubrey Jaffer.\n\
SCM comes with ABSOLUTELY NO WARRANTY; for details type `(warranty)'.\n\
This is free software, and you are welcome to redistribute it\n\
under certain conditions; type `(terms)' for details.");
  }
  init_tables();
  init_storage();
  init_subrs();
  init_io();
  init_scl();
  init_scm();
  if (verbose)
  {
    fputs(";heap_size = ",stdout);
    iprint(heap_size,10,stdout);
    fputs(" cells, ",stdout);
    iprint(heap_size*sizeof(cell),10,stdout);
    puts(" bytes.");
  }
  repl_driver(argc,argv,1);
  if (verbose) puts(";EXIT");
  exit(0);
}

#ifdef unix
void bus_signal(sig)
int sig;
{
	signal(SIGBUS,bus_signal);
	err("bus error",UNDEFINED);
}
#endif
void segv_signal(sig)
int sig;
{
	signal(SIGSEGV,segv_signal);
	err("segment violation",UNDEFINED);
}
void int_signal(sig)
int sig;
{
	signal(SIGINT,int_signal);
	if (sig_disabled) sig_deferred = 1;
	else err_ctrl_c();
}

repl_driver(argc,argv,want_sigint)
int argc;
char **argv;
int want_sigint;
{
	long i;
	stack_start_ptr = &i;
	switch (setjmp(errjmp)) {
	case 0:
		if (want_sigint) signal(SIGINT,int_signal);
#ifdef unix
		signal(SIGBUS,bus_signal);
#endif
		signal(SIGSEGV,segv_signal);
		errjmp_ok = 1;
		sig_deferred = 0;
		sig_disabled = 0;
		for(i=1;i<argc;++i)if(argv[i][0] != '-') iload(argv[i]);
	default:
		errjmp_ok = 1;
		sig_deferred = 0;
		sig_disabled = 0;
		load_name = NULL;
		repl();
	case 2:
		;
	}
}

growth_mon(obj, size, units)
char *obj;
long size;
char *units;
{
	if (verbose)
	{
		DEFER_SIGINT;
		fputs("; grew ",stdout);
		fputs(obj,stdout);
		fputs(" to ",stdout);
		iprint(size,10,stdout);
		putc(' ',stdout);
		puts(units);
		ALLOW_SIGINT;
	}
}

gc_start()
{
	gc_rt = myruntime();
	gc_cells_collected = 0;
	gc_malloc_collected = 0;
	gc_ports_collected = 0;
	if (verbose) fputs(";GC ",stdout);
}
gc_end()
{
	gc_rt = myruntime() - gc_rt;
	gc_time_taken = gc_time_taken + gc_rt;
	if (verbose) {
		iprint(TIME_IN_MSEC(gc_rt),10,stdout);
		fputs(" cpu mSec, ",stdout);
		iprint(gc_cells_collected,10,stdout);
		fputs(" cells, ",stdout);
		iprint(gc_malloc_collected,10,stdout);
		fputs(" malloc, ",stdout);
		iprint(gc_ports_collected,10,stdout);
		puts(" ports collected");
	}
}

SCM gc_status(arg)
SCM arg;
{
  register SCM l;
  register long n;
  DEFER_SIGINT;
  if (!UNBNDP(arg))
    if FALSEP(arg) verbose = 0;
    else verbose = 1;
  fputs(verbose?";verbose ":";silent ",stdout);
  puts("garbage collection.");
  {
    for(n=0,l=freelist;NNULLP(l); ++n) l = CDR(l);
    putc(';',stdout);
    iprint(heap_size-n,10,stdout);
    fputs(" allocated ",stdout);
    iprint(n,10,stdout);
    puts(" free");
  }
  ALLOW_SIGINT;
  return BOOL_F;
}

repl()
{
  SCM x;
  long rt;
  DEFER_SIGINT;
  while(1) {
    fputs("> ",stdout);
    fflush(stdout);
    ALLOW_SIGINT;
    x = lread(cur_inp);
    if (x == EOF_VAL) break;
    rt = myruntime();
    cells_allocated = 0;
    gc_time_taken = 0;
    x = EVAL(x,EOL);
    {
      DEFER_SIGINT;
      fputs(";Evaluation took ",stdout);
      iprint(TIME_IN_MSEC(myruntime()-rt),10,stdout);
      fputs(" mSec (",stdout);
      iprint(TIME_IN_MSEC(gc_time_taken),10,stdout);
      fputs(" in gc) ",stdout);
      iprint(cells_allocated,10,stdout);
      puts(" cons work");
    }
    lprin1f(x,stdout,1);
    putc('\n',stdout);
  }

}
SCM quit()
{
	longjmp(errjmp,2);
}

err_head()
{
	fflush(stdout);
	putc('\n',stderr);
	if(load_name) {
		putc('"',stderr);
		fputs(load_name,stderr);
		fputs("\", line ",stderr);
		iprint((long)line_num,10,stderr);
		fputs(": ",stderr);
	}
}
warn(str1,str2)
char *str1,*str2;
{
	DEFER_SIGINT;
	err_head();
	fputs("WARNING: ",stderr);
	fputs(str1,stderr);
	fputs(str2,stderr);
	putc('\n',stderr);
	fflush(stderr);
	ALLOW_SIGINT;
}
extern cell dummy_cell;
everr(sexp,env,arg,pos,s_subr)
SCM sexp,env,arg;
long pos;
char *s_subr;
{
	DEFER_SIGINT;
	err_head();
	fputs("ERROR: ",stderr);
	switch (pos) {
	case ARG1:
	case ARG2:
	case ARG3:
		fputs("Wrong type in arg ",stderr);
		iprint(pos,10,stderr);
		fputs(" to ",stderr);
		goto putsname;
	case WNA:
		fputs("Wrong number of args to ",stderr);
		lprin1f(arg,stderr,1);
		if (s_subr && *s_subr)
			fputs(" from ",stderr);
		arg = UNDEFINED;
		goto putsname;
	case NALLOC:
		fputs("could not allocate ",stderr);
		iprint(INUM(arg),10,stderr);
		putc(' ',stderr);
		goto putsname;
	case NOFILE:
		perror((char *)arg);
		fputs("could not open file in ",stderr);
		fputs(s_subr,stderr);
		arg = UNDEFINED;
		break;
	default:
		fputs((char *)pos,stderr);
		if INUMP(arg) {
		  fputs(s_subr,stderr);
		  fputs(": ",stderr);
		  lprin1f(arg,stderr,1);
		  arg = UNDEFINED;
		  break;
		}
putsname:
		fputs(s_subr,stderr);
		if UNBNDP(sexp) break;
		if NIMP(sexp) {
		  fputs("\n; in expression: ",stderr);
		  if (sexp == (SCM)&dummy_cell) lprin1f(CAR(sexp),stderr,1);
		  if ECONSP(sexp)
		    iprlist("(... ",CAR(sexp),CDR(sexp),')',stderr,1);
		  else lprin1f(sexp,stderr,1);
		}
		if NULLP(env) {
		  fputs("\n; in top level environment.",stderr);
		  break;
		}
		fputs("\n; in scope:",stderr);
		while NNULLP(env) {
		  putc('\n',stderr);
		  fputs(";   ",stderr);
		  lprin1f(CAR(CAR(env)),stderr,1);
		  env = CDR(env);
		}
	}
	if (errjmp_ok && !UNBNDP(arg))
		fputs("\n; (see errobj)",stderr);
	putc('\n',stderr);
	fflush(stderr);
	if (errjmp_ok) {
		VCELL(sym_errobj)=arg;
		longjmp(errjmp,1);
	}
	lprin1f(arg,stderr,1);
	fputs("\nFATAL ERROR DURING CRITICAL CODE SECTION\n",stderr);
	exit(1);
}
wta(arg,pos,s_subr)
SCM arg;
long pos;
char *s_subr;
{
  everr(UNDEFINED,EOL,arg,pos,s_subr);
}
err(message,x)
char *message;
SCM x;
{
	wta(x,message,"");
}
err_ctrl_c()
{
	wta(UNDEFINED,"control-c interrupt","");
}

SCM iiopen(filename)
char *filename;
{
	FILE *f = fopen(filename,"r");
	if (!f) {
		if (filextension[0]) {
			char buf[64];
			register int i,j;
			for(i=0;filename[i];i++) buf[i] = filename[i];
			for(j=0;filextension[j];j++)buf[i++] = filextension[j];
			buf[i] = 0;
			f = fopen(buf,"r");
		}
		if (!f) wta(filename,NOFILE,s_load);
	}
	return makport(f,tc_inport);
}
iload(filename)
char *filename;
{
	char *olonam = load_name;
	long olninum = line_num;
	SCM form,port = iiopen(filename);
	load_name = filename;
	line_num = 1;
	DEFER_SIGINT;
	fputs(";loading ",stdout);
	fputs(filename,stdout);
	ALLOW_SIGINT;
	while(1) {
		form = lread(port);
		if (EOF_VAL == form) break;
		SIDEVAL(form,EOL);
	}
	close_port(port);
	DEFER_SIGINT;
	fputs(" ;done loading ",stdout);
	ALLOW_SIGINT;
	puts(filename);
	line_num = olninum;
	load_name = olonam;
}
SCM load(filename)
SCM filename;
{
	ASSERT(NIMP(filename) && STRINGP(filename),filename,ARG1,s_load);
	iload(CHARS(filename));
	return UNSPECIFIED;
}
icat(in,out)
SCM in, out;
{
  FILE *fin = STREAM(in);
  FILE *fout = STREAM(out);
  int c;
  DEFER_SIGINT;
  while (EOF != (c = getc(fin))) {
    CHECK_SIGINT;
    putc(c, fout);
  }
  ALLOW_SIGINT;
}
SCM warranty()
{
  SCM port = iiopen(WARRANTY);
  icat(port,cur_outp);
  return close_port(port);
}
SCM list_file(file)
SCM file;
{
  SCM port = open_input_file(file);
  icat(port,cur_outp);
  return close_port(port);
}
/* This is cfib, for compiled fib. Test to see what the overhead
   of interpretation actually is in a given implementation
;(define (fib x)
;  (if (< x 2)
;      x
;    (+ (fib (- x 1))
;       (fib (- x 2))))) */

SCM cfib(x)
SCM x;
{
  if NFALSEP(lessp(x,MAKINUM(2L),EOL)) return(x);
  else return(sum(cfib(difference(x,MAKINUM(1L))),
		  cfib(difference(x,MAKINUM(2L)))));
}

#ifdef vms
#include <descrip.h>
#include <ssdef.h>
char s_sys_edit[]="edit";
SCM sys_edit(fname)
SCM fname;
{
	struct dsc$descriptor_s d;
	ASSERT(STRINGP(fname),fname,ARG1,s_sys_edit);
	d.dsc$b_dtype = DSC$K_DTYPE_T;
	d.dsc$b_class = DSC$K_CLASS_S;
	d.dsc$w_length = LENGTH(fname);
	d.dsc$a_pointer = CHARS(fname);
	DEFER_SIGINT;
	edt$edit(&d);
	ALLOW_SIGINT;
	return(fname);
}

SCM vms_debug(v)
SCM v;
{
	lib$signal(SS$_DEBUG);
	return(v);
}
#endif
char s_system[]="system";
SCM lsystem(cmd)
SCM cmd;
{
	ASSERT(STRINGP(cmd),cmd,ARG1,s_system);
	return MAKINUM(system(CHARS(cmd)));
}

static iproc subr0s[]={
	{"quit",quit},
	{"warranty",warranty},
	{"terms",warranty},
      	{0,0}};

static iproc subr1os[]={
	{"verbose",gc_status},
      	{0,0}};

static iproc subr1s[]={
	{s_load,load},
	{"list-file",list_file},
	{"cfib",cfib},
#ifdef vms
	{s_sys_edit,sys_edit},
	{"vms-debug",vms_debug},
#endif
	{"system",lsystem},
      	{0,0}};
	
init_scm()
{
	sym_errobj=intern("errobj",-6L);
	VCELL(sym_errobj)=UNDEFINED;
	init_iprocs(subr0s, tc7_subr_0);
	init_iprocs(subr1os, tc7_subr_1o);
	init_iprocs(subr1s, tc7_subr_1);
}
