/* Scheme implementation intended for JACAL.
   Copyright (C) 1990, 1991 Aubrey Jaffer.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

The author can be reached at jaffer@ai.mit.edu or
Aubrey Jaffer, 84 Pleasant St., Wakefield MA 01880
*/

#include <stdio.h>
#include <ctype.h>
#ifdef vms
# ifndef CHEAP_CONTINUATIONS
#  include "setjump.h"
# else
#  include <setjmp.h>
# endif
#else
# include <setjmp.h>
#endif
#define IN_SYS
#include "scm.h"

#ifdef EBCDIC
char *charnames[]={
  "nul","soh","stx","etx", "pf", "ht", "lc","del",
   0   , 0   ,"smm", "vt", "ff", "cr", "so", "si",
  "dle","dc1","dc2","dc3","res", "nl", "bs", "il",
  "can", "em", "cc", 0   ,"ifs","igs","irs","ius",
   "ds","sos", "fs", 0   ,"byp", "lf","eob","pre",
   0   , 0   , "sm", 0   , 0   ,"enq","ack","bel",
   0   , 0   ,"syn", 0   , "pn", "rs", "uc","eot",
   0   , 0   , 0   , 0   ,"dc4","nak", 0   ,"sub",
  "space",s_newline,"tab","backspace","return","page","null"};
char charnums[]=
"\000\001\002\003\004\005\006\007\
\010\011\012\013\014\015\016\017\
\020\021\022\023\024\025\026\027\
\030\031\032\033\034\035\036\037\
\040\041\042\043\044\045\046\047\
\050\051\052\053\054\055\056\057\
\060\061\062\063\064\065\066\067\
\070\071\072\073\074\075\076\077\
 \n\t\b\r\f\0";
#endif
#ifdef ASCII
char *charnames[] = {
  "nul","soh","stx","etx","eot","enq","ack","bel",
   "bs", "ht", "nl", "vt", "np", "cr", "so", "si",
  "dle","dc1","dc2","dc3","dc4","nak","syn","etb",
  "can", "em","sub","esc", "fs", "gs", "rs", "us",
  "space",s_newline,"tab","backspace","return","page","null","del"};
char charnums[]=
"\000\001\002\003\004\005\006\007\
\010\011\012\013\014\015\016\017\
\020\021\022\023\024\025\026\027\
\030\031\032\033\034\035\036\037\
 \n\t\b\r\f\0\177";
#endif
char *isymnames[]={
				/* Special Forms */
  "and", "begin", "case", "cond", "define", "do", "if", "lambda",
  "let", "let*", "letrec", "or", "quote", "set!",
				/* IXSYMS go here */
  "quasiquote",
				/* other keywords */
  "=>", "else", "unquote", "unquote-splicing", ".",
				/* Flags */
  "#f", "#t", "#[undefined]", "#[eof]", "()", "#[unspecified]"
  };

size_t tok_buf_len = 30;
char *tok_buf;
SCMPTR stack_start_ptr = 0;

char	s_make_vector[]="make-vector";
static char	s_cw_input_file[]="call-with-input-file";
static char	s_cw_output_file[]="call-with-output-file";
static char	s_input_portp[]="input-port?", s_output_portp[]="output-port?";
static char	s_open_input_file[]="open-input-file";
static char	s_open_output_file[]="open-output-file";
static char	s_close_port[]="close-input-port";
static char	s_read_char[]="read-char", s_peek_char[]="peek-char";
char	s_read[]="read", s_write[]="write", s_newline[]="newline";
static char	s_display[]="display", s_write_char[]="write-char";

#ifdef IO_EXTENSIONS
static char	s_open_rw_file[]="open-io-file",
		s_file_position[]="file-position",
		s_file_set_pos[]="file-set-position";
static char	s_read_to_str[]="read-string!";
#endif
char	s_eofin[]="end of file in ";
char	s_bytes[]="bytes",s_bad_type[]="unknown type in ";
static char	s_unknown_sharp[]="unknown # object";
static char	s_heap[]="heap", s_hplims[]="hplims";

SCM lreadr(),lreadparen(),istring2number();
size_t read_token();
void grow_throw(), gc_mark(), mark_locations(), gc_sweep();

char num_buf[NUMBUFLEN];
void iprint(n,radix,f)
long n;
int radix;
FILE *f;
{
  fwrite(num_buf,(size_t)1,iint2str(n,radix,num_buf),f);
}
void ipruk(hdr,ptr,f)
     char *hdr;
     SCM ptr;
     FILE *f;
{
  fputs("#[unknown-",f);
  fputs(hdr,f);
  if CELLP(ptr) {
    fputs(" (0x",f);
    iprint(CAR(ptr),16,f);
    fputs(" . 0x",f);
    iprint(CDR(ptr),16,f);
    fputs(") @",f);
  }
  fputs(" 0x",f);
  iprint(ptr,16,f);
  putc(']',f);
}

void iprlist(hdr,carexp,cdrexp,tlr,f,writing)
     char *hdr, tlr;
     SCM carexp, cdrexp;
     FILE *f;
     int writing;
{
  fputs(hdr,f);
  CHECK_SIGINT;
  iprin1(carexp,f,writing);
  for(;NIMP(cdrexp);cdrexp=CDR(cdrexp)) {
    if NECONSP(cdrexp) break;
    putc(' ',f);
    CHECK_SIGINT;
    iprin1(CAR(cdrexp),f,writing);
  }
  if NNULLP(cdrexp) {
    fputs(" . ",f);
    iprin1(cdrexp,f,writing);
  }
  putc(tlr,f);
}
void iprin1(exp,f,writing)
SCM exp;
FILE *f;
int writing;
{
  register long i;
taloop:
  switch (7 & (int)exp) {
  case 2:
  case 6:
    iprint(INUM(exp),10,f);
    break;
  case 4:
    if ICHRP(exp) {
      i = ICHR(exp);
      if (writing) fputs("#\\",f);
      if (!writing) putc((int)i,f);
      else if ((i<=' ') && charnames[i]) fputs(charnames[i],f);
#ifndef EBCDIC
      else if (i=='\177')
	fputs(charnames[(sizeof charnames/sizeof(char *))-1],f);
#endif
      else putc((int)i,f);
    }
    else if (IFLAGP(exp) && (ISYMNUM(exp)<(sizeof isymnames/sizeof(char *))))
      fputs(ISYMCHARS(exp),f);
    else if ILOCP(exp) {
      fputs("#@",f);
      iprint((long)IFRAME(exp),10,f);
      putc(ICDRP(exp)?'-':'+',f);
      iprint((long)IDIST(exp),10,f);
    }
    else goto idef;
    break;
  case 1:			/* gloc */
    fputs("#@",f);
    exp--;
    goto taloop;
  default:
  idef:
    ipruk("immediate",exp,f);
    break;
  case 0:
    switch TYP7(exp) {
    case tcs_cons_gloc:
    case tcs_cons_imcar:
    case tcs_cons_nimcar:
      iprlist("(",CAR(exp),CDR(exp),')',f,writing);
      break;
    case tcs_closures:
      exp = CODE(exp);
      iprlist("#[CLOSURE ",CAR(exp),CDR(exp),']',f,writing);
      break;
    case tc7_string:
      if (writing) {
	putc('\"',f);
	for(i=0;i<LENGTH(exp);++i) switch (CHARS(exp)[i]) {
	case '"':
	case '\\':
	  putc('\\',f);
	default:
	  putc(CHARS(exp)[i], f);
	}
	putc('\"',f);
      }
      else
      dispstr:
	fwrite(CHARS(exp),(size_t)1,(size_t)LENGTH(exp),f);
/* was     for(i=0;i<LENGTH(exp);++i) putc(CHARS(exp)[i], f); */
      break;
    case tc7_vector:
      fputs("#(",f);
      for(i=0;i<(LENGTH(exp)-1);++i) {
	CHECK_SIGINT;
	iprin1(VELTS(exp)[i],f,writing);
	putc(' ',f);
      }
      if (i<LENGTH(exp)) {
	CHECK_SIGINT;
	iprin1(VELTS(exp)[i],f,writing);
      }
      putc(')',f);
      break;
    case tcs_symbols:
      exp = NAMESTR(exp);
      goto dispstr;
    case tcs_subrs:
      fputs("#[primitive-procedure ",f);
      fputs(CHARS(SNAME(exp)),f);
      putc(']',f);
      break;
    case tc7_contin:
      fputs("#[continuation ",f);
      iprint(LENGTH(exp),10,f);
      fputs(" @ ",f);
      iprint((long)CHARS(exp),16,f);
      putc(']',f);
      break;
    case tc7_smob:
      switch TYP16(exp) {
      case tc16_port:
	fputs("#[",f);
	if (RDNG & CAR(exp))
	  fputs("input-",f);
	if (WRTNG & CAR(exp))
	  fputs("output-",f);
	fputs("port ",f);
	if CLOSEDP(exp) fputs("closed",f);
	else iprint((long)fileno(STREAM(exp)),10,f);
	putc(']',f);
	break;
#ifdef FLOATS
      case tc16_flo:
	fwrite(num_buf,(size_t)1,iflo2str(exp,num_buf),f);
	break;
#endif
      default:
	goto cdef;
      }
      break;
    default:
    cdef:
      ipruk("type",exp,f);
    }
  }
}

#ifdef vms
int lgetc(f)			/* This version of getc handles ^c */
FILE *f;
{
	int c;
	long old_sig_deferred;
	DEFER_SIGINT;
	old_sig_deferred = sig_deferred;
	c = getc(f);
	if ((old_sig_deferred == 0) && sig_deferred && (f == stdin))
		while(c && (c != EOF)) c = getc(f);
	ALLOW_SIGINT;
	return c;
}
#else
#define lgetc getc
#endif

char *grow_tok_buf();

int flush_ws(f,eoferr)
FILE *f;
char *eoferr;
{
	register int c;
	while(1) switch (c = lgetc(f)) {
	case EOF:
goteof:
		if (eoferr) wta(UNDEFINED,s_eofin,eoferr);
		return c;
	case ';':
lp:
		switch (c = lgetc(f)) {
		case EOF:
			goto goteof;
		case LINE_INCREMENTORS:
			line_num++;
			break;
		default:
			goto lp;
		}
		break;
	case LINE_INCREMENTORS:
		line_num++;
	case WHITE_SPACES:
		break;
	default:
		return c;
	}
}
SCM lread(port)
SCM port;
{
	FILE *f;
	int c;
	if UNBNDP(port) port = cur_inp;
	else ASSERT(NIMP(port) && OPINPORTP(port),port,ARG1,s_read);
	f = STREAM(port);
	c = flush_ws(f,(char *)NULL);
	if (c == EOF) return EOF_VAL;
	ungetc(c,f);
	return lreadr(f);
}
SCM lreadr(f)
FILE *f;
{
	int c;
	size_t j;
	SCM p;
tryagain:
	c = flush_ws(f,s_read);
	switch (c) {
	case '(':
		return lreadparen(f,s_list);
	case ')':
		warn("unexpected \")\"","");
		goto tryagain;
	case '\'':
		return cons2(I_QUOTE,lreadr(f),EOL);
	case '`':
		return cons2(I_QUASIQUOTE,lreadr(f),EOL);
	case ',':
		c = lgetc(f);
		if (c == '@') p = I_UQ_SPLICING;
		else {
			ungetc(c,f);
			p = I_UNQUOTE;
		}
		return cons2(p,lreadr(f),EOL);
	case '#':
		c = lgetc(f);
		switch (c) {
		case '(':
			return vector(lreadparen(f,s_vector));
		case 't':
		case 'T':
			return BOOL_T;
		case 'f':
		case 'F':
			return BOOL_F;
		case 'b':
		case 'B':
		case 'o':
		case 'O':
		case 'd':
		case 'D':
		case 'x':
		case 'X':
		case 'i':
		case 'I':
		case 'e':
		case 'E':
			ungetc(c,f);
			c = '#';
			goto num;
		case '\\':
			c = lgetc(f);
			j = read_token(c,f);
			if (j==1) return MAKICHR(c);
			for (c=0;c<sizeof charnames/sizeof(char *);c++)
				if (charnames[c]  &&
				    (0==strcmp(charnames[c],tok_buf)))
				  return MAKICHR(charnums[c]);
			wta(UNDEFINED,"unknown # object: #\\",tok_buf);
		case '|':
			j = 1;	/* here j is the comment nesting depth */
lp:
			c = lgetc(f);
lpc:
			switch (c) {
			case EOF:
			  wta(UNDEFINED,s_eofin,"balanced comment");
			case LINE_INCREMENTORS:
			  line_num++;
			default:
			  goto lp;
			case '|':
			  if ('#' != (c = lgetc(f))) goto lpc;
			  if (--j) goto lp;
			  break;
			case '#':
			  if ('|' != (c = lgetc(f))) goto lpc;
			  ++j; goto lp;
			}
			goto tryagain;
		default:
			wta(MAKICHR(c),s_unknown_sharp,"");
		}
	case '\"':
		j = 0;
		while ((c = lgetc(f)) != '\"') {
			ASSERT(c != EOF,UNDEFINED,"eof in ",s_string);
			if (j+1 >= tok_buf_len) grow_tok_buf(s_string);
			if (c == '\\') c = lgetc(f);
			tok_buf[j] = c;
			++j;
		}
		if (j == 0) return nullstr;
		tok_buf[j] = 0;
		return makfromstr(tok_buf,j);
	case DIGITS:
	case '.': case '-': case '+':
num:
		j = read_token(c,f);
		p = istring2number(tok_buf, (long)j, 10L);
		if (p != BOOL_F) return p;
		ASSERT(c != '#',UNDEFINED,s_unknown_sharp,tok_buf);
		goto tok;
	default:
		j = read_token(c,f);
tok:
		return intern(tok_buf,(long)j);
	}
}
size_t read_token(ic,f)
int ic;
FILE *f;
{
	register size_t j = 1;
	register int c = ic;
	register char *p = tok_buf;
	p[0] = downcase[c];
	while(1) {
		if (j+1 >= tok_buf_len) p = grow_tok_buf("symbol");
		switch (c = lgetc(f)) {
		case '(':
		case ')':
		case '\"':
		case ';':
/*		case '\'':
		case '`':
		case ',':
		case '#': */
			ungetc(c,f);
		case EOF:
		case WHITE_SPACES:
getout:
			p[j] = 0;
			return j;
		case LINE_INCREMENTORS:
			line_num++;
			goto getout;
		default:
			p[j++] = downcase[c];
		}
	}
}
SCM lreadparen(f,name)
FILE *f;
char *name;
{
	SCM tmp;
	int c;
	c = flush_ws(f,name);
	if (c == ')') return EOL;
	ungetc(c,f);
	tmp = lreadr(f);
	if (tmp != I_DOT) return cons(tmp,lreadparen(f,name));
	tmp = lreadr(f);
	c = flush_ws(f,name);
	if (c != ')') wta(UNDEFINED,"missing close paren","");
	return tmp;
}
SCM open_input_file(filename)
SCM filename;
{
	FILE *f;
	ASSERT(NIMP(filename) && STRINGP(filename),
	       filename,ARG1,s_open_input_file);
	f = fopen(CHARS(filename),OPEN_READ);
	if (!f) wta((SCM)CHARS(filename),(char *)NOFILE,s_open_input_file);
	return makport(f,tc_inport);
}
SCM open_output_file(filename)
SCM filename;
{
	FILE *f;
	ASSERT(NIMP(filename) && STRINGP(filename),
	       filename,ARG1,s_open_output_file);
	f = fopen(CHARS(filename),OPEN_WRITE);
	if (!f) wta((SCM)CHARS(filename),(char *)NOFILE,s_open_output_file);
	return makport(f,tc_outport);
}
#ifdef IO_EXTENSIONS
SCM open_rw_file(filename)
SCM filename;
{
	FILE *f;
	ASSERT(NIMP(filename) && STRINGP(filename),
	       filename,ARG1,s_open_rw_file);
	f = fopen(CHARS(filename),OPEN_BOTH);
	if (!f) wta((SCM)CHARS(filename),(char *)NOFILE,s_open_rw_file);
	return makport(f,tc_ioport);
}
SCM file_position(port)
SCM port;
{
	ASSERT(NIMP(port) && OPPORTP(port), port,ARG1,s_file_position);
	return MAKINUM(ftell(STREAM(port)));
}
SCM file_set_position(port, pos)
SCM port, pos;
{
	ASSERT(NIMP(port) && OPPORTP(port), port,ARG1,s_file_set_pos);
	if (fseek(STREAM(port),INUM(pos),0)) return BOOL_F;
	return BOOL_T;
}
#endif

SCM close_port(f)
SCM f;
{
	ASSERT(NIMP(f) && PORTP(f),f,ARG1,s_close_port);
	if CLOSEDP(f) return UNSPECIFIED;
	DEFER_SIGINT;
	fclose(STREAM(f));
	SETSTREAM(f,0);
	CAR(f) &= ~OPN;
	ALLOW_SIGINT;
	return UNSPECIFIED;
}
SCM cw_input_file(str,proc)
SCM str,proc;
{
	SCM file,res;
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_cw_input_file);
	ASSERT(NFALSEP(procedurep(proc)),proc,ARG2,s_cw_input_file);
	file = open_input_file(str);
	res = apply(proc,file,listofnull);
	close_port(file);
	return res;
}
SCM cw_output_file(str,proc)
SCM str,proc;
{
	SCM file,res;
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_cw_output_file);
	ASSERT(NFALSEP(procedurep(proc)),proc,ARG2,s_cw_output_file);
	file = open_output_file(str);
	res = apply(proc,file,listofnull);
	close_port(file);
	return res;
}
SCM input_portp(x)
SCM x;
{
	if IMP(x) return BOOL_F;
	return INPORTP(x) ? BOOL_T : BOOL_F;
}
SCM output_portp(x)
SCM x;
{
	if IMP(x) return BOOL_F;
	return OUTPORTP(x) ? BOOL_T : BOOL_F;
}
SCM cur_input_port()
{
	return cur_inp;
}
SCM cur_output_port()
{
	return cur_outp;
}

SCM read_char(port)
SCM port;
{
	int c;
	if UNBNDP(port) port = cur_inp;
	else ASSERT(NIMP(port) && OPINPORTP(port),port,ARG1,s_read_char);
	c = lgetc(STREAM(port));
	if (c == EOF) return EOF_VAL;
	return MAKICHR(c);
}
SCM peek_char(port)
SCM port;
{
	FILE *f;
	int c;
	if UNBNDP(port) port = cur_inp;
	else ASSERT(NIMP(port) && OPINPORTP(port), port,ARG1,s_peek_char);
	f = STREAM(port);
	c = lgetc(f);
	if (c == EOF) return EOF_VAL;
	ungetc(c,f);
	return MAKICHR(c);
}
#ifdef IO_EXTENSIONS
SCM read_to_string(str,port)
SCM str,port;
{
  if UNBNDP(port) port = cur_inp;
  else
    ASSERT(NIMP(port) && OPINPORTP(port),port,ARG2,s_read_to_str);
  ASSERT(NIMP(str) && STRINGP(str),
	 str,ARG1,s_read_to_str);
  return MAKINUM(fread(CHARS(str),(size_t)1,(size_t)LENGTH(str),STREAM(port)));
}
#endif
SCM eof_objectp(x)
SCM x;
{
	return (EOF_VAL == x) ? BOOL_T : BOOL_F;
}
SCM lwrite(obj,port)
SCM obj,port;
{
	if UNBNDP(port) port = cur_outp;
	else ASSERT(NIMP(port) && OPOUTPORTP(port),port,ARG2,s_write);
	DEFER_SIGINT;
	iprin1(obj,STREAM(port),1);
	ALLOW_SIGINT;
	return UNSPECIFIED;
}
SCM display(obj,port)
SCM obj,port;
{
	if UNBNDP(port) port = cur_outp;
	else ASSERT(NIMP(port) && OPOUTPORTP(port),port,ARG2,s_display);
	DEFER_SIGINT;
	iprin1(obj,STREAM(port),0);
	ALLOW_SIGINT;
	return UNSPECIFIED;
}
SCM newline(port)
SCM port;
{
	if UNBNDP(port) port = cur_outp;
	else ASSERT(NIMP(port) && OPOUTPORTP(port),port,ARG1,s_newline);
	DEFER_SIGINT;
	putc('\n',STREAM(port));
	if (port == cur_outp) fflush(STREAM(port));
	ALLOW_SIGINT;
	return UNSPECIFIED;
}
SCM write_char(chr,port)
SCM chr,port;
{
	if UNBNDP(port) port = cur_outp;
	else ASSERT(NIMP(port) && OPOUTPORTP(port),port,ARG2,s_write_char);
	ASSERT(ICHRP(chr),chr,ARG1,s_write_char);
	DEFER_SIGINT;
	putc((int)ICHR(chr),STREAM(port));
	ALLOW_SIGINT;
	return UNSPECIFIED;
}

static iproc subr0s[]={
	{"gc",gc},
	{"current-input-port",cur_input_port},
	{"current-output-port",cur_output_port},
	{0,0}};

static iproc subr1s[]={
	{"call-with-current-continuation",call_cc},
	{s_input_portp,input_portp},
	{s_output_portp,output_portp},
	{s_open_input_file,open_input_file},
	{s_open_output_file,open_output_file},
	{s_close_port,close_port},
	{"close-output-port",close_port},
	{"eof-object?",eof_objectp},
#ifdef IO_EXTENSIONS
	{s_open_rw_file,open_rw_file},
	{"close-io-port",close_port},
	{s_file_position,file_position},
#endif
	{0,0}};

static iproc subr1os[]={
	{s_read,lread},
	{s_read_char,read_char},
	{s_peek_char,peek_char},
	{s_newline,newline},
	{0,0}};

static iproc subr2s[]={
	{s_cw_input_file,cw_input_file},
	{s_cw_output_file,cw_output_file},
#ifdef IO_EXTENSIONS
	{s_file_set_pos,file_set_position},
#endif
	{0,0}};

static iproc subr2os[]={
	{s_write,lwrite},
	{s_display,display},
	{s_write_char,write_char},
#ifdef IO_EXTENSIONS
	{s_read_to_str,read_to_string},
#endif
	{0,0}};

void init_io(){
  init_iprocs(subr0s, tc7_subr_0);
  init_iprocs(subr1s, tc7_subr_1);
  init_iprocs(subr1os, tc7_subr_1o);
  init_iprocs(subr2s, tc7_subr_2);
  init_iprocs(subr2os, tc7_subr_2o);
}

size_t hplim_ind = 0,num_heap_segs = 1;
long heap_size = 0;
CELLPTR *hplims, heap_org;
SCM freelist = EOL;

char *must_malloc(len,what)
long len;
char *what;
{
	char *ptr;
	size_t size = len;
	if (len != size)
malerr:
		wta(MAKINUM(len),(char *)NALLOC,what);
	ptr = malloc(size);
	if (ptr != NULL) return ptr;
	gc();
	ptr = malloc(size);
	if (ptr != NULL) return ptr;
	goto malerr;
}
#define NEWCELL(_into) {if IMP(freelist) _into = gc_for_newcell();\
	else {_into = freelist;freelist = CDR(freelist);++cells_allocated;}}
int symhash_dim = NUM_HASH_BUCKETS;

void init_isyms()
{
	int hash,i = NUM_ISYMS,n = symhash_dim;
	char *cname,c;
	while (0 <= --i) {
		hash = 0;
		cname = isymnames[i];
		while(c = *cname++) hash = ((hash * 17) ^ c) % n;
		VELTS(symhash)[hash] =
			cons((i<15)?MAKSPCSYM(i):MAKISYM(i),
			     VELTS(symhash)[hash]);
	}
}
/* if length is negative, use the given string directly if possible */
SCM intern(name,len)
char *name;
long len;
{
	SCM lsym;
	size_t alen = (len < 0) ? -len:len;
	register size_t i = alen;
	register char *tmp = name;
	register int hash = 0, n = symhash_dim;
	while(i--) hash = ((hash * 17) ^ *tmp++) % n;
	for(lsym=VELTS(symhash)[hash];NIMP(lsym);lsym=CDR(lsym)) {
		if ISYMP(CAR(lsym)) {
			tmp = ISYMCHARS(CAR(lsym));
			for(i = 0;i < alen;i++) {
				if (tmp[i] == 0) goto trynext;
				if (name[i] != tmp[i]) goto trynext;
				}
			if (tmp[i] == 0) return CAR(lsym);
		}
		else {
			tmp = CHARS(NAMESTR(CAR(lsym)));
			if (alen != LENGTH(NAMESTR(CAR(lsym)))) goto trynext;
			for(i = alen;i--;)
				if (name[i] != tmp[i]) goto trynext;
			return CAR(lsym);
		}
trynext:
		;
	}
	DEFER_SIGINT;
	if ((len < 0)/* && ( 1 & (long)name == 0)*/) {
		NEWCELL(lsym);
		SETLENGTH(lsym,alen,tc7_string);
		SETCHARS(lsym,name);
	}
	else lsym = makfromstr(name, alen);
	{
		SCM z = lsym;
		NEWCELL(lsym);
		SETNAMESTR(lsym,z);
		VCELL(lsym) = UNDEFINED;
	}
	VELTS(symhash)[hash] = cons(lsym,VELTS(symhash)[hash]);
	ALLOW_SIGINT;
	return lsym;
}
SCM cons(x,y)
SCM x,y;
{
	register SCM z;
	NEWCELL(z);
	CAR(z) = x;
	CDR(z) = y;
	return z;
}
SCM cons2(w,x,y)
SCM w,x,y;
{
	register SCM z;
	NEWCELL(z);
	CAR(z) = x;
	CDR(z) = y;
	x = z;
	NEWCELL(z);
	CAR(z) = w;
	CDR(z) = x;
	return z;
}
SCM cons2r(w,x,y)
SCM w,x,y;
{
	register SCM z;
	NEWCELL(z);
	CAR(z) = w;
	CDR(z) = x;
	x = z;
	NEWCELL(z);
	CAR(z) = x;
	CDR(z) = y;
	return z;
}

SCM makstr(len)
long len;
{
	register SCM s;
	char *str = must_malloc(len+1,s_string);
	str[len] = 0;
	NEWCELL(s);
	DEFER_SIGINT;
	SETLENGTH(s,len,tc7_string);
	SETCHARS(s,str);
	ALLOW_SIGINT;
	return s;
}
SCM makfromstr(src, len)
char *src;
size_t len;
{
	SCM s;
	register char *dst;
	s = makstr((long)len);
	dst = CHARS(s);
	while (len--) *dst++ = *src++;
	return s;
}
SCM make_vector(k,fill)
SCM k,fill;
{
	register SCM v;
	register long i;
	register char *str;
	ASSERT(INUMP(k),k,ARG1,s_make_vector);
	i = INUM(k);
	if (i == 0) return nullvect;
	str = must_malloc(i*sizeof(SCM),s_vector);
	NEWCELL(v);
	DEFER_SIGINT;
	SETLENGTH(v,i,tc7_vector);
	SETCHARS(v,str);
	while(--i>=0) ((SCM *)str)[i] = fill;
	ALLOW_SIGINT;
	return v;
}
#ifdef FLOATS
SCM makdbl (x,y)
double x,y;
{
  register SCM z;
  double *nums;
  if (y == 0.0) {
/*    if (x == 0.0) return dbl0; */
    nums = (double *)must_malloc(1L*sizeof(double),"real");
  }
  else {
    nums = (double *)must_malloc(2L*sizeof(double),"complex");
    nums[1] = y;
  }
  nums[0] = x;
  NEWCELL(z);
  DEFER_SIGINT;
  CAR(z) = (y == 0.0) ? tc_dblr : tc_dblc;
  SETCDR(z,nums);
  ALLOW_SIGINT;
  return z;
}

#ifdef SINGLES
SCM makflo (x)
float x;
{
  register SCM z;
  NEWCELL(z);
  CAR(z) = tc_flo;
  FLO(z) = x;
  return z;
}
#endif /* SINGLES */
#endif /* FLOATS */

void make_subr(name,type,fcn)
char *name;
int type;
SCM (*fcn)();
{
	SCM sym = intern(name,-(long)strlen(name));
	register SCM z;
	NEWCELL(z);
	SETSNAME(z,NAMESTR(sym),type);
	SUBRF(z) = fcn;
	VCELL(sym) = z;
}
SCM closure(code,env)
SCM code,env;
{
	register SCM z;
	NEWCELL(z);
	SETCODE(z,code);
	ENV(z) = env;
	return z;
}
long stack_size(start)
SCMPTR start;
{
	long stack;
#ifdef STACK_GROWS_UP
	stack = (SCMPTR)&stack - start;
#else
	stack = start - (SCMPTR)&stack;
#endif
	return stack;
}

typedef struct {jmp_buf jmpbuf;} regs;
#define JMPBUF(x) (((regs *)CHARS(x))->jmpbuf)
#define SETJMPBUF(x,v) SETCDR(x,v)

SCM throwval = UNDEFINED;
SCM call_cc(proc)
SCM proc;
{
	long j;
	SCM cont;
#ifdef CHEAP_CONTINUATIONS
	NEWCELL(cont);
	DEFER_SIGINT;
	SETLENGTH(cont,0,tc7_contin);
	SETJMPBUF(cont,must_malloc((long)sizeof(regs),"continuation"));
	ALLOW_SIGINT;
#else
	register SCM *src,*dst;
	NEWCELL(cont);
	DEFER_SIGINT;
	FLUSH_REGISTER_WINDOWS;
	SETLENGTH(cont,stack_size(stack_start_ptr),tc7_contin);
	SETJMPBUF(cont,must_malloc(sizeof(regs)+LENGTH(cont)*sizeof(SCM *)
				   ,"continuation"));
	ALLOW_SIGINT;
	src = stack_start_ptr;
#ifndef STACK_GROWS_UP
	src -= LENGTH(cont);
#endif
	dst = (SCM *)(CHARS(cont)+sizeof(regs));
	for (j = LENGTH(cont);0 <= --j;) *dst++ = *src++;
#endif
	if (setjmp(JMPBUF(cont))) return throwval;
	return apply(proc,cont,listofnull);
}

#define PTR_GT(x,y) PTR_LT(y,x)
#define PTR_LE(x,y) (!PTR_GT(x,y))
#define PTR_GE(x,y) (!PTR_LT(x,y))

void throw(cont,val)
SCM cont,val;
{
#ifndef CHEAP_CONTINUATIONS
	register long j;
	register SCM *src;
	register SCMPTR dst = stack_start_ptr;
#ifdef STACK_GROWS_UP
	if PTR_GE(dst + LENGTH(cont),(SCMPTR)&cont) grow_throw(cont,val);
#else
	dst -= LENGTH(cont);
	if PTR_LE(dst,(SCMPTR)&cont) grow_throw(cont,val);
#endif
	FLUSH_REGISTER_WINDOWS;
	src = (SCM *)(CHARS(cont)+sizeof(regs));
	for (j = LENGTH(cont);0 <= --j;) *dst++ = *src++;
#endif
	throwval = val;
	longjmp(JMPBUF(cont),1);
}
#ifndef CHEAP_CONTINUATIONS
void grow_throw(cont,val)	/* Grow the stack so that there is room */
SCM cont,val;			/* to copy in the continuation.  Then */
{				/* retry the throw. */
	long growth[100];
	throw(cont,val);
}
#endif
SCM makport(stream,type)
FILE *stream;
long type;
{
	register SCM z;
	DEFER_SIGINT;
	NEWCELL(z);
	SETLENGTH(z,0L,type);
	SETSTREAM(z,stream);
	ALLOW_SIGINT;
	return z;
}

void fixconfig(s1,s2)
char *s1, *s2;
{
	fputs(s1,stdout);
	puts(s2);
	puts("in config.h and recompile scm");
	exit(1);
}

size_t init_heap_seg(base,size)
CELLPTR base;
size_t size;
{
	register CELLPTR ptr = base;
	CELLPTR seg_org, seg_end;
	size_t i = 0,ni = hplim_ind;
	if (base == NULL) return 0;
	seg_org = CELL_UP(ptr);
	seg_end = CELL_DN((char *)ptr + size);
	while((i < ni) && PTR_LE(hplims[i],seg_org)) i++;
	ni = i;
	hplim_ind += 2;
	for(i+=2;i < hplim_ind;i++) hplims[i] = hplims[i-2];
	hplims[ni++] = seg_org;
	hplims[ni++] = seg_end;
	ni = seg_end - seg_org;
	ptr = seg_org;
	for (i=ni;i--;ptr++) {
		CAR(ptr) = (SCM)tc_free_cell;
		CDR(ptr) = (SCM)(ptr+1);
	}
	CDR(--ptr) = freelist;
	freelist = (SCM)seg_org;
	heap_size += ni;
	growth_mon(s_heap,heap_size,"cells");
	return size;
}
void alloc_some_heap()
{
	size_t len = num_heap_segs*2;
	CELLPTR ptr;
	if (hplim_ind >= len)
		if(!(hplims = (CELLPTR *)realloc((char *)hplims,
				len*2*sizeof(CELLPTR))))
			wta(UNDEFINED,"could not realloc ",s_hplims);
		else growth_mon("number of heaps",
				(long)(num_heap_segs = len),"segments");
	len = HEAP_SEG_SIZE;
	if (len != HEAP_SEG_SIZE)
		fixconfig("reduce","size of HEAP_SEG_SIZE");
trya:
	if (len >= MIN_HEAP_SEG_SIZE) {
		ptr = (CELLPTR) malloc(len);
		if (ptr == NULL) {
			len /= 2;
			goto trya;
		}
		init_heap_seg(ptr,len);
	}
}
char *grow_tok_buf(typstr)
char *typstr;
{
	size_t len = tok_buf_len;
	long llen = len;
	llen += len / 2;
	len = llen;
	if(len != llen) goto dontdo;
	tok_buf = (char *)realloc(tok_buf,len);
	if (tok_buf == NULL) {
		tok_buf = (char *)malloc(tok_buf_len);
		if(tok_buf == NULL) errjmp_ok = 0;
dontdo:
		wta(MAKINUM(llen),(char *)NALLOC,(char *)typstr);
	}
	else tok_buf_len = len;
	growth_mon("tok_buf",(long)tok_buf_len,s_bytes);
	return tok_buf;
}
void init_storage()
{
	size_t j;
#ifdef SINGLES
	if (sizeof (float) != sizeof (long))
	  fixconfig("remove\n#","define SINGLES");
#endif
	if (stack_start_ptr==0)
	  wta(INUM0,"stack_start_ptr not ",ISYMCHARS(I_SET));
	if (stack_size(stack_start_ptr) < 0)
		fixconfig(
#ifdef STACK_GROWS_UP
			"remove\n#"
#else
			"add\n#"
#endif
			,"define STACK_GROWS_UP");

	tok_buf = must_malloc((long)tok_buf_len,"tok_buf");

	hplims = (CELLPTR *)
		must_malloc((long)num_heap_segs*2*sizeof(CELLPTR),s_hplims);
	j = INIT_HEAP_SIZE;
	if ((j != INIT_HEAP_SIZE) || !init_heap_seg((CELLPTR) malloc(j),j))
		alloc_some_heap();
	heap_org = hplims[0];
		/* hplims[0] can change. do not remove heap_org */


	cur_inp=makport(stdin,tc_inport);
	cur_outp=makport(stdout,tc_outport);
	listofnull = cons(EOL,EOL);
	undefineds = cons(UNDEFINED,EOL);
	CDR(undefineds) = undefineds;
	nullstr = makstr(0L);
	NEWCELL(nullvect);
	SETLENGTH(nullvect,0L,tc7_vector);
	SETCHARS(nullvect,NULL);
	symhash = make_vector(MAKINUM(symhash_dim),EOL);
	init_isyms();
}
/* The way of garbage collecting which allows use of the cstack is due to */
/* Scheme In One Defun, but in C this time.

 *			  COPYRIGHT (c) 1989 BY				    *
 *	  PARADIGM ASSOCIATES INCORPORATED, CAMBRIDGE, MASSACHUSETTS.	    *
 *			   ALL RIGHTS RESERVED				    *

Permission to use, copy, modify, distribute and sell this software
and its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all copies
and that both that copyright notice and this permission notice appear
in supporting documentation, and that the name of Paradigm Associates
Inc not be used in advertising or publicity pertaining to distribution
of the software without specific, written prior permission.

PARADIGM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
PARADIGM BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

gjc@paradigm.com

Paradigm Associates Inc		 Phone: 617-492-6079
29 Putnam Ave, Suite 6
Cambridge, MA 02138
*/

SCM gc_for_newcell()
{
	SCM fl;
	gc();
	if ((gc_cells_collected < MIN_GC_YIELD) || IMP(freelist))
		alloc_some_heap();
	++cells_allocated;
	fl = freelist;
	freelist = CDR(fl);
	return fl;
}

SCM sys_protects[NUM_PROTECTS];
jmp_buf save_regs_gc_mark;

SCM gc()
{
	long j = NUM_PROTECTS;
	DEFER_SIGINT;
	errjmp_ok = 0;
	gc_start();
	while(0 <= --j) gc_mark(sys_protects[j]);
	/* This assumes that all registers are saved into the jmp_buf */
	setjmp(save_regs_gc_mark);
	mark_locations((SCM *) save_regs_gc_mark,
			(size_t) sizeof(save_regs_gc_mark)/sizeof(SCM *));
#ifdef STACK_GROWS_UP
	mark_locations(stack_start_ptr,(SCMPTR)&j - stack_start_ptr);
#else
	mark_locations((SCMPTR)&j,stack_start_ptr - (SCMPTR)&j);
#endif
#ifdef SHORT_ALIGN
	mark_locations((SCM *) (((char *)save_regs_gc_mark)+2),
		       (size_t)(sizeof(save_regs_gc_mark)-2)/sizeof(SCM *));
#ifdef STACK_GROWS_UP
	mark_locations((SCMPTR)(((char *)stack_start_ptr)+2),
		       (SCMPTR)&j - stack_start_ptr);
#else
	mark_locations((SCMPTR)(((char *)&j)+2),
		       stack_start_ptr - (SCMPTR)&j);
#endif
#endif
	gc_sweep();
	gc_end();
	errjmp_ok = 1;
	ALLOW_SIGINT;
	return UNSPECIFIED;
}

void gc_mark(p)
SCM p;
{
  register long i;
  register SCM ptr = p;
 gc_mark_loop:
  if IMP(ptr) return;
  if (NCELLP(ptr)
      /* #ifndef RECKLESS
	 || PTR_GT(hplims[0], (CELLPTR)ptr)
	 || PTR_GE((CELLPTR)ptr, hplims[hplim_ind-1])
	 #endif */
      ) wta(ptr,"rogue pointer in ",s_heap);
  switch TYP7(ptr) {
  case tcs_cons_nimcar:
    if GCMARKP(ptr) break;
    SETGCMARK(ptr);
    gc_mark(CAR(ptr));
    ptr = GCCDR(ptr);
    goto gc_mark_loop;
  case tcs_cons_imcar:
  case tcs_cons_gloc:
    if GCMARKP(ptr) break;
    SETGCMARK(ptr);
    ptr = GCCDR(ptr);
    goto gc_mark_loop;
  case tcs_symbols:
    if GCMARKP(ptr) break;
    SETGCMARK(ptr);
    gc_mark(NAMESTR(ptr));
    ptr = GCCDR(ptr);
    goto gc_mark_loop;
  case tcs_closures:
    if GCMARKP(ptr) break;
    SETGCMARK(ptr);
    gc_mark(CODE(ptr));
    ptr = GCCDR(ptr);
    goto gc_mark_loop;
  case tc7_vector:
    if GC8MARKP(ptr) break;
    SETGC8MARK(ptr);
    i=LENGTH(ptr);
    if (i == 0) break;
    while(--i>0) gc_mark(VELTS(ptr)[i]);
    ptr = VELTS(ptr)[0];
    goto gc_mark_loop;
  case tc7_contin:
    if GC8MARKP(ptr) break;
    SETGC8MARK(ptr);
    mark_locations(VELTS(ptr),
		   (size_t)
		   (LENGTH(ptr) + sizeof(regs)/sizeof(SCM *)));
#ifdef SHORT_ALIGN
    mark_locations(CHARS(ptr)+2,
		   (size_t)
		   (LENGTH(ptr) + (sizeof(regs)-2)/sizeof(SCM *)));
#endif
    break;
  case tc7_string:
    /*		if GC8MARKP(ptr) break;*/
    SETGC8MARK(ptr);
  case tcs_subrs:
    break;
  case tc7_smob:
    if GC8MARKP(ptr) break;
    switch TYP16(ptr) {
#ifdef FLOATS
    case tc16_flo:
#endif
    case tc16_port:
      SETGC8MARK(ptr);
      break;
    default:
      goto def;
    }
    break;
  default:
  def:
    wta(ptr,s_bad_type,"gc_mark");
  }
}

void mark_locations(x,n)
SCM x[];
size_t n;
{
	register long m = n;
	register int i,j;
	register CELLPTR ptr;
	while(0 <= --m) if CELLP(x[m]) {
		ptr = (CELLPTR)x[m];
		i=0;
		j=hplim_ind;
		do {
			if PTR_GT(hplims[i++], ptr) break;
			if PTR_LE(hplims[--j], ptr) break;
			if ((i != j)  &&
			    PTR_LE(hplims[i++], ptr)  &&
			    PTR_GT(hplims[--j], ptr)) continue;
			if NFREEP(x[m]) gc_mark(x[m]);
			break;
		} while(i<j);
	}
}

void gc_sweep()
{
  register cell *ptr;
  register SCM nfreelist = EOL;
  register long n=0,m=0;
  register size_t j;
  int i=0;
  size_t seg_size;
  while (i<hplim_ind) {
    ptr=hplims[i++];
    seg_size=hplims[i++] - (CELLPTR)ptr;
    for(j=seg_size;j--;++ptr) {
      switch TYP7(ptr) {
      case tcs_cons_imcar:
      case tcs_cons_nimcar:
      case tcs_cons_gloc:
      case tcs_closures:
	if GCMARKP(ptr) goto cmrkcontinue;
	break;
      case tc7_vector:
	if GC8MARKP(ptr) goto c8mrkcontinue;
	m += (LENGTH(ptr)*sizeof(SCM));
	free(CHARS(ptr));
	break;
#ifdef BIGDIG
      case tc7_bignum:
	if GC8MARKP(ptr) goto c8mrkcontinue;
	m += (LENGTH(ptr)*sizeof(short));
	free(CHARS(ptr));
	break;
#endif
      case tc7_string:
	if GC8MARKP(ptr) goto c8mrkcontinue;
	m += LENGTH(ptr)+1;
	free(CHARS(ptr));
	break;
      case tc7_contin:
	if GC8MARKP(ptr) goto c8mrkcontinue;
	m += LENGTH(ptr) + sizeof(regs);
	free(CHARS(ptr));
	break;
      case tcs_symbols:
	goto cmrkcontinue;
      case tcs_subrs:
	continue;
      case tc7_smob:
	switch GCTYP16(ptr) {
	case tc16_port:
	  if GC8MARKP(ptr) goto c8mrkcontinue;
	  if OPENP(ptr) {
	    gc_ports_collected++;
	    fclose(STREAM(ptr));
	    SETSTREAM(ptr,0);
	    CAR(ptr) &= ~OPN;
	  }
	case tc_free_cell:
	  break;
#ifdef FLOATS
	case tc16_flo:
	  if GC8MARKP(ptr) goto c8mrkcontinue;
	  switch ((int)(CAR(ptr)>>16)) {
	  case (IMAG_PART | REAL_PART)>>16:
	    m += 2*sizeof(double);
	  case REAL_PART>>16:
	  case IMAG_PART>>16:
	    m += sizeof(double);
	    free(CHARS(ptr));
#ifdef SINGLES
	  case 0:
#endif
	    break;
	  default:
	    goto sweeperr;
	  }
	  break;
#endif				/* FLOATS */
	default:
	  goto sweeperr;
	}
	break;
      default:
      sweeperr:
	wta((SCM)ptr,s_bad_type,"gc_sweep");
      }
      ++n;
      CAR(ptr) = (SCM)tc_free_cell;
      CDR(ptr) = nfreelist;
      nfreelist = (SCM)ptr;
      continue;
    c8mrkcontinue:
      CLRGC8MARK(ptr);
      continue;
    cmrkcontinue:
      CLRGCMARK(ptr);
    }
    if (n==seg_size) growth_mon("heap seg",0L,"cells");
    gc_cells_collected += n;
    n=0;
  }
  gc_malloc_collected = m;
  freelist = nfreelist;
}
