; "hash.scm", hashing functions for Scheme.
; Copyright (c) 1992, Aubrey Jaffer

;This hashing functions are for use in quickly classifying objects.
;Hash tables use these functions.

;  (hashq obj k)					procedure
;  (hashv obj k)					procedure
;  (hash obj k)						procedure

;Returns an exact non-negative integer less than k.  For each
;non-negative integer less than k there are arguments obj for which
;the hashing functions applied to obj and k returns that integer.
;For HASHQ (EQ? obj1 obj2) implies (= (HASHQ obj1 k) (HASHQ obj2)).
;For HASHV (EQV? obj1 obj2) implies (= (HASHV obj1 k) (HASHV obj2)).
;For HASH (EQUAL? obj1 obj2) implies (= (HASH obj1 k) (HASH obj2)).
;HASH, HASHV, and HASHQ return in time bounded by a constant.  Notice
;that HASH implies HASHV implies HASHQ.

;  (predicate->hash pred)				procedure

;Returns a hash function (like HASHQ, HASHV, or HASH) corresponding to
;the equality predicate pred.  Pred should be EQ?, EQV?, EQUAL?, =,
;CHAR=?, CHAR-CI=?, STRING=?, or STRING-CI=?.
 
(define (hash:hash-char char n)
  (modulo (char->integer char) n))

(define (hash:hash-char-ci char n)
  (modulo (char->integer (char-downcase char)) n))

(define (hash:hash-symbol sym n)
  (hash:hash-string (symbol->string sym) n))

;;; I am trying to be careful about overflow and underflow here.
(define (hash:hash-number num n)
  (if (exact? num)
      (modulo (cond ((integer? num) num)
		    ((rational? num)
		     (- (abs (numerator num)) (denominator num))))
	      n)
      (let ((anum (abs (if (real? num) num
			   (abs (- (real-part num) (imag-part num)))))))
	(inexact->exact
	 (floor (if (< anum n) anum (* n (/ n anum))))))))

(define (hash:hash-string str n)
  (let ((len (string-length str)))
    (if (> len 5)
	(let loop ((h 0) (i 5))
	  (if (positive? i)
	      (loop (modulo (+ (* h 256)
			       (char->integer
				(string-ref str (modulo h len))))
			    n)
		    (- i 1))
	      h))
	(let loop ((h (modulo 264 n)) (i (- len 1)))
	  (if (>= i 0)
	      (loop (modulo (+ (* h 256)
			       (char->integer (string-ref str i)))
			    n)
		    (- i 1))
	      h)))))

(define (hash:hash-string-ci str n)
  (let ((len (string-length str)))
    (if (> len 5)
	(let loop ((h 0) (i 5))
	  (if (positive? i)
	      (loop (modulo (+ (* h 256)
			       (char->integer
				(char-downcase
				 (string-ref str (modulo h len)))))
			    n)
		    (- i 1))
	      h))
	(let loop ((h (modulo 264 n)) (i (- len 1)))
	  (if (>= i 0)
	      (loop (modulo (+ (* h 256)
			       (char->integer
				(char-downcase (string-ref str i))))
			    n)
		    (- i 1))
	      h)))))

(define (hash:hash obj n)
  (let hs ((d 10) (obj obj))
    (cond
     ((number? obj)      (hash:hash-number obj n))
     ((char? obj)        (modulo (char->integer obj) n))
     ((symbol? obj)      (hash:hash-symbol obj n))
     ((string? obj)      (hash:hash-string obj n))
     ((vector? obj)
      (let ((len (vector-length obj)))
	(if (> len 5)
	    (let lp ((h 1) (i (quotient d 2)))
	      (if (positive? i)
		  (lp (modulo (+ (* h 256)
				 (hs 2 (vector-ref obj (modulo h len))))
			      n)
		      (- i 1))
		  h))
	    (let loop ((h (- n 1)) (i (- len 1)))
	      (if (>= i 0)
		  (loop (modulo (+ (* h 256) (hs (quotient d len)
						 (vector-ref obj i)))
				n)
			(- i 1))
		  h)))))
     ((pair? obj)
      (if (positive? d) (modulo (+ (hs (quotient d 2) (car obj))
				   (hs (quotient d 2) (cdr obj)))
				n)
	  1))
     (else
      (modulo
       (cond
	((null? obj)        256)
	((boolean? obj)     (if obj 257 258))
	((eof-object? obj)  259)
	((input-port? obj)  260)
	((output-port? obj) 261)
	((procedure? obj)   262)
	((and (provided? 'RECORD) (record? obj))
	 (let* ((rtd (record-type-descriptor obj))
		(fns (record-type-field-names rtd))
		(len (length fns)))
	   (if (> len 5)
	       (let lp ((h (modulo 266 n)) (i (quotient d 2)))
		 (if (positive? i)
		     (lp (modulo
			  (+ (* h 256)
			     (hs 2 ((record-accessor
				     rtd (list-ref fns (modulo h len)))
				    obj)))
			  n)
			 (- i 1))
		     h))
	       (let loop ((h (- n 1)) (i (- len 1)))
		 (if (>= i 0)
		     (loop (modulo
			    (+ (* h 256)
			       (hs (quotient d len)
				   ((record-accessor
				     rtd (list-ref fns (modulo h len)))
				    obj)))
			    n)
			   (- i 1))
		     h)))))
	(else               263))
       n)))))

(define hash hash:hash)
(define hashv hash:hash)

;;; Object-hash is somewhat expensive on copying GC systems (like
;;; PC-Scheme and MITScheme).  We use it only on strings, pairs,
;;; vectors, and records.  This also allows us to use it for both
;;; hashq and hashv.

(if (provided? 'object-hash)
    (set! hashv
	  (if (provided? 'record)
	      (lambda (obj k)
		(if (or (string? obj) (pair? obj) (vector? obj) (record? obj))
		    (modulo (object-hash obj) k)
		    (hash:hash obj k)))
	      (lambda (obj k)
		(if (or (string? obj) (pair? obj) (vector? obj))
		    (modulo (object-hash obj) k)
		    (hash:hash obj k))))))

(define hashq hashv)

(define (predicate->hash pred)
  (cond ((eq? pred eq?) hashq)
	((eq? pred eqv?) hashv)
	((eq? pred equal?) hash)
	((eq? pred =) hash:hash-number)
	((eq? pred char=?) hash:hash-char)
	((eq? pred char-ci=?) hash:hash-char-ci)
	((eq? pred string=?) hash:hash-string)
	((eq? pred string-ci=?) hash:hash-string-ci)
	(else (slib:error "unknown predicate for hash" pred))))
