; "hashtab.scm", hash tables for Scheme.
; Copyright (c) 1992, Aubrey Jaffer

;A hash table is a vector of association lists.

;   (make-hash-table k)					procedure

;Returns a vector of empty (association) lists.

;Hash table functions provide utilities for an associative database.
;These functions take an equality predicate, pred, as an argument.
;Pred should be EQ?, EQV?, EQUAL?, =, CHAR=?, CHAR-CI=?, STRING=?, or
;STRING-CI=?.   

;  (predicate->hash-asso pred)				procedure

;Returns an hash association function of 2 arguments, key and hashtab,
;corresponding to pred.  The returned function returns a key-value
;pair whose key is pred equal to its first argument or #f if no key in
;hashtab is pred equal to the first argument.

;  (hash-inquirer pred)					procedure

;Returns a procedure of 2 arguments, hashtab and key, which returns
;the value associated with key in hashtab or #f if key does not appear
;in hash.

;  (hash-associator pred)				procedure

;Returns a procedure of 3 arguments, hashtab, key, and value, which
;modifies hashtab so that key and value associated.  Any previous
;value associated with key will be lost.

;  (hash-remover pred)					procedure

;Returns a procedure of 2 arguments, hashtab and key, which modifies
;hashtab so that the association whose key is key removed.

(require 'hash)
(require 'alist)

(define (make-hash-table k) (make-vector k '()))

(define (predicate->hash-asso pred)
  (let ((hashfun (predicate->hash pred))
	(asso (predicate->asso pred)))
    (lambda (key hashtab)
      (asso key
	    (vector-ref hashtab (hashfun key (vector-length hashtab)))))))

(define (hash-inquirer pred)
  (let ((hashfun (predicate->hash pred))
	(ainq (alist-inquirer pred)))
    (lambda (hashtab key)
      (ainq (vector-ref hashtab (hashfun key (vector-length hashtab)))
	    key))))

(define (hash-associator pred)
  (let ((hashfun (predicate->hash pred))
	(asso (alist-associator pred)))
    (lambda (hashtab key val)
      (let* ((num (hashfun key (vector-length hashtab))))
	(vector-set! hashtab num
		     (asso (vector-ref hashtab num) key val)))
      hashtab)))

(define (hash-remover pred)
  (let ((hashfun (predicate->hash pred))
	(arem (alist-remover pred)))
    (lambda (hashtab key)
      (let* ((num (hashfun key (vector-length hashtab))))
	(vector-set! hashtab num
		     (arem (vector-ref hashtab num) key)))
      hashtab)))
