# Bindings for incremental and regexp search


# Find_forward and find_reverse can be used by other procedures; they have
# no visible effects.

# Finds first occurrence of string after point in text widget t
# Returns start and end of occurance or "" if unsuccessful.
proc find_forward {t point string} {
	set answer [string first $string [$t get $point end]]
	if {($answer < 0)} {return ""
	} else {return [list "$point +$answer chars" "$point +$answer chars +[string length $string] chars"]
}}

# Finds first occurrence of string before point in text widget t
# Returns start and end of occurance or "" if unsuccessful.
proc find_reverse {t point string} {
	set answer [string last $string [$t get 1.0 $point]]
	if {($answer < 0)} {return ""
	} else {return [list "1.0 +$answer chars" "1.0 +$answer chars +[string length $string] chars"]
}}

# A regular expression searcher
# Finds first occurrence of regular expression after point in text widget t
# Returns start and end of first occurance or "" if unsuccessful.
proc regexp_find_forward {t point exp} {
	if {[catch {set success [regexp -indices $exp [$t get $point end] where]}]} {set success 0}
	if {($success == 0)} {return ""
	} else {return [list "$point +[lindex $where 0] chars" \
				"$point +[lindex $where 1] chars +1 chars"]
}}

# Informs user of success or failure of search and resets search tag.
proc search_any {t f success_msg failure_msg search_fn search_string direction} {
	set was_search_area [$t tag nextrange search 1.0 end]
	if {[llength $was_search_area] == 2} {
		eval $t tag remove search $was_search_area
		eval $t tag add was_search $was_search_area
	}
	set length [string length $search_string]
	if {($length == 0)} {return 0}

	set slist [$search_fn $t insert $search_string]
	if {($slist != "")} {
		$t tag add search [lindex $slist 0] [lindex $slist 1]
		$f.s configure -text $success_msg
		if {($direction == "s")} {	$t mark set insert search.first
		} else {			$t mark set insert search.last}
		$t yview -pickplace insert
		return 1
	} else {$f.s configure -text $failure_msg ; beep
		return 0
}}

# Direction we're searching. "s" == forward, "r" == reverse
set direction "s"

proc re_search {t f success_msg failure_msg search_fn search_string d} {
	global direction
	set direction $d
	set new_index [$t index insert]

	if {($d == "s")} {
		if {[lindex [$f.s configure -text] 4] == $failure_msg} {
			$t mark set insert 1.0
		} else {$t mark set insert {insert +1 chars}
			if {[catch {set new_index [$t index search.last]}]} {
				set new_index [$t index "insert +1 chars"]}}
	} else {
		if {[lindex [$f.s configure -text] 4] == $failure_msg} {
			$t mark set insert end
		} else {$t mark set insert {insert -1 chars}
			if {[catch {set new_index [$t index search.first]}]} {
				set new_index [$t index "insert -1 chars"]}}
	}

	search_any $t $f $success_msg $failure_msg $search_fn \
				$search_string $direction
	if {[lindex [$f.s configure -text] 4] == $failure_msg} {
		$t mark set insert $new_index}
}

proc exit_search {t f c} {
	if {![regexp . $c]} {return}
	$t tag remove search 1.0 end
	$t tag remove was_search 1.0 end
	destroy_f_entry $t $f.s $f.ss
	global menu ; menuentries_change_state $menu.search.m normal 3 4 6
}

# Messages
set search_msg "Search: "
set reverse_search_msg "Reverse Search: "
set search_failed_msg "Search failed: "
set reverse_search_failed_msg "Reverse Search failed: "
set regexp_msg "Regexp Search: "
set regexp_failed_msg "Regexp Search Failed: "

# The string currently being searched for (it's bound to the search entry)
set search_string ""

# Called whenever key is hit in entry during incremental search.
proc revise_search {t f c} {
	global search_msg reverse_search_msg direction
	global search_failed_msg reverse_search_failed_msg search_string
	if {![regexp . $c]} {return}
	if {($direction == "s")} {set result [search_any $t $f $search_msg $search_failed_msg find_forward $search_string $direction]
	}  else {$t mark set insert {insert +1 chars}
		set result [search_any $t $f $reverse_search_msg $reverse_search_failed_msg find_reverse $search_string $direction]}
}

proc make_search_incremental {t f} {
	foreach binding [bind Entry] {
		if {([bind $f.ss $binding] == "")} {
			bind $f.ss $binding "[bind Entry $binding]
						revise_search $t $f %A"}}
}

# For any general type of search
proc search_setup {t f success_msg failure_msg search_function} {
	global Keys
	create_f_entry $t $f.s $f.ss
	$f.ss configure -textvariable search_string
	parse_bindings $f.ss \
$Keys(C_m)	"exit_search $t $f x" \
$Keys(C_s)	"re_search $t $f \"$success_msg\" \"$failure_msg\" \
					 $search_function \[$f.ss get\] s"
}

# For regular expression searches
proc regexp_search_setup {t f m} {
	global regexp_msg regexp_failed_msg
	search_setup $t $f $regexp_msg $regexp_failed_msg regexp_find_forward
	$f.s configure -text $regexp_msg
}

# For forward and reverse searching
proc bidirectional_search_setup {t f d} {
	global search_msg reverse_search_msg direction Keys
	global search_failed_msg reverse_search_failed_msg search_string

	$t tag lower was_search
	search_setup $t $f $search_msg $search_failed_msg find_forward

	if {($d == "s")} {$f.s configure -text $search_msg
	} else {	$f.s configure -text $reverse_search_msg}
	set direction $d

	parse_bindings $f.ss $Keys(C_r) "re_search $t $f \"$reverse_search_msg\" \"$reverse_search_failed_msg\" find_reverse \[$f.ss get\] r"
}

proc reset_search {} {	global search_string ; set search_string ""}


# Search bindings. f is a frame widget to put messages in.
proc searchbind {f t m} {
	global Keys
	parse_bindings all \
C-g		"+ catch \{exit_search $t $f x\}" \
$Keys(C_r)	"reset_search ; \
	bidirectional_search_setup $t $f r ; make_search_incremental $t $f" \
$Keys(C_s)	"reset_search ; \
	bidirectional_search_setup $t $f s ; make_search_incremental $t $f" \
C-R "bidirectional_search_setup $t $f r ; make_search_incremental $t $f" \
C-S "bidirectional_search_setup $t $f s ; make_search_incremental $t $f" \
C-M-s		"set search_string {} ; regexp_search_setup $t $f $m"

	if {[winfo exists $m]} {parse_menu $m \
{Search 0			{"Search Forward" 0 C-s}
				{"Reverse Search" 0 C-r}
			separator
				{"Search Previous" 5 C-S}
				{"Reverse Search Previous" 2 C-R}
			separator
				{"Search Regular Expression" 15 M-C-s}}

# Set the menuoptions to C-S, C-R, and M-C-s to be disabled upon entering search
		global search_bindings
		parse_bindings all \
$search_bindings "+ menuentries_change_state $m.search.m disabled 3 4 6"
}}


set search_bindings "$Keys(C_r) $Keys(C_s) M-C-s"
searchbind $frame $text $menu
