# seditBind.tcl
#
# Support routines to define a set of consistent editing bindings for
# Text and Entry widgets
#
# Copyright (c) 1993 Xerox Corporation.
# Use and copying of this software and preparation of derivative works based
# upon this software are permitted. Any distribution of this software or
# derivative works must comply with all applicable United States export
# control laws. This software is made available AS IS, and Xerox Corporation
# makes no warranty about the software, its performance or its conformity to
# any specification.

proc Sedit_BindInit {} {
    global sedit

    set sedit(key,selpaste) <Control-y>
    set sedit(key,seldelete) <Control-w>
    set sedit(key,backspace) <Control-h>
    set sedit(key,backspace2) <Key-Delete>
    set sedit(key,backspace3) <Key-BackSpace>
    set sedit(key,openline) <Control-o>
    set sedit(key,deleol) <Control-k>
    set sedit(key,delword) <Escape>d	;# forwardly
    set sedit(key,delchar) <Control-d>

    set sedit(key,bof)	<Escape><Key-less>
    set sedit(key,eof)	<Escape><Key-greater>
    set sedit(key,linestart) <Control-a>
    set sedit(key,lineend) <Control-e>
    set sedit(key,up1line) <Control-p>
    set sedit(key,down1line) <Control-n>
    set sedit(key,backword) <Escape>b
    set sedit(key,forwword) <Escape>f
    set sedit(key,backchar) <Control-b>
    set sedit(key,forwchar) <Control-f>
    set sedit(key,up1page) <Mod1-v>
    set sedit(key,down1page) <Control-v>

    set sedit(dotfile) ~/.exmhsedit

    SeditReadPref
    Sedit_ClassBindings
}
proc SeditReadPref {} {
    global sedit
    if [file exists $sedit(dotfile)] {
	if [catch {uplevel #0 source [glob $sedit(dotfile)]} msg] {
	    Exmh_Status "Error in $file: $msg"
	    return
	} 
    }
}
proc SeditBind { class key body } {
    global sedit
    if [catch {
	bind $class $sedit(key,$key) $body
    } err] {
	if ![info exists sedit(key,$key)] {
	    puts stderr "Bind $class $key: $err"
	} else {
	    puts stderr "Bind $class $key $sedit(key,$key): $err"
	}
    }
}
proc Sedit_ClassBindings { } {
    global sedit

    # Modification bindings

    bind Text <Return> {
	%W insert insert \n; %W yview -pickplace insert
	SeditDirty %W
    }
    bind Text <Escape> { } ;# no-op
    bind Entry <Escape> { } ;# no-op

    SeditBind Text selpaste {
	if [catch {
	    %W insert insert [selection get]
	    %W yview -pickplace insert
	    SeditDirty %W
	}] {
	    if [catch {%W insert insert [cutbuffer get]}] {
		catch {%W insert insert $sedit(killbuf)}
	    }
	}
    }
    SeditBind Entry selpaste {
	if [catch {
	    %W insert insert [selection get]
	}] {
	    if [catch {%W insert insert [cutbuffer get]}] {
		catch {%W insert insert $sedit(killbuf)}
	    }
	}
    }

    SeditBind Text seldelete {
	catch {
	    set sedit(killbuf) [%W get sel.first sel.last]
	    %W delete sel.first sel.last
	}
	SeditDirty %W
    }
    SeditBind Entry seldelete {
	catch {%W delete sel.first sel.last}
    }

    foreach bs {backspace backspace2 backspace3} {
	SeditBind Text $bs {
	    tk_textBackspace %W; %W yview -pickplace insert
	    SeditDirty %W
	}
	SeditBind Entry $bs {
	    tk_entryBackspace %W
	}
    }

    SeditBind Text openline {
	set _foo [%W index insert]
	%W insert insert \n; %W yview -pickplace insert
	%W mark set insert $_foo
	unset _foo
	SeditDirty %W
    }
    SeditBind Entry openline { info library }

    SeditBind Text deleol {
	if {[%W index insert] == [%W index "insert lineend"]} {
	    %W delete insert "insert + 1 chars"
	} else {
	    set sedit(killbuf) [%W get insert "insert lineend"]
	    %W delete insert "insert lineend"
	}
	SeditDirty %W
    }
    SeditBind Entry deleol {
	%W delete insert end
    }

    SeditBind Text delword {
	set sedit(killbuf) [%W get insert "insert wordend"]
	%W delete insert "insert wordend"
	SeditDirty %W
    }
    SeditBind Entry delword { info library }

    SeditBind Text delchar {
	%W delete insert
	SeditDirty %W
    }
    SeditBind Entry delchar {
	%W delete insert
    }

    # Motion bindings
    SeditBind Text bof {
	%W mark set insert 1.0
	%W yview -pickplace insert
    }
    SeditBind Entry bof { info library }

    SeditBind Text eof {
	%W mark set insert end
	%W yview -pickplace insert
    }
    SeditBind Entry eof { info library }

    SeditBind Text linestart {
	%W mark set insert "insert linestart"
    }
    SeditBind Entry linestart {
	%W icursor 0
    }

    SeditBind Text lineend {
	%W mark set insert "insert lineend"
    }
    SeditBind Entry lineend {
	%W icursor end
    }

    set sedit(lastpos,Text) {}
    SeditBind Text up1line {
	if ![info exist sedit(lastpos,%W)] {
	    set sedit(lastpos,%W) {}
	}
	if {$sedit(lastpos,%W) != [%W index insert]} {
	    set sedit(lastpos,%W) [%W index insert]
	    set parts [split $sedit(lastpos,%W) .]
	    set sedit(col,%W) [lindex $parts 1]
	    set sedit(line,%W) [lindex $parts 0]
	}
	if {$sedit(line,%W) > 1} {
	    incr sedit(line,%W) -1
	    set parts [split [%W index $sedit(line,%W).end] .]
	    set lastcol [lindex $parts 1]
	    set column $sedit(col,%W)
	    if {$column > $lastcol} {
		set column end
	    }
	    %W mark set insert $sedit(line,%W).$column
	    %W yview -pickplace insert
	    set sedit(lastpos,%W) [%W index insert]
	}
    }
    SeditBind Entry up1line { info library }

    SeditBind Text down1line {
	if ![info exist sedit(lastpos,%W)] {
	    set sedit(lastpos,%W) {}
	}
	if {$sedit(lastpos,%W) != [%W index insert]} {
	    set sedit(lastpos,%W) [%W index insert]
	    set parts [split $sedit(lastpos,%W) .]
	    set sedit(col,%W) [lindex $parts 1]
	    set sedit(line,%W) [lindex $parts 0]
	}
	set parts [split [%W index end] .]
	set lastline [lindex $parts 0]
	if {$sedit(line,%W) < $lastline} {
	    incr sedit(line,%W)
	}
	set parts [split [%W index $sedit(line,%W).end] .]
	set lastcol [lindex $parts 1]
	set column $sedit(col,%W)
	if {$sedit(line,%W) <= $lastline} {
	    if {$column > $lastcol} {
		set column end
	    }
	    %W mark set insert $sedit(line,%W).$column
	    %W yview -pickplace insert
	} else {
	    set column end
	    %W mark set insert $line.end
	}
	set sedit(lastpos,%W) [%W index insert]
    }
    SeditBind Entry down1line { info library }

    SeditBind Text backword {
	while {[%W index insert] == [%W index "insert wordstart"]} {
	    if {([%W index insert]-1) <= 0} {
		break
	    }
	    %W mark set insert "insert -1 chars"
	}
	%W mark set insert "insert wordstart"
	%W yview -pickplace insert
    }
    SeditBind Entry backword {
	set string [%W get]
	set curs [expr [%W index insert]-1]
	if {$curs < 0} return
	for {set x $curs} {$x > 0} {incr x -1} {
	    if {([string first [string index $string $x] " \t"] < 0)
		    && ([string first [string index $string [expr $x-1]] " \t"]
		    >= 0)} {
		break
	    }
	}
	%W icursor $x
    }

    SeditBind Text forwword {
	while {[%W index insert] == [%W index "insert wordend -1 chars"]} {
	    if {[%W index insert] >= [%W index end]} {
		break
	    }
	    %W mark set insert "insert +1 chars"
	}
	%W mark set insert "insert wordend"
	%W yview -pickplace insert
    }
    SeditBind Entry forwword {
	set string [%W get]
	set curs [expr [%W index insert]+1]
	set len [string length $string]
	if {$curs < 0} return
	for {set x $curs} {$x < $len} {incr x} {
	    if {([string first [string index $string $x] " \t"] < 0)
		    && ([string first [string index $string [expr $x+1]] " \t"]
		    >= 0)} {
		break
	    }
	}
	%W icursor $x	
    }

    SeditBind Text backchar {
	%W mark set insert "insert - 1 chars"
	%W yview -pickplace insert
    }
    SeditBind Entry backchar {
	set x [%W index insert]
	if {$x > 0} {
	    incr x -1
	    %W icursor $x
	}
    }

    SeditBind Text forwchar {
	%W mark set insert "insert + 1 chars"
	%W yview -pickplace insert
    }
    SeditBind Entry forwchar {
	set x [%W index insert]
	incr x
	%W icursor $x
    }

    SeditBind Text up1page {
	Widget_TextPageUp %W
    }
    SeditBind Entry up1page { } ;# no-op

    SeditBind Text down1page {
	Widget_TextPageDown %W
    }
    SeditBind Entry down1page { } ;# no-op

    bind Text <Any-Key> {
	if {"%A" != ""} {
	    %W insert insert %A
	    %W yview -pickplace insert
	    SeditDirty %W
	}
    }
}
proc SeditMarkClean { t } {
    global sedit
    set sedit($t,dirty) 0
}
proc SeditDirty { t } {
    global sedit
    set sedit($t,dirty) 1
}
proc SeditIsDirty { t } {
    global sedit
    return $sedit($t,dirty)
}

proc Sedit_Pref {} {
    global sedit
    if [catch {	Widget_Toplevel .seditpref "Simple Edit Preferences" Pref}] {
	raise .seditpref
	return
    }
    Widget_Frame .seditpref b Menubar {top fill}
    Widget_AddBut .seditpref.b quit "Dismiss" {Exmh_Focus ; destroy .seditpref}
    Widget_AddBut .seditpref.b save "Save" {SeditPrefSave}
    Widget_Label .seditpref.b label {left fill} \
	-text "Text and Entry class bindings"
    set f [Widget_Frame .seditpref p Dialog]
    $f configure -bd 10
    set lr [Widget_SplitFrame $f Left Right]
    set left [lindex $lr 0]
    set right [lindex $lr 1]
    set width 0
    foreach item [array names sedit] {
	if [regexp key $item] {
	    set name [lindex [split $item ,] 1]
	    set w [string length $name]
	    if {$w > $width} { set width $w }
	}
    }
    set size 0
    foreach item [lsort [array names sedit]] {
	if [regexp key $item] {
	    incr size
	    set name [lindex [split $item ,] 1]
	    set keystroke $sedit($item)
	    set frame [lindex $lr [expr {$size % 2}]]
	    SeditPrefItem $frame $width $name $keystroke
	}
    }
}
proc SeditPrefItem { frame width name keystroke } {
    global sedit
    Widget_Frame $frame $name Pref
    Widget_Label $frame.$name label {left} -text $name -width $width
    Widget_Entry $frame.$name entry {right expand fill} -background white
    set sedit(entry,$name) $frame.$name.entry
    $frame.$name.entry insert 0 $keystroke
}
proc SeditPrefSave { } {
    global sedit
    # Save it
    set out [open $sedit(dotfile) w]
    foreach item [array names sedit] {
	if [regexp key $item match] {
	    set name [lindex [split $item ,] 1]
	    set entry $sedit(entry,$name)
	    set keystroke [$entry get]
	    puts $out "set sedit($match,$name) $keystroke"
	}
    }
    close $out
    Exmh_Focus
    destroy .seditpref
    # Apply it to current session
    SeditReadPref
    Sedit_ClassBindings
}


