/*
@(#)  FILE: ncl_net.c  RELEASE: 1.2  DATE: 5/27/94, 13:12:22
*/
/*******************************************************************************

File:

    ncl_net.c

    "ncl_net" Command Functions.


Author:    Alex Measday, ISI


Purpose:

    This file contains the functions that process the NICL-specific, Tcl
    command "ncl_net".


Public Procedures:

    NCL_NET - interprets the "ncl_net" Tcl extension.

Private Procedures:

    NCL_NET_ANSWER - answers a network connection request from a client.
    NCL_NET_CALL - attempts to establish a network connection with a server.
    NCL_NET_INPUT - reads and processes input from a network connection.
    NCL_NET_OBJECT - interprets a Tcl net object command.

*******************************************************************************/


#include  <errno.h>			/* System error definitions. */
#include  <stdio.h>			/* Standard I/O definitions. */
#include  <stdlib.h>			/* Standard C library definitions. */
#include  <string.h>			/* C Library string functions. */
#ifdef VMS
#    include  <unixio.h>		/* VMS-emulation of UNIX I/O. */
#else
#    include  <unistd.h>		/* UNIX-specific definitions. */
#endif
#include  "libncl.h"			/* NICL library definitions. */
#include  "libutilgen.h"		/* LIBUTILGEN definitions. */
#ifdef vms
#    include  "libutilvms.h"		/* LIBUTILVMS definitions. */
#endif
#include  "net_util.h"			/* Network utility definitions. */
#include  "nix_util.h"			/* Network I/O Handler definitions. */
#include  "opt_util.h"			/* Option scanning definitions. */
#include  "xnet_util.h"			/* XNET definitions. */


/*******************************************************************************
    Net Objects - contain information about a network connection.
*******************************************************************************/

typedef  struct  net_object {
    char  *name;			/* "<server>[@<host>]" */
    int  listening_socket ;		/* To listen for connection requests. */
    int  data_socket ;			/* To exchange data. */
    XnetStream  stream ;		/* XNET handle. */
    int  more_input ;			/* True if more input can be read from the connection. */
#ifdef vms
    int  event_flag ;			/* Event flag for I/O notification. */
#endif
    Tcl_Interp  *interpreter ;		/* Tcl interpreter. */
    char  *connect_command ;		/* Tcl command to execute when connection is established. */
    char  *error_command ;		/* Tcl command to execute upon I/O error. */
    char  *input_command ;		/* Tcl command to execute when input is received. */
    NxAppContext  *appl_context ;	/* NIX application context. */
}  net_object ;


/*******************************************************************************
    Private Functions
*******************************************************************************/

static  int  ncl_net_answer (
#    if defined(vms)			/* Timer callback function (VMS). */
        NxAppContext  appl_context,
        NxIntervalId  timer_ID,
        void  *client_data
#    elif __STDC__			/* Input callback function (UNIX). */
        NxAppContext  appl_context,
        NxInputId  source_ID,
        int  source,
        void  *client_data
#    endif
    ) ;

static  int  ncl_net_call (
#    if __STDC__ || defined(vaxc)
        NxAppContext  appl_context,
        NxIntervalId  timer_ID,
        void  *client_data
#    endif
    ) ;

static  int  ncl_net_input (
#    if __STDC__ || defined(vaxc)
        NxAppContext  appl_context,
        NxInputId  source_ID,
        int  source,
        void  *client_data
#    endif
    ) ;

static  int  ncl_net_object (
#    if __STDC__ || defined(vaxc)
        ClientData  clientData,
        Tcl_Interp  *interp,
        int  argc,
        char  *argv[]
#    endif
    ) ;

/*******************************************************************************

Procedure:

    ncl_net ()

    Create a Net Object.


Purpose:

    Function NCL_NET interprets the "ncl_net" Tcl command.  This
    command establishes an XDR network connection with a specified
    client or server.  The "ncl_net" command is entered as follows:

        ncl_net call <name> -server <server>[@<host>] [-nowait] \
            [-connect <command>] [-error <command>] [-input <command>]

        ncl_net answer <name> -server <server> [-nowait] \
            [-connect <command>] [-error <command>] [-input <command>]

    where:

        "call"
            attempts to establish a network connection with the specified
            server (possibly on a remote host).
        "answer"
            listens for and accepts a network connection request from a client.
        "<name>"
            is a logical name to associated with the network connection.  This
            name is used as the command keyword for a dynamically-assigned Tcl
            command that must be called to read from or write to the connection.
        "-server <server>[@<host>]"
            specifies the server name.  This name should map to a network
            port number in the local host's "/etc/services" file.
        "-[no]wait"
            indicates whether or not the "ncl_net" command should wait until
            the network connection is established before returning.  "-nowait"
            puts the connection attempt(s) into "background" mode: ANSWERs are
            answered when a connection request is received; CALLs are re-tried
            periodically.
        "-connect <command>"
            is a Tcl command to be executed when the connection is established.
            This may be immediately (for CALLs) or at a later time (for
            ANSWERs).
        "-error <command>"
            is a Tcl command to be executed when an I/O error is detected.
            Such an error generally indicates a broken network connection.
        "-input <command>"
            is a Tcl command to be executed when input is received on the
            connection.  The command is executed for each line of input read;
            the text of the input line is appended to the command as a string.


    Invocation:

        status = ncl_net (&appl_context, interpreter, arg_count, arg_list) ;

    where:

        <appl_context>	- I/O
            is the address of a variable containing (an opaque pointer to) the
            NIX application context.
        <interpreter>	- I
            is the handle for the Tcl interpreter.
        <arg_count>	- I
            is the number of entries in the argument list.
        <arg_list>	- I
            is the list of arguments (stored in an array) for the command.
            The zero-th element in the list is always the command keyword.
        <status>	- O
            returns the status of interpreting the command, TCL_OK if no
            errors occurred and TCL_ERROR otherwise.

*******************************************************************************/


int  ncl_net (

#    if __STDC__ || defined(vaxc)
        ClientData  clientData,
        Tcl_Interp  *interp,
        int  argc,
        char  *argv[])
#    else
        clientData, interp, argc, argv)

        ClientData  clientData ;
        Tcl_Interp  *interp ;
        int  argc ;
        char  *argv[] ;
#    endif

{    /* Local variables. */
    char  *action, *argument, *connect_command, *error_command ;
    char  *input_command, *name, *s, *server_name ;
    int  errflg, option, wait_for_connection ;
    net_object  *nob ;
    static  char  *option_list[] = {
        "{connect:}", "{error:}", "{input:}",
        "{server:}", "{wait}", "{nowait}", NULL
    } ;
    static  void  *context = NULL ;




/*******************************************************************************
    Scan the command's arguments.
*******************************************************************************/

    action = NULL ;
    connect_command = NULL ;  error_command = NULL ;  input_command = NULL ;
    name = NULL ;  server_name = NULL ;
    wait_for_connection = 1 ;

    if (context == NULL)  opt_init (argc, argv, 1, option_list, &context) ;
    opt_reset (context, argc, argv) ;
    errflg = 0 ;
    while (option = opt_get (context, &argument)) {
        switch (option) {
        case 1:			/* "-connect <command>" */
            connect_command = argument ;  break ;
        case 2:			/* "-error <command>" */
            error_command = argument ;  break ;
        case 3:			/* "-input <command>" */
            input_command = argument ;  break ;
        case 4:			/* "-server <server>" */
            server_name = argument ;  break ;
        case 5:			/* "-wait" */
            wait_for_connection = 1 ;  break ;
        case 6:			/* "-nowait" */
            wait_for_connection = 0 ;  break ;
        case NONOPT:
            if (action == NULL) {
                if ((strcmp (argument, "call") == 0) ||
                    (strcmp (argument, "answer") == 0))
                    action = argument ;
                else
                    errflg++ ;
            } else {
                name = argument ;
            }
            break ;
        case OPTERR:
            errflg++ ;  break ;
        default:
            break ;
        }
    }

    if (errflg || (action == NULL) || (name == NULL)) {
        Tcl_SetResult (interp, "missing or invalid argument", TCL_STATIC) ;
        return (TCL_ERROR) ;
    }

/*******************************************************************************
    Create a net object structure for the network connection.
*******************************************************************************/

    nob = (net_object *) malloc (sizeof (net_object)) ;
    if (nob == NULL) {
        vperror ("(ncl_net) Error allocating net object for %s:%s.\nmalloc: ",
                 (server_name == NULL) ? name : server_name) ;
        Tcl_SetResult (interp, "error allocating net object", TCL_STATIC) ;
        return (TCL_ERROR) ;
    }

    nob->name = str_dupl ((server_name == NULL) ? name : server_name, -1) ;
    nob->listening_socket = -1 ;
    nob->data_socket = -1 ;
    nob->stream = NULL ;
    nob->more_input = 0 ;
#ifdef vms
    nob->event_flag = ncl_get_ef () ;
#endif
    nob->interpreter = interp ;
    nob->connect_command = (connect_command == NULL) ? NULL :
                           str_dupl (connect_command, -1) ;
    nob->error_command = (error_command == NULL) ? NULL :
                         str_dupl (error_command, -1) ;
    nob->input_command = (input_command == NULL) ? NULL :
                         str_dupl (input_command, -1) ;
    nob->appl_context = (NxAppContext *) clientData ;


/*******************************************************************************
    If this application is the client calling a server, then do so.
*******************************************************************************/

    if (strcmp (action, "call") == 0) {

/* If the "-nowait" option was NOT specified, call NCL_NET_CALL() directly to
   establish the network connection; if NCL_NET_CALL() is unsuccessful, no more
   attempts are made.  If the "-nowait" option WAS specified, set a timer to
   initiate the periodic invocation of NCL_NET_CALL(), thus putting the attempts
   at establishing a connection in the "background". */

        if (wait_for_connection) {	/* Try once? */
            if (ncl_net_call (NULL, NULL, (void *) nob)) {
                vperror ("(ncl_net) Error connecting to %s.\nncl_net_call: ",
                         nob->name) ;
                Tcl_SetResult (interp, "error calling server", TCL_STATIC) ;
                return (TCL_ERROR) ;
            }
        } else {			/* Periodically attempt in background. */
            NxAddTimeOut (nob->appl_context, 0.0, ncl_net_call, (void *) nob) ;
        }

    }

/*******************************************************************************
    If this application is the server, then listen for and accept a connection
    request from a client.
*******************************************************************************/

    else {

/* Create a socket used to listen for connection requests from clients. */

        if (net_answer (nob->name, "-listen", -1.0,
                        &nob->listening_socket, NULL)) {
            vperror ("(ncl_net) Error creating listening socket for %s.\nnet_answer: ",
                     nob->name) ;
            Tcl_SetResult (interp, "error creating listening socket", TCL_STATIC) ;
            return (TCL_ERROR) ;
        }

/* If the "ncl_net" command must wait for the connection to be established,
   then fake a "callback" invocation of NCL_NET_ANSWER() and let it wait for
   the connection request.  Note that, under VMS, NCL_NET_ANSWER() is a timer
   callback instead of an I/O callback. */

        if (wait_for_connection) {
#ifdef vms
            if (ncl_net_answer (NULL, NULL, (void *) nob)) {
#else
            if (ncl_net_answer (NULL, NULL, nob->listening_socket,
                                (void *) nob)) {
#endif
                vperror ("(ncl_net) Error answering %s client.\nncl_net_answer: ",
                         nob->name) ;
                Tcl_SetResult (interp, "error answering client", TCL_STATIC) ;
                return (TCL_ERROR) ;
            }
        }

/* If "-nowait" was specified, register the listening socket with the NIX
   I/O handler.  When a connection request is received, the NIX handler
   will automatically invoke the NCL_NET_ANSWER() function to accept the
   connection request.  Note that, under VMS, NXADDTIMEOUT() is used.
   QIO "peek" requests posted on a listening socket complete immediately,
   whether or not a connection request has actually really been received.
   Consequently, NXADDINPUT() could not be used (although it still works
   for data sockets).  Instead, NCL_NET_ANSWER() periodically polls the
   listening socket for an incoming connection request. */

        else {

#ifdef vms
            NxAddTimeOut (nob->appl_context,
                          0.0,			/* Check immediately. */
                          ncl_net_answer,	/* Routine to answer connection request. */
                          (void *) nob) ;	/* Arbitrary data. */
#else
            NxAddInput (nob->appl_context,
                        nob->listening_socket,	/* File descriptor for listening socket. */
                        NxInputReadMask,	/* Monitor input. */
                        ncl_net_answer,		/* Routine to answer connection request. */
                        (void *) nob) ;		/* Arbitrary data. */
#endif

            if (libncl_debug)
                printf ("(ncl_net) Listening on %s, connection %d.\n",
                        nob->name, nob->listening_socket) ;

        }

    }


/* Create a new Tcl command for the network connection.  The dynamically-defined
   Tcl command can be used to perform I/O on the connection. */

    s = strchr (name, '@') ;
    if (s == NULL)				/* "<server>" */
        name = str_dupl (name, -1) ;
    else					/* "<server>@<host>" */
        name = str_dupl (name, (s - name)) ;

    Tcl_CreateCommand (interp, name, (Tcl_CmdProc *) ncl_net_object,
                       (ClientData) nob, NULL) ;


/* Return the object's name as the Tcl command's result string. */

    Tcl_SetResult (interp, name, TCL_DYNAMIC) ;

    return (TCL_OK) ;

}

/*******************************************************************************

Procedure:

    ncl_net_answer ()

    Answer a Network Connection Request.


Purpose:

    Function NCL_NET_ANSWER answers a network connection request from
    a client.  When an "answer" connection is created by NCL_NET(), the
    server's listening socket is registered with the NIX I/O handler as
    an input source.  Then, when a connection request is received from a
    potential client, the NIX handler automatically invokes NCL_NET_ANSWER()
    to field the request.  NCL_NET_ANSWER() accepts the connection request
    and registers the newly-created data socket with the NIX I/O handler
    as an input source.

    NOTE:  Under VMS, registering the listening socket with the NIX I/O
    handler does not work as desired.  The "lookahead" QIO posted on the
    listening socket completes immediately, rather than waiting until a
    connection request is actually received.  Consequently, NCL_NET_ANSWER()
    is invoked immediately and its call to XNET_ANSWER() blocks until a
    connection request is received, thus defeating the whole purpose of
    using NXADDINPUT()!  The work-around implemented below is to set a
    timer with NXADDTIMEOUT() and periodically poll the listening socket.


    Invocation (UNIX):

        status = ncl_net_answer (appl_context, source_ID, source, client_data) ;

    Invocation (VMS):

        status = ncl_net_answer (appl_context, timer_ID, client_data) ;

    where:

       <appl_context>	- I
            is the application context used when the network connection's
            listening socket (UNIX) or timeout timer (VMS) was registered
            with the NIX I/O handler.
        <source_ID>	- I	(UNIX)
            is the identifier returned by NXADDINPUT() when the network
            connection's listening socket was registered with the NIX I/O
            handler.  A NULL source ID indicates that NCL_NET_ANSWER() was
            called directly by NCL_NET() rather than indirectly via the NIX
            I/O handler.
        <source>	- I	(UNIX)
            is the socket on which the NIX I/O handler is listening for
            connection requests.
        <timer_ID>	- I	(VMS)
            is the identifier returned by NXADDTIMEOUT() when the ANSWER
            timeout was registered with the NIX I/O handler.  A NULL source
            ID indicates that NCL_NET_ANSWER() was called directly by NCL_NET()
            rather than indirectly via the NIX I/O handler.
        <client_data>	- I
            is the address of the network object structure created for the
            network connection by NCL_NET().
        <status>	- O
            returns the status of answering a connection request, zero if
            there were no errors and ERRNO otherwise.  The status value
            is ignored by the NIX I/O handler, but it may be of use if the
            application calls NCL_NET_ANSWER() directly.

*******************************************************************************/


static  int  ncl_net_answer (

#    if defined(vms)			/* Timer callback function (VMS). */
        NxAppContext  appl_context,
        NxIntervalId  timer_ID,
        void  *client_data)
#    elif __STDC__			/* Input callback function (UNIX). */
        NxAppContext  appl_context,
        NxInputId  source_ID,
        int  source,
        void  *client_data)
#    else
        appl_context, source_ID, source, client_data)

        NxAppContext  appl_context ;
        NxInputId  source_ID ;
        int  source ;
        void  *client_data ;
#    endif

{    /* Local variables. */
    int  status ;
    net_object  *nob ;




/* When running under VMS, poll the listening socket to see if a connection
   request has been received.  I originally tried posting a QIO on the
   listening socket, but, under VMS, QIOs on a listening socket return
   immediately, whether or not a connection request has actually been
   received.  The only alternative seems to be this periodic polling of
   the socket. */

    nob = (net_object *) client_data ;
#ifdef vms
	/* Notice that we're looking for an "error" here.  When a connection
	   request is received on a listening socket, NET_POLL() thinks the
	   connection is broken because SELECT(2) says there is data while
	   IOCTL(2) says there isn't.  See NET_POLL() for more information. */
    if (!net_poll (nob->listening_socket, NULL)) {
        NxAddTimeOut (nob->appl_context, 1.0, ncl_net_answer, (void *) nob) ;
        return (0) ;
    }
#endif


/* Accept the connection request from a new client. */

    if (xnet_answer (nob->name, NULL, -1.0,
                     &nob->listening_socket, &nob->stream)) {
        status = errno ;
        vperror ("(ncl_net_answer) Error answering connection request for %s.\nxnet_answer: ",
                 nob->name) ;
        if (source_ID != NULL)  NxRemoveInput (nob->appl_context, source_ID) ;
        if (nob->error_command != NULL)
            Tcl_Eval (nob->interpreter, nob->error_command) ;
        errno = status ;
        return (errno) ;
    }
    nob->data_socket = xnet_socket (nob->stream) ;

    if (libncl_debug)  printf ("(ncl_net_answer) Answered %s, connection %d.\n",
                               nob->name, nob->data_socket) ;


/* Close and remove the listening socket from the group of input sources
   monitored by the NIX I/O handler. */

    close (nob->listening_socket) ;
    nob->listening_socket = -1 ;
#ifndef vms
    if (source_ID != NULL)  NxRemoveInput (nob->appl_context, source_ID) ;
#endif


/* Execute the Tcl command that is defined for connection completion. */

    if (libncl_debug)  printf ("(ncl_net_answer) \"Answer\" command: %s\n",
                               (nob->connect_command == NULL) ? "<none>" :
                               nob->connect_command) ;
    if (nob->connect_command != NULL)
        Tcl_Eval (nob->interpreter, nob->connect_command) ;


/* Register the data socket with the NIX I/O handler.  When incoming data is
   detected, the NIX handler will automatically invoke the NCL_NET_INPUT()
   function to read and process the input. */

#ifdef vms
					/* Post "read" on data socket. */
    vnet_post (nob->data_socket, nob->event_flag) ;
    NxAddInput (nob->appl_context,
                nob->event_flag,	/* Set by incoming data. */
#else
    NxAddInput (nob->appl_context,
                nob->data_socket,	/* File descriptor for data socket. */
#endif
                NxInputReadMask,	/* Monitor input. */
                ncl_net_input,		/* Routine to handle input. */
                (void *) nob) ;		/* Arbitrary data. */


    return (0) ;

}

/*******************************************************************************

Procedure:

    ncl_net_call ()

    Establish a Connection with a Network Server.


Purpose:

    Function NCL_NET_CALL attempts to establish a connection with a network
    server.  NCL_NET_CALL() operates in either of two modes:

        One-Shot - If NCL_NET_CALL() was called directly by NCL_NET()
            (indicated by a NULL timer ID), NCL_NET_CALL() tries just
            once to connect to the server.

        Periodic - If NCL_NET_CALL() was invoked as a result of the timer
            firing (indicated by a non-NULL timer ID), NCL_NET_CALL() tries
            to connect to the server and, if unsuccessful, reschedules a
            timeout timer for the next try.

    Once the network connection is established, the new data socket is
    registered with the NIX I/O handler as an input source.


    Invocation:

        status = ncl_net_call (appl_context, timer_ID, client_data) ;

    where:

       <appl_context>	- I
            is the application context used when the CALL timeout was
            registered with the NIX I/O handler.
        <timer_ID>	- I
            is the identifier returned by NXADDTIMEOUT() when the CALL timeout
            was registered with the NIX I/O handler.  If the timer ID is NULL
            (i.e., when NCL_NET_CALL() is invoked directly by NCL_NET()),
            NCL_NET_CALL() makes a single attempt to connect to the network
            server.  If the timer ID is NOT NULL, NCL_NET_CALL() tries to
            establish the connection and, if unsuccessful, re-registers a
            timeout timer with the NIX I/O handler.  When the timeout expires,
            NCL_NET_CALL() will attempt the connection again (and again, etc.).
        <client_data>	- I
            is the address of the network object structure created for the
            network connection by NCL_NET().
        <status>	- O
            returns the status of establishing the network connection, zero
            if there were no errors and ERRNO otherwise.  The status value
            is ignored by the NIX I/O handler, but it may be of use if the
            application calls NCL_NET_CALL() directly.

*******************************************************************************/


static  int  ncl_net_call (

#    if __STDC__ || defined(vaxc)
        NxAppContext  appl_context,
        NxIntervalId  timer_ID,
        void  *client_data)
#    else
        appl_context, timer_ID, client_data)

        NxAppContext  appl_context ;
        NxIntervalId  timer_ID ;
        void  *client_data ;
#    endif

{    /* Local variables. */
    int  status ;
    net_object  *nob ;




    nob = (net_object *) client_data ;


/* Establish a network connection with the server. */

    if (xnet_call (nob->name, NULL, 0.0, &nob->stream)) {
        status = errno ;
        vperror ("(ncl_net_call) Error calling %s.\nxnet_call: ",
                 nob->name) ;
        if (timer_ID != NULL)		/* Schedule the next attempt. */
            NxAddTimeOut (nob->appl_context, 1.0, ncl_net_call, (void *) nob) ;
        errno = status ;
        return (errno) ;
    }

    nob->data_socket = xnet_socket (nob->stream) ;

    if (libncl_debug)  printf ("(ncl_net_call) Called %s, connection %d.\n",
                               nob->name, nob->data_socket) ;


/* Execute the Tcl command that is defined for connection completion. */

    if (libncl_debug)  printf ("(ncl_net_call) \"Call\" command: %s\n",
                               (nob->connect_command == NULL) ? "<none>" :
                               nob->connect_command) ;
    if (nob->connect_command != NULL)
        Tcl_Eval (nob->interpreter, nob->connect_command) ;


/* Register the data socket with the NIX I/O handler.  When incoming data is
   detected, the NIX handler will automatically invoke the NCL_NET_INPUT()
   function to read and process the input. */

#ifdef vms
					/* Post "read" on data socket. */
    vnet_post (nob->data_socket, nob->event_flag) ;
    NxAddInput (nob->appl_context,
                nob->event_flag,	/* Set by incoming data. */
#else
    NxAddInput (nob->appl_context,
                nob->data_socket,	/* File descriptor for data socket. */
#endif
                NxInputReadMask,	/* Monitor input. */
                ncl_net_input,		/* Routine to handle input. */
                (void *) nob) ;		/* Arbitrary data. */


    return (0) ;

}

/*******************************************************************************

Procedure:

    ncl_net_input ()

    Read and Process Data from a Network Connection.


Purpose:

    Function NCL_NET_INPUT reads and processes lines of input from a network
    connection.  When a network connection is established by NCL_NET(), the
    connection's data socket is registered with the NIX I/O handler as an
    input source.  Thereafter, whenever input is detected on the connection,
    the NIX handler automatically invokes NCL_NET_INPUT() to read and process
    the input.  A single invocation of NCL_NET_INPUT() will read/process one
    or more lines of input, however much is immediately available to be read.
    For each line of input, NCL_NET_INPUT() executes the Tcl command defined
    for this connection's input.


    Invocation:

        status = ncl_net_input (appl_context, source_ID, source, client_data) ;

    where:

        <appl_context>	- I
            is the application context used when the network connection
            was registered with the NIX I/O handler.
        <source_ID>	- I
            is the identifier returned by NXADDINPUT() when the network
            connection was registered with the NIX I/O handler.
        <source>	- I
            is the socket (UNIX) or event flag (VMS) used by the NIX I/O
            handler to listen for input on the network connection.
        <client_data>	- I
            is the address of the network object structure created for the
            network connection by NCL_NET().
        <status>	- O
            returns the status of reading/processing the input, zero if
            there were no errors and ERRNO otherwise.  The status value
            is ignored by the NIX I/O handler, but it may be of use if the
            application calls NCL_NET_INPUT() directly.

*******************************************************************************/


static  int  ncl_net_input (

#    if __STDC__ || defined(vaxc)
        NxAppContext  appl_context,
        NxInputId  source_ID,
        int  source,
        void  *client_data)
#    else
        appl_context, source_ID, source, client_data)

        NxAppContext  appl_context ;
        NxInputId  source_ID ;
        int  source ;
        void  *client_data ;
#    endif

{    /* Local variables. */
    char  *string ;
    net_object  *nob ;




    nob = (net_object *) client_data ;

/* While more input is available, read the next line of input and execute
   the Tcl command defined for input on this connection.  The input text,
   enclosed in braces so it will be interpreted as a single string, is
   appended to the Tcl command before the command is executed. */

    do {
					/* Read the next line of input. */
        if (xnet_read (nob->stream, -1.0, &string, &nob->more_input)) {
            vperror ("(ncl_net_input) Error reading from connection %d.\nxnet_read: ",
                     nob->data_socket) ;
            break ;
        }
        if (libncl_debug)  printf ("(ncl_net_input) Read from connection %d: \"%s\"\n",
                                   nob->data_socket, string) ;

        if (nob->input_command != NULL) {
					/* Append the input to the Tcl
					   command and execute it. */
            if (libncl_debug)
                printf ("(ncl_net_input) Connection %d input command: %s {%s}\n",
                        nob->data_socket, nob->input_command, string) ;
            Tcl_VarEval (nob->interpreter, nob->input_command,
                         " {", string, "}", NULL) ;
        }

    } while (nob->more_input) ;


/* Check to see if the network connection has been broken.  If so, close the
   connection, remove it from the group of input sources monitored by the NIX
   I/O handler, and execute the Tcl error command defined for the connection. */

    if (net_poll (nob->data_socket, NULL)) {
        errno = EPIPE ;
        vperror ("(ncl_net_input) Broken connection to %s.\nnet_poll: ",
                 nob->name) ;
        NxRemoveInput (nob->appl_context, source_ID) ;
#ifdef vms
        ncl_free_ef (nob->event_flag) ;
#endif
        xnet_close (nob->stream) ;  nob->stream = NULL ;
        Tcl_Eval (nob->interpreter, nob->error_command) ;
        errno = EPIPE ;
        return (errno) ;
    }

#ifdef vms
    else {	/* For the NIX I/O handler's benefit, post a "read" on the socket. */
        vnet_post (nob->data_socket, nob->event_flag) ;
    }
#endif


    return (0) ;

}

/*******************************************************************************

Procedure:

    ncl_net_object ()

    Responds to Commands for a Net Object.


Purpose:

    Function NCL_NET_OBJECT interprets the Tcl command defined for a network
    connection.  This command, whose keyword is the logical name associated
    with the network connection, is registered with the Tcl interpreter when
    the network connection is established (see the "ncl_net" command).
    NCL_NET_OBJECT is used to perform I/O on the network connection and is
    entered as follows:

        <name> connected
        <name> poll
        <name> read
        <name> write <string>

    where:

        "<name>"
            is the logical name assigned to the network connection when the
            connection was established by a "ncl_net call" or "ncl_net answer"
            command.
        "connected"
            returns 1 if the object's network connection has been established
            and zero otherwise.
        "poll"
            returns 1 if the object's network connection has data waiting to
            be read and zero otherwise.
        "read"
            reads the next line of input from the network connection.  The
            text of the line is returned as this command's result string.
        "write <string>"
            writes the specified string to the network connection as an
            XDR string.


    Invocation:

        status = ncl_net_object (netobj, interpreter, arg_count, arg_list) ;

    where:

        <netobj>	- I
            is the address of the network object structure created for the
            network connection by NCL_NET().
        <interpreter>	- I
            is the handle for the Tcl interpreter.
        <arg_count>	- I
            is the number of entries in the argument list.
        <arg_list>	- I
            is the list of arguments (stored in an array) for the command.
            The zero-th element in the list is always the command keyword.
        <status>	- O
            returns the status of interpreting the command, TCL_OK if no
            errors occurred and TCL_ERROR otherwise.

*******************************************************************************/


static  int  ncl_net_object (

#    if __STDC__ || defined(vaxc)
        ClientData  clientData,
        Tcl_Interp  *interp,
        int  argc,
        char  *argv[])
#    else
        clientData, interp, argc, argv)

        ClientData  clientData ;
        Tcl_Interp  *interp ;
        int  argc ;
        char  *argv[] ;
#    endif

{    /* Local variables. */
    char  *action, *string, *text ;
    int  length ;
    net_object  *nob ;




/*******************************************************************************
    Scan the command's arguments.
*******************************************************************************/

    nob = (net_object *) clientData ;

    action = (argc > 1) ? argv[1] : NULL ;
    text = (argc > 2) ? argv[2] : NULL ;

    if (action == NULL) {
        Tcl_SetResult (interp, "missing argument", TCL_STATIC) ;
        return (TCL_ERROR) ;
    }


/*******************************************************************************
    If the application is checking if the network connection has been
    established, then check that a valid data socket has been assigned
    and that the connection socket hasn't been broken.
*******************************************************************************/

    if (strcmp (action, "connected") == 0) {

        if ((nob->data_socket < 0) || net_poll (nob->data_socket, NULL))
            Tcl_SetResult (interp, "0", TCL_STATIC) ;	/* Not connected. */
        else
            Tcl_SetResult (interp, "1", TCL_STATIC) ;	/* Connected. */

    }


/*******************************************************************************
    If the application is checking for outstanding input on the network
    connection, then poll the connection.
*******************************************************************************/

    else if (strcmp (action, "poll") == 0) {

        if ((nob->data_socket < 0) || net_poll (nob->data_socket, &length))
            Tcl_SetResult (interp, "0", TCL_STATIC) ;	/* Not connected. */
        else if ((length > 0) || nob->more_input)
            Tcl_SetResult (interp, "1", TCL_STATIC) ;	/* Data available. */
        else
            Tcl_SetResult (interp, "0", TCL_STATIC) ;	/* No pending data. */

    }

/*******************************************************************************
    If a line of input is to be read from the network connection, then do so
    and return the input line as the result string for this Tcl command.
*******************************************************************************/

    else if (strcmp (action, "read") == 0) {

        if (xnet_read (nob->stream, -1.0, &string, &nob->more_input)) {
            vperror ("(ncl_net_object) Error reading from connection %d.\nxnet_read: ",
                     nob->data_socket) ;
            Tcl_SetResult (interp, "error reading connection", TCL_STATIC) ;
            return (TCL_ERROR) ;
        }
        if (libncl_debug)  printf ("(ncl_net) Read from connection %d: \"%s\"\n",
                                   nob->data_socket, string) ;
        Tcl_SetResult (interp, str_dupl (string, -1), TCL_DYNAMIC) ;

    }


/*******************************************************************************
    Otherwise, write a line of text out to the network connection.
*******************************************************************************/

    else if (strcmp (action, "write") == 0) {

        if (libncl_debug)  printf ("(ncl_net) Writing to connection %d: \"%s\"\n",
                                   nob->data_socket, text) ;
        if (xnet_write (nob->stream, -1.0, "%s", text)) {
            vperror ("(ncl_net_object) Error writing to connection %d.\nxnet_write: ",
                     nob->data_socket) ;
            Tcl_SetResult (interp, "error writing connection", TCL_STATIC) ;
            return (TCL_ERROR) ;
        }

    }


/*******************************************************************************
    Invalid keyword.
*******************************************************************************/

    else {
        Tcl_SetResult (interp, "invalid argument", TCL_STATIC) ;
        return (TCL_ERROR) ;
    }


    return (TCL_OK) ;

}
