/* 
 * mxCmdUtils.c --
 *
 *	This file contains a collection of utility procedures that
 *	are used by many of the top-level command procedures for Mx.
 *
 * Copyright (C) 1986 Regents of the University of California
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 * Copyright (c) 1992 Xerox Corporation.
 * Use and copying of this software and preparation of derivative works based
 * upon this software are permitted. Any distribution of this software or
 * derivative works must comply with all applicable United States export
 * control laws. This software is made available AS IS, and Xerox Corporation
 * makes no warranty about the software, its performance or its conformity to
 * any specification.
 */

#ifndef lint
static char rcsid[] = "$Header: /import/tcl7/src/mxedit/RCS/mxCmdUtils.c,v 2.3 1994/02/01 18:16:20 welch Exp $ SPRITE (Berkeley)";
#endif not lint


#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <sys/stat.h>

#include "mxWidget.h"
#include "tk.h"

/*
 * The information below is used to create the command table.
 * These commands are all widget-instance commands, not quite top level
 * TCL commands.  They are kept in a hash table and branched to
 * from the MxWidgetInstanceCmd procedure.
 */
static Tcl_HashTable mxInstanceCmdTable;
static int mxWidgetInstanceCmdInit = 0;

typedef struct {
    char *name;			/* Command name. */
    int (*proc)();		/* Procedure to process command. */
} CmdInfo;

static CmdInfo commands[] = {

    {"caret",		Mx_CaretCmd},
    {"clean",		Mx_CleanCmd},
    {"column",		Mx_ColumnCmd},
    {"configure",	Mx_ConfigureCmd},
    {"control",		Mx_ControlCmd},
    {"delete",		Mx_DeleteCmd},
    {"extract",		Mx_ExtractCmd},
    {"gridsize",	Mx_GridsizeCmd},
    {"history",		Mx_HistoryCmd},
    {"indent",		Mx_IndentCmd},
    {"index",		Mx_IndexCmd},
    {"insert", 		Mx_InsertCmd},
    {"mark",		Mx_MarkCmd},
    {"markparen",	Mx_MarkParenCmd},
    {"message",		Mx_MessageCmd},
    {"newline",		Mx_NewlineCmd},
    {"quit",		Mx_QuitCmd},
    {"quote",		Mx_QuoteCmd},
    {"read",		Mx_ReadCmd},
    {"replace",		Mx_ReplaceCmd},
    {"reset",		Mx_ResetCmd},
    {"scan", 		Mx_ScanCmd},
    {"search",		Mx_SearchCmd},
    {"see",		Mx_SeeCmd},
    {"selection",	Mx_SelectionCmd},
    {"switch",		Mx_SwitchCmd},
    {"taginfo",		Mx_TaginfoCmd},
    {"undo", 		Mx_UndoCmd},
    {"view", 		Mx_ViewCmd},
    {"write",		Mx_WriteCmd},
    {"written",		Mx_WrittenCmd},

    {(char *) NULL, (int (*)()) NULL}
};

/*
 * Information used for parenthesis matching:
 */

char *(mxOpenParens[]) =  {"(", "{", "[", "<", NULL};
char *(mxCloseParens[]) = {")", "}", "]", ">", NULL};

/*
 * Information used for word-boundary finding:
 */

char mxWordMask[16] =  {0, 0, 0, 0, 0, 0, 0xff, 0x3,
	0xfe, 0xff, 0xff, 0x87, 0xfe, 0xff, 0xff, 0x7};

/*
 * Forward references to procedures defined later in this file:
 */

static void		MxCollapseMotion();

/*
 *----------------------------------------------------------------------
 *
 * MxCmdInit --
 *
 *	This sets up a hash table of widget instance commands.  Because
 *	all widget operations are through the MxWidgetInstanceCommand,
 *	it has to deal with a large number of sub-commands.  Those are
 *	put into a hash table here.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	Allocates and initializes the command indirection table.
 *
 *----------------------------------------------------------------------
 */
void
MxCmdInit(mxwPtr)
    MxWidget *mxwPtr;
{
    CmdInfo *cmd;

    /*
     * Create a hash table of widget commands to be invoked from
     * MxWidgetInstanceCmd.  This maps from argv[1] to C-procedures.
     */
    if (!mxWidgetInstanceCmdInit) {
	mxWidgetInstanceCmdInit = 1;
	Tcl_InitHashTable(&mxInstanceCmdTable, TCL_STRING_KEYS);
	for (cmd = commands; cmd->name != NULL ; cmd++) {
	    int newEntry;
	    Tcl_HashEntry *entryPtr;
	    entryPtr = Tcl_CreateHashEntry(&mxInstanceCmdTable,
			cmd->name, &newEntry);
	    Tcl_SetHashValue(entryPtr, cmd->proc);
	}
    }

    return;
}

/*
 *----------------------------------------------------------------------
 *
 * MxCmdFind --
 *
 *	Map from a command string to a handler procedure.
 *
 * Results:
 *	A pointer to procedure.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
int (*
MxCmdFind(string))()
    char *string;
{
    Tcl_HashEntry *entryPtr;
    int (*handler)();

    entryPtr = Tcl_FindHashEntry(&mxInstanceCmdTable, string);
    if (entryPtr == NULL) {
	return NULL;
    }
    handler = (int(*)())Tcl_GetHashValue(entryPtr);
    return handler;
}

/*
 *----------------------------------------------------------------------
 *
 * MxCheckWritten --
 *
 *	See if closing a particular window will result in file modifications
 *	being lost.  If so, give the user a chance to either abort the
 *	command in progress or write out the file.
 *
 * Results:
 *	TCL_OK is returned if either a) changingAll is zero and there's
 *	another window open on mxwPtr's file, b) mxwPtr's file isn't
 *	dirty, or c) the user doesn't mind losing the changes to
 *	mxwPtr's file.  Otherwise, TCL_ERROR is returned and
 *	mxwPtr->interp->result will point to a message indicating why
 *	it's not safe to continue.
 *
 * Side effects:
 *	If the user asks for it, then the file will get written.
 *
 *----------------------------------------------------------------------
 */

int
MxCheckWritten(interp, mxwPtr, changingAll)
    Tcl_Interp *interp;
    register MxWidget *mxwPtr;	/* Window that's about to be recycled. */
    int changingAll;		/* Non-zero means all of the windows on
				 * this window's files are about to be
				 * closed.  Zero means only mxwPtr is
				 * about to be closed. */
{
    if (*mxwPtr->fileInfoPtr->name == '\0') {
	return TCL_OK;	/* How do you get here? */
    }
    if (Mx_FileModified(mxwPtr->fileInfoPtr->file)
	    && (changingAll || ((mxwPtr->nextPtr == NULL)
	    && (mxwPtr == mxwPtr->fileInfoPtr->mxwPtr)))) {
	if (*mxwPtr->fileInfoPtr->name != 0) {
	    sprintf(interp->result, "dirty file: %s",
		    mxwPtr->fileInfoPtr->name);
	    return TCL_ERROR;
	} else {
	    sprintf(interp->result, "unsaved work: (no file)");
	    return TCL_ERROR;	    
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * MxGetMark --
 *
 *	Given a textual description of a position in a file, return
 *	an Mx_Position that describes the same point.  Marks are
 *	normally represented as two decimal numbers separated by
 *	any number of non-digits;  the first number is the line number
 *	and the second number is the character position within the
 *	line.  In addition, a number of keywords are recognized.  See
 *	the Mx man page (or the code below) for the keywords and what
 *	they mean.
 *
 * Results:
 *	The return value is TCL_OK if the mark was successfully converted
 *	to a position.  In this case, *posPtr will contain the position
 *	corresponding to mark.  If there was a problem (bad syntax for
 *	mark, or selection requested but doesn't exist in the file), TCL_ERROR
 *	is returned, an error message is left at mxwPtr->interp->result, and
 *	*posPtr is undefined.  If the position described by mark doesn't
 *	exist in the file (line number greater than # lines in file, or
 *	char number greater than # chars in the line), then it is rounded
 *	downward to the nearest existing position in the file.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
MxGetMark(mxwPtr, mark, posPtr)
    MxWidget *mxwPtr;			/* Window to whose file mark refers,
					 * and in which an error notifier
					 * should be displayed, if needed. */
    char *mark;				/* Textual description of position. */
    register Mx_Position *posPtr;	/* Where to store mark's position. */
{
    register MxFileInfo *fileInfoPtr = mxwPtr->fileInfoPtr;
    int nameLength;
    
    /*
     * Handle first the common case of a numerical mark specification.
     */
     
    if (isdigit(*mark)) {
	char *p;
	int length;

	posPtr->lineIndex = strtoul(mark, &p, 10);
	posPtr->lineIndex--;
	if (posPtr->lineIndex < 0) {
	    posPtr->lineIndex = 0;
	}
	while (!isdigit(*p)) {
	    if (*p == 0) {
		goto error;
	    }
	    p++;
	}
	posPtr->charIndex = atoi(p);
	if (Mx_GetLine(fileInfoPtr->file, posPtr->lineIndex, &length)
		== NULL) {
	    *posPtr = Mx_EndOfFile(fileInfoPtr->file);
	} else {
	    if (posPtr->charIndex >= length) {
		posPtr->charIndex = length-1;
	    }
	}
	return TCL_OK;
    }

    /*
     * Check for a screen position mark.
     */
    if (mark[0] == '@') {
	char *p, *end;
	int x, y;

	p = mark+1;
	x = strtol(p, &end, 0);
	if ((end == p) || (*end != ',')) {
	    goto error;
	}
	p = end+1;
	y = strtol(p, &end, 0);
	if (end == p) {
	    goto error;
	}
	(void)MxFindPosition(mxwPtr, x, y, posPtr);
	return TCL_OK;
    }
    /*
     * Not a numeric specifier.  Now check for the predefined mark names.
     */

    nameLength = strlen(mark);
    if (nameLength > 5) {
	nameLength = 5;
    }
    if (strncmp(mark, "caret", nameLength) == 0) {
	*posPtr = fileInfoPtr->caretFirst;
	return TCL_OK;
    }
    if (strncmp(mark, "eof", nameLength) == 0) {
	*posPtr = Mx_EndOfFile(fileInfoPtr->file);
	return TCL_OK;
    }
    if (strncmp(mark, "sel.left", nameLength) == 0) {
	Mx_Position dummy;
	return MxGetSelRange(mxwPtr, posPtr, &dummy);
    }
    if (strncmp(mark, "sel.right", nameLength) == 0) {
	Mx_Position dummy;
	return MxGetSelRange(mxwPtr, &dummy, posPtr);
    }
    if (strncmp(mark, "top", nameLength) == 0) {
	*posPtr = mxwPtr->linePtr[0].position;
	return TCL_OK;
    }
    if (strncmp(mark, "center", nameLength) == 0) {
	*posPtr = mxwPtr->linePtr[mxwPtr->heightLines/2].position;
	return TCL_OK;
    }
    if (strncmp(mark, "bottom", nameLength) == 0) {
	*posPtr = mxwPtr->linePtr[mxwPtr->heightLines-1].position;
	return TCL_OK;
    }

    error:
    sprintf(mxwPtr->interp->result,
	    "\"%.100s\" doesn't make sense as a mark",
	    mark);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * MxCvtToPrintable --
 *
 *	Given a keystroke binding that may contain control characters
 *	and/or meta characters, this routine produces a printable version
 *	of the string.
 *
 * Results:
 *	Up to length characters are stored at *result (including the
 *	terminating NULL character).
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
MxCvtToPrintable(string, length, result)
    char *string;		/* Binding string to be converted. */
    int length;			/* No. of bytes available at result. */
    char *result;		/* Where to store printable form. */
{
    int chunkSize;
    char chunk[20];
    char *p;

    /*
     * Process the input string one character at a time to do the
     * conversion.
     */

    p = result;
    for ( ; *string != 0; string++) {
	int i;

	/*
	 * Figure out how to represent this particular character.
	 */

	i = *string & 0377;
	if (i <= 040) {
	    if (i == 033) {
		strcpy(chunk, "ESC");
	    } else if (i == '\n') {
		strcpy(chunk, "RET");
	    } else if (i == '\t') {
		strcpy(chunk, "TAB");
	    } else if (i == ' ') {
		strcpy(chunk, "SPACE");
	    } else {
		chunk[0] = 'C';
		chunk[1] = '-';
		chunk[2] = i - 1 + 'a';
		chunk[3] = 0;
	    }
	} else if (i < 0177) {
	    chunk[0] = i;
	    chunk[1] = 0;
	} else if (i == 0177) {
	    sprintf(chunk, "DEL");
	} else if ((i > 0240) && (i < 0377)) {
	    chunk[0] = 'M';
	    chunk[1] = '-';
	    chunk[2] = i & 0177;
	    chunk[3] = 0;
	} else {
	    sprintf(chunk, "%#x", i);
	}

	/*
	 * Add this chunk onto the result string (if it fits), with a
	 * preceding space if this isn't the first chunk.
	 */

	if (p != result) {
	    if (length < 1) {
		break;
	    }
	    *p = ' ';
	    p++;
	    length--;
	}
	chunkSize = strlen(chunk);
	if (length < chunkSize) {
	    strncpy(p, chunk, length);
	    p += length;
	    length = 0;
	    break;
	} else {
	    strcpy(p, chunk);
	    p += chunkSize;
	    length -= chunkSize;
	}
    }

    if (length == 0) {
	p--;
    }
    *p = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * MxMarkParens --
 *
 *	See if position indicates a parenthesis (or part of one).  If it
 *	does, then change the parenthesis highlights for the file to mark
 *	the pair.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The parenthesis highlights for fileInfoPtr may be modified.  Only
 *	the 100 lines on either side of position are searched for a matching
 *	parenthesis, in order to save time when there's no match.
 *
 *----------------------------------------------------------------------
 */

void
MxMarkParens(fileInfoPtr, position)
    register MxFileInfo *fileInfoPtr;		/* File in which to look for
						 * matching parentheses. */
    Mx_Position position;			/* Position that may contain
						 * one parenthesis. */
{
    Mx_Position start, stop, open1, open2, close1, close2;

    start.lineIndex = position.lineIndex - 100;
    start.charIndex = 0;
    stop.lineIndex = position.lineIndex + 100;
    stop.charIndex = 0;
    if (Mx_SearchParen(fileInfoPtr->file, position, start, stop,
	    mxOpenParens, mxCloseParens, &open1, &open2,
	    &close1, &close2)) {
	if (fileInfoPtr->openParenPtr == NULL) {
	    fileInfoPtr->openParenPtr = MxHighlightCreate(fileInfoPtr->mxwPtr,
		    open1, open2, MX_UNDERLINE);
	    fileInfoPtr->closeParenPtr =
		    MxHighlightCreate(fileInfoPtr->mxwPtr, close1, close2,
		    MX_UNDERLINE);
	} else {
	    MxHighlightSetRange(fileInfoPtr->openParenPtr, open1,
		    open2);
	    MxHighlightSetRange(fileInfoPtr->closeParenPtr,
		    close1, close2);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * MxFocusProc --
 *
 *	This procedure is invoked by the TK focus manager when an
 *	Mx file window gets or loses the focus.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The caret may be turned on or off.  Also, we remember that
 *	we've got the focus.
 *
 *----------------------------------------------------------------------
 */

void
MxFocusProc(mxwPtr, gotFocus)
    register MxWidget *mxwPtr;		/* Window that just got the focus. */
    int gotFocus;			/* 1 means window just got focus, 0
					 * means it just lost it. */
{
    if (gotFocus) {
	mxwPtr->flags |= FOCUS_WINDOW;
	MxEventuallyRedraw(mxwPtr);
    } else {
	mxwPtr->flags &= ~FOCUS_WINDOW;
	MxCaretRedisplayText(mxwPtr->fileInfoPtr);
    }
}

#ifdef notdef

/*
 *----------------------------------------------------------------------
 *
 * MxMouseProc --
 *
 *	Invoked by the Tk dispatcher whenever mouse-related events
 *	occur in an Mx or Tx window.  This procedure implements
 *	"power-scrolling", also known as "shift-left-dragging"
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The caret may be turned on or turned off or moved.  The
 *	selection may change.
 *
 *----------------------------------------------------------------------
 */
typedef struct {
    MxWidget *mxwPtr;
    int delta, speed;
    Tk_TimerToken token;
    char active;
    char posted;
} ScrollState;
ScrollState scrollState;
#define SCROLL_MSEC	50

void MxPowerScroll(), MxScroll();

int mxMouseProcEnabled = 0;

void
MxMouseProc(mxwPtr, eventPtr)
    register MxWidget *mxwPtr;		/* Window to which event pertains. */
    register XButtonEvent *eventPtr;	/* What happened. */
{
    register MxFileInfo *fileInfoPtr = mxwPtr->fileInfoPtr;
    int x, y;
    Mx_Position pointer, first, last, anchor;
    XCrossingEvent *crossEventPtr;	/* For Enter and Leave events */
    static int dragCaret;		/* Should caret drag with pointer?
					 * Set when buttons go down. */
    static int dragSelection;		/* Should selection drag with pointer?
					 * Set when buttons go down. */
    static int button = -1;		/* The button that's currently down
					 * (if any). */
    static MxWidget *lastWindow;	/* Last window in which button was
					 * pressed. */
    static int lastX, lastY;		/* Last location where button was
					 * pressed. */
    static unsigned long lastTime;	/* Time when last button was pressed */
    static int repeatCount;		/* Number of times button has been
					 * pressed in the same place.  Cycles
					 * between 1 and 3. */
    static int anchorRight;		/* 1 means anchor point for
					 * selection is its right side.  0
					 * means it's the left side.  Can't
					 * save anchor position itself because
					 * the selection could float between
					 * button clicks. */
    static int scrollAction;		/* 1 means the shift key was down
					 * when the first button was
					 * pressed. */
    static int delta, speed;		/* Values for drag scrolling */

    if ((eventPtr->subwindow != NULL)
	    && (eventPtr->subwindow != Tk_WindowId(mxwPtr->tkwin))) {
	return;
    }

    if (! mxMouseProcEnabled) {
	goto done;
    }
    if (mxwPtr->flags & DESTROYED) {
	goto done;
    }

    /*
     * Collapse mouse motion events, so that when many motion events
     * pile up we immediately skip to the last one.
     */

    if (eventPtr->type == MotionNotify) {
	MxCollapseMotion(eventPtr);
    }
    /*
     * Before checking for "normal" button actions (for selection)
     * check for the shift key.  If it's down, then this is a scrolling
     * action.  Handle it right now and then return.
     */

    if ((eventPtr->state & ShiftMask) || scrollAction) {
	static int lastPosition;	/* The mouse y-position on which the
					 * current screen view is based. */
	if (eventPtr->type == ButtonPress) {
	    if (button != -1) {
		return;
	    }
	    button = eventPtr->button;
	    scrollAction = 1;
	    lastPosition = eventPtr->y;
	} else if (eventPtr->type == ButtonRelease) {
	    if (eventPtr->button == button) {
		scrollAction = 0;
		button = -1;
	    }
	} else if ((eventPtr->type == MotionNotify) &&
		   (eventPtr->state & (Button1Mask|Button3Mask))) {
	    if (!scrollAction) {
		/*
		 * It's possible to get a motion event before
		 * a mouse down event.  Thanks OLWM!
		 */
		lastPosition = eventPtr->y;
	    } else {
		if (eventPtr->state & Button1Mask) {
		    speed = 4;
		} else {
		    speed = 15;
		}
		delta = ((eventPtr->y - lastPosition)*speed)
		    /(mxwPtr->fontPtr->ascent + mxwPtr->fontPtr->descent);
		MxScroll(mxwPtr, speed, delta, &lastPosition);
	    }
	}
	return;
    }

    /*
     * (It's a little easier to see what you're pointing to if the
     * mouse doesn't have to be right on top of it to select it.
     * Offset the hot spot to accomplish this).
     */
    
    x = eventPtr->x - 2;
    y = eventPtr->y - 4;

    /*
     * When the left button goes up, search for a parenthesis matching
     * the current position, and highlight it.  If the button is anything
     * but the middle button, then save the commands invoked since the
     * last button press.
     */

    (void) MxFindPosition(mxwPtr, x, y, &pointer);
    if (eventPtr->type == ButtonRelease) {
	if (eventPtr->button == button) {
	    if (button == Button1) {
		MxMarkParens(fileInfoPtr, pointer);
	    }
	    if (button != Button2) {
		MxHistoryNext(mxwPtr, "mxHistory", 1);
	    }
	    button = -1;
	}
	Undo_Mark(mxwPtr->fileInfoPtr->log);
	return;
    }

    /*
     * When a button goes down, count how many consecutive clicks there
     * have been in the same place (if it's the left button), and decide
     * whether we're dragging the caret or the selection or both.
     */
    
    if (eventPtr->type == ButtonPress) {
	button = eventPtr->button;
	if (button == Button1) {
	    if ((lastWindow == mxwPtr) && ((eventPtr->time - lastTime) < 500)
		    && ((lastX + 1) >= x) && ((lastX - 1) <= x)
		    && ((lastY + 1) >= y) && ((lastY - 1) <= y)) {
		repeatCount += 1;
		if (repeatCount > 3) {
		    repeatCount = 1;
		}
	    } else {
		repeatCount = 1;
	    }
	    if (eventPtr->state & ControlMask) {
		dragCaret = 0;
		dragSelection = 1;
	    } else {
		dragCaret = 1;
		dragSelection = 0;
	    }
	    if (repeatCount != 1) {
		dragSelection = 1;
	    }
	} else if (button == Button3) {
	    dragSelection = 1;
	    if (eventPtr->state & Button1Mask) {
		dragCaret = 1;
	    } else {
		dragCaret = 0;
	    }
	} else {
	    button = -1;
	}
	lastX = x;
	lastY = y;
	lastTime = eventPtr->time;
	lastWindow = mxwPtr;
    }

    if (button == -1) {
	goto done;			/* We didn't see the down event. */
    }

    if (dragSelection) {

	/*
	 * There are three possibilities when setting the selection:
	 * drag a new selection; drag one edge of the selection with
	 * the caret as the other edge, and drag one edge with the
	 * original selection point as the other edge.
	 */

	if ((button == Button1) || dragCaret) {
	    first = last = anchor = pointer;
	} else {
	    Mx_Position dummy;

	    if ((fileInfoPtr != MxSelectedFile)
		    || (!(mxwPtr->flags & TX_WIDGET)
		    && !(eventPtr->state & ControlMask))) {
		anchor = fileInfoPtr->caretFirst;
		if (MX_POS_LESS(pointer, anchor)) {
		    anchor = Mx_Offset(fileInfoPtr->file, anchor, -1);
		}
	    } else if (anchorRight) {
		(void) MxGetSelRange(mxwPtr, &dummy, &anchor);
	    } else {
		(void) MxGetSelRange(mxwPtr, &anchor, &dummy);
	    }
	    if (MX_POS_LESS(pointer, anchor)) {
		first = pointer;
		last = anchor;
	    } else {
		first = anchor;
		last = pointer;
	    }
	}
	anchorRight = MX_POS_LESS(pointer, anchor);

	/*
	 * Depending on how many clicks there have been, round the selection up:
	 * 1st click:		no rounding
	 * 2nd click:		round to word boundary or parenthesized group
	 * 3rd click:		round out to line boundary
	 */

	switch (repeatCount) {
    
	    case 2: {
		Mx_Position tmp, oldFirst,  oldLast;

		tmp.lineIndex = tmp.charIndex = 0;
		oldFirst = first;
		oldLast = last;
		(void) Mx_SearchMask(fileInfoPtr->file, first, tmp,
			mxWordMask, &first);
		(void) Mx_SearchMask(fileInfoPtr->file, last,
			Mx_EndOfFile(fileInfoPtr->file), mxWordMask, &last);
		if ((button == Button1) && MX_POS_EQUAL(first, oldFirst)
			&& MX_POS_EQUAL(last, oldLast)) {
		    Mx_Position tmp1, tmp2;

		    (void) Mx_SearchParen(fileInfoPtr->file, first,
			    Mx_ZeroPosition, Mx_EndOfFile(fileInfoPtr->file),
			    mxOpenParens, mxCloseParens, &first, &tmp1, &tmp2,
			    &last);
		}
		break;
	    }
    
	    case 3: {
		first.charIndex = 0;
		(void) Mx_GetLine(fileInfoPtr->file, last.lineIndex,
			&last.charIndex);
		last.charIndex -= 1;
		break;
	    }
	}
	MxSelectionSet(mxwPtr, first, last);
    }

    if (dragCaret) {
	if (!(mxwPtr->flags & TX_WIDGET)) {
	    if (dragSelection) {
		pointer = first;
	    }
	    MxCaretSetPosition(mxwPtr, pointer, 0);
	}
	MxFocusProc(mxwPtr, 1);
    }

    done:
    return;
}

/*
 *----------------------------------------------------------------------
 *
 * MxPowerScroll --
 *
 *	TimerCallback callback that implements continuous scrolling.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Scrolls the window.
 *
 *----------------------------------------------------------------------
 */
static void
MxPowerScroll(scrollStatePtr)
    ScrollState *scrollStatePtr;
{
    scrollStatePtr->token = 0;
    if (! scrollStatePtr->active) {
	return;
    }
    MxScroll(scrollStatePtr->mxwPtr, scrollStatePtr->speed,
	     scrollStatePtr->delta, NULL);
    scrollStatePtr->token =
	    Tk_CreateTimerHandler(20, MxPowerScroll,
		    (ClientData) scrollStatePtr);

}
#endif

/*
 *----------------------------------------------------------------------
 *
 * MxScroll --
 *
 *	Scroll the window by a given delta.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Scrolls the window.
 *
 *----------------------------------------------------------------------
 */
void
MxScroll(mxwPtr, speed, delta, lastPositionPtr)
    MxWidget *mxwPtr;
    int speed;
    int delta;
    int *lastPositionPtr;
{
    Mx_Position pos, eof;

    if (lastPositionPtr) {
	*lastPositionPtr += (delta*(mxwPtr->fontPtr->ascent
		+ mxwPtr->fontPtr->descent))/speed;
    }
    if (delta < 0) {
	if (delta > -mxwPtr->heightLines) {
	    pos = mxwPtr->linePtr[-delta].position;
	} else {
	    pos = mxwPtr->linePtr[mxwPtr->heightLines-1].position;
	    delta += mxwPtr->heightLines;
	    pos.lineIndex -= delta;
	    pos.charIndex = 0;
	}
	eof = Mx_EndOfFile(mxwPtr->fileInfoPtr->file);
	if (MX_POS_LESS(eof, pos)) {
	    pos = eof;
	}
	MxGetInWindow(mxwPtr, pos, 0, 1);
    } else if (delta > 0) {
	if (delta < mxwPtr->heightLines) {
	    pos = mxwPtr->linePtr[mxwPtr->heightLines
		    - 1 - delta].position;
	} else {
	    pos = mxwPtr->linePtr[mxwPtr->heightLines-1].position;
	    delta -= mxwPtr->heightLines - 1;
	    pos.lineIndex -= delta;
	    pos.charIndex = 0;
	    if (pos.lineIndex < 0) {
		pos.lineIndex = 0;
	    }
	}
	MxGetInWindow(mxwPtr, pos, mxwPtr->heightLines-1, 1);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * MxCollapseMotion --
 *
 *	Given a MouseMotion event, this procedure searches the event
 *	queue for any other MouseMotion events.  If any are found that
 *	match the current one in all respects except mouse position,
 *	the pending event is eliminated and its position information
 *	is substituted into *eventPtr.
 *
 * Results:
 *	*EventPtr may be modified.
 *
 * Side effects:
 *	Collapsable mouse motion events are removed from the input
 *	queue.
 *
 *----------------------------------------------------------------------
 */

static int abortCollapse;

static void
MxCollapseMotion(eventPtr)
    XMotionEvent *eventPtr;	/* Event to try to collpase with others. */

{
    extern Bool CollapseProc();
    XEvent dummy;

    abortCollapse = 0;
    while (1) {
	if (XCheckIfEvent(eventPtr->display, &dummy, CollapseProc,
		(char *) eventPtr) == 0) {
	    return;
	}
    }
}

/*
 * Predicate procedure to see if any given event can be collapsed with
 * the given event.  If a non-matching MouseMotion event is found, abort
 * the rest of the search.
 */

	/* ARGSUSED */
Bool
CollapseProc(display, eventPtr, matchEventPtr)
    Display *display;			/* Connection to X server. */
    register XMotionEvent *eventPtr;	/* Potentially-collapsible event. */
    register XMotionEvent *matchEventPtr;
					/* Event to collapse into (the event
					 * passed to MxCollapseMotion). */
{
    if ((eventPtr->type != MotionNotify) || (abortCollapse)) {
	return 0;
    }
    if ((eventPtr->window != matchEventPtr->window)
	    || (eventPtr->root != matchEventPtr->root)
	    || (eventPtr->subwindow != matchEventPtr->subwindow)
	    || (eventPtr->state != matchEventPtr->state)
	    || (eventPtr->is_hint != matchEventPtr->is_hint)
	    || (eventPtr->same_screen != matchEventPtr->same_screen)) {
	abortCollapse = 1;
	return 0;
    }

    /*
     * Collapse the events by replacing info in the original event
     * with information from this later event.
     */

    matchEventPtr->time = eventPtr->time;
    matchEventPtr->x = eventPtr->x;
    matchEventPtr->y = eventPtr->y;
    matchEventPtr->x_root = eventPtr->x_root;
    matchEventPtr->y_root = eventPtr->y_root;
    return 1;
}

/*
 *----------------------------------------------------------------------
 *
 * MxDoCmd --
 *
 *	Execute a given Tcl command in a given window, and display
 *	error information if the command doesn't complete successfully.
 *
 * Results:
 *	Returns the result code from the command:  TCL_OK etc.
 *
 * Side effects:
 *	The command itself will probably have side effects, plus this
 *	procedure will display in the message window the command's
 *	result.
 *
 *----------------------------------------------------------------------
 */

int
MxDoCmd(mxwPtr, command)
    MxWidget *mxwPtr;		/* Context in which to execute command. */
    char *command;		/* Command to execute.  NULL means do
				 * nothing. */
{
    int result;
    char c;

    if (command == NULL) {
	return TCL_OK;
    }
    result = Tcl_Eval(mxwPtr->interp, command);
    if (result == TCL_OK) {
	if (*mxwPtr->interp->result != 0) {
	    MxOutputMsg(mxwPtr, mxwPtr->interp->result);
	}
	return result;
    }

    if (mxwPtr->flags & DESTROYED) {
	return result;
    }

    /*
     * Just for prettiness' sake, capitalize the first character of
     * the error message.
     */

    c = mxwPtr->interp->result[0];
    if (islower(c)) {
	char *msg;

	msg = (char *)malloc((unsigned) (strlen(mxwPtr->interp->result) + 1));
	strcpy(msg, mxwPtr->interp->result);
	msg[0] = toupper(c);
	MxOutputMsg(mxwPtr, msg);
	ckfree(msg);
    } else {
	MxOutputMsg(mxwPtr, mxwPtr->interp->result);
    }
    return result;
}


/*
 *----------------------------------------------------------------------
 *
 * MxModTime --
 *
 *	Gets the time of the last modification of the file.
 *
 * Results:
 *	Returns the time of the last modification of the file.
 *	Returns 0 if the modification time can't be obtained,
 *	for instance if the file doesn't exist.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

time_t
MxLastMod(name)
    char *name;			/* File name */
{
    struct stat buf;		/* File status buffer */

    if (stat(name,&buf)<0) {
	return 0;
    }
    else {
	return buf.st_mtime;
    }
}
/*
 * Mx_Panic --
 */
void
Mx_Panic(msg)
    char *msg;
{
    fprintf(stderr, "Mx_Panic: %s\n", msg);
    abort();
}

/*
 *----------------------------------------------------------------------
 *
 * MxOutputMsg --
 *
 * 	Display a one-line message in the message window.  This knows
 *	how the feedback widget is set up by feedbackSetup defined in
 *	/project/tcl/lib/utils.tk.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If the message window doesn't exist, then it is created.
 *
 *----------------------------------------------------------------------
 */

void
MxOutputMsg(mxwPtr, msg)
    register MxWidget *mxwPtr;		/* Window in which to create. */
    char *msg;				/* Message to display.  If it has
					 * more than MAX_STRING_LENGTH chars,
					 * only the first ones will appear. */
{
    char cmd[1024];
    if (mxwPtr->feedbackCmd == NULL) {
	fprintf(stderr, "MxOutputMsg (no feedback widget): %s\n", msg);
    } else {
	sprintf(cmd, "%s {%s}", mxwPtr->feedbackCmd, msg);
	if (Tcl_Eval(mxwPtr->interp, cmd) != TCL_OK) {
	    fprintf(stderr, "ERROR: %s\n", msg);
	}
    }
}

