/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/*
 * OT 3.0.2
 */

/*
 *	otCompare.c -- compare library routines
 *
 */

#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"

#ifndef LINT
static char RCSid_otCompare[] =
    "$RCSfile: otCompare.c,v $ $Revision: 1.1.4.2 $ $Date: 1993/10/07 19:10:10 $";

#endif


/*
 *		List of functions defined in the otCompare.c
 *
 *
 *
 */



/*      Table of Contents of Procedures defined and used        */

OTErr    otCompDate();     /* compare dates */
OTErr 	 otCompEnum();	 /* compare enumerated type */
OTErr    otCompText();     /* compare text */
OTErr    otCompList();     /* look for a match in a list */
OTErr    otCompNum();     /* compare numbers */

/* Compare() --  compare a field against a set of values or value templates
 *
 *  Used by validation and the report filtering stuff.
 *
 * The type field contains the type of test to be done, including the
 * whether it is optional (surrounded by brackets), or required
 * (surrounded by parens).
 *
 *
 *  type        valid refs  action
 *  ======      ==========  ======================================
 *  date        mm/dd/yy    make sure value fits mm/dd/yy form
 *  mm/dd/yy    >mm/dd/yy   match dates after mm/dd/yy
 *              <mm/dd/yy   match dates before mm/dd/yy
 *              mm/dd/yy-mm/dd/yy
 *                          matches dates range
 *
 *  mailname     <none>     validate as ok userid
 *
 *  list        <list>      make sure value is one of <list>
 *                          (doesn't require wildcard)
 *  text        <text>      regular expression match:
 *                          no validation for right now.
 *
 * Possible errors returned:
 * OT_SUCCESS		  - success
 * OT_NOMATCH		  - no match
 * OT_ILLEGAL_DATE_FORMAT - illegal date format ( date )
 * OT_ILLEGAL_DATE_COMPAR - illegal date comparison characters ( date )
 * OT_MALLOC_LOCATION     - problem w/ malloc ( text, enum )
 * OT_OTTYPE              - illegal type ( compare itself )
 */
OTErr
otCompare(value, type, ref)
char * value;               /* the value of the field to be tested */
otType type;                /* the type of field 		   */
char * ref;                 /* the comparison specs 		   */
{
    char *cp, *tmcp;
    OTErr matchStatus;

    DBUG_MED((stderr, "otCompare: value '%s' type '%d' ref '%s'\n", value, type, ref));
    /*
     * Find the right validation list.
     */
    if (!ref[0])
	return OT_SUCCESS;      /* always ok if nothing to validate against */
    if (!value || !value[0] || (value[0] == '#'))
	return OT_NOMATCH;      /* but fail on validate with no arg 	    */

    cp = strdup(value);
    if ( tmcp = strchr(cp, '#') )
        *tmcp = 0;
    otTrimWs(cp);
    
    switch ( type ) {
	case TYPE_DATE:     matchStatus = otCompDate(cp,ref);     break;
	case TYPE_TEXT:     matchStatus = otCompText(cp,ref);     break;
	case TYPE_ENUM:     matchStatus = otCompEnum(cp,ref);     break;
	case TYPE_IDNUM:   
	case TYPE_NUM:	    matchStatus = otCompNum(cp,ref);     break;
	case TYPE_MAILNAME:
	case TYPE_LIST:
	case TYPE_NULL:
			    matchStatus = otCompList(cp,ref);     break;
	default:            {
	  			otPutPostedMessage( OT_OTTYPE );
				matchStatus = OT_OTTYPE;
			    }
    }
    free(cp);
    return matchStatus;
}


/* 
 * otCompDate() -- compare dates.
 *
 * Calls parse_date to fill in DateComparison structures.
 *
 * Compares DateComparison structures based on comparison character (<>-)
 *
 *      valid refs          action
 *      ==========          ======================================
 *      mm/dd/yy            exact date match
 *      >mm/dd/yy           match dates after mm/dd/yy
 *      <mm/dd/yy           match dates before mm/dd/yy
 *  mm/dd/yy-mm/dd/yy       matches date range
 *
 * In place of mm/dd/yy
 *	.		    today's date
 *	-3		    three days ago
 *      +3		    three days from now
 *
 * Return statuses:
 *	OT_SUCCESS 	       - perfect match
 *	OT_NOMATCH 	       - match failed
 *	OT_ILLEGAL_DATE_FORMAT - from otParseDate()
 */

OTErr
otCompDate(value, ref)
char * value;       /* the value of the field to be tested */
char * ref;         /* the comparison specs */
{
    OTErr dateErr, res;
    OTDateComparison valueD, refD, refD1;
    register char *cp;
    register char *tmp, *sp, *vp, *wp, *qp, *rp, *xp;
    bool oneDate;
    bool eos = FALSE;

    /*
     * Remember, skip over both space AND newline now that there is
     * multiline support.
     */
    res = OT_NOMATCH;

    vp = wp = xp = strdup(value);
    while ( res == OT_NOMATCH && !eos ) {

        /* This is because strtok() is not re-entrant.	           */
        /* Find the start of the substring, then null terminate it */

	for( ;  *xp && strchr(" ,\n", *xp); xp++ );
	for( vp = xp;  *xp && !strchr(" ,\n", *xp); xp++ );

	if ( !*xp )
	    eos = TRUE;
	else
	    *xp = 0;

	qp = rp = strdup(ref); 
	DBUG_MED((stderr, "vp = %s\n", vp));

	for (qp = strtok(rp, ", \n"); *vp && qp && (res == OT_NOMATCH); qp = strtok(0, ", \n")) {

	    oneDate = FALSE;
	    if ( dateErr = otParseDate(vp, &valueD) )
		res = dateErr;

	    /* Check first date value in passed reference. */
	    sp = cp = tmp = strdup(qp);
	    if (*cp == '-')
		cp++;

	    for( ; *cp && *cp != '-'; cp++ );
	    if ( !*cp || *sp == '>' || *sp == '<' )
		oneDate = TRUE;
	    else
		*cp++ = '\000';

	    if ( dateErr = otParseDate(sp, &refD) )
		res = dateErr;
	    else if ( !oneDate ) {
		/* Check for the second date if range.	 */
		if ( dateErr = otParseDate(cp, &refD1) )
		    res = dateErr;
		else
		    refD.comparison = '-';
	    }

	    DBUG_MAX((stderr, "CompDate: dates comp '%c' value.days %d ref.days %d,%d\n", refD.comparison, valueD.days, refD.days, refD1.days));

	    if (res == OT_NOMATCH) {
		switch (refD.comparison) {
		    case '-':   if (valueD.days >= refD.days && valueD.days <= refD1.days)
				    res = OT_SUCCESS;
				break;

		    case '<':   if (valueD.days < refD.days)
				    res = OT_SUCCESS;
				break;

		    case '>':   if (valueD.days > refD.days)
				    res = OT_SUCCESS;
				break;

		    case '=':   if (valueD.days == refD.days)
				    res = OT_SUCCESS;
				break;

		    default :   res = OT_ILLEGAL_DATE_COMPAR;
				break;
		}
	    }
	    free(tmp);
	}
	free(rp);
	xp++;
    }
    free(wp);
    return res;

}



#ifdef POSIX_REGEXP

/* char regexp_pattern[MAXREGEXP+1]; */
char *regexp;

#define INIT		register char *sp=regexp;
#define GETC()		(*sp++)
#define PEEKC()		(*sp)
#define UNGETC(c)	(--sp)
#define RETURN(c)	return
#define ERROR( c )	otPutPostedMessage(OT_INTERNAL_REGEXP, (c))

#include <regexp.h>



/*
 * otCompEnum
 *
 */
OTErr
otCompEnum(value, ref)
char * value;       /* the value of the field to be tested */
char * ref;         /* the comparison specs */
{
    char *tp, *startp, *nextp;
    char *comp;
    OTErr status = OT_NOMATCH;

    startp = nextp = tp = strdup(ref);

    if ( (comp = (char *)malloc(MAXREGEXP)) == NULL) {
	otPutPostedMessage( OT_MALLOC_LOCATION, "CompEnum()" );
	status = OT_MALLOC_LOCATION;
    }
    else {
	while ( nextp && *nextp ) {
	    startp = nextp;
	    if ( nextp = strchr(startp, ',') ) {
		*nextp = 0;
		nextp++;
		while (isspace(*nextp))
		    nextp++;
	    }

	    /*
	     * The ^ and $ are necessary so project definition baseline pattern
	     * "1\.1" doesn't match user string "11.111" etc.
	     */
	    if ( (regexp = (char *)malloc(strlen(startp)+3)) == NULL) {
	        otPutPostedMessage( OT_MALLOC_LOCATION, "CompEnum()" );
		status = OT_MALLOC_LOCATION;
	    }
	    sprintf(regexp, "^%s$", startp );
	    (void)compile(0, comp, (char *)comp + MAXREGEXP, '\000');
	    if (step(value, comp))
		status = OT_SUCCESS;
	    free(regexp);
	}
    }

    free(comp);
    free(tp);
    return status;
}


/*  otCompText()  - compare text
 *
 *  Return:  OT_NOMATCH if not match if found
 */

OTErr
otCompText(value, ref)
char * value;       /* the value of the field to be tested */
char * ref;         /* the comparison specs */
{
    char *comp;
    OTErr status = OT_NOMATCH;

    if ( (regexp = (char *)malloc(strlen(ref)+3)) == NULL) {
        otPutPostedMessage( OT_MALLOC_LOCATION, "CompText()" );
	status = OT_MALLOC_LOCATION;
    }
    else if ( (comp = (char *)malloc(MAXREGEXP)) == NULL ) {
        free(regexp);
        otPutPostedMessage( OT_MALLOC_LOCATION, "CompText()" );
	status = OT_MALLOC_LOCATION;
    }
    else {
	sprintf(regexp, "%s", ref );
	(void)compile(0, comp, (char *)comp + MAXREGEXP, '\000');
	if (step(value, comp))
	    status = OT_SUCCESS;
	free(regexp);
	free(comp);
    }

    return status;
}


#endif

/*
 * otCompEnum
 *
 */
OTErr
otCompEnum(value, ref)
char * value;       /* the value of the field to be tested */
char * ref;         /* the comparison specs */
{
#ifdef notdef
    extern char *tclRegexpError;
    regexp *result;
#endif
    int match;
    register char *sp, *regp, *vp, *xp, *rp;
    char *wp, *tp;
    OTErr status = OT_NOMATCH;
    bool eos = FALSE;
    bool refeos;

    vp = wp = xp = strdup(value);

    while ( status == OT_NOMATCH && !eos ) {

        /* This is because strtok() is not re-entrant.	           */
        /* Find the start of the substring, then null terminate it */
	for( ;  *xp && strchr(" ,\n", *xp); xp++ );
	for( vp = xp;  *xp && !strchr(",", *xp); xp++ );

	if ( !*xp )
	    eos = TRUE;
	else
	    *xp = 0;

	sp = tp = rp = strdup(ref);
	DBUG_MED((stderr, "vp = %s\n", vp));

	refeos = FALSE;
	while ( status == OT_NOMATCH && !refeos && *vp) {

	    for( ; *rp && strchr(" ,\n", *rp); rp++ );
	    for( sp = rp; rp && !strchr(",", *rp); rp++ );

	    if ( !*rp )
		refeos = TRUE;
	    else
		*rp = 0;

	    DBUG_MED((stderr, "sp = %s\n", sp));
	    /*
	     * The ^ and $ are necessary so project definition baseline pattern
	     * "1\.1" doesn't match user string "11.111" etc.
	     */
	    if ( !(regp = (char *)malloc(strlen(sp) + LONGVALUE)) ) {
		otPutPostedMessage( OT_MALLOC_LOCATION, "CompEnum()" );
		status = OT_MALLOC_LOCATION;
	    }
	    sprintf(regp, "^%s$", sp);
            DBUG_MED((stderr, "regp = %s, |%s| = vp \n", regp, vp));	
	
#ifdef notdef
	    tclRegexpError = NULL;
	    result = regcomp(regp);
	    if ( (tclRegexpError != NULL) || (result == NULL) ) {
		otPutPostedMessage(OT_INTERNAL_ERROR, "error in regcomp()");
		status = OT_INTERNAL_ERROR;
	    } else {
		tclRegexpError = NULL;
		match = regexec(result, vp);
		if ( result )
		    free(result);	

		if (tclRegexpError != NULL) {
		    otPutPostedMessage(OT_INTERNAL_ERROR, "error in regcomp");
		    status = OT_INTERNAL_ERROR;
		}
		if (match)
		    status = OT_SUCCESS;
	    }
#endif
	    if ( (match = Tcl_RegExpMatch(otCB->cb_pcb->pcb_interp, vp, regp))
		< 0 ) {
		otPutPostedMessage(OT_INTERNAL_ERROR, otCB->cb_pcb->pcb_interp->result);
		status = OT_INTERNAL_ERROR;
	    } else if ( match ) {
		status = OT_SUCCESS;
	    }

	    free(regp);
	    rp++;
	}
	free(tp);
	xp++;
    }
    free(wp);

    return status;

}


/*  otCompText()  - compare text
 *
 *  Return:  OT_NOMATCH if not match if found
 */

OTErr
otCompText(value, ref)
char * value;       /* the value of the field to be tested */
char * ref;         /* the comparison specs */
{
    OTErr status;
#ifdef notdef
    extern char *tclRegexpError;
    regexp *result;
#endif
    int match;

    status = OT_NOMATCH;

#ifdef notdef
    tclRegexpError = NULL;
    result = regcomp(ref);
    if ( (tclRegexpError != NULL) || (result == NULL) ) {
	otPutPostedMessage(OT_INTERNAL_ERROR, "error in regcomp()");
	status = OT_INTERNAL_ERROR;
    } else {
	tclRegexpError = NULL;
	match = regexec(result, value);
	if ( result )
	    free(result);	

	if (tclRegexpError != NULL) {
	    otPutPostedMessage(OT_INTERNAL_ERROR, "error in regcomp");
	    status = OT_INTERNAL_ERROR;
	} else {
	    if (match)
		status = OT_SUCCESS;
        }
    }
#endif

    if ((match = Tcl_RegExpMatch(otCB->cb_pcb->pcb_interp, value, ref)) < 0) {
	otPutPostedMessage(OT_INTERNAL_ERROR, otCB->cb_pcb->pcb_interp->result);
	status = OT_INTERNAL_ERROR;
    } else if ( match ) {
	status = OT_SUCCESS;
    }

    return status;
}




/* CompList()   - look for a match through the comma and null terminated list
 *
 *  Return:  OT_SUCCESS if not match if found
 */

OTErr
otCompList(value, ref)
char * value;       /* the value of the field to be tested */
char * ref;         /* the comparison specs */
{
    register int i, j, refI;

    refI = 0;
    for (i=0;  ref[i]; ) {

	for (j=0; value[j]; ) {

	    refI = i;
	    for ( ; ref[refI] && (ref[refI] != ',') && (value[j] == ref[refI]) && 
		    value[j]  && (value[j] != ','); j++, refI++) ;

	    if ( (!value[j] || value[j] == ',') && (!ref[refI] || (ref[refI] == ',')) )
		return OT_SUCCESS;

	    while (value[j] && (value[j] != ','))
		j++;

	    if (value[j]) {
		while ( (value[j] == ' ') || (value[j] == ',') || (value[j] == '\n') )
		    j++;
	    }

	}

	i = refI;
	while (ref[i] && (ref[i] != ','))
	    i++;

        if (ref[i]) {
	    while ( (ref[i] == ' ') || (ref[i] == ',') || (ref[i] == '\n') )
		i++;
	}

    }
    return OT_NOMATCH;

}


/*
 *  otCompNum() -- compare numbers.
 *  
 *  Allows to compare for an exact match and for a less/greater than,
 *  or a series of digits separated by commas.
 *
 * Return statuses:
 *      OT_SUCCESS             - perfect match
 * 	OT_NOMATCH             - match failed
 *	OT_ILLEGAL_NUMERIC_SEARCH  - illegal numeric search
 */

OTErr
otCompNum(value, ref)
char * value;       /* the value of the field to be tested */
char * ref;         /* the comparison specs */
{
    register int i;
    register char  *cp, *str, *startstr, *dp;
    OTErr res = OT_NOMATCH;

#define LESS_GREATER_THAN  "<>"
    for (i=0; ref[i]; i++) {
	if ( (!isdigit(ref[i])) && !(ref[i] == ',') &&
	    (!strchr(LESS_GREATER_THAN, ref[i])) )
	    return OT_ILLEGAL_NUMERIC_SEARCH;
	
    }

    startstr = dp = str = strdup(ref);

    while ( *startstr && res == OT_NOMATCH ) {

	for( ; *dp && *dp != ','; dp++);
	if ( *dp == ',' )
	    *dp++ = 0;

	if ( cp = strpbrk(startstr, LESS_GREATER_THAN) ) {
	    switch (cp[0]) {
		case '>':
		    cp++;
		    if (atol(value) > atol(cp))
			res = OT_SUCCESS;
		    break;
		case '<':
		    cp++;
		    if (atol(value) < atol(cp))
			res = OT_SUCCESS;
		    break;
		}
	} else {
	    if (atol(value) == atol(startstr))
		res = OT_SUCCESS;
	}
	startstr = dp;
    }

    free(str);
    return res;
}
