/*
 *	mcmodel.c 
 *	A bunch of functions for calculating a 2D-isingmodel.
 *	These functions will be used in tkIsing.c
 *
 *	HUFFI 16-03-94
 *
 */

/*	include constants structtypes and functionprototypes	*/
#include "mcmodel.h"


/*
 *	InitRandom must be called before Random to initialize the
 *	randomgenerator.
 *	
 *	IN:
 *	    prandstruct: pointer to a struct of randtype, memory should be
 *	    allocated before calling this function.
 *	    seed: seed to be used for the randomgenerator.
 *	OUT:
 *	    pointer to a struct of randtype. The arrays in the struct have
 *	    been filled with the appropriate values. 
 */
struct randtype *InitRandom(struct randtype *prandstruct, int seed)
{
    int mult=32781;
    int fdb=471;
    unsigned int k=pow(3,18)+2*seed;
    int i;

    for(i=1;i<=lenr;i++)
    {        
        k=k*mult;
        prandstruct->irs[i]=k;
        prandstruct->inxt[i]=i+1;
    }
    prandstruct->inxt[lenr]=1;
    prandstruct->point=1;
    prandstruct->poinf=fdb;
    prandstruct->k=k;

    return prandstruct;
}

/*
 *	Random is the random generator which returns an array of random-
 *	values in the range of INT_MIN to INT_MAX. The routine returns 
 *	n randomvalues at a time.
 *	
 *	WARNING: You must have called InitRandom once before using this
 *	    function!
 *
 *	IN:
 *	    prandstruct: pointer to a struct of randtype which should be
 *	    initialized by InitRandom.
 *	    n: number of randomvalues you need.
 *	OUT:
 *	    pointer to an array of n randomvalues.
 */
int *Random(struct randtype *prandstruct, int n)
{
    int mult=32781;
    unsigned int i, k, l;
    unsigned int point, poinf;
    static int *irn = NULL;
    
    if(irn!=NULL)
        free(irn);
    
        irn=(int*)malloc(sizeof(int)*(n+1));  /* array met random getallen */

    k=prandstruct->k;

    for(i=1;i<=n;i++)
    {
        point=prandstruct->point;
        poinf=prandstruct->poinf;

        l=prandstruct->irs[point]^prandstruct->irs[poinf];
        k=k*mult;

        irn[i]=k^l;
        prandstruct->irs[point]=l;
        prandstruct->point=prandstruct->inxt[point];
        prandstruct->poinf=prandstruct->inxt[poinf];
    }
    
    return &irn[0];
}

/*
 *	MonteCarlo calculates one sweep of the hard square 2d-ising model.
 *	
 *	WARNING: You should call InitMonteCarlo once before using this
 *	    function, to initialise the model!
 *	IN: 
 *	    pmodelstruct: pointer to a struct of modeltype, which holds
 *	    the current data for the model.
 *	OUT:
 *	    pointer to a struct of modeltype with the newly calculated
 *	    spins.
 */
struct modeltype *MonteCarlo( struct modeltype *pmodelstruct )
/*	calculate one sweep	*/
{
    int ns = 1;
    int *randtab;  /* tabel van randomgetallen */
    int i1, i2;
    unsigned int v1, v2, v3, v4;
    int n1, n2;
    struct modeltype *p;
    int plpr;

    p = pmodelstruct;
    n1 = p->n1;
    n2 = p->n2;
    
    randtab = Random( pmodelstruct->prandstruct, n1*n2 );
    
    for( i2 = 0; i2 < n2; i2++ )
        for( i1 = 0; i1 < n1; i1++ )
        {
            ns++;
	    v1 = *(p->isp + *(p->nxtx+i1) + n1*i2)
		+ *(p->isp + *(p->nprx+i1) + n1*i2);
	    v2 = *(p->isp + i1 + *(p->nxty+i2)*n1)
		+ *(p->isp + i1 + *(p->npry+i2)*n1);
	    v3 = *(p->isp + *(p->nxtx+i1) + *(p->nxty+i2)*n1) 
		+ *(p->isp + *(p->nprx+i1) + *(p->npry+i2)*n1);
	    v4 = *(p->isp + *(p->nxtx+i1) + *(p->npry+i2)*n1)
		+ *(p->isp + *(p->nprx+i1) + *(p->nxty+i2)*n1);
	    plpr = p->ippr[v1][v2][v3][v4];
            if( randtab[ns] > plpr )
                *(p->isp + i1 + i2*n1) = 1;
            else
                *(p->isp + i1 + i2*n1) = 0;
        }

    return pmodelstruct;
}

/*
 *	InitMonteCarlo initializes the modelstruct of the model.
 *
 *	IN:
 *	    pmodelstruct a pointer to a struct of type modeltype, 
 *	    memory should already be allocated before calling this
 *	    function.
 *	OUT:
 *	    pointer to a struct of modeltype which is initialized.
 */
struct modeltype *InitMonteCarlo( struct modeltype *pmodelstruct )
/*	initialize monte carlo simulation	*/
{
    double bf, bfm1, bfm2, bfm3, bfm4, bfm5;
    int i, i1, i2, i3, i4;
    int n1 = pmodelstruct->n1;
    int n2 = pmodelstruct->n2;

/*	make cyclic arrays, so we don't have to check on lattice
 *	boundaries. (The lattice has been made cyclic).
 */
    for( i = 0; i < n1; i++ )
    {
        pmodelstruct->nxtx[i] = i + 1;
        pmodelstruct->nprx[i] = i - 1;
    }
    pmodelstruct->nxtx[n1-1] = 0;
    pmodelstruct->nprx[0] = n1 - 1;

    for( i = 0; i < n2; i++ )
    {
        pmodelstruct->nxty[i] = i + 1;
        pmodelstruct->npry[i] = i - 1;
    }
    pmodelstruct->nxty[n2-1] = 0;
    pmodelstruct->npry[0] = n2 - 1;

/*	Calculate the change distribution for the neighbours.	    */
    bfm1 = exp(pmodelstruct->beta[0]);
    bfm2 = exp(pmodelstruct->beta[1]);
    bfm3 = exp(pmodelstruct->beta[2]);
    bfm4 = exp(pmodelstruct->beta[3]);
    bfm5 = exp(pmodelstruct->beta[4]);
    bf = 1.0;

    for( i1 = 0; i1 <= ILUL1; i1++ )
        for( i2 = 0; i2 <= ILUL2; i2++ )
            for( i3 = 0; i3 <= ILUL3; i3++ )
		for( i4 = 0; i4 <= ILUL4; i4++ )
            	{
                   	bf = bfm1 * ( pow( bfm2, i1 )) * ( pow( bfm3, i2 ))
						* ( pow( bfm4, i3 )) * ( pow( bfm5, i4 ));
                	pmodelstruct->ippr[i1][i2][i3][i4] = ( int ) (( 1.0 / ( bf + 1.0 ) ) * UINT_MAX + INT_MIN);
		}
            
	return pmodelstruct;
}

/*
 *	InitSpins initializes the spins at random.
 *	
 *	IN:
 *	    pmodelstruct: pointer to a struct of modeltype, memory for
 *	    this struct should already be allocated before using this
 *	    function.
 *	OUT:
 *	    returns a pointer to a struct of modeltype, which contains
 *	    a pointer to an array of random spins.
 */
struct modeltype *InitSpins( struct modeltype *pmodelstruct )
{
	int i1, i2;
	int n1 = pmodelstruct->n1;
	int n2 = pmodelstruct->n2;
	double rnd, ipm;
	int *randtab;

	/* initieer de spins random */
	randtab = Random( pmodelstruct->prandstruct, n1*n2 );
	for( i2 = 0; i2 < n2; i2++ )
	    for( i1 = 0; i1 < n1; i1++ )
	    {
		rnd = (randtab[i1+i2*n1+1]+INT_MIN) / (double) UINT_MAX;
		ipm = rnd + rnd;
		*(pmodelstruct->isp + i1 + i2*n1) = (int) ipm;
	    }
	return pmodelstruct;
}

/*
 *	InitModel is used to initialize the model. This functions calls
 *	InitRandom to initialize the randomgenarator, 
 *	Initspins to initialize some spins, 
 *	and InitMonteCarlo to do some initialization for the MonteCarlo
 *	simulation.
 *
 *	IN:
 *	    n1: x-size of the lattice.
 *	    n2: y-size of the lattice.
 *	    beta1, beta2, beta3, beta4, beta5 are the betas to be used in the model.
 *	OUT:
 *	    returns 0. 
 */
int InitModel( struct modeltype *pmodelstruct,
		int n1, int n2, double beta1, double beta2, double beta3,
		double beta4, double beta5 )
{
    static struct randtype randstruct; 
    int seed = NR;

    /* initialiseer het model */
	if(pmodelstruct->isp != NULL)
	    free(pmodelstruct->isp);
	if(pmodelstruct->nxtx != NULL)
	    free(pmodelstruct->nxtx);
	if(pmodelstruct->nxty != NULL)
	    free(pmodelstruct->nxty);
	if(pmodelstruct->nprx != NULL)
	    free(pmodelstruct->nprx);
	if(pmodelstruct->npry != NULL)
	    free(pmodelstruct->npry);
	    
    pmodelstruct->n1 = n1;
    pmodelstruct->n2 = n2;
    pmodelstruct->prandstruct = &randstruct;
    pmodelstruct->isp = (int*)malloc(sizeof(int)*n1*n2);
    pmodelstruct->nxtx = (int*)malloc(sizeof(int)*n1);
    pmodelstruct->nxty = (int*)malloc(sizeof(int)*n2);
    pmodelstruct->nprx = (int*)malloc(sizeof(int)*n1);
    pmodelstruct->npry = (int*)malloc(sizeof(int)*n2);
    pmodelstruct->beta[0] = beta1;
    pmodelstruct->beta[1] = beta2;
    pmodelstruct->beta[2] = beta3;
    pmodelstruct->beta[3] = beta4;
    pmodelstruct->beta[4] = beta5;

    pmodelstruct->prandstruct = InitRandom( pmodelstruct->prandstruct, seed );

    pmodelstruct = InitSpins( pmodelstruct );

    pmodelstruct = InitMonteCarlo( pmodelstruct ); 

    return 0;
}

/*
 *	ModelData calculates a number of sweeps.
 *	If there has been a change in betas since the last time the
 *	function was called, it calls InitMonteCarlo to reinitialize
 *	the MonteCarlo simulation.
 *
 *	IN:
 *	    sweeps: number of sweeps to be calculated.
 *	    beta1, beta2, beta3, beta4, beta5 betas to be used in the model.
 *	OUT:
 *	    pointer to an array of integers containing the calculated
 *	    spins.
 */
int *ModelData( struct modeltype *pmodelstruct,
		int sweeps, double beta1, double beta2, double beta3,
		double beta4, double beta5 )
{
    static double oldbeta1;
    static double oldbeta2;
    static double oldbeta3;
    static double oldbeta4;
    static double oldbeta5;
    int i;

    if( (oldbeta1 != beta1) || (oldbeta2 != beta2) || (oldbeta3 != beta3)
		|| (oldbeta4 != beta4) || (oldbeta5 != beta5))
	{
		pmodelstruct->beta[0] = beta1;
		pmodelstruct->beta[1] = beta2;
		pmodelstruct->beta[2] = beta3; 
		pmodelstruct->beta[3] = beta4;
		pmodelstruct->beta[4] = beta5;
		pmodelstruct = InitMonteCarlo( pmodelstruct );

 	}
 
	for( i=1; i<=sweeps; i++ )
	{

    	pmodelstruct = MonteCarlo( pmodelstruct );

    }

    oldbeta1 = beta1;
    oldbeta2 = beta2;
    oldbeta3 = beta3;
    oldbeta4 = beta4;
    oldbeta5 = beta5; 
  
	return pmodelstruct->isp;  /* geef de pointer naar isp terug */
}

/*
 *	GetSpinData returns a handle to spindata.
 *
 *	IN:
 *	    nothing.
 *	    uses the global pointer pmodelstruct.
 *
 *	OUT:
 *	    returns a pointer to an array with spins
 *
 */
int *GetSpinData( struct modeltype *pmodelstruct )
{
	return pmodelstruct->isp;    
}
