static char rcsid[] = "$Id: functions.c,v 1.4 1994/08/04 23:32:00 sls Exp $";

/*
 * This software is copyright (C) 1994 by the Lawrence Berkeley Laboratory.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

/* This file contains the built-in narray functions  */

#include "tcl.h"
#include <assert.h>
#include <math.h>
#include "narray.h"
#include "narrayInt.h"

#define UNARY_FN(fn) \
    static char* fn_##fn(int argc, int* sp, NArrayOperand* stack, \
			 Tcl_Interp* interp) \
    { \
        if (argc != 1) return #fn " takes only 1 argument"; \
        if (stack[*sp].type != NARRAY_TYPE_FLOAT) \
	    return #fn " only takes numeric arguments"; \
        stack[*sp].value.n = fn(stack[*sp].value.n); \
	return 0; \
    }

UNARY_FN(sin)
UNARY_FN(cos)
UNARY_FN(tan)
UNARY_FN(asin)
UNARY_FN(acos)
UNARY_FN(atan)
UNARY_FN(cosh)
UNARY_FN(sinh)
UNARY_FN(tanh)
UNARY_FN(exp)
UNARY_FN(log)
UNARY_FN(log10)
UNARY_FN(sqrt)
UNARY_FN(ceil)
UNARY_FN(fabs)
UNARY_FN(floor)

#define BINARY_FN(fn) \
    static char* fn_##fn(int argc, int* sp, NArrayOperand* stack, \
			 Tcl_Interp* interp) \
    { \
        if (argc != 2) return #fn " takes exactly 2 arguments"; \
	if (stack[*sp].type != NARRAY_TYPE_FLOAT \
	    || stack[*sp-1].type != NARRAY_TYPE_FLOAT) \
	    return #fn " only takes numeric arguments"; \
        --*sp; \
	stack[*sp].value.n = fn(stack[*sp].value.n, stack[*sp + 1].value.n); \
        return 0; \
    }

BINARY_FN(atan2)
BINARY_FN(pow)
BINARY_FN(fmod)

static char* fn_doprintf(FILE* out, char* format,
			 int argc, int bp, NArrayOperand* stack,
			 Tcl_Interp* interp)
{
    char* fp;
    int arg, i, tmp;

    arg = 0;
    for (fp = format; *fp; fp++) {
	if (fp[0] == '%') {
	    if (fp[1] == '%') {
		putc('%', out);
	    } else {
		for (i = 1; fp[i]; i++) {
		    switch (fp[i]) {
		    case 'd': case 'f': case 'g': case 's':
			if (arg == argc)
			    return "fewer arguments than formats";
			tmp = fp[i+1];
			fp[i+1] = '\0';
			switch (fp[i]) {
			case 'd':
			    if (stack[bp+arg].type != NARRAY_TYPE_FLOAT)
				return "%d argument must be numeric";
			    fprintf(out, fp, (int) stack[bp+arg].value.n);
			    break;
			case 'f':
			case 'g':
			    if (stack[bp+arg].type != NARRAY_TYPE_FLOAT)
				return "%f or %g argument must be numeric";
			    fprintf(out, fp, (double) stack[bp+arg].value.n);
			    break;
			case 's':
			    if (stack[bp+arg].type != NARRAY_TYPE_STRING)
				return "%s argument must be a string";
			    fprintf(out, fp, stack[bp+arg].value.s);
			    break;
			}
			fp[i+1] = tmp;
			arg++;
			fp += i;
			goto next_arg;
		    }
		}
		return "unknown format, must be %d %f %g or %s";
	    }
	} else {
	    putc(fp[0], out);
	}
    next_arg:
    }
    return 0;
}

static char* fn_fprintf(int argc, int* sp, NArrayOperand* stack,
			Tcl_Interp* interp)
{
    int bp;
    char buf[20];
    FILE* fp;

    if (argc < 2)
	return \
       "wrong # args, should be: \"fprintf(file_handle, format, ?args ...?)\"";
    bp = *sp + 1 - argc;
    if (stack[bp].type != NARRAY_TYPE_FLOAT)
	return "arg 0 (file handle) to fprintf must numeric";
    sprintf(buf, "file%.0f", stack[bp].value.n);
    if (Tcl_GetOpenFile(interp, buf, 0, 0, &fp) != TCL_OK)
	return interp->result;
    if (stack[bp+1].type != NARRAY_TYPE_STRING)
	return "arg 1 (format) to fprintf must be a string";
    *sp -= argc;
    return fn_doprintf(fp, stack[bp+1].value.s, argc-2, bp+2, stack, interp);
}

static char* fn_printf(int argc, int* sp, NArrayOperand* stack,
		       Tcl_Interp* interp)
{
    int bp;

    if (argc < 1)
	return "wrong # args, should be: \"printf(format, ?args ...?)\"";
    bp = *sp + 1 - argc;
    if (stack[bp].type != NARRAY_TYPE_STRING)
	return "arg 0 (format) to printf must be a string";
    *sp -= argc;
    return fn_doprintf(stdout, stack[bp].value.s, argc-1, bp+1, stack,
		       interp);
}

static char* fn_tcl_eval(int argc, int* sp, NArrayOperand* stack,
			 Tcl_Interp* interp)
{
    int bp, i, result;
    char buf[TCL_DOUBLE_SPACE+1];
    Tcl_DString string;
    if (argc < 1)
	return "wrong # args, should be: \"tcl_eval(string, ?args ...?)\"";
    bp = *sp + 1 - argc;
    if (stack[bp].type != NARRAY_TYPE_STRING)
	return "arg 0 (string) to tcl_eval must be a string";
    Tcl_DStringInit(&string);
    Tcl_DStringAppend(&string, stack[bp].value.s, -1);
    for (i = 1; i < argc; i++) {
	switch (stack[bp+i].type) {
	case NARRAY_TYPE_FLOAT:
	    buf[0] = ' ';
	    Tcl_PrintDouble(interp, stack[bp+i].value.n, &buf[1]);
	    Tcl_DStringAppend(&string, buf, -1);
	    break;
	case NARRAY_TYPE_STRING:
	    Tcl_DStringAppend(&string, stack[bp+i].value.s, -1);
	    break;
	default:
	    assert(("unknown type", 0));
	}
    }
    result = Tcl_Eval(interp, Tcl_DStringValue(&string));
    Tcl_DStringFree(&string);
    if (result == TCL_ERROR)
	return interp->result;
    *sp -= argc;
    return 0;
}

int NArray_FunctionsInit(Tcl_Interp* interp)
{
    NArray_CreateFn("sin", fn_sin);
    NArray_CreateFn("cos", fn_cos);
    NArray_CreateFn("tan", fn_tan);
    NArray_CreateFn("asin", fn_asin);
    NArray_CreateFn("acos", fn_acos);
    NArray_CreateFn("atan", fn_atan);
    NArray_CreateFn("cosh", fn_cosh);
    NArray_CreateFn("sinh", fn_sinh);
    NArray_CreateFn("tanh", fn_tanh);
    NArray_CreateFn("exp", fn_exp);
    NArray_CreateFn("log", fn_log);
    NArray_CreateFn("log10", fn_log10);
    NArray_CreateFn("sqrt", fn_sqrt);
    NArray_CreateFn("ceil", fn_ceil);
    NArray_CreateFn("fabs", fn_fabs);
    NArray_CreateFn("floor", fn_floor);
    NArray_CreateFn("atan2", fn_atan2);
    NArray_CreateFn("pow", fn_pow);
    NArray_CreateFn("fmod", fn_fmod);
    NArray_CreateFn("printf", fn_printf);
    NArray_CreateFn("fprintf", fn_fprintf);
    NArray_CreateFn("tcl_eval", fn_tcl_eval);
    return TCL_OK;
}
