static char rcsid[] = "$Id: narray.c,v 1.12 1994/08/04 23:28:03 sls Exp $";

/*
 * This software is copyright (C) 1994 by the Lawrence Berkeley Laboratory.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include "tcl.h"
#include <assert.h>
#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include "narray.h"
#include "narrayInt.h"
#ifdef WITH_NETCDF
#include <netcdf.h>
#endif

#define state narray_compile_state

static int NArrayObjectCmd(ClientData, Tcl_Interp*, int, char**);
static void NArrayFree(ClientData data);

static int narray_flags;
#define NARRAY_FLAG_CRIPPLED 1	/* disable ncload/ncsave */

#ifdef WITH_NETCDF
static void AppendNCError(Tcl_Interp*);
#endif

/*
 * narray create name dim0 ?dim1? ...
 *
 * Create a new array named "name" with dimensions of length
 * dim0 ... dimN.  Creates a new command "name" that handles
 * operations on the narray.  Returns "name".
 *
 */

static int NArrayCmd(ClientData data, Tcl_Interp* interp,
		     int argc, char* argv[])
{
    NArray* array;
    int i, n, len;
#ifdef WITH_NETCDF
    char* src;
    int nc, ncvar, ndims, natts, old_ncopts;
    int dimids[MAX_NC_DIMS];
    long start[MAX_NC_DIMS], count[MAX_NC_DIMS];
    nc_type type;
    long size;
    char* tmp_storage;
#endif

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
			 " option ?args ... ?\"", 0);
	return TCL_ERROR;
    }
    len = strlen(argv[1]);
    if (!strncmp(argv[1], "create", len)) {
	if (argc < 4) {
	    Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
			     " create name dim0 ?dim1? ...\"", 0);
	    return TCL_ERROR;
	}
	n = argc - 3;
	array = (NArray*) ckalloc(sizeof(NArray) + (n - 1) * sizeof(int));
	array->n_dims = n;
	array->length = 1;
	array->debug = 0;
#ifdef WITH_NETCDF
	array->ncopts = NC_VERBOSE;
#endif
	array->closure.interp = interp;
	array->closure.vars = 0;
	array->closure.double_table = 0;
	array->closure.n_doubles = 0;
	array->closure.id_table = 0;
	array->closure.alloced_ids = 0;
	for (i = 0; i < n; i++) {
	    if (Tcl_GetInt(interp, argv[3+i], &array->dim_length[i]) != TCL_OK) {
		ckfree(array);
		return TCL_ERROR;
	    }
	    array->length *= array->dim_length[i];
	}
	array->storage =
	    (NArrayFloat*) ckalloc(sizeof(NArrayFloat) * array->length);
	memset(array->storage, 0, sizeof(NArrayFloat) * array->length);
	Tcl_CreateCommand(interp, argv[2], NArrayObjectCmd,
			  (ClientData) array, NArrayFree);
	Tcl_SetResult(interp, argv[2], TCL_VOLATILE);
	return TCL_OK;
    }
#ifdef WITH_NETCDF
    if (!strncmp(argv[1], "ncload", len)) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
			     " ncload name filename variable\"", 0);
	    return TCL_ERROR;
	}
	if (narray_flags & NARRAY_FLAG_CRIPPLED) {
	    Tcl_AppendResult(interp, "ncload disabled", 0);
	    return TCL_ERROR;
	}
	old_ncopts = ncopts;
	ncopts = NC_VERBOSE;
	if ((nc = ncopen(argv[3], NC_NOWRITE)) == -1) {
	    AppendNCError(interp);
	    ncopts = old_ncopts;
	    return TCL_ERROR;
	}
	if ((ncvar = ncvarid(nc, argv[4])) == -1) {
	    AppendNCError(interp);
	    ncopts = old_ncopts;
	    return TCL_ERROR;
	}
	if (ncvarinq(nc, ncvar, 0, &type, &ndims, dimids, &natts) == -1) {
	    AppendNCError(interp);
	    ncopts = old_ncopts;
	    return TCL_ERROR;
	}
	array = (NArray*) ckalloc(sizeof(NArray) + (ndims - 1) * sizeof(int));
	array->n_dims = ndims;
	array->length = 1;
	array->debug = 0;
	array->ncopts = 0;
	array->closure.interp = interp;
	array->closure.vars = 0;
	array->closure.double_table = 0;
	array->closure.n_doubles = 0;
	array->closure.id_table = 0;
	array->closure.alloced_ids = 0;
	for (i = 0; i < ndims; i++) {
	    if (ncdiminq(nc, dimids[i], 0, &size) == -1) {
		ckfree(array);
		AppendNCError(interp);
		ncopts = old_ncopts;
		return TCL_ERROR;
	    }
	    array->dim_length[i] = size;
	    array->length *= size;
	}
	array->storage =
	    (NArrayFloat*) ckalloc(sizeof(NArrayFloat) * array->length);
	len = nctypelen(type);
	if (len != sizeof(NArrayFloat)) {
	    tmp_storage = (char*) ckalloc(len * array->length);
	} else {
	    tmp_storage = (char*) array->storage;
	}
	for (i = 0; i < ndims; i++) {
	    start[i] = 0;
	    count[i] = array->dim_length[i];
	}
	if (ncvarget(nc, ncvar, start, count, tmp_storage) == -1) {
	    if (tmp_storage != (char*) array->storage)
		ckfree(tmp_storage);
	    ckfree(array->storage);
	    ckfree(array);
	    AppendNCError(interp);
	    ncopts = old_ncopts;
	    return TCL_ERROR;
	}
	if (type != NARRAY_NCFLOAT) {
	    for (i = 0; i < array->length; i++) {
		src = tmp_storage + len * i;
		switch (type) {
		case NC_BYTE:
		case NC_CHAR:
		    array->storage[i] = (NArrayFloat) *((unsigned char*) src);
		    break;
		case NC_SHORT:
		    array->storage[i] = (NArrayFloat) *((short*) src);
		    break;
		case NC_LONG:
		    array->storage[i] = (NArrayFloat) *((long*) src);
		    break;
		case NC_FLOAT:
		    array->storage[i] = (NArrayFloat) *((float*) src);
		    break;
		case NC_DOUBLE:
		    array->storage[i] = (NArrayFloat) *((double*) src);
		    break;
		default:
		    Tcl_AppendResult(interp, "unknown netCDF type", 0);
		    if (tmp_storage != (char*) array->storage)
			ckfree(tmp_storage);
		    ckfree(array->storage);
		    ckfree(array);
		    return TCL_ERROR;
		}
	    }
	    if (tmp_storage != (char*) array->storage)
		ckfree(tmp_storage);
	}
	ncclose(nc);
	Tcl_CreateCommand(interp, argv[2], NArrayObjectCmd,
			  (ClientData) array, NArrayFree);
	Tcl_SetResult(interp, argv[2], TCL_VOLATILE);
	ncopts = old_ncopts;
	return TCL_OK;
    }
#endif
    if (!strncmp(argv[1], "cripple", len)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
			     " cripple\"", 0);
	    return TCL_ERROR;
	}
	narray_flags |= NARRAY_FLAG_CRIPPLED;
	return TCL_OK;
    }
    Tcl_AppendResult(interp, "bad option \"", argv[1],
		     "\", should be one of: create, ncload, cripple.", 0);
    return TCL_ERROR;
}

#ifdef WITH_NETCDF
static char* ncstrerr[] = {
    "no error", "not a netcdf id", "too man netcdfs open",
    "netcdf file exists", "invalid argument", "write to read-only file",
    "operation not allowed in data mode",
    "operation not allowed in define mode", "coordinates out of domain",
    "MAX_NC_DIMS exceeded", "string match to name in use",
    "MAX_NC_ATTRS exceeded", "not a netcdf data type", "invalid dimension id",
    "NC_UNLIMITED in wrong index", "MAX_NC_VARS exceeded",
    "variable not found", "action prohibited on NC_GLOBAL varid",
    "not a netcdf file", "string too short", "MAX_NC_NAME exceeded",
    "NC_UNLIMITED size already in use"
};

static void AppendNCError(Tcl_Interp* interp)
{
    if (ncerr < 0 || ncerr > (sizeof(ncstrerr) / sizeof(ncstrerr[0]))) {
	Tcl_AppendResult(interp, "unknown netCDF error", 0);
    } else {
	Tcl_AppendResult(interp, ncstrerr[ncerr], 0);
    }
}
#endif

static void NArrayFree(ClientData data)
{
    NArray* array = (NArray*) data;
    int i;
    if (array->closure.vars) {
	ckfree(array->closure.vars);
    }
    if (array->closure.double_table) ckfree(array->closure.double_table);
    if (array->closure.id_table) {
	for (i = 0; i < array->closure.alloced_ids; i++)
	    if (array->closure.id_table[i].flags & NARRAY_SLOT_IN_USE)
		ckfree(array->closure.id_table[i].id);
	ckfree(array->closure.id_table);
    }
    ckfree(array->storage);
    ckfree(array);
}

/*
 * NArrayObjectCmd is the command that handles operations on
 * a narray.
 *
 * $na aref n0 ?n1? ...
 *    Return the value at (n0, n1, ..., nN).
 *
 * $na aset n0 ?n1? ... val
 *    Set (n0, n1, ..., nN) to val.
 *
 * $na vref var
 *    Return the value of var
 *
 * $na vset var val
 *    Set var to val.
 *
 * $na vars
 *    Return a list of variables in this narray.
 *
 * $na map code ?{var1 narray1} ...
 *    Map code over the each element, making narray1 available in var1.
 *
 * $na dimensions
 *    Return the length of each dimension in a list.
 *
 * $na status
 *    Return some information about the array.
 */

static int NArrayObjectCmd(ClientData data, Tcl_Interp* interp,
			   int argc, char* argv[])
{
    int len, n, i;
    NArrayFloat* result;
    NArray* array = (NArray*) data;
    char buf[TCL_DOUBLE_SPACE];
    double d;
#ifdef WITH_NETCDF
    int nc, ncvar, old_ncopts;
    int dimids[MAX_NC_DIMS];
    long start[MAX_NC_DIMS], count[MAX_NC_DIMS];
#endif
    
    if (argc == 1) {
	Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
			 " option ?args?\"", 0);
	return TCL_ERROR;
    }
    len = strlen(argv[1]);
    if (!strncmp(argv[1], "aref", len)) {
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
			     " index n0 ?n1? ...\"", 0);
	    return TCL_ERROR;
	}
	if ((argc - 2) > array->n_dims) {
	    sprintf(buf, "%d", array->n_dims);
	    Tcl_AppendResult(interp, "too many indexes, should be <= ",
			     buf, 0);
	    return TCL_ERROR;
	}
	result = array->storage;
	for (i = 0; i < array->n_dims; i++) {
	    if (Tcl_GetInt(interp, argv[i + 2], &n) != TCL_OK)
		return TCL_ERROR;
	    if (i == 0)
		result += n;
	    else
		result += array->dim_length[i - 1] * n;
	}
	if (result >= (array->storage + array->length)) {
	    Tcl_AppendResult(interp, "indexes are out of range", 0);
	    return TCL_ERROR;
	}
	Tcl_PrintDouble(interp, *result, buf);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	return TCL_OK;
    }
    if (!strncmp(argv[1], "aset", len)) {
	if (argc < 4) {
	    Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
			     " set n0 ?n1? ... val\"", 0);
	    return TCL_ERROR;
	}
	if ((argc - 3) > array->n_dims) {
	    sprintf(buf, "%d", array->n_dims);
	    Tcl_AppendResult(interp, "too many indexes, should be <= ",
			     buf, 0);
	    return TCL_ERROR;
	}
	if (Tcl_GetDouble(interp, argv[argc - 1], &d) != TCL_OK)
	    return TCL_ERROR;
	result = array->storage;
	for (i = 0; i < argc - 3; i++) {
	    if (Tcl_GetInt(interp, argv[i + 2], &n) != TCL_OK)
		return TCL_ERROR;
	    if (i == 0)
		result += n;
	    else
		result += array->dim_length[i - 1] * n;
	    if (result >= (array->storage + array->length)) {
		Tcl_AppendResult(interp, "indexes are out of range", 0);
		return TCL_ERROR;
	    }
	}
	*result = d;
	return TCL_OK;
    }
    if (!strncmp(argv[1], "vref", len)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args, should be \"",
			     argv[0], " vref var\"", 0);
	    return TCL_ERROR;
	}
	for (i = 0; i < array->closure.alloced_ids; i++) {
	    if (((array->closure.id_table[i].flags & NARRAY_SLOT_IN_USE)
		 && (array->closure.id_table[i].flags & NARRAY_SLOT_VARIABLE)
		 && !strcmp(argv[2], array->closure.id_table[i].id))) {
		Tcl_PrintDouble(interp, array->closure.vars[i], buf);
		Tcl_SetResult(interp, buf, TCL_VOLATILE);
		return TCL_OK;
	    }
	}
	Tcl_AppendResult(interp, "variable \"", argv[2], "\" does not exist",
			 0);
	return TCL_ERROR;
    }
    if (!strncmp(argv[1], "vset", len)) {
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args, should be \"",
			     argv[0], " vset var val\"", 0);
	    return TCL_ERROR;
	}
	if (Tcl_GetDouble(interp, argv[3], &d) != TCL_OK)
	    return TCL_ERROR;
	i = NArray_CreateClosureIdSlot(&array->closure, argv[2]);
	array->closure.vars[i] = d;
	array->closure.id_table[i].flags |= NARRAY_SLOT_VARIABLE;
	Tcl_PrintDouble(interp, d, buf);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	return TCL_OK;
    }
    if (!strncmp(argv[1], "vars", len)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args, should be \"",
			     argv[0], " vars\"", 0);
	    return TCL_ERROR;
	}
	Tcl_ResetResult(interp);
	for (i = 0; i < array->closure.alloced_ids; i++) {
	    if (((array->closure.id_table[i].flags & NARRAY_SLOT_IN_USE)
		 && (array->closure.id_table[i].flags & NARRAY_SLOT_VARIABLE))) {
		Tcl_AppendElement(interp, array->closure.id_table[i].id);
	    }
	}
	return TCL_OK;
    }
    if (!strncmp(argv[1], "map", len)) {
	Code* code;
	int n, result;
	char** split_argv;
	Tcl_CmdInfo info;
	
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args, should be\"", argv[0],
			     " map code ?{var0 arrary0} ...?\"", 0);
	    return TCL_ERROR;
	}
	result = TCL_OK;
	array->closure.n_arrays = 0;
	for (i = 0; i < (argc - 3); i++) {
	    if (Tcl_SplitList(interp, argv[3+i], &n, &split_argv) != TCL_OK) {
		result = TCL_ERROR;
		goto map_error;
	    }
	    if (!Tcl_GetCommandInfo(interp, split_argv[1], &info)
		|| info.proc != NArrayObjectCmd) {
		Tcl_AppendResult(interp, "\"", split_argv[1],
				 "\" is not an narray", 0);
		ckfree(split_argv);
		result = TCL_ERROR;
		goto map_error;
	    }
	    assert(i < NARRAY_MAX_BOUND_ARRAYS);
	    array->closure.array_table[i].array = (NArray*) info.clientData;
	    array->closure.array_table[i].name = ckalloc(strlen(split_argv[0])
							 + 1);
	    strcpy(array->closure.array_table[i].name, split_argv[0]);
	    ckfree(split_argv);
	    array->closure.n_arrays = i + 1;
	}
	code = NArray_Compile(array, argv[2]);
	if (code == 0) {
	    Tcl_AppendResult(interp, state->error_msg, " in compiling \"",
			     argv[2], "\"", 0);
	    result = TCL_ERROR;
	    goto map_error;
	}
	if (array->debug & DEBUG_DUMP) {
	    printf("Compiled code:\n");
	    NArray_PrintCode(array, code);
	}
	if (NArray_ApplyCode(array, code) == 0) {
	    Tcl_AppendResult(interp, array->errmsg, 0);
	    result = TCL_ERROR;
	}
	NArray_FreeCode(code);

    map_error:
	for (i = 0; i < array->closure.n_arrays; i++)
	    ckfree(array->closure.array_table[i].name);
	return result;
    }
    if (!strncmp(argv[1], "dimensions", len)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args, should be\"", argv[0],
			     " dimensions\"", 0);
	    return TCL_ERROR;
	}
	Tcl_ResetResult(interp);
	for (i = 0; i < array->n_dims; i++) {
	    sprintf(buf, "%d", array->dim_length[i]);
	    Tcl_AppendElement(interp, buf);
	}
	return TCL_OK;
    }
    if (!strncmp(argv[1], "status", len)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args, should be\"", argv[0],
			     " dimensions\"", 0);
	    return TCL_ERROR;
	}
	Tcl_ResetResult(interp);
	sprintf(buf, "%8.2fKB used, debug %d",
		(sizeof(NArray) + array->length * sizeof(NArrayFloat)
		 + (array->n_dims - 1) * sizeof(int)) / 1024.0,
		array->debug);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	return TCL_OK;
    }
    if (!strncmp(argv[1], "debug", len)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
			     " debug level\"", 0);
	    return TCL_ERROR;
	}
	if (Tcl_GetInt(interp, argv[2], &array->debug) != TCL_OK)
	    return TCL_ERROR;
	return TCL_OK;
    }
#ifdef WITH_NETCDF
    if (!strncmp(argv[1], "ncsave", len)) {
	if (argc < 5) {
	    Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
	       " ncsave filename variable_name dim0_name ?dim1_name? ...\"",
			     0);
	    return TCL_ERROR;
	}
	if (narray_flags & NARRAY_FLAG_CRIPPLED) {
	    Tcl_AppendResult(interp, "ncsave disabled", 0);
	    return TCL_ERROR;
	}
	if ((argc - 4) != array->n_dims) {
	    Tcl_AppendResult(interp, "wrong # dimension names", 0);
	    return TCL_ERROR;
	}
	old_ncopts = ncopts;
	ncopts = array->ncopts;
	if ((nc = nccreate(argv[2], NC_CLOBBER)) == -1) {
	    AppendNCError(interp);
	    ncopts = old_ncopts;
	    return TCL_ERROR;
	}
	for (i = 0; i < array->n_dims; i++) {
	    if ((dimids[i] = ncdimdef(nc, argv[4 + i], array->dim_length[i])) == -1) {
		AppendNCError(interp);
		ncabort(nc);
		ncopts = old_ncopts;
		return TCL_ERROR;
	    }
	    start[i] = 0;
	    count[i] = array->dim_length[i];
	}
	if ((ncvar = ncvardef(nc, argv[3], NARRAY_NCFLOAT, array->n_dims, dimids)) == -1) {
	    AppendNCError(interp);
	    ncabort(nc);
	    ncopts = old_ncopts;
	    return TCL_ERROR;
	}
	if (ncendef(nc) == -1) {
	    AppendNCError(interp);
	    ncabort(nc);
	    ncopts = old_ncopts;
	    return TCL_ERROR;
	}
	if (ncvarput(nc, ncvar, start, count, array->storage) == -1) {
	    AppendNCError(interp);
	    ncabort(nc);
	    ncopts = old_ncopts;
	    return TCL_ERROR;
	}
	ncclose(nc);
	ncopts = old_ncopts;
	return TCL_OK;
    }
#endif    
    Tcl_AppendResult(interp, "unknown option \"", argv[1],
	  "\", should be one of: aref, aset, vref, vset, vars, dimensions, map, status, debug",
		     0);
    return TCL_ERROR;
}

int NArray_Init(Tcl_Interp* interp)
{
    char* lib_dir;
    NArray_CodeInit(interp);
    NArray_FunctionsInit(interp);
    Tcl_CreateCommand(interp, "narray", NArrayCmd, 0, 0);
    if ((lib_dir = getenv("NARRAY_LIBRARY")) == 0)
	lib_dir = LIBRARY_DIR;
    if (Tcl_SetVar(interp, "auto_path", lib_dir,
		   (TCL_APPEND_VALUE|TCL_LIST_ELEMENT|TCL_LEAVE_ERR_MSG
		    |TCL_GLOBAL_ONLY)) == 0)
	return TCL_ERROR;
    if (Tcl_SetVar(interp, "narray_library", lib_dir,
		   TCL_LEAVE_ERR_MSG|TCL_GLOBAL_ONLY) == 0)
	return TCL_ERROR;
    if (Tcl_SetVar(interp, "narray_version", NARRAY_VERSION,
		   TCL_LEAVE_ERR_MSG|TCL_GLOBAL_ONLY) == 0)
	return TCL_ERROR;
    return TCL_OK;
}
