/*
 *  -----------------------------------------------------------------
 *    Copyright 1993 D.I.S. - Universita` di Pavia - Italy
 *  -----------------------------------------------------------------
 *
 *  Permission to  use,  copy,   modify,   distribute  this  software
 *  and  its  documentation for any purpose is hereby granted without
 *  fee, provided that the above copyright  notice   appear   in  all
 *  copies   and  that both that copyright notice and this permission
 *  notice appear in supporting documentation, and that the  name  of
 *  D.I.S.   not  be  used  in advertising or publicity pertaining to
 *  distribution of the software without specific, written prior per-
 *  mission.   D.I.S.  makes no representations about the suitability
 *  of this software for any purpose.  It is provided "as is" without
 *  express or implied warranty.
 *
 *  D.I.S. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, IN-
 *  CLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 *  NO EVENT SHALL D.I.S.  BE LIABLE FOR  ANY  SPECIAL,  INDIRECT  OR
 *  CONSEQUENTIAL  DAMAGES  OR  ANY DAMAGES WHATSOEVER RESULTING FROM
 *  LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION  OF  CONTRACT,
 *  NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNEC-
 *  TION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <setjmp.h>

#include <tcl.h>

extern int getopt(int argc, char *argv[], char *opts);
extern int errno, optind;
extern char *optarg;


int Tcl_AppInit(Tcl_Interp *intrep);

void interrupt(void);
jmp_buf loop;

#define Prompt	"tcl> "



int main( int argc, char *argv[])
{
	char usage[]="[-H] [-e command] [-f source] [args ...]";
	Tcl_Interp *interp;
	Tcl_CmdBuf cmdbuf;
	char *source, *command;
	int c, errF;
	char *cmd, *tail, line[1024];


	/*
	 *		source defaults to stdin
	 */
	source=NULL;
	command=NULL;

	/*
	 *		Options parsing
	 */
	for ( errF=0; (c=getopt( argc, argv, "e:f:H")) != EOF; )
		switch ( c ) {
			case 'e' :	if ( command != NULL || source != NULL ) ++errF;
						command=optarg;
						break;

			case 'f' :	if ( command != NULL || source != NULL ) ++errF;
						source=optarg;
						break;

			case 'H' :
			default  :	++errF;
						break;
		}

	if ( errF ) {
		fprintf( stderr, "usage : %s %s\n", argv[0], usage);
		return -1;
	}


	/*
	 *		Create interpreter and command buffer
	 */
	if ( (interp=Tcl_CreateInterp()) == NULL
			|| (cmdbuf=Tcl_CreateCmdBuf()) == NULL )
		return -1;

	/*
	 *		Set argc/argv variables
	 */

	sprintf(line, "%d", argc-optind+1);

	if ( Tcl_SetVar(interp, "argc", line, TCL_LEAVE_ERR_MSG) == NULL ) {
		fprintf(stderr, "internal error : interp->result\n");
		return -1;
	}

	if ( Tcl_SetVar(interp, "argv", argv[0],
			TCL_LIST_ELEMENT | TCL_LEAVE_ERR_MSG) == NULL ) {
		fprintf(stderr, "internal error : interp->result\n");
		return -1;
	}

	for (; optind < argc; ++optind)
		if ( Tcl_SetVar(interp, "argv", argv[optind], TCL_LIST_ELEMENT
				| TCL_APPEND_VALUE | TCL_LEAVE_ERR_MSG) == NULL ) {
			fprintf(stderr, "internal error : interp->result\n");
			return -1;
		}


#ifdef EXTENDED

	/*
	 *		Application-specific initialization
	 */
	if ( Tcl_AppInit(interp) != TCL_OK ) {
		fprintf(stderr, "initialization error : interp->result\n");
		return -1;
	}

#endif

    /*
     *  Command string
     */
    if ( command != NULL ) {
        if ( Tcl_GlobalEval(interp, command) != TCL_OK ) {
            fprintf(stderr, "error : %s\n", interp->result);
            return -1;
        }

    /*
     *  Interactive
     */
    } else if ( source == NULL && isatty(fileno(stdin)) ) {

        if ( setjmp(loop) == 0 )
            signal(SIGINT, interrupt);

        while ( (fputs(Prompt, stdout), fgets(line, sizeof(line), stdin))
                != NULL ) {

            if ( line[0] == '!' ) system(line+1);
            else if ( (cmd=Tcl_AssembleCmd(cmdbuf, line)) != NULL )
                if ( Tcl_GlobalEval(interp, cmd) != TCL_OK )
                    fprintf(stderr, "error : %s\n", interp->result);

                    /*
                     *  print if result is not blank and command is
                     *  not terminatd by ';'
                     */
                else if ( strlen(interp->result) > 0
                        && ( (tail=strrchr(cmd, ';')) == NULL ||
                            strspn(tail+1, " \t\n") != strlen(tail+1) ) )
                    printf("%s\n", interp->result);
        }

        printf("\nbye!\n");


    /*
     *  Non-interactive stdin
     */
    } else if ( source == NULL ) {

        while ( fgets(line, sizeof(line), stdin) != NULL ) {

            if ( (cmd=Tcl_AssembleCmd(cmdbuf, line)) != NULL
                    && Tcl_GlobalEval(interp, cmd) != TCL_OK )
                fprintf(stderr, "error : %s\n", interp->result);
        }

        if ( ferror(stdin) ) {
            fprintf( stderr, "stdin : %s\n", strerror(errno));
            return errno;
        }

    /*
     *  Input file
     */
    } else
        if ( Tcl_EvalFile(interp, source) != TCL_OK ) {
            fprintf(stderr, "error : %s\n", interp->result);
            return -1;
        }


    /*
     *  Clean-Up
     */
    Tcl_DeleteInterp(interp);
    Tcl_DeleteCmdBuf(cmdbuf);
    return 0;
}


void interrupt()
{
    putchar('\n');
    longjmp(loop, 1);
}

