/* 
 * tkSticker.c --
 *
 *	This file implements sticker items for canvas widgets.
 *	Much of the code is from tkCanvText.c and tkRectOval.c
 *	because a sticker item is basically text inside an rectangle item,
 *	but only that many characters will be drawn that fit into the bbox.
 *
 * Copyright (c) 1994 Heribert Dahms (dahms@ifk20.mach.uni-karlsruhe.de)
 * All rights reserved. The following UCB regulations shall apply accordingly.
 *
 * Copyright (c) 1991-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

/* $Log: tkSticker.c,v $
 * Revision 1.0  1994/08/12  00:07:23  dahms
 * BUG FIX: Reordered clipping code. relx<0 and/or rely<0 didn't work right.
 * First public release!
 *
 * Revision 0.9  1994/08/09  23:30:47  dahms
 * Bug Fix: Check length of version number and snip off trailing blank.
 *
 * Revision 0.8  1994/08/09  22:43:52  dahms
 * Improved copyright.
 * TkStickerType is now static, need not be global.
 * Reworked initialization routine. This is an incompatible change to name,
 * arguments and return value, which I hope is more compatible and uniform
 * to other packages 8-) Should ease dynamic loading. Now a global variable
 * "tksticker_version" is created.
 *
 * Revision 0.7  1994/08/09  19:02:41  dahms
 * The bar is now relative to the inner free space instead of base coords, so
 * it's not obscured by a fat outline (text was already handled this way).
 * Bug Fix: Postscript text was misplaced compared to screen.
 *
 * Revision 0.6  1994/06/20  08:03:57  dahms
 * Restricted operations on rotated text to only actual width and height.
 *
 * Revision 0.5  1994/06/19  22:04:07  dahms
 * Added check for display where cached resources are valid.
 * Forgot to credit: Inspiration for rotated text and some code is from BLT !
 *
 * Revision 0.4  1994/06/16  21:53:40  dahms
 * Added -orient to support rotated, vertical text.
 *
 * Revision 0.3  1994/06/07  16:18:54  dahms
 * Added -version to return rcsid string.
 *
 * Revision 0.2  1994/04/27  10:20:35  dahms
 * Functionally complete BETA version. Seems to have some rounding
 * problems with postscript text position inside outline rectangle.
 *
 * Revision 0.1  1994/04/25  10:48:48  dahms
 * First halfworking ALPHA version. Not everything yet implemented.
 *
 * Revision 0.0  1994/04/21  12:56:32  dahms
 * Initial revision
 *
 */

#ifndef lint
static char rcsid[] =
 "$Id: tkSticker.c,v 1.0 1994/08/12 00:07:23 dahms Exp $";
#endif

#include <stdio.h>
#include "tkInt.h"
#include "tkCanvas.h"
#include "tkConfig.h"

/*
 * The structure below is used for rotating text.
 */

typedef struct Rotate_Stuff {
    int maxHeight;		/* Height of Bitmap */
    int maxWidth;		/* Width of Bitmap */
    Pixmap hBitmap;		/* To draw text invisibly here. */
    GC clearGC;			/* To clear Bitmap. */
    GC writeGC;			/* To write Bitmap. */
    Display *display;		/* Where it's valid. */
} Rotate_Stuff;

/*
 * Set sizes for initial allocation to some reasonable binary power.
 * Note: A single static, permanent buffer is nonreentrant, but efficient.
 */

static Rotate_Stuff RotateStuff = { 32, 128, None, None, None, NULL };

/*
 * The structure below defines the record for each sticker item.
 */

typedef struct StickerItem {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    double bbox[4];		/* Coordinates of bounding box for sticker's
				 * rectangle (x1, y1, x2, y2).  Item includes
				 * x1 and x2 but not y1 and y2. */
    int width;			/* Width of outline. */
    XColor *outlineColor;	/* Color for outline. */
    XColor *fillColor;		/* Color for filling sticker. */
    XColor *barColor;		/* Additional color for filling bar. */
    Pixmap fillStipple;		/* Stipple bitmap for filling item. */
    GC outlineGC;		/* Graphics context for outline. */
    GC fillGC;			/* Graphics context for filling item. */
    GC barGC;			/* Graphics context for filling bar. */
    double relheight, relwidth, /* Relative geometry of bar inside sticker. */
	relx, rely;
    char *text;			/* Text for item (malloc-ed). */
    int numChars;		/* Number of non-NULL characters in text. */
    char *start;		/* Start of visible portion of text. */
    int visChars;		/* Number of visible characters of text. */
    int space;			/* Space pixels between repeated text. */
    int repeats;		/* How many times text is repeated. */
    int horSize;		/* Horizontal size of text+space in Pixels. */
    Tk_Anchor anchor;		/* Where to anchor text relative to sticker. */
    int lefttrunc;		/* Non-zero means truncate text on the left. */
    XFontStruct *fontPtr;	/* Font for drawing text. */
    XColor *textColor;		/* Color for text. */
    GC textGC;			/* Graphics context for drawing text. */
    int x, y;			/* Start coordinates of the text. */
    Tk_Uid orientUid;		/* Orientation for text ("vertical" or
				 * "horizontal"). */
    int vertical;		/* Non-zero means vertical orientation
				 * requested, zero means horizontal. */
    int numPixels;		/* Used text width in Pixels. */
} StickerItem;

/*
 * Information used for parsing configuration specs:
 */

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_ANCHOR, "-anchor", (char *) NULL, (char *) NULL,
	"center", Tk_Offset(StickerItem, anchor),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-bar", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(StickerItem, barColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-color", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(StickerItem, textColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(StickerItem, fillColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_FONT, "-font", (char *) NULL, (char *) NULL,
	"-Adobe-Helvetica-Bold-R-Normal--*-120-*",
	Tk_Offset(StickerItem, fontPtr), 0},
    {TK_CONFIG_BOOLEAN, "-lefttrunc", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, lefttrunc), 0},
    {TK_CONFIG_COLOR, "-outline", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(StickerItem, outlineColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-relheight", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, relheight), 0},
    {TK_CONFIG_DOUBLE, "-relwidth", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, relwidth), 0},
    {TK_CONFIG_DOUBLE, "-relx", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, relx), 0},
    {TK_CONFIG_DOUBLE, "-rely", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, rely), 0},
    {TK_CONFIG_PIXELS, "-space", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, space), 0},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(StickerItem, fillStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_STRING, "-text", (char *) NULL, (char *) NULL,
	"", Tk_Offset(StickerItem, text), 0},
    {TK_CONFIG_PIXELS, "-width", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, width), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_UID, "-orient", (char *) NULL, (char *) NULL,
	"horizontal", Tk_Offset(StickerItem, orientUid), 0},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */

int			TkSticker_Init _ANSI_ARGS_((Tcl_Interp *interp));
static void             ComputeStickerBbox _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    StickerItem *stickerPtr));
static int		ConfigureSticker _ANSI_ARGS_((
			    Tk_Canvas *canvasPtr, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateSticker _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    struct Tk_Item *itemPtr, int argc, char **argv));
static void		DeleteSticker _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr));
static void		DisplaySticker _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Drawable dst));
static void		LineToPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, int numChars));
static void		ScaleSticker _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static int		StickerCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, int argc, char **argv));
static int		StickerToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *rectPtr));
static double		StickerToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *pointPtr));
static int		StickerToPostscript _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Tk_PostscriptInfo *psInfoPtr));
static void		TranslateSticker _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * The structures below defines the sticker item type
 * by means of procedures that can be invoked by generic item code.
 */

static Tk_ItemType TkStickerType = {
    "sticker",				/* name */
    sizeof(StickerItem),		/* itemSize */
    CreateSticker,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureSticker,			/* configureProc */
    StickerCoords,			/* coordProc */
    DeleteSticker,			/* deleteProc */
    DisplaySticker,			/* displayProc */
    0,					/* alwaysRedraw */
    StickerToPoint,			/* pointProc */
    StickerToArea,			/* areaProc */
    StickerToPostscript,		/* postscriptProc */
    ScaleSticker,			/* scaleProc */
    TranslateSticker,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,	        /* insertProc */
    (Tk_ItemDCharsProc *) NULL,	        /* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};

/*
 *--------------------------------------------------------------
 *
 * Tk_StickerInit --
 *
 *	This procedure is invoked once for each interpreter to hook
 *	sticker items into the canvas code (from e.g. tkAppInit).
 *
 * Results:
 *	Returns a standard Tcl completion code and sets interp->result
 *	if there is an error.
 *
 * Side effects:
 *	The list of known canvas item types is modified.
 *	The version number is put in a global variable.
 *
 *--------------------------------------------------------------
 */

int
TkSticker_Init(interp)
    Tcl_Interp *interp;		/* Interpreter to initialize. */
{
    /*
     * Nice to have a prototype, but it's not in any *.h ! Alas; isn't this
     * the best, most portable interface in this case, avoiding any patches?
     */
    extern void Tk_CreateItemType _ANSI_ARGS_((Tk_ItemType *typePtr));

    static char version[] = "$Revision: 1.0 $";		/* Automated by RCS. */
    static char initialized = 0;
    char *p;

    /* More than once would neither hurt much nor help! */
    if (!initialized) {
	initialized = 1;
	Tk_CreateItemType(&TkStickerType);
    }

    /* Isolate version number. See RCS man pages: co(1) and rcsintro(1) */
    p = version + sizeof(version) - 1;	/* Address of null byte. */
    if (sizeof(version) < 3) goto error;	/* Known at compile time! */
    *(p-2) = 0;				/* Cut off trailing ' $'. */
    p = strchr(version, ' ');		/* One char before version number. */ 
    if (p == NULL) goto error;
    Tcl_SetVar(interp, "tksticker_version", p+1, TCL_GLOBAL_ONLY);

    return TCL_OK;

error:
    Tcl_SetResult(interp,
	"TkSticker_Init: Bogus RCS info \"$Revision: 1.0 $\".",
	TCL_STATIC);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * CreateSticker --
 *
 *	This procedure is invoked to create a new sticker item
 *	in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized so it can be safely freed by the
 *	caller.
 *      This message consists of the rcsid string if invoked as e.g.
 *              .canvas create sticker -version
 *      which seems weird, but I think it's the best solution.
 *
 * Side effects:
 *	A new sticker item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateSticker(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing sticker. */
{
    register StickerItem *stickerPtr = (StickerItem *) itemPtr;

    if (argc < 4) {
	if (argc==1 && !strcmp(argv[0], "-version")) {
	    Tcl_SetResult(canvasPtr->interp, rcsid, TCL_STATIC);
	} else

	Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
		Tk_PathName(canvasPtr->tkwin), "\" create ",
		itemPtr->typePtr->name, " x1 y1 x2 y2 ?options?",
		(char *) NULL);

	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    stickerPtr->width = 0;
    stickerPtr->outlineColor = NULL;
    stickerPtr->fillColor = NULL;
    stickerPtr->barColor = NULL;
    stickerPtr->fillStipple = None;
    stickerPtr->outlineGC = None;
    stickerPtr->fillGC = None;
    stickerPtr->barGC = None;
    stickerPtr->relheight = 0;
    stickerPtr->relwidth = 0;
    stickerPtr->relx = 0;
    stickerPtr->rely = 0;
    stickerPtr->text = NULL;
    stickerPtr->numChars = 0;
    stickerPtr->start = NULL;
    stickerPtr->visChars = 0;
    stickerPtr->space = 0;
    stickerPtr->repeats = 1;
    stickerPtr->horSize = 1;
    stickerPtr->anchor = TK_ANCHOR_CENTER;
    stickerPtr->lefttrunc = 0;
    stickerPtr->fontPtr = NULL;
    stickerPtr->textColor = NULL;
    stickerPtr->textGC = None;
    stickerPtr->x = 0;
    stickerPtr->y = 0;
    stickerPtr->orientUid = NULL;
    stickerPtr->vertical = 0;
    stickerPtr->numPixels = 0;

    /*
     * Process the arguments to fill in the item record.
     */

    if ((TkGetCanvasCoord(canvasPtr, argv[0], &stickerPtr->bbox[0]) != TCL_OK)
	    || (TkGetCanvasCoord(canvasPtr, argv[1],
		&stickerPtr->bbox[1]) != TCL_OK)
	    || (TkGetCanvasCoord(canvasPtr, argv[2],
		&stickerPtr->bbox[2]) != TCL_OK)
	    || (TkGetCanvasCoord(canvasPtr, argv[3],
		&stickerPtr->bbox[3]) != TCL_OK)) {
        return TCL_ERROR;
    }

    if (ConfigureSticker(canvasPtr, itemPtr, argc-4, argv+4, 0) != TCL_OK) {
	DeleteSticker(canvasPtr, itemPtr);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * StickerCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on sticker items.  See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
StickerCoords(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    register StickerItem *stickerPtr = (StickerItem *) itemPtr;
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE];
    char c2[TCL_DOUBLE_SPACE], c3[TCL_DOUBLE_SPACE];

    if (argc == 0) {
        Tcl_PrintDouble(canvasPtr->interp, stickerPtr->bbox[0], c0);
        Tcl_PrintDouble(canvasPtr->interp, stickerPtr->bbox[1], c1);
        Tcl_PrintDouble(canvasPtr->interp, stickerPtr->bbox[2], c2);
        Tcl_PrintDouble(canvasPtr->interp, stickerPtr->bbox[3], c3);
        Tcl_AppendResult(canvasPtr->interp, c0, " ", c1, " ",
                c2, " ", c3, (char *) NULL);
    } else if (argc == 4) {
        if ((TkGetCanvasCoord(canvasPtr, argv[0],
			&stickerPtr->bbox[0]) != TCL_OK)
                || (TkGetCanvasCoord(canvasPtr, argv[1],
			&stickerPtr->bbox[1]) != TCL_OK)
                || (TkGetCanvasCoord(canvasPtr, argv[2],
			&stickerPtr->bbox[2]) != TCL_OK)
                || (TkGetCanvasCoord(canvasPtr, argv[3],
			&stickerPtr->bbox[3]) != TCL_OK)) {
            return TCL_ERROR;
        }
        ComputeStickerBbox(canvasPtr, stickerPtr);
    } else {
        sprintf(canvasPtr->interp->result,
                "wrong # coordinates:  expected 0 or 4, got %d",
                argc);
        return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureSticker --
 *
 *	This procedure is invoked to configure various aspects
 *	of a sticker item, such as its border and background colors.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureSticker(canvasPtr, itemPtr, argc, argv, flags)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Sticker item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    register StickerItem *stickerPtr = (StickerItem *) itemPtr;
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;
    int length;

    if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
	    configSpecs, argc, argv, (char *) stickerPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    stickerPtr->numChars = strlen(stickerPtr->text);
    newGC = None;
    if ((stickerPtr->textColor != NULL) && (stickerPtr->fontPtr != NULL)) {
	gcValues.foreground = stickerPtr->textColor->pixel;
	gcValues.font = stickerPtr->fontPtr->fid;
	mask = GCForeground|GCFont;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (stickerPtr->textGC != None) {
	Tk_FreeGC(canvasPtr->display, stickerPtr->textGC);
    }
    stickerPtr->textGC = newGC;

    if (stickerPtr->outlineColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = stickerPtr->outlineColor->pixel;
	gcValues.cap_style = CapProjecting;
	if (stickerPtr->width < 0) {
	    stickerPtr->width = 1;
	}
	gcValues.line_width = stickerPtr->width;
	mask = GCForeground|GCCapStyle|GCLineWidth;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (stickerPtr->outlineGC != None) {
	Tk_FreeGC(canvasPtr->display, stickerPtr->outlineGC);
    }
    stickerPtr->outlineGC = newGC;

    if (stickerPtr->fillColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = stickerPtr->fillColor->pixel;
	if (stickerPtr->fillStipple != None) {
	    gcValues.stipple = stickerPtr->fillStipple;
	    gcValues.fill_style = FillStippled;
	    mask = GCForeground|GCStipple|GCFillStyle;
	} else {
	    mask = GCForeground;
	}
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (stickerPtr->fillGC != None) {
	Tk_FreeGC(canvasPtr->display, stickerPtr->fillGC);
    }
    stickerPtr->fillGC = newGC;

    if (stickerPtr->barColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = stickerPtr->barColor->pixel;
	if (stickerPtr->fillStipple != None) {
	    gcValues.stipple = stickerPtr->fillStipple;
	    gcValues.fill_style = FillStippled;
	    mask = GCForeground|GCStipple|GCFillStyle;
	} else {
	    mask = GCForeground;
	}
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (stickerPtr->barGC != None) {
	Tk_FreeGC(canvasPtr->display, stickerPtr->barGC);
    }
    stickerPtr->barGC = newGC;

    length = strlen(stickerPtr->orientUid);
    if (strncmp(stickerPtr->orientUid, "horizontal", length) == 0) {
	stickerPtr->vertical = 0;
    } else if (strncmp(stickerPtr->orientUid, "vertical", length) == 0) {
	stickerPtr->vertical = 1;
    } else {
	Tcl_AppendResult(canvasPtr->interp, "bad orientation \"",
		stickerPtr->orientUid,
		"\": must be vertical or horizontal", (char *) NULL);
	return TCL_ERROR;
    }

    ComputeStickerBbox(canvasPtr, stickerPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteSticker --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a sticker item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteSticker(canvasPtr, itemPtr)
    Tk_Canvas *canvasPtr;		/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
{
    register StickerItem *stickerPtr = (StickerItem *) itemPtr;

    if (stickerPtr->text != NULL) {
	ckfree(stickerPtr->text);
    }
    if (stickerPtr->fontPtr != NULL) {
	Tk_FreeFontStruct(stickerPtr->fontPtr);
    }
    if (stickerPtr->textColor != NULL) {
	Tk_FreeColor(stickerPtr->textColor);
    }
    if (stickerPtr->fillColor != NULL) {
	Tk_FreeColor(stickerPtr->fillColor);
    }
    if (stickerPtr->barColor != NULL) {
	Tk_FreeColor(stickerPtr->barColor);
    }
    if (stickerPtr->outlineColor != NULL) {
	Tk_FreeColor(stickerPtr->outlineColor);
    }
    if (stickerPtr->fillStipple != None) {
	Tk_FreeBitmap(canvasPtr->display, stickerPtr->fillStipple);
    }
    if (stickerPtr->textGC != None) {
	Tk_FreeGC(canvasPtr->display, stickerPtr->textGC);
    }
    if (stickerPtr->fillGC != None) {
	Tk_FreeGC(canvasPtr->display, stickerPtr->fillGC);
    }
    if (stickerPtr->barGC != None) {
	Tk_FreeGC(canvasPtr->display, stickerPtr->barGC);
    }
    if (stickerPtr->outlineGC != None) {
	Tk_FreeGC(canvasPtr->display, stickerPtr->outlineGC);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeStickerBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a sticker item.
 *	In addition, it recomputes all of the geometry information
 *	used to display a sticker item or check for mouse hits.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
ComputeStickerBbox(canvasPtr, stickerPtr)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    register StickerItem *stickerPtr;	/* Item whose bbox is to be
					 * recomputed. */
{
    int maxPixels, leftX, topY;
    int lineHeight, maxHeight;
    int bloat;
    double factor;

    /*
     * Make sure that the first coordinates are the lowest ones.
     */

    if (stickerPtr->bbox[1] > stickerPtr->bbox[3]) {
	double tmp;
	tmp = stickerPtr->bbox[3];
	stickerPtr->bbox[3] = stickerPtr->bbox[1];
       	stickerPtr->bbox[1] = tmp;
    }
    if (stickerPtr->bbox[0] > stickerPtr->bbox[2]) {
	double tmp;
	tmp = stickerPtr->bbox[2];
	stickerPtr->bbox[2] = stickerPtr->bbox[0];
	stickerPtr->bbox[0] = tmp;
    }

    bloat = (stickerPtr->width+1)/2 + 1;
    stickerPtr->header.x1 = stickerPtr->bbox[0] - bloat;
    stickerPtr->header.y1 = stickerPtr->bbox[1] - bloat;
    stickerPtr->header.x2 = stickerPtr->bbox[2] + bloat;
    stickerPtr->header.y2 = stickerPtr->bbox[3] + bloat;

    /*
     * Work through the text computing the starting point, number of
     * visible characters, and number of pixels.
     */

    if (stickerPtr->vertical) {
	maxPixels = stickerPtr->bbox[3] - stickerPtr->bbox[1] - 1 - 2*bloat;
	maxHeight = stickerPtr->bbox[2] - stickerPtr->bbox[0] - 1 - 2*bloat;
    } else {
	maxPixels = stickerPtr->bbox[2] - stickerPtr->bbox[0] - 1 - 2*bloat;
	maxHeight = stickerPtr->bbox[3] - stickerPtr->bbox[1] - 1 - 2*bloat;
    }

    if (!stickerPtr->lefttrunc) {
	stickerPtr->visChars = TkMeasureChars(stickerPtr->fontPtr,
		stickerPtr->text,
		stickerPtr->numChars, 0,
		maxPixels, 0, &stickerPtr->numPixels);
    } else {
	char *ptr;
	int i, j;
	char buf[100];

	ptr = (stickerPtr->numChars <= sizeof(buf)) ? buf :
		ckalloc(stickerPtr->numChars);

	/*
	 * Reverse all characters of the text.
	 */

	for (i = stickerPtr->numChars - 1, j = 0; i >= 0; --i, j++) {
		ptr[i] = stickerPtr->text[j];
	}
	
	stickerPtr->visChars = TkMeasureChars(stickerPtr->fontPtr,
                ptr,
                stickerPtr->numChars, 0,
                maxPixels, 0, &stickerPtr->numPixels);
	if (ptr != buf) ckfree(ptr);
    }


    stickerPtr->start = stickerPtr->text + (stickerPtr->lefttrunc ?
	(stickerPtr->numChars - stickerPtr->visChars) : 0);

    /*
     * Compute repeat count and spacing information, if desired.
     */

    if (stickerPtr->space > 0) {
	stickerPtr->horSize = stickerPtr->numPixels + stickerPtr->space;
	stickerPtr->repeats = (maxPixels + stickerPtr->space) /
				stickerPtr->horSize;
	stickerPtr->numPixels = stickerPtr->repeats * stickerPtr->horSize -
			stickerPtr->space;
    } else {
	stickerPtr->repeats = 1;
    }

    /*
     * Use overall geometry information to compute the top-left corner
     * of the bounding box for the text of the sticker item.
     * If rotated (vertical), the roles of x and y exchange temporarily,
     * so the anchor coordinate system is always relative to the text.
     */

    if (stickerPtr->vertical) {
	leftX = stickerPtr->bbox[3] - 0.5 - bloat;
	topY = stickerPtr->bbox[0] + 0.5 + bloat;
    } else {
	leftX = stickerPtr->bbox[0] + 0.5 + bloat;
	topY = stickerPtr->bbox[1] + 0.5 + bloat;
    }
    lineHeight = stickerPtr->fontPtr->ascent + stickerPtr->fontPtr->descent;
    factor = 0;		/* Only to prevent compiler warnings */
    switch (stickerPtr->anchor) {
	case TK_ANCHOR_NW:
	case TK_ANCHOR_N:
	case TK_ANCHOR_NE:
	    factor = 0;
	    break;

	case TK_ANCHOR_W:
	case TK_ANCHOR_CENTER:
	case TK_ANCHOR_E:
	    factor = 0.5;
	    break;

	case TK_ANCHOR_SW:
	case TK_ANCHOR_S:
	case TK_ANCHOR_SE:
	    factor = 1;
	    break;
    }
    topY += (maxHeight - lineHeight) * factor;
    switch (stickerPtr->anchor) {
	case TK_ANCHOR_NW:
	case TK_ANCHOR_W:
	case TK_ANCHOR_SW:
	    factor = 0;
	    break;

	case TK_ANCHOR_N:
	case TK_ANCHOR_CENTER:
	case TK_ANCHOR_S:
	    factor = 0.5;
	    break;

	case TK_ANCHOR_NE:
	case TK_ANCHOR_E:
	case TK_ANCHOR_SE:
	    factor = 1;
	    break;
    }

    if (stickerPtr->vertical) {
	leftX -= (maxPixels - stickerPtr->numPixels) * factor;
	stickerPtr->y = leftX;
	stickerPtr->x = topY;
    } else {
	leftX += (maxPixels - stickerPtr->numPixels) * factor;
	stickerPtr->x = leftX;
	stickerPtr->y = topY;
    }
    if (lineHeight > maxHeight) stickerPtr->visChars = 0;
}

/*
 *--------------------------------------------------------------
 *
 * DisplaySticker --
 *
 *	This procedure is invoked to draw a sticker item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

static void
DisplaySticker(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
    register StickerItem *stickerPtr = (StickerItem *) itemPtr;
    Display *display = Tk_Display(canvasPtr->tkwin);
    int x1, y1, x2, y2;
    int bx1, by1, bx2, by2;
    int i, w;

    /*
     * Compute the screen coordinates of the bounding box for the item.
     * Make sure that the bbox is at least one pixel large, since some
     * X servers will die if it isn't.
     */

    x1 = SCREEN_X(canvasPtr, stickerPtr->bbox[0]);
    y1 = SCREEN_Y(canvasPtr, stickerPtr->bbox[1]);
    x2 = SCREEN_X(canvasPtr, stickerPtr->bbox[2]);
    y2 = SCREEN_Y(canvasPtr, stickerPtr->bbox[3]);
    if (x2 <= x1) {
	x2 = x1+1;
    }
    if (y2 <= y1) {
	y2 = y1+1;
    }

    /*
     * A fat outline produces a smaller remaining inner rectangle, which
     * is then partitioned in a relative manner inspired by the Tk Placer.
     */

    w = stickerPtr->width;
    bx1 = x1 + w/2 + stickerPtr->relx * (x2 - x1 - w);
    by1 = y1 + w/2 + stickerPtr->rely * (y2 - y1 - w);
    bx2 = bx1 + stickerPtr->relwidth * (x2 - x1 - w);
    by2 = by1 + stickerPtr->relheight * (y2 - y1 - w);

    /*
     * Clip to bbox.
     */

    if (bx1 < x1) bx1 = x1; else if (bx1 > x2) bx1 = x2;
    if (by1 < y1) by1 = y1; else if (by1 > y2) by1 = y2;
    if (bx2 < x1) bx2 = x1; else if (bx2 > x2) bx2 = x2;
    if (by2 < y1) by2 = y1; else if (by2 > y2) by2 = y2;

    /*
     * Display filled parts first (if wanted), then outline.  If we're
     * stippling, then modify the stipple offset in the GC.  Be sure to
     * reset the offset when done, since the GC is supposed to be
     * read-only.
     */

    if (stickerPtr->fillGC != None) {
	if (stickerPtr->fillStipple != None) {
	    XSetTSOrigin(display, stickerPtr->fillGC,
		-canvasPtr->drawableXOrigin, -canvasPtr->drawableYOrigin);
	}
	XFillRectangle(display, drawable, stickerPtr->fillGC,
		x1, y1, (unsigned int) (x2-x1), (unsigned int) (y2-y1));
	if (stickerPtr->fillStipple != None) {
	    XSetTSOrigin(display, stickerPtr->fillGC, 0, 0);
	}
    }

    if (stickerPtr->barGC != None && bx1 < bx2 && by1 < by2) {
        if (stickerPtr->fillStipple != None) {
            XSetTSOrigin(display, stickerPtr->barGC,
                -canvasPtr->drawableXOrigin, -canvasPtr->drawableYOrigin);
        }
        XFillRectangle(display, drawable, stickerPtr->barGC,
                bx1, by1, (unsigned int) (bx2-bx1), (unsigned int) (by2-by1));
        if (stickerPtr->fillStipple != None) {
            XSetTSOrigin(display, stickerPtr->barGC, 0, 0);
        }
    }

    if (stickerPtr->outlineGC != None) {
	XDrawRectangle(display, drawable, stickerPtr->outlineGC,
		x1, y1, (x2-x1-1), (y2-y1-1));
    }

    /*
     * Finally display the text, optionally repeated and/or rotated.
     */

    if (stickerPtr->textGC != None && stickerPtr->visChars > 0) {
	int w, h;
	int lineHeight;
	XImage *hImage;

	lineHeight = stickerPtr->fontPtr->ascent + stickerPtr->fontPtr->descent;
	w = stickerPtr->numPixels;	/* More is useless */
	h = lineHeight;

	/*
	 * Check for enough resources if rotated text is desired.
	 * Also ensure cached resources are valid for current display.
	 */

	if (stickerPtr->vertical && (RotateStuff.display != display ||
		w > RotateStuff.maxWidth ||
		h > RotateStuff.maxHeight)) {
	    if (RotateStuff.hBitmap != None) {
		XFreePixmap(RotateStuff.display, RotateStuff.hBitmap);
	    }

	    /* Double sizes until fit. */
	    while (w > RotateStuff.maxWidth)
		RotateStuff.maxWidth <<= 1;
	    while (h > RotateStuff.maxHeight)
		RotateStuff.maxHeight <<= 1;

	    RotateStuff.hBitmap = XCreatePixmap(display, drawable,
		RotateStuff.maxWidth, RotateStuff.maxHeight, 1);

	    if (RotateStuff.display != display) {
	        XGCValues gcValues;
	        unsigned long mask;

		if (RotateStuff.clearGC != None) {
		    XFreeGC(RotateStuff.display, RotateStuff.clearGC);
		    XFreeGC(RotateStuff.display, RotateStuff.writeGC);
		}
		gcValues.foreground = gcValues.background = 0;
		mask = GCForeground | GCBackground;
		RotateStuff.clearGC = XCreateGC(display, RotateStuff.hBitmap,
					mask, &gcValues);

		gcValues.foreground = 1;
		mask = GCForeground | GCBackground; 
		RotateStuff.writeGC = XCreateGC(display, RotateStuff.hBitmap,
					mask, &gcValues);
		RotateStuff.display = display;
	    }
	}

	hImage = None;
	for (i = 0; i < stickerPtr->repeats; i++) {
	    if (stickerPtr->vertical && !i) {
		XFillRectangle(display, RotateStuff.hBitmap,
			RotateStuff.clearGC, 0, 0, w, h);
		XSetFont(display, RotateStuff.writeGC,
			stickerPtr->fontPtr->fid);
	    }
	    if (!stickerPtr->vertical) {
		TkDisplayChars(display,
		    drawable,
		    stickerPtr->textGC,
		    stickerPtr->fontPtr,
		    stickerPtr->start,
		    stickerPtr->visChars,
		    stickerPtr->x - canvasPtr->drawableXOrigin +
			i * stickerPtr->horSize,
		    stickerPtr->y - canvasPtr->drawableYOrigin +
			stickerPtr->fontPtr->ascent,
		    0);
	    } else {
		int x, y;

		if (!i) {
		   TkDisplayChars(display,
			RotateStuff.hBitmap,
			RotateStuff.writeGC,
			stickerPtr->fontPtr,
			stickerPtr->start,
			stickerPtr->visChars,
			0,
			stickerPtr->fontPtr->ascent,
			0);
		   hImage = XGetImage(display, RotateStuff.hBitmap, 0, 0,
				w, h, 1, ZPixmap);
		}

		for (x = 0; x < w; x++) {
		    for (y = 0; y < h; y++) {
			unsigned pixel;

			pixel = XGetPixel(hImage, x, y);
			if (pixel) {
			    XDrawPoint(display, drawable, stickerPtr->textGC,
				stickerPtr->x - canvasPtr->drawableXOrigin + y,
				stickerPtr->y - canvasPtr->drawableYOrigin -
					i * stickerPtr->horSize - x);
			}
		    }
		}
	    }
	}
	if (hImage != None) XDestroyImage(hImage);
    }
}

/*
 *--------------------------------------------------------------
 *
 * StickerToPoint --
 *
 *	Computes the distance from a given point to a given
 *	sticker item, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the sticker. If the
 *	point isn't inside the sticker then the return value is the
 *	distance from the point to the sticker.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
StickerToPoint(canvasPtr, itemPtr, pointPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;
    double x1, x2, y1, y2, xDiff, yDiff;

    x1 = stickerPtr->header.x1;
    x2 = stickerPtr->header.x2;
    y1 = stickerPtr->header.y1;
    y2 = stickerPtr->header.y2;

    /*
     * Point is outside rectangle.
     */

    if (pointPtr[0] < x1) {
	xDiff = x1 - pointPtr[0];
    } else if (pointPtr[0] >= x2)  {
	xDiff = pointPtr[0] + 1 - x2;
    } else {
	xDiff = 0;
    }

    if (pointPtr[1] < y1) {
	yDiff = y1 - pointPtr[1];
    } else if (pointPtr[1] >= y2)  {
	yDiff = pointPtr[1] + 1 - y2;
    } else {
	yDiff = 0;
    }

    return hypot(xDiff, yDiff);
}

/*
 *--------------------------------------------------------------
 *
 * StickerToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangle.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by rectPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
StickerToArea(canvasPtr, itemPtr, rectPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item to check against rectangle. */
    double *rectPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;

    if ((rectPtr[2] <= stickerPtr->header.x1)
            || (rectPtr[0] >= stickerPtr->header.x2)
            || (rectPtr[3] <= stickerPtr->header.y1)
            || (rectPtr[1] >= stickerPtr->header.y2)) {
        return -1;
    }
    if ((rectPtr[0] <= stickerPtr->header.x1)
            && (rectPtr[1] <= stickerPtr->header.y1)
            && (rectPtr[2] >= stickerPtr->header.x2)
            && (rectPtr[3] >= stickerPtr->header.y2)) {
        return 1;
    }
    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleSticker --
 *
 *	This procedure is invoked to rescale a sticker item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Scales the position and size of the sticker, but not the size
 *	of the font for the text.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
ScaleSticker(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas *canvasPtr;		/* Canvas containing sticker. */
    Tk_Item *itemPtr;			/* Sticker item to be scaled. */
    double originX, originY;		/* Origin about which to scale item. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    register StickerItem *stickerPtr = (StickerItem *) itemPtr;

    stickerPtr->bbox[0] = originX + scaleX*(stickerPtr->bbox[0] - originX);
    stickerPtr->bbox[1] = originY + scaleY*(stickerPtr->bbox[1] - originY);
    stickerPtr->bbox[2] = originX + scaleX*(stickerPtr->bbox[2] - originX);
    stickerPtr->bbox[3] = originY + scaleY*(stickerPtr->bbox[3] - originY);
    ComputeStickerBbox(canvasPtr, stickerPtr);
    return;
}

/*
 *--------------------------------------------------------------
 *
 * TranslateSticker --
 *
 *	This procedure is called to move a sticker item by a
 *	given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the sticker item is offset by (xDelta, yDelta),
 *	and the bounding box is updated in the generic part of the
 *	item structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateSticker(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    register StickerItem *stickerPtr = (StickerItem *) itemPtr;

    stickerPtr->bbox[0] += deltaX;
    stickerPtr->bbox[1] += deltaY;
    stickerPtr->bbox[2] += deltaX;
    stickerPtr->bbox[3] += deltaY;
    ComputeStickerBbox(canvasPtr, stickerPtr);
}

/*
 *--------------------------------------------------------------
 *
 * StickerToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	sticker items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in canvasPtr->interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
StickerToPostscript(canvasPtr, itemPtr, psInfoPtr)
    Tk_Canvas *canvasPtr;		/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    Tk_PostscriptInfo *psInfoPtr;	/* Information about the Postscript;
					 * must be passed back to Postscript
					 * utility procedures. */
{
    register StickerItem *stickerPtr = (StickerItem *) itemPtr;
    char buffer[500];
    double x1, x2, y1, y2;
    int i;

    x1 = stickerPtr->bbox[0];
    y1 = TkCanvPsY(psInfoPtr, stickerPtr->bbox[1]);
    x2 = stickerPtr->bbox[2];
    y2 = TkCanvPsY(psInfoPtr, stickerPtr->bbox[3]);


    /*
     * First draw the filled areas of the rectangle.
     */

    sprintf(buffer, "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto \
%.15g 0 rlineto closepath\n",
		x1, y1,
		x2-x1, y2-y1,
		x1-x2);

    if (stickerPtr->fillColor != NULL) {
	Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
	if (TkCanvPsColor(canvasPtr, psInfoPtr, stickerPtr->fillColor)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
	if (stickerPtr->fillStipple != None) {
	    if (TkCanvPsStipple(canvasPtr, psInfoPtr,
		    stickerPtr->fillStipple, 1) != TCL_OK) {
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(canvasPtr->interp, "fill\n", (char *) NULL);
	}
    }

    if (stickerPtr->barColor != NULL) {
	char buffer[500];
	double y1, y2;
	double bx1, by1, bx2, by2;
	int w;

	y1 = stickerPtr->bbox[1];
	y2 = stickerPtr->bbox[3];

	w = stickerPtr->width;
	bx1 = x1 + w/2 + stickerPtr->relx * (x2 - x1 - w);
	if (bx1 < x1) bx1 = x1; else if (bx1 > x2) bx1 = x2;
	by1 = y1 + w/2 + stickerPtr->rely * (y2 - y1 - w);
	if (by1 < y1) by1 = y1; else if (by1 > y2) by1 = y2;
	bx2 = bx1 + stickerPtr->relwidth * (x2 - x1 - w);
	if (bx2 < x1) bx2 = x1; else if (bx2 > x2) bx2 = x2;
	by2 = by1 + stickerPtr->relheight * (y2 - y1 - w);
	if (by2 < y1) by2 = y1; else if (by2 > y2) by2 = y2;

	by1 = TkCanvPsY(psInfoPtr, by1);
	by2 = TkCanvPsY(psInfoPtr, by2);

	sprintf(buffer, "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto \
%.15g 0 rlineto closepath\n",
		bx1, by1,
		bx2-bx1, by2-by1,
		bx1-bx2);

	Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
	if (TkCanvPsColor(canvasPtr, psInfoPtr, stickerPtr->barColor)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
	if (stickerPtr->fillStipple != None) {
	    if (TkCanvPsStipple(canvasPtr, psInfoPtr,
		    stickerPtr->fillStipple, 1) != TCL_OK) {
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(canvasPtr->interp, "fill\n", (char *) NULL);
	}
    }

    /*
     * Now draw the outline, if there is one.
     */

    if (stickerPtr->outlineColor != NULL) {
        Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
        sprintf(buffer, "%d setlinewidth", stickerPtr->width);
        Tcl_AppendResult(canvasPtr->interp, buffer,
                " 0 setlinejoin 2 setlinecap\n", (char *) NULL);
        if (TkCanvPsColor(canvasPtr, psInfoPtr, stickerPtr->outlineColor)
                != TCL_OK) {
            return TCL_ERROR;
        }
        Tcl_AppendResult(canvasPtr->interp, "stroke\n", (char *) NULL);
    }

    /*
     * Finally draw the text, if there is one, optionally repeated.
     */

    if (stickerPtr->textColor != NULL && stickerPtr->visChars > 0) {
	if (TkCanvPsFont(canvasPtr, psInfoPtr, stickerPtr->fontPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (TkCanvPsColor(canvasPtr, psInfoPtr, stickerPtr->textColor) !=
		TCL_OK) {
	    return TCL_ERROR;
	}
	for (i = 0; i < stickerPtr->repeats; i++) {
	    static char* PSrotate[] = { "", "translate 90 rotate 0 0" };
	    sprintf(buffer, "gsave %.15g %.15g %s [\n",
		(double) stickerPtr->x + (stickerPtr->vertical ?
			stickerPtr->fontPtr->descent :
				i * stickerPtr->horSize),
		TkCanvPsY(psInfoPtr, (double) stickerPtr->y -
			(stickerPtr->vertical ? i * stickerPtr->horSize :
				- stickerPtr->fontPtr->descent)),
		PSrotate[stickerPtr->vertical]);
	    Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
	    Tcl_AppendResult(canvasPtr->interp, "    ", (char *) NULL);
	    LineToPostscript(canvasPtr->interp, stickerPtr->start,
		stickerPtr->visChars);
	    Tcl_AppendResult(canvasPtr->interp, "\n", (char *) NULL);
	    sprintf(buffer, "] %d 0 0 0 () false DrawText grestore\n",
		stickerPtr->fontPtr->ascent + stickerPtr->fontPtr->descent);
	    Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
	}
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * LineToPostscript --
 *
 *	This procedure generates a parenthesized Postscript string
 *	describing one line of text.
 *
 * Results:
 *	None. The parenthesized string is appended to
 *	interp->result.  It generates proper backslash notation so
 *	that Postscript can interpret the string correctly.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
LineToPostscript(interp, string, numChars)
    Tcl_Interp *interp;		/* Interp whose result is to be appended to. */
    register char *string;	/* String to Postscript-ify. */
    int numChars;		/* Number of characters in the string. */
{
#define BUFFER_SIZE 100
    char buffer[BUFFER_SIZE+5];
    int used, c;

    buffer[0] = '(';
    used = 1;
    for ( ; numChars > 0; string++, numChars--) {
	c = (*string) & 0xff;
	if ((c == '(') || (c == ')') || (c == '\\') || (c < 0x20)
		|| (c >= 0x7f)) {
	    /*
	     * Tricky point:  the "03" is necessary in the sprintf below,
	     * so that a full three digits of octal are always generated.
	     * Without the "03", a number following this sequence could
	     * be interpreted by Postscript as part of this sequence.
	     */
	    sprintf(buffer+used, "\\%03o", c);
	    used += strlen(buffer+used);
	} else {
	    buffer[used] = c;
	    used++;
	}
	if (used >= BUFFER_SIZE) {
	    buffer[used] = 0;
	    Tcl_AppendResult(interp, buffer, (char *) NULL);
	    used = 0;
	}
    }
    buffer[used] = ')';
    buffer[used+1] = 0;
    Tcl_AppendResult(interp, buffer, (char *) NULL);
}
