# Functions for keeping a list of all active BETH interpreters, and switching
# between them.

# Interpreter management

# Code remote interpreter must successfully execute to be added.
set shiboleth {info exists beth_dir}
# List of interpreters
set all_interps {}
set me [winfo name .]

# Ensures that interp is still active, removes it if it isn't
# Returns 1 if interp is OK, 0 if not.
proc check_interp {interp} {
	global shiboleth
	if {[catch {send $interp $shiboleth}]} {
		delete_interp $interp
		return 0
	}
	return 1
}

# Like send, but checks interp first. Returns 0 if send failed.
proc check_send {interp args} {
	if {[llength $args] == 1} {	set cmd [lindex $args 0]
	} else {			set cmd $args}
	if {[catch {send $interp $cmd} result]} {
		delete_interp $interp
		return 0
	}
	return $result
}

# Removes interpreter from all_interps list
proc delete_interp {interp} {
	global all_interps
	set place [lsearch $all_interps $interp]
	if {($place >= 0)} {set all_interps [lreplace $all_interps $place $place]
}}

proc add_interp {interp} {
	global all_interps
	if {[lsearch $all_interps $interp] < 0} {
		lappend all_interps $interp
}}

# When I quit, notify the other interps
proc delete_me {f} {
	global me
	if {([grab status .] != "none")} {catch {give_control +1 0 $f}}
	delete_interp $me
	send_interp_updates
}


# Interpreter registration -- not pretty, but it works

# Figures out the list of all currently active interpreters.
# (w/o contacting another interp)
proc refresh_interpreter_list {} {
	global all_interps ; set all_interps ""
	foreach interp [winfo interps] {
		if {[check_interp $interp]} {add_interp $interp}}
	global registered ; set registered 1
}

# Sends my interp/title lists to every other interp
proc send_interp_updates {} {
	global all_interps me
	foreach interp $all_interps {
		if {($interp != $me) && [check_interp $interp]} {
			send $interp "set all_interps \{$all_interps\} ; \
				set im_last 0"
}}}


# Interpreter switching

proc wm_iconify {f} {
	if {([grab status .] != "none")} {
		give_control +1 0 $f
	} else {relinquish_control $f}
	wm iconify .
}

# Given an interp, returns the next/prev one. (d is +1/-1 for next/prev)
# If deiconify_flag is 1, the interp returned is deiconifed first, if it's 0
# iconified windows are skipped.
proc another_interp {d deiconify_flag} {
	global all_interps me
	set index [lsearch $all_interps $me]
	set first_index $index
	if {($first_index < 0)} {set first_index 0}
	while {1} {	incr index $d
		if {($index >= [llength $all_interps])} {set index 0}
		if {($index == -1)} {set index [llength $all_interps] ; incr index -1}
		if {($index == $first_index)} {return ""}

		set next [lindex $all_interps $index]
		if {[check_send $next {winfo ismapped .}]} {
			return $next
		} elseif {($deiconify_flag == 1)} {
			check_send $next {wm deiconify .}
			return $next
}}}

proc transfer_control {interp f} {
	if {([grab status .] != "none")} {
		relinquish_control $f
		check_send $interp {wm deiconify . ; take_control $frame}
	} else {check_send $interp {wm_raise $frame}
}}

# Gives control to another interpreter (see another_interp for parm details)
proc give_control {d deiconify_flag f} {
	set next_interp [another_interp $d $deiconify_flag]
	if {($next_interp == "")} {beep ; return}
	transfer_control $next_interp $f
}

# Given an interp's title, returns the interp
proc which_interp_is {int_title all_titles} {
	global all_interps
	set i [lsearch $all_titles $int_title*]
	if {($i == 0)} {beep ; return}
	set int [lindex $all_interps $i]
	if {([check_interp $int] == 1)} {return $int
	} else {return ""
}}

proc choose_beth_which {t f all_titles} {
	set which [$f.choosee get]
	destroy_f_entry $t $f.choosel $f.choosee

	set next_interp [which_interp_is $which $all_titles]
	if {($next_interp == "")} {beep ; return}
	transfer_control $next_interp $f
}

proc get_all_titles {} {
	global all_interps
	set list ""
	foreach interp $all_interps {
		lappend list [check_send $interp wm title .]}
	return $list
}

proc choose_beth_interp {t f} {
	global Keys
	set all_titles [get_all_titles]
	create_f_entry $t $f.choosel $f.choosee
	$f.choosel configure -text "Which:"
	parse_bindings $f.choosee \
Tab			"e_complete_string %W [list $all_titles]" \
$Keys(C_m)		"choose_beth_which $t $f [list $all_titles]"
}


# Interpreter bindings. f is a frame widget to put messages in.
proc interpbind {f t m} {
	parse_bindings all \
C-g			"+catch \{destroy_f_entry $t $f.choosel $f.choosee\}" \
M-I			"wm_iconify $f" \
M-C-b			"choose_beth_interp $t $f" \
M-N			"give_control +1 1 $f" \
M-n			"give_control +1 0 $f" \
M-P			"give_control -1 1 $f" \
M-p			"give_control -1 0 $f"

	if {[winfo exists $m]} {
		parse_menuentries $m.window.m {
					{Iconify 0 M-I}}
		parse_menuentries $m.interpreter.m {
				separator
					{"Next Interpreter" 1 M-N}
					{"Previous Interpreter" 1 M-P}
					{"Next Shown Interpreter" 0 M-n}
					{"Previous Shown Interpreter" 0 M-p}
					{"Choose Interpreter" 7 M-C-b}
}}}


interpbind $frame $text $menu

# Let the other interps know if I get destroyed
bind . <Destroy> "+delete_me $frame"

# If a bunch of beth interpreters were started at once, they may be confused
# about who all is out there. If I am the last one started, then I'll figure
# out who's out there & tell 'em.
set im_last 1
after 2000 {if {($im_last)} {
	refresh_interpreter_list
	send_interp_updates
}}
