# jrichtext.tcl - procedures for dealing with rich text
# 
# Copyright 1992-1994 by Jay Sekora.  All rights reserved, except 
# that this file may be freely redistributed in whole or in part 
# for nonprofit, noncommercial use.
# 
# 
# these procedures are required by
#     help.tk
#     edit.tk
#     browser.tk
#     people.tk
# they may be located in the file "~/.tk/jrt.tcl" (where they will
# be source'd by those applications on startup), or in the site-wide
# tk library directory, where they will be found (and loaded) by the
# default tk  unknown  procedure.
######################################################################

# j:tagged_insert w text args - insert tagged text into a text widget
# j:rt text dest - prepare to write rich text to text widget dest
# j:rt:type - return type of current rich text destination (text, TeX)
# j:rt:destination - return current rich text destination (widget, file)
# j:rt:textfonts {style font}... - set fonts for text widget
# j:rt:done - finish writing rich text (clear vars, close files)
# j:rt:rm text - write rich text (roman)
# j:rt:it text - write rich text (italic)
# j:rt:bf text - write rich text (bold face)
# j:rt:bi text - write rich text (bisexual)
# j:rt:tt text - write rich text (typewriter - monospaced)
# j:rt:hl text - write rich text (`headline' - larger bold)
# j:rt:tab - tab in rich text
# j:rt:cr - line break in rich text
# j:rt:par - paragraph break in rich text
# j:rt:mkabbrevs - make shorter convenience procs, for text-intensive apps
# rm - dummy do-nothing procedure to prevent unknown from calling /bin/rm
#   if you forget to j:rt:mkabbrevs

######################################################################
# j:tagged_insert - append to a text widget with a particular tag
#   (lifted from mkStyles.tcl demo, where it was insertWithTags)
######################################################################

# The procedure below inserts text into a given text widget and
# applies one or more tags to that text.  The arguments are:
#
# w		Window in which to insert
# text		Text to insert (it's inserted at the "insert" mark)
# args		One or more tags to apply to text.  If this is empty
#		then all tags are removed from the text.

proc j:tagged_insert {w text args} {
  set start [$w index insert]
  $w insert insert $text
  foreach tag [$w tag names $start] {
    $w tag remove $tag $start insert
  }
  foreach i $args {
    $w tag add $i $start insert
  }
}

######################################################################
# j:rt text dest - prepare to write rich text to text widget dest
#   future versions will support PostScript, TeX, maybe canvas, etc.
######################################################################

proc j:rt { {type {}} {destination stdout} } {
  global j_rt
  
  case $type in {
    {text} {			;# output to a text widget
      set j_rt(type) $type
      set j_rt(destination) $destination
      $j_rt(destination) delete 0.0 end
      $j_rt(destination) configure -wrap word
      catch {
        $j_rt(destination) configure -font \
          -adobe-helvetica-medium-r-normal--*-120-*
        $j_rt(destination) tag configure j:rt:rm -font \
          -adobe-helvetica-medium-r-normal--*-120-*
        $j_rt(destination) tag configure j:rt:it -font \
          -adobe-helvetica-medium-o-normal--*-120-*
        $j_rt(destination) tag configure j:rt:bf -font \
          -adobe-helvetica-bold-r-normal--*-120-*
        $j_rt(destination) tag configure j:rt:bi -font \
          -adobe-helvetica-bold-o-normal--*-120-*
        $j_rt(destination) tag configure j:rt:tt -font \
          -adobe-courier-medium-r-normal--*-120-*
        $j_rt(destination) tag configure j:rt:hl -font \
          -adobe-helvetica-bold-o-normal--*-180-*
      }
    }
    default {
      tkerror "j:rt $type $destination: only type \"text\" is supported."
    }
  }
}

######################################################################
# j:rt:textfonts {style font}... - set fonts for text widget
#   style is one of {rm it bf bi tt hl}; font is an X font
#   future versions will support PostScript, TeX, maybe canvas, etc.
######################################################################

proc j:rt:textfonts { args } {
  case [j:rt:type] in {
    {text} {			;# output to a text widget
      set w [j:rt:destination]
      foreach pair $args {
        set tag "j:rt:[lindex $pair 0]"
        set font [lindex $pair 1]
        catch {$w tag configure $tag -font $font}
        if {$tag == "j:rt:rm"} {
          # configure text widget itself, so it sizes properly
          catch {$w configure -font $font}
        }
      }
    }
    default {
      tkerror "j:rt:textfonts called when type of output was [j:rt:type]."
    }
  }
}

######################################################################
# j:rt:type - return type of current rich text destination (text, TeX)
######################################################################

proc j:rt:type {} {
  global j_rt
  
  if { (! [info exists j_rt(type)])} {
    # this might be considered an error
    return {}
  } else {
    return $j_rt(type)
  }
}

######################################################################
# j:rt:destination - return current rich text destination (widget, file)
######################################################################

proc j:rt:destination {} {
  global j_rt
  
  if { (! [info exists j_rt(destination)]) } {
    # this might be considered an error
    return {}
  } else {
    return $j_rt(destination)
  }
}

######################################################################
# j:rt:done - finish writing rich text (clear vars, close files)
######################################################################

proc j:rt:done {} {
  global j_rt

  # to start, would close files if appropriate
  
  set j_rt(type) {}
  set j_rt(destination) {}
}
  
######################################################################
# CREATE PROCEDURES FOR:
# j:rt:rm text - write rich text (roman)
# j:rt:it text - write rich text (italic)
# j:rt:bf text - write rich text (bold face)
# j:rt:bi text - write rich text (bisexual)
# j:rt:tt text - write rich text (typewriter - monospaced)
# j:rt:hl text - write rich text (`headline' - larger bold)
######################################################################

set tmp_body {
  set type [j:rt:type]
  
  case $type in {
    {text} {			;# output to a text widget
      j:tagged_insert [j:rt:destination] $text $tag
    }
    default {
      tkerror "j:rt type \"$type\" is not supported."
    }
  }
}

foreach style {rm it bf bi tt hl} {
  proc j:rt:$style {text} "  set tag j:rt:$style$tmp_body"
}

######################################################################
# j:rt:tab - tab in rich text
######################################################################

proc j:rt:tab {} {
  j:rt:rm "\t"
}

######################################################################
# j:rt:cr - line break in rich text
######################################################################

proc j:rt:cr {} {
  j:rt:rm "\n"
}

######################################################################
# j:rt:par - paragraph break in rich text
######################################################################

proc j:rt:par {} {
  j:rt:rm "\n\n"
}

######################################################################
# j:rt:mkabbrevs - make shorter convenience procs, for text-intensive apps
######################################################################

# this creates shorter aliases rm, it, bf, bi, tt, hl, tab, cr, and
# par identical to the corresponding procedures starting with "j:rt:"

proc j:rt:mkabbrevs {} {
  foreach proc {rm it bf bi tt hl tab cr par} {
    proc $proc [info args j:rt:$proc] [info body j:rt:$proc]
  }
}

######################################################################
# rm - dummy do-nothing procedure to prevent unknown from calling /bin/rm
#   if you forget to j:rt:mkabbrevs
######################################################################

proc rm {args} {
  tkerror "Called `rm' without calling `j:rt:mkabbrevs'."
}
