static char rcsid[]="$Id: tkDRectOval.c,v 1.1 94/02/09 10:14:02 mangin Exp $";
/* 
 * tkDRectOval.c --
 *
 *	This file implements drectangle and doval items for canvas
 *	widgets.
 *
 * Copyright (c) 1991-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <stdio.h>
#include <tk.h>
#include <tkInt.h>
#include "tkCanvas.h"
#include <tkConfig.h>
     
/*
 * The structure below defines the record for each drectangle/doval item.
 */

typedef struct DRectOvalItem  {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    double bbox[4];		/* Coordinates of bounding box for drectangle
				 * or doval (x1, y1, x2, y2).  Item includes
				 * x1 and x2 but not y1 and y2. */
    int width;			/* Width of outline. */
    XColor *outlineColor;	/* Color for outline. */
    XColor *fillColor;		/* Color for filling drectangle/doval. */
    Pixmap fillStipple;		/* Stipple bitmap for filling item. */
    GC outlineGC;		/* Graphics context for outline. */
    GC fillGC;			/* Graphics context for filling item. */
    int dashNumber;		/* Number of elements in dashList */
    char *dashList;		/* Specifies dash style. Malloc'd, a value of */
      				/* NULL means no dashes */
} DRectOvalItem;

/*
 * Information used for parsing configuration specs:
 */

static int		ParseDashes _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, Tk_Window tkwin, char *value,
			    char *recordPtr, int offset));
static char *		PrintDashes _ANSI_ARGS_((ClientData clientData,
			    Tk_Window tkwin, char *recordPtr, int offset,
			    Tcl_FreeProc **freeProcPtr));
     
static Tk_CustomOption dashesOption = {ParseDashes, PrintDashes, (ClientData) NULL};
     
static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DRectOvalItem, fillColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-outline", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(DRectOvalItem, outlineColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DRectOvalItem, fillStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_PIXELS, "-width", (char *) NULL, (char *) NULL,
	"1", Tk_Offset(DRectOvalItem, width), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-dashes", (char *) NULL, (char *) NULL,
       "", Tk_Offset(DRectOvalItem, dashList),
       TK_CONFIG_DONT_SET_DEFAULT, &dashesOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */

static void		ComputeDRectOvalBbox _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    DRectOvalItem *DRectOvalPtr));
static int		ConfigureDRectOval _ANSI_ARGS_((
			    Tk_Canvas *canvasPtr, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateDRectOval _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    struct Tk_Item *itemPtr, int argc, char **argv));
static void		DeleteDRectOval _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr));
static void		DisplayDRectOval _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Drawable dst));
static int		OvalToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *areaPtr));
static double		OvalToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *pointPtr));
static int		DRectOvalCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, int argc, char **argv));
static int		DRectOvalToPostscript _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Tk_PostscriptInfo *psInfoPtr));
static int		RectToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *areaPtr));
static double		RectToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *pointPtr));
static void		ScaleDRectOval _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static void		TranslateDRectOval _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * The structures below defines the drectangle and doval item types
 * by means of procedures that can be invoked by generic item code.
 */

Tk_ItemType TkDRectangleType = {
    "drectangle",			/* name */
    sizeof(DRectOvalItem),		/* itemSize */
    CreateDRectOval,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureDRectOval,			/* configureProc */
    DRectOvalCoords,			/* coordProc */
    DeleteDRectOval,			/* deleteProc */
    DisplayDRectOval,			/* displayProc */
    0,					/* alwaysRedraw */
    RectToPoint,			/* pointProc */
    RectToArea,				/* areaProc */
    DRectOvalToPostscript,		/* postscriptProc */
    ScaleDRectOval,			/* scaleProc */
    TranslateDRectOval,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};

Tk_ItemType TkDOvalType = {
    "doval",				/* name */
    sizeof(DRectOvalItem),		/* itemSize */
    CreateDRectOval,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureDRectOval,			/* configureProc */
    DRectOvalCoords,			/* coordProc */
    DeleteDRectOval,			/* deleteProc */
    DisplayDRectOval,			/* displayProc */
    0,					/* alwaysRedraw */
    OvalToPoint,			/* pointProc */
    OvalToArea,				/* areaProc */
    DRectOvalToPostscript,		/* postscriptProc */
    ScaleDRectOval,			/* scaleProc */
    TranslateDRectOval,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* cursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};


/***************************************************************
 ****          Canvas drectoval item creation               ****
 ***************************************************************/

void DRectOvalItemTypeCreate()
{
  Tk_CreateItemType(&TkDRectangleType);
  Tk_CreateItemType(&TkDOvalType);
}

/*
 *--------------------------------------------------------------
 *
 * CreateDRectOval --
 *
 *	This procedure is invoked to create a new drectangle
 *	or oval item in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized, so it can be safely freed by the
 *	caller.
 *
 * Side effects:
 *	A new drectangle or oval item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateDRectOval(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing drectangle. */
{
    register DRectOvalItem *DRectOvalPtr = (DRectOvalItem *) itemPtr;

    if (argc < 4) {
	Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
		Tk_PathName(canvasPtr->tkwin), "\" create ",
		itemPtr->typePtr->name, " x1 y1 x2 y2 ?options?",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    DRectOvalPtr->width = 1;
    DRectOvalPtr->outlineColor = NULL;
    DRectOvalPtr->fillColor = NULL;
    DRectOvalPtr->fillStipple = None;
    DRectOvalPtr->outlineGC = None;
    DRectOvalPtr->fillGC = None;
    DRectOvalPtr->dashNumber = 0;
    DRectOvalPtr->dashList = (char *)0;

    /*
     * Process the arguments to fill in the item record.
     */

    if ((TkGetCanvasCoord(canvasPtr, argv[0], &DRectOvalPtr->bbox[0]) != TCL_OK)
	    || (TkGetCanvasCoord(canvasPtr, argv[1],
		&DRectOvalPtr->bbox[1]) != TCL_OK)
	    || (TkGetCanvasCoord(canvasPtr, argv[2],
		    &DRectOvalPtr->bbox[2]) != TCL_OK)
	    || (TkGetCanvasCoord(canvasPtr, argv[3],
		    &DRectOvalPtr->bbox[3]) != TCL_OK)) {
	return TCL_ERROR;
    }

    if (ConfigureDRectOval(canvasPtr, itemPtr, argc-4, argv+4, 0) != TCL_OK) {
	DeleteDRectOval(canvasPtr, itemPtr);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DRectOvalCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on drectangles and ovals.  See the user documentation
 *	for details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
DRectOvalCoords(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    register DRectOvalItem *DRectOvalPtr = (DRectOvalItem *) itemPtr;
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE];
    char c2[TCL_DOUBLE_SPACE], c3[TCL_DOUBLE_SPACE];

    if (argc == 0) {
	Tcl_PrintDouble(canvasPtr->interp, DRectOvalPtr->bbox[0], c0);
	Tcl_PrintDouble(canvasPtr->interp, DRectOvalPtr->bbox[1], c1);
	Tcl_PrintDouble(canvasPtr->interp, DRectOvalPtr->bbox[2], c2);
	Tcl_PrintDouble(canvasPtr->interp, DRectOvalPtr->bbox[3], c3);
	Tcl_AppendResult(canvasPtr->interp, c0, " ", c1, " ",
		c2, " ", c3, (char *) NULL);
    } else if (argc == 4) {
	if ((TkGetCanvasCoord(canvasPtr, argv[0],
		    &DRectOvalPtr->bbox[0]) != TCL_OK)
		|| (TkGetCanvasCoord(canvasPtr, argv[1],
		    &DRectOvalPtr->bbox[1]) != TCL_OK)
		|| (TkGetCanvasCoord(canvasPtr, argv[2],
			&DRectOvalPtr->bbox[2]) != TCL_OK)
		|| (TkGetCanvasCoord(canvasPtr, argv[3],
			&DRectOvalPtr->bbox[3]) != TCL_OK)) {
	    return TCL_ERROR;
	}
	ComputeDRectOvalBbox(canvasPtr, DRectOvalPtr);
    } else {
	sprintf(canvasPtr->interp->result,
		"wrong # coordinates:  expected 0 or 4, got %d",
		argc);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureDRectOval --
 *
 *	This procedure is invoked to configure various aspects
 *	of a drectangle or oval item, such as its border and
 *	background colors.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureDRectOval(canvasPtr, itemPtr, argc, argv, flags)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Drectangle item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    register DRectOvalItem *DRectOvalPtr = (DRectOvalItem *) itemPtr;
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;

    if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
	    configSpecs, argc, argv, (char *) DRectOvalPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    if (DRectOvalPtr->outlineColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = DRectOvalPtr->outlineColor->pixel;
	gcValues.cap_style = CapProjecting;
	if (DRectOvalPtr->width < 0) {
	    DRectOvalPtr->width = 1;
	}
	gcValues.line_width = DRectOvalPtr->width;
	mask = GCForeground|GCCapStyle|GCLineWidth;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    
    if (DRectOvalPtr->outlineGC != None) {
	Tk_FreeGC(canvasPtr->display, DRectOvalPtr->outlineGC);
    }
    DRectOvalPtr->outlineGC = newGC;

    if (DRectOvalPtr->fillColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = DRectOvalPtr->fillColor->pixel;
	if (DRectOvalPtr->fillStipple != None) {
	    gcValues.stipple = DRectOvalPtr->fillStipple;
	    gcValues.fill_style = FillStippled;
	    mask = GCForeground|GCStipple|GCFillStyle;
	} else {
	    mask = GCForeground;
	}
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (DRectOvalPtr->fillGC != None) {
	Tk_FreeGC(canvasPtr->display, DRectOvalPtr->fillGC);
    }
    DRectOvalPtr->fillGC = newGC;
    
    ComputeDRectOvalBbox(canvasPtr, DRectOvalPtr);

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteDRectOval --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a drectangle or oval item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteDRectOval(canvasPtr, itemPtr)
    Tk_Canvas *canvasPtr;		/* Info about overall widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
{
    register DRectOvalItem *DRectOvalPtr = (DRectOvalItem *) itemPtr;

    if (DRectOvalPtr->outlineColor != NULL) {
	Tk_FreeColor(DRectOvalPtr->outlineColor);
    }
    if (DRectOvalPtr->fillColor != NULL) {
	Tk_FreeColor(DRectOvalPtr->fillColor);
    }
    if (DRectOvalPtr->fillStipple != None) {
	Tk_FreeBitmap(canvasPtr->display, DRectOvalPtr->fillStipple);
    }
    if (DRectOvalPtr->outlineGC != None) {
	Tk_FreeGC(canvasPtr->display, DRectOvalPtr->outlineGC);
    }
    if (DRectOvalPtr->fillGC != None) {
	Tk_FreeGC(canvasPtr->display, DRectOvalPtr->fillGC);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeDRectOvalBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a drectangle
 *	or oval.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
ComputeDRectOvalBbox(canvasPtr, DRectOvalPtr)
    Tk_Canvas *canvasPtr;		/* Canvas that contains item. */
    register DRectOvalItem *DRectOvalPtr;	/* Item whose bbox is to be
					 * recomputed. */
{
    int bloat;

    /*
     * Make sure that the first coordinates are the lowest ones.
     */

    if (DRectOvalPtr->bbox[1] > DRectOvalPtr->bbox[3]) {
	double tmp;
	tmp = DRectOvalPtr->bbox[3];
	DRectOvalPtr->bbox[3] = DRectOvalPtr->bbox[1];
	DRectOvalPtr->bbox[1] = tmp;
    }
    if (DRectOvalPtr->bbox[0] > DRectOvalPtr->bbox[2]) {
	double tmp;
	tmp = DRectOvalPtr->bbox[2];
	DRectOvalPtr->bbox[2] = DRectOvalPtr->bbox[0];
	DRectOvalPtr->bbox[0] = tmp;
    }

    bloat = (DRectOvalPtr->width+1)/2 + 1;
    DRectOvalPtr->header.x1 = DRectOvalPtr->bbox[0] - bloat;
    DRectOvalPtr->header.y1 = DRectOvalPtr->bbox[1] - bloat;
    DRectOvalPtr->header.x2 = DRectOvalPtr->bbox[2] + bloat;
    DRectOvalPtr->header.y2 = DRectOvalPtr->bbox[3] + bloat;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayDRectOval --
 *
 *	This procedure is invoked to draw a drectangle or oval
 *	item in a given drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

static void
DisplayDRectOval(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
    register DRectOvalItem *DRectOvalPtr = (DRectOvalItem *) itemPtr;
    Display *display = Tk_Display(canvasPtr->tkwin);
    int x1, y1, x2, y2;
    unsigned long valuemask;
    XGCValues savedLineStyle;

    /*
     * Compute the screen coordinates of the bounding box for the item.
     * Make sure that the bbox is at least one pixel large, since some
     * X servers will die if it isn't.
     */

    x1 = SCREEN_X(canvasPtr, DRectOvalPtr->bbox[0]);
    y1 = SCREEN_Y(canvasPtr, DRectOvalPtr->bbox[1]);
    x2 = SCREEN_X(canvasPtr, DRectOvalPtr->bbox[2]);
    y2 = SCREEN_Y(canvasPtr, DRectOvalPtr->bbox[3]);
    if (x2 <= x1) {
	x2 = x1+1;
    }
    if (y2 <= y1) {
	y2 = y1+1;
    }

    /*
     * Display filled part first (if wanted), then outline.  If we're
     * stippling, then modify the stipple offset in the GC.  Be sure to
     * reset the offset when done, since the GC is supposed to be
     * read-only.
     * If dash style != solid, then modify the GC dash style.
     * Be sure to reset the dash style when done, since the
     * GC is supposed to be read-only. Note that if an XSetDashes had been
     * performed on the GC before this point, it won't be restored.
     */

    if (DRectOvalPtr->dashNumber > 0) {
      XGCValues newLineStyle;

      valuemask = GCLineStyle;
      XGetGCValues(canvasPtr->display, DRectOvalPtr->outlineGC,
                   valuemask, &savedLineStyle);
      newLineStyle.line_style = LineOnOffDash;
      XChangeGC(canvasPtr->display, DRectOvalPtr->outlineGC,
                valuemask, &newLineStyle);
      XSetDashes(canvasPtr->display, DRectOvalPtr->outlineGC,
                 0, DRectOvalPtr->dashList, DRectOvalPtr->dashNumber);
    }

    if (DRectOvalPtr->fillGC != None) {
	if (DRectOvalPtr->fillStipple != None) {
	    XSetTSOrigin(display, DRectOvalPtr->fillGC,
		    -canvasPtr->drawableXOrigin, -canvasPtr->drawableYOrigin);
	}
	if (DRectOvalPtr->header.typePtr == &TkDRectangleType) {
	    XFillRectangle(display, drawable, DRectOvalPtr->fillGC,
		    x1, y1, (unsigned int) (x2-x1), (unsigned int) (y2-y1));
	} else {
	    XFillArc(display, drawable, DRectOvalPtr->fillGC,
		    x1, y1, (x2-x1), (y2-y1), 0, 360*64);
	}
	if (DRectOvalPtr->fillStipple != None) {
	    XSetTSOrigin(display, DRectOvalPtr->fillGC, 0, 0);
	}
    }
    if (DRectOvalPtr->outlineGC != None) {
	if (DRectOvalPtr->header.typePtr == &TkDRectangleType) {
	    XDrawRectangle(display, drawable, DRectOvalPtr->outlineGC,
		    x1, y1, (x2-x1-1), (y2-y1-1));
	} else {
	    XDrawArc(display, drawable, DRectOvalPtr->outlineGC,
		    x1, y1, (x2-x1-1), (y2-y1-1), 0, 360*64);
	}
    }

    if (DRectOvalPtr->dashNumber > 0) {
      XChangeGC(canvasPtr->display, DRectOvalPtr->outlineGC,
                valuemask, &savedLineStyle);
      /* Here's the place for a XSetDashes ( savedDashes ). */
    }
}

/*
 *--------------------------------------------------------------
 *
 * RectToPoint --
 *
 *	Computes the distance from a given point to a given
 *	rectangle, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are coordPtr[0] and coordPtr[1] is inside the rectangle.  If the
 *	point isn't inside the rectangle then the return value is the
 *	distance from the point to the rectangle.  If itemPtr is filled,
 *	then anywhere in the interior is considered "inside"; if
 *	itemPtr isn't filled, then "inside" means only the area
 *	occupied by the outline.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
RectToPoint(canvasPtr, itemPtr, pointPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    register DRectOvalItem *rectPtr = (DRectOvalItem *) itemPtr;
    double xDiff, yDiff, x1, y1, x2, y2, inc, tmp;

    /*
     * Generate a new larger rectangle that includes the border
     * width, if there is one.
     */

    x1 = rectPtr->bbox[0];
    y1 = rectPtr->bbox[1];
    x2 = rectPtr->bbox[2];
    y2 = rectPtr->bbox[3];
    if (rectPtr->outlineGC != None) {
	inc = rectPtr->width/2.0;
	x1 -= inc;
	y1 -= inc;
	x2 += inc;
	y2 += inc;
    }

    /*
     * If the point is inside the rectangle, handle specially:
     * distance is 0 if rectangle is filled, otherwise compute
     * distance to nearest edge of rectangle and subtract width
     * of edge.
     */

    if ((pointPtr[0] >= x1) && (pointPtr[0] < x2)
		&& (pointPtr[1] >= y1) && (pointPtr[1] < y2)) {
	if ((rectPtr->fillGC != None) || (rectPtr->outlineGC == None)) {
	    return 0.0;
	}
	xDiff = pointPtr[0] - x1;
	tmp = x2 - pointPtr[0];
	if (tmp < xDiff) {
	    xDiff = tmp;
	}
	yDiff = pointPtr[1] - y1;
	tmp = y2 - pointPtr[1];
	if (tmp < yDiff) {
	    yDiff = tmp;
	}
	if (yDiff < xDiff) {
	    xDiff = yDiff;
	}
	xDiff -= rectPtr->width;
	if (xDiff < 0.0) {
	    return 0.0;
	}
	return xDiff;
    }

    /*
     * Point is outside rectangle.
     */

    if (pointPtr[0] < x1) {
	xDiff = x1 - pointPtr[0];
    } else if (pointPtr[0] > x2)  {
	xDiff = pointPtr[0] - x2;
    } else {
	xDiff = 0;
    }

    if (pointPtr[1] < y1) {
	yDiff = y1 - pointPtr[1];
    } else if (pointPtr[1] > y2)  {
	yDiff = pointPtr[1] - y2;
    } else {
	yDiff = 0;
    }

    return hypot(xDiff, yDiff);
}

/*
 *--------------------------------------------------------------
 *
 * OvalToPoint --
 *
 *	Computes the distance from a given point to a given
 *	oval, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are coordPtr[0] and coordPtr[1] is inside the oval.  If the
 *	point isn't inside the oval then the return value is the
 *	distance from the point to the oval.  If itemPtr is filled,
 *	then anywhere in the interior is considered "inside"; if
 *	itemPtr isn't filled, then "inside" means only the area
 *	occupied by the outline.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
OvalToPoint(canvasPtr, itemPtr, pointPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    register DRectOvalItem *ovalPtr = (DRectOvalItem *) itemPtr;
    double width;
    int filled;

    width = ovalPtr->width;
    filled = ovalPtr->fillGC != None;
    if (ovalPtr->outlineGC == None) {
	width = 0.0;
	filled = 1;
    }
    return TkOvalToPoint(ovalPtr->bbox, width, filled, pointPtr);
}

/*
 *--------------------------------------------------------------
 *
 * RectToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangle.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by rectPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
RectToArea(canvasPtr, itemPtr, areaPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against rectangle. */
    double *areaPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    register DRectOvalItem *rectPtr = (DRectOvalItem *) itemPtr;
    double halfWidth;

    halfWidth = rectPtr->width/2.0;
    if (rectPtr->outlineGC == None) {
	halfWidth = 0.0;
    }

    if ((areaPtr[2] <= (rectPtr->bbox[0] - halfWidth))
	    || (areaPtr[0] >= (rectPtr->bbox[2] + halfWidth))
	    || (areaPtr[3] <= (rectPtr->bbox[1] - halfWidth))
	    || (areaPtr[1] >= (rectPtr->bbox[3] + halfWidth))) {
	return -1;
    }
    if ((rectPtr->fillGC == None) && (rectPtr->outlineGC != None)
	    && (areaPtr[0] >= (rectPtr->bbox[0] + halfWidth))
	    && (areaPtr[1] >= (rectPtr->bbox[1] + halfWidth))
	    && (areaPtr[2] <= (rectPtr->bbox[2] - halfWidth))
	    && (areaPtr[3] <= (rectPtr->bbox[3] - halfWidth))) {
	return -1;
    }
    if ((areaPtr[0] <= (rectPtr->bbox[0] - halfWidth))
	    && (areaPtr[1] <= (rectPtr->bbox[1] - halfWidth))
	    && (areaPtr[2] >= (rectPtr->bbox[2] + halfWidth))
	    && (areaPtr[3] >= (rectPtr->bbox[3] + halfWidth))) {
	return 1;
    }
    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * OvalToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by rectPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
OvalToArea(canvasPtr, itemPtr, areaPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against oval. */
    double *areaPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    register DRectOvalItem *ovalPtr = (DRectOvalItem *) itemPtr;
    double oval[4], halfWidth;
    int result;

    /*
     * Expand the oval to include the width of the outline, if any.
     */

    halfWidth = ovalPtr->width/2.0;
    if (ovalPtr->outlineGC == None) {
	halfWidth = 0.0;
    }
    oval[0] = ovalPtr->bbox[0] - halfWidth;
    oval[1] = ovalPtr->bbox[1] - halfWidth;
    oval[2] = ovalPtr->bbox[2] + halfWidth;
    oval[3] = ovalPtr->bbox[3] + halfWidth;

    result = TkOvalToArea(oval, areaPtr);

    /*
     * If the rectangle appears to overlap the oval and the oval
     * isn't filled, do one more check to see if perhaps all four
     * of the rectangle's corners are totally inside the oval's
     * unfilled center, in which case we should return "outside".
     */

    if ((result == 0) && (ovalPtr->outlineGC != None)
	    && (ovalPtr->fillGC == None)) {
	double centerX, centerY, width, height;
	double xDelta1, yDelta1, xDelta2, yDelta2;

	centerX = (ovalPtr->bbox[0] + ovalPtr->bbox[2])/2.0;
	centerY = (ovalPtr->bbox[1] + ovalPtr->bbox[3])/2.0;
	width = (ovalPtr->bbox[2] - ovalPtr->bbox[0])/2.0 - halfWidth;
	height = (ovalPtr->bbox[3] - ovalPtr->bbox[1])/2.0 - halfWidth;
	xDelta1 = (areaPtr[0] - centerX)/width;
	xDelta1 *= xDelta1;
	yDelta1 = (areaPtr[1] - centerY)/height;
	yDelta1 *= yDelta1;
	xDelta2 = (areaPtr[2] - centerX)/width;
	xDelta2 *= xDelta2;
	yDelta2 = (areaPtr[3] - centerY)/height;
	yDelta2 *= yDelta2;
	if (((xDelta1 + yDelta1) < 1.0)
		&& ((xDelta1 + yDelta2) < 1.0)
		&& ((xDelta2 + yDelta1) < 1.0)
		&& ((xDelta2 + yDelta2) < 1.0)) {
	    return -1;
	}
    }
    return result;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleDRectOval --
 *
 *	This procedure is invoked to rescale a rectangle or oval
 *	item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The rectangle or oval referred to by itemPtr is rescaled
 *	so that the following transformation is applied to all
 *	point coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

static void
ScaleDRectOval(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas *canvasPtr;		/* Canvas containing rectangle. */
    Tk_Item *itemPtr;			/* Rectangle to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    register DRectOvalItem *DRectOvalPtr = (DRectOvalItem *) itemPtr;

    DRectOvalPtr->bbox[0] = originX + scaleX*(DRectOvalPtr->bbox[0] - originX);
    DRectOvalPtr->bbox[1] = originY + scaleY*(DRectOvalPtr->bbox[1] - originY);
    DRectOvalPtr->bbox[2] = originX + scaleX*(DRectOvalPtr->bbox[2] - originX);
    DRectOvalPtr->bbox[3] = originY + scaleY*(DRectOvalPtr->bbox[3] - originY);
    ComputeDRectOvalBbox(canvasPtr, DRectOvalPtr);
}

/*
 *--------------------------------------------------------------
 *
 * TranslateDRectOval --
 *
 *	This procedure is called to move a rectangle or oval by a
 *	given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the rectangle or oval is offset by
 *	(xDelta, yDelta), and the bounding box is updated in the
 *	generic part of the item structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateDRectOval(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    register DRectOvalItem *DRectOvalPtr = (DRectOvalItem *) itemPtr;

    DRectOvalPtr->bbox[0] += deltaX;
    DRectOvalPtr->bbox[1] += deltaY;
    DRectOvalPtr->bbox[2] += deltaX;
    DRectOvalPtr->bbox[3] += deltaY;
    ComputeDRectOvalBbox(canvasPtr, DRectOvalPtr);
}



/*
 *--------------------------------------------------------------
 *
 * ParseDashes
 *
 *	This procedure is called back during option parsing to
 *	parse dash information
 *
 * Results:
 *	The return value is a standard Tcl result:  TCL_OK means
 *	that the arrow shape information was parsed ok, and
 *	TCL_ERROR means it couldn't be parsed.
 *
 * Side effects:
 *	Dash information in recordPtr is updated.
 *
 *--------------------------------------------------------------
 */

static int
ParseDashes(clientData, interp, tkwin, value, recordPtr, offset)
    ClientData clientData;	/* Not used. */
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Window tkwin;		/* Not used. */
    char *value;		/* Textual specification of arrow shape. */
    char *recordPtr;		/* Pointer to item record in which to
				 * store arrow information. */
    int offset;			/* Offset of shape information in widget
				 * record. */
{
  DRectOvalItem *DRectOvalPtr = (DRectOvalItem *) recordPtr;
  int argc;
  char **argv = NULL;
  
  if (offset != Tk_Offset(DRectOvalItem, dashList)) {
    panic("ParseDashes received bogus offset");
  }
  
  if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
  syntaxError:
    Tcl_ResetResult(interp);
    Tcl_AppendResult(interp, "bad dash list \"", value,
		     "\": must be even length list of numbers", (char *) NULL);
    if (argv != NULL) {
      ckfree((char *) argv);
    }
    return TCL_ERROR;
  }
  if (argc%2 != 0) {
    goto syntaxError;
  }
  
  if (DRectOvalPtr->dashList != (char *)0)
    free(DRectOvalPtr->dashList);
  DRectOvalPtr->dashList = (char *)0;
  DRectOvalPtr->dashNumber = argc;
  
  if (argc > 0) {
    /**  fill in the dashList array  **/
    int ibuf, i;
    DRectOvalPtr->dashList = (char *)malloc(argc * sizeof(char));
    for (i = 0; i < argc; i++) {
      if (Tcl_GetInt(interp, argv[i], &ibuf) != TCL_OK) {
	free(DRectOvalPtr->dashList);
	DRectOvalPtr->dashList = (char *)0;
	goto syntaxError;
      }
      DRectOvalPtr->dashList[i] = (char)ibuf;
    }
  }
  
  ckfree((char *) argv);
  
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PrintDashes
 *
 *	This procedure is a callback invoked by the configuration
 *	code to return a printable value describing a dash list.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static char *
PrintDashes(clientData, tkwin, recordPtr, offset, freeProcPtr)
    ClientData clientData;	/* Not used. */
    Tk_Window tkwin;		/* Window associated with DRectOvalPtr's widget. */
    char *recordPtr;		/* Pointer to item record containing current
				 * shape information. */
    int offset;			/* Offset of arrow information in record. */
    Tcl_FreeProc **freeProcPtr;	/* Store address of procedure to call to
				 * free string here. */
{
    DRectOvalItem *DRectOvalPtr = (DRectOvalItem *) recordPtr;
    char *buffer;

    if (DRectOvalPtr->dashNumber == 0) {
      buffer = ckalloc(1);
      *buffer = '\0';
    } else {
      int i;
      
      buffer = ckalloc((4*DRectOvalPtr->dashNumber+1) * sizeof(char));
      sprintf(buffer, "%d", DRectOvalPtr->dashList[0]);
      for (i = 1; i < DRectOvalPtr->dashNumber; i++)
	sprintf(buffer + strlen(buffer), " %d", DRectOvalPtr->dashList[i]);
    }
    *freeProcPtr = (Tcl_FreeProc *) free;
    return buffer;
}

/*
 *--------------------------------------------------------------
 *
 * DRectOvalToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	rectangle and oval items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in canvasPtr->interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	rectangle is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
DRectOvalToPostscript(canvasPtr, itemPtr, psInfoPtr)
    Tk_Canvas *canvasPtr;		/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    Tk_PostscriptInfo *psInfoPtr;	/* Information about the Postscript;
					 * must be passed back to Postscript
					 * utility procedures. */
{
    char pathCmd[500], string[100];
    register DRectOvalItem *DRectOvalPtr = (DRectOvalItem *) itemPtr;
    double y1, y2;

    y1 = TkCanvPsY(psInfoPtr, DRectOvalPtr->bbox[1]);
    y2 = TkCanvPsY(psInfoPtr, DRectOvalPtr->bbox[3]);

    /*
     * Generate a string that creates a path for the rectangle or oval.
     * This is the only part of the procedure's code that is type-
     * specific.
     */


    if (DRectOvalPtr->header.typePtr == &TkDRectangleType) {
	sprintf(pathCmd, "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto %.15g 0 rlineto closepath\n",
		DRectOvalPtr->bbox[0], y1,
		DRectOvalPtr->bbox[2]-DRectOvalPtr->bbox[0], y2-y1,
		DRectOvalPtr->bbox[0]-DRectOvalPtr->bbox[2]);
    } else {
	sprintf(pathCmd, "matrix currentmatrix\n%.15g %.15g translate %.15g %.15g scale 1 0 moveto 0 0 1 0 360 arc\nsetmatrix\n",
		(DRectOvalPtr->bbox[0] + DRectOvalPtr->bbox[2])/2, (y1 + y2)/2,
		(DRectOvalPtr->bbox[2] - DRectOvalPtr->bbox[0])/2, (y1 - y2)/2);
    }

    /*
     * First draw the filled area of the rectangle.
     */

    if (DRectOvalPtr->fillColor != NULL) {
	Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
	if (TkCanvPsColor(canvasPtr, psInfoPtr, DRectOvalPtr->fillColor)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
	if (DRectOvalPtr->fillStipple != None) {
	    if (TkCanvPsStipple(canvasPtr, psInfoPtr,
		    DRectOvalPtr->fillStipple, 1) != TCL_OK) {
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(canvasPtr->interp, "fill\n", (char *) NULL);
	}
    }

    /*
     * Now draw the outline, if there is one.
     */

    if (DRectOvalPtr->outlineColor != NULL) {
	Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
	sprintf(string, "%d setlinewidth", DRectOvalPtr->width);
	Tcl_AppendResult(canvasPtr->interp, string,
		" 0 setlinejoin 2 setlinecap\n", (char *) NULL);
	if (TkCanvPsColor(canvasPtr, psInfoPtr, DRectOvalPtr->outlineColor)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
	if (DRectOvalPtr->dashNumber > 0) {
	  char buffer[256];
	  int i;
	  sprintf(buffer, "[%d", DRectOvalPtr->dashList[0]);
	  for (i = 1; i < DRectOvalPtr->dashNumber; i++)
	    sprintf(buffer + strlen(buffer), " %d", DRectOvalPtr->dashList[i]);
	  strcat(buffer, "] 0 setdash\n");
	  Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
	}
	Tcl_AppendResult(canvasPtr->interp, "stroke\n", (char *) NULL);
      }
    return TCL_OK;
  }
