static char rcsid[]="$Id: tkSwitch.c,v 1.2 94/02/14 14:53:17 mangin Exp $";
/* 
 * tkSwitch.c --
 *
 *	This file contains the Tk_SwitchInfo procedure.
 *
 * Copyright (c) 1990-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <tkConfig.h>
#include <tk.h>

/*
 * Values for "flags" field of Tk_ConfigSpec structures.  Be sure
 * to coordinate these values with those defined in tk.h
 * (TK_CONFIG_COLOR_ONLY, etc.).  There must not be overlap!
 *
 * INIT -		Non-zero means (char *) things have been
 *			converted to Tk_Uid's.
 */

#define INIT		0x20

/*
 * Forward declarations for procedures defined later in this file:
 */

static Tk_ConfigSpec *  FindConfigSpec _ANSI_ARGS_ ((Tcl_Interp *interp,
                            Tk_ConfigSpec *specs, char *argvName,
                            int needFlags, int hateFlags));
static char *		FormatSwitchInfo _ANSI_ARGS_ ((Tcl_Interp *interp,
			    Tk_Window tkwin, Tk_ConfigSpec *specPtr,
			    char *widgRec));

/*
 *--------------------------------------------------------------
 *
 * Tk_SwitchInfo --
 *
 *	Return information about the switches
 *	for a canvas item, and their current values.
 *
 * Results:
 *	Always returns TCL_OK.  Interp->result will be modified
 *	hold a description of either a single configuration option
 *	available for "widgRec" via "specs", or all the configuration
 *	options available.  In the "all" case, the result will
 *	available for "widgRec" via "specs".  The result will
 *	be a list, each of whose entries describes one option.
 *	Each entry will itself be a list containing the option's
 *	name for use on command lines, and current value. The list is
 *      empty if current value is empty. For options that are
 *      synonyms, the list will be empty. If the
 *	"name" argument is non-NULL, then the only information
 *	returned is that for the named argument (i.e. the corresponding
 *	entry in the overall list is returned).
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
Tk_SwitchInfo(interp, tkwin, specs, widgRec, argvName, flags)
    Tcl_Interp *interp;		/* Interpreter for error reporting. */
    Tk_Window tkwin;		/* Window corresponding to widgRec. */
    Tk_ConfigSpec *specs;	/* Describes legal options. */
    char *widgRec;		/* Record whose fields contain current
				 * values for options. */
    char *argvName;		/* If non-NULL, indicates a single option
				 * whose info is to be returned.  Otherwise
				 * info is returned for all options. */
    int flags;			/* Used to specify additional flags
				 * that must be present in config specs
				 * for them to be considered. */
{
    register Tk_ConfigSpec *specPtr;
    int needFlags, hateFlags;
    char *list;
    char *leader = "";
    
    needFlags = flags & ~(TK_CONFIG_USER_BIT - 1);
    if (Tk_GetColorModel(tkwin) != TK_COLOR) {
	hateFlags = TK_CONFIG_COLOR_ONLY;
    } else {
	hateFlags = TK_CONFIG_MONO_ONLY;
    }

    /*
     * If information is only wanted for a single configuration
     * spec, then handle that one spec specially.
     */

    Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
    if (argvName != NULL) {
	specPtr = FindConfigSpec(interp, specs, argvName, needFlags,
		hateFlags);
	if (specPtr == NULL) {
	    return TCL_ERROR;
	}
	interp->result = FormatSwitchInfo(interp, tkwin, specPtr, widgRec);
	interp->freeProc = TCL_DYNAMIC;
	return TCL_OK;
    }

    /*
     * Loop through all the specs, creating a big list with all
     * their information.
     */

    for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	if ((argvName != NULL) && (specPtr->argvName != argvName)) {
	    continue;
	}
	if (((specPtr->specFlags & needFlags) != needFlags)
		|| (specPtr->specFlags & hateFlags)) {
	    continue;
	}
	if (specPtr->argvName == NULL) {
	    continue;
	}
	list = FormatSwitchInfo(interp, tkwin, specPtr, widgRec);
	Tcl_AppendResult(interp, leader, list, (char *) NULL);
	ckfree(list);
	leader = " ";
      }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * FormatSwitchInfo --
 *
 *	Create a valid Tcl list holding the switches information
 *	for a single switch, or nothing.
 *
 * Results:
 *	A Tcl list, dynamically allocated.  The caller is expected to
 *	arrange for this list to be freed eventually.
 *
 * Side effects:
 *	Memory is allocated.
 *
 *--------------------------------------------------------------
 */

static char *
FormatSwitchInfo(interp, tkwin, specPtr, widgRec)
    Tcl_Interp *interp;			/* Interpreter to use for things
					 * like floating-point precision. */
    Tk_Window tkwin;			/* Window corresponding to widget. */
    register Tk_ConfigSpec *specPtr;	/* Pointer to information describing
					 * option. */
    char *widgRec;			/* Pointer to record holding current
					 * values of info for widget. */
{
  char *argv[2], *ptr, *result;
  char buffer[200];
  Tcl_FreeProc *freeProc = (Tcl_FreeProc *) NULL;

  argv[0] = specPtr->argvName;
  argv[1] = "";
  if (specPtr->type == TK_CONFIG_SYNONYM) {
    return argv[1];
  }
  ptr = widgRec + specPtr->offset;
  switch (specPtr->type) {
  case TK_CONFIG_BOOLEAN:
    if (*((int *) ptr) == 0) {
      argv[1] = "0";
    } else {
      argv[4] = "1";
    }
    break;
  case TK_CONFIG_INT:
    sprintf(buffer, "%d", *((int *) ptr));
    argv[1] = buffer;
    break;
  case TK_CONFIG_DOUBLE:
    Tcl_PrintDouble(interp, *((double *) ptr), buffer);
    argv[1] = buffer;
    break;
  case TK_CONFIG_STRING:
    argv[1] = (*(char **) ptr);
    break;
  case TK_CONFIG_UID: {
    Tk_Uid uid = *((Tk_Uid *) ptr);
    if (uid != NULL) {
      argv[1] = uid;
    }
    break;
  }
  case TK_CONFIG_COLOR: {
    XColor *colorPtr = *((XColor **) ptr);
    if (colorPtr != NULL) {
      argv[1] = Tk_NameOfColor(colorPtr);
    }
    break;
  }
  case TK_CONFIG_FONT: {
    XFontStruct *fontStructPtr = *((XFontStruct **) ptr);
    if (fontStructPtr != NULL) {
      argv[1] = Tk_NameOfFontStruct(fontStructPtr);
    }
    break;
  }
  case TK_CONFIG_BITMAP: {
    Pixmap pixmap = *((Pixmap *) ptr);
    if (pixmap != None) {
      argv[1] = Tk_NameOfBitmap(Tk_Display(tkwin), pixmap);
    }
    break;
  }
  case TK_CONFIG_BORDER: {
    Tk_3DBorder border = *((Tk_3DBorder *) ptr);
    if (border != NULL) {
      argv[1] = Tk_NameOf3DBorder(border);
    }
    break;
  }
  case TK_CONFIG_RELIEF:
    argv[1] = Tk_NameOfRelief(*((int *) ptr));
    break;
  case TK_CONFIG_CURSOR:
  case TK_CONFIG_ACTIVE_CURSOR: {
    Cursor cursor = *((Cursor *) ptr);
    if (cursor != None) {
      argv[1] = Tk_NameOfCursor(Tk_Display(tkwin), cursor);
    }
    break;
  }
  case TK_CONFIG_JUSTIFY:
    argv[1] = Tk_NameOfJustify(*((Tk_Justify *) ptr));
    break;
  case TK_CONFIG_ANCHOR:
    argv[1] = Tk_NameOfAnchor(*((Tk_Anchor *) ptr));
    break;
  case TK_CONFIG_CAP_STYLE:
    argv[1] = Tk_NameOfCapStyle(*((int *) ptr));
    break;
  case TK_CONFIG_JOIN_STYLE:
    argv[1] = Tk_NameOfJoinStyle(*((int *) ptr));
    break;
  case TK_CONFIG_PIXELS:
    sprintf(buffer, "%d", *((int *) ptr));
    argv[1] = buffer;
    break;
  case TK_CONFIG_MM:
    Tcl_PrintDouble(interp, *((double *) ptr), buffer);
    argv[1] = buffer;
    break;
  case TK_CONFIG_WINDOW: {
    Tk_Window tkwin;
    
    tkwin = *((Tk_Window *) ptr);
    if (tkwin != NULL) {
      argv[1] = Tk_PathName(tkwin);
    }
    break;
  }
  case TK_CONFIG_CUSTOM:
    argv[1] =
      (*specPtr->customPtr->printProc)(specPtr->customPtr->clientData, tkwin, widgRec,
				       specPtr->offset, &freeProc);
    break;
  default: 
    argv[1] = "?? unknown type ??";
  }

  if ((argv[1] == NULL) || (*argv[1] == '\0')) {
    result = "";
  } else {
    result = Tcl_Merge(2, argv);
  }
  
  if (freeProc != NULL) {
    if (freeProc == (Tcl_FreeProc *) free) {
      ckfree(argv[1]);
    } else {
      (*freeProc)(argv[1]);
    }
  }
  return result;
}
/*
 *--------------------------------------------------------------
 *
 * FindConfigSpec --
 *
 *      Search through a table of configuration specs, looking for
 *      one that matches a given argvName.
 *
 * Results:
 *      The return value is a pointer to the matching entry, or NULL
 *      if nothing matched.  In that case an error message is left
 *      in interp->result.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

static Tk_ConfigSpec *
FindConfigSpec(interp, specs, argvName, needFlags, hateFlags)
    Tcl_Interp *interp;         /* Used for reporting errors. */
    Tk_ConfigSpec *specs;       /* Pointer to table of configuration
                                 * specifications for a widget. */
    char *argvName;             /* Name (suitable for use in a "config"
                                 * command) identifying particular option. */
    int needFlags;              /* Flags that must be present in matching
                                 * entry. */
    int hateFlags;              /* Flags that must NOT be present in
                                 * matching entry. */
{
    register Tk_ConfigSpec *specPtr;
    register char c;            /* First character of current argument. */
    Tk_ConfigSpec *matchPtr;    /* Matching spec, or NULL. */
    int length;

    c = argvName[1];
    length = strlen(argvName);
    matchPtr = NULL;
    for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
        if (specPtr->argvName == NULL) {
            continue;
        }
        if ((specPtr->argvName[1] != c)
                || (strncmp(specPtr->argvName, argvName, length) != 0)) {
            continue;
        }
        if (((specPtr->specFlags & needFlags) != needFlags)
                || (specPtr->specFlags & hateFlags)) {
            continue;
        }
        if (specPtr->argvName[length] == 0) {
            matchPtr = specPtr;
            goto gotMatch;
        }
        if (matchPtr != NULL) {
            Tcl_AppendResult(interp, "ambiguous option \"", argvName,
                    "\"", (char *) NULL);
            return (Tk_ConfigSpec *) NULL;
        }
        matchPtr = specPtr;
    }

    if (matchPtr == NULL) {
        Tcl_AppendResult(interp, "unknown option \"", argvName,
                "\"", (char *) NULL);
        return (Tk_ConfigSpec *) NULL;
    }

    /*
     * Found a matching entry.  If it's a synonym, then find the
     * entry that it's a synonym for.
     */

    gotMatch:
    specPtr = matchPtr;
    if (specPtr->type == TK_CONFIG_SYNONYM) {
        for (specPtr = specs; ; specPtr++) {
            if (specPtr->type == TK_CONFIG_END) {
                Tcl_AppendResult(interp,
                        "couldn't find synonym for option \"",
                        argvName, "\"", (char *) NULL);
                return (Tk_ConfigSpec *) NULL;
            }
            if ((specPtr->dbName == matchPtr->dbName) 
                    && (specPtr->type != TK_CONFIG_SYNONYM)
                    && ((specPtr->specFlags & needFlags) == needFlags)
                    && !(specPtr->specFlags & hateFlags)) {
                break;
            }
        }
    }
    return specPtr;
}
