#!/usr/bin/wish -f
#
# $Id: TdDebug.tcl,v 3.1 1993/12/06 01:40:13 schmid Exp schmid $
#
# tdebug.tcl - A simple debugger for tcl scripts
# Version 0.3
#
# Copyright (C) 1993 Gregor Schmid 
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software

# Please send bug-reports, suggestions etc. to
#
# 		schmid@fb3-s7.math.tu-berlin.de
#

# This file was written with emacs using Jamie Lokier's folding mode
# That's what the funny # {{{ marks are there for

# {{{ setup global variables

# If we can't use send, .tdebugrc was sourced by
# TdChoose.tcl
if {! [info exists td_priv(scrollbarside)]} {
    if {[file exists "~/.tdebugrc" ]} {source "~/.tdebugrc"}
}

# Setup default values for variables not set from
# .tdebugrc

if {! [info exists td_priv(scrollbarside)] || $td_priv(scrollbarside) != "left"} {
    set td_priv(scrollbarside) right
}

if {! [info exists td_priv(wrap)]} {
    set td_priv(wrap) none
}
trace variable td_priv(wrap) w td_configure

if {! [info exists td_priv(wrapback)]} {
    set td_priv(wrapback) none
}
trace variable td_priv(wrapback) w td_configure

if {! [info exists td_priv(fullnames)]} {
    set td_priv(fullnames) 0
}
trace variable td_priv(fullnames) w td_configure

if {! [info exists td_priv(update)]} {
    set td_priv(update) slow
}

if {! [info exists td_priv(delay)]} {
    set td_priv(delay) 300
}

if {! [info exists td_priv(detail)]} {
    set td_priv(detail) high
}

if {![info exists td_priv(constrainscroll)]} {
    set td_priv(constrainscroll) 1
}

# the execution state of the debugger
set td_priv(state) stop

# miscellaneous
set td_priv(eval) ""
set td_priv(proc) ""
set td_priv(current) ""
set td_priv(listheight) {0 10 0 0}

# }}}
# {{{ debugger procs

# {{{ td_prepareProc

# Prepare a procedure for debugging.
# This is done by inserting calls to td_eval into the body of
# the procedure.
# The original body is preserved and can be restored via
# td_restoreProc.
#
# Args:
# proc		Name of the procedure to debug

proc td_prepareProc proc {
    global td_priv

    set script [info body $proc]
    # check whether proc has allready been debugged
    if {[string match #tdebug* $script]} {
	# if so, use original script and redebug
	set script $td_priv(body.$proc)
    } else {
	set td_priv(body.$proc) $script
    }
    # clear breakpoints for proc
    set td_priv(break.$proc) ""
    set td_priv(result.$proc) ""
    set script [split $script \n]
    set res [td_parseScript $script $proc]
    eval "proc $proc \{[info args $proc]\} \{#tdebug \n\
	    $res \n td_eval $proc end end \{\}\}"
}

# }}}
# {{{ td_restoreProc

# Restore the original body of a procedure that has been modified
# by td_prepareProc
#
# Args:
# proc		Name of the procedure to restore.

proc td_restoreProc proc {
    global td_priv td_Listing td_Result td_Vars

    if {[string match #tdebug* [info body $proc]]} {
	eval "proc $proc \{[info args $proc]\} \{$td_priv(body.$proc)\}"
	unset td_priv(body.$proc)
	unset td_priv(break.$proc)
	unset td_priv(result.$proc)
	# if proc is currently displayed, remove it
	if {$proc == $td_priv(current)} {
	    $td_Listing configure -state normal
	    $td_Listing delete 1.0 end
	    $td_Listing configure -state disabled
	    $td_Result configure -state normal
	    $td_Result delete 0 end
	    $td_Result configure -state disabled
	    $td_Vars delete 0 end
	    set td_priv(proc) ""
	    set td_priv(current) ""
	    if {$td_priv(state) == "waiting"} {
		set td_priv(state) break
		update
    }   }   }
}

# }}}
# {{{ td parseScript

# Parse a tcl script and insert calls to td_eval at appropriate
# places.
# Don't try to parse subexpressions if td_priv(detail)
# is set to low.
#
# Args:
# script   	A list of lines of tcl script
# name		The name of the procedure being prepared
#
# Result:	A string, the modified script

proc td_parseScript {script name} {
    global td_priv
    set switch -1
    set result ""

    for {set lnum 0} {$lnum < [llength $script]} {incr lnum} {
	set line [lindex $script $lnum]
	if {[regexp "^\[ \t\]*(#|$)" $line]}  {
	    # skip empty lines
	    continue
	} elseif {[regexp "^\[ \t\]*\}" $line]} {
	    # closing brace
	    if {$td_priv(detail) != "low"} {
		set line [lindex [td_parseLine $line "" $name [expr $lnum + 1] 0] 1]
	    }
	    set rline $line
	} elseif {[regexp "^\[ \t\]*(break|continue|return)( |\t|;|$)" $line]} {
	    # avoid uplevel with these
	    set rline "td_eval $name [expr $lnum + 1].0 [expr $lnum + 1].end {} ;\
		    $line"
	} elseif { [info complete $line] } {
	    if {$td_priv(detail) != "low"} {
		set line [lindex [td_parseLine $line "" $name [expr $lnum + 1] 0] 1]
	    }
	    set rline "td_eval $name [expr $lnum + 1].0 [expr $lnum + 1].end \{$line\}"
	} elseif {[regexp "^\[ \t\]*bind( |\t|;|$)" $line]} {
	    #don't debug bindings
	    set rline $line
	    while {$lnum < [llength $script] - 1} {
		incr lnum
		append rline \n[lindex $script $lnum]
		if {[info complete rline]} {
		    break
	}   }   } elseif {[regexp "^\[ \t\]*switch( |\t|;|$)" $line]} {
	    # try to avoid switch cases - needs better handling !!
	    set temp $line
	    if {$td_priv(detail) != "low"} {
		set line [lindex [td_parseLine $line "" $name [expr $lnum + 1] 0] 1]
	    }
	    set rline "td_eval $name [expr $lnum + 1].0 [expr $lnum + 1].end {} ;\
		    $line"
	    set l $lnum
	    while {$l < [llength $script] - 1} {
		incr l
		append temp \n[lindex $script $l]
		if {[info complete $temp]} {break}
	    }
	    set switch $l
	} elseif {$lnum > $switch} {
	    if {$td_priv(detail) != "low"} {
		set line [lindex [td_parseLine $line "" $name [expr $lnum + 1] 0] 1]
	    }
	    set rline "td_eval $name [expr $lnum + 1].0 [expr $lnum + 1].end {} ;\
		    $line"
	} else {
	    set rline $line
	}
	append result "$rline\n"
    }
    return $result
}

# }}}
# {{{ td_parseLine

# Search for bracketed command expressions in a line of tcl script.
# Insert a call to td_eval for each of those.
# This procedure is called recusirvely to handle nested expressions

# Args:
# line		The string to be parsed
# state		If state is "[", this is a subexpression
# name		Name of the procedure this line belongs to
# lnum		Current line number
# start		Start of current subexpression
#
# Result:	A list. First element is length of parsed subexpression,
#		second element is the modified expression

proc td_parseLine {line state name lnum start} {
    global td_priv

    # regular expression to search for brackets and backslashes
    set r "((\[^\]\[\\\\\]*)(\\\[|\\\]|\\\\))(.*)"

    # start of current expression
    set last 0
    # length of current expression
    set length 0
    # rest of expression to be parsed
    set m4 $line
    # The modified expression
    set pline ""		
    
    while {[regexp $r $m4 m m1 m2 m3 m4] != 0} {
	incr length [string length $m1]
	switch -exact $m3 {
	    "\\" {
		# skip next character since it's quoted
		set m4 [string range $m4 1 end]
		incr length 1
	    }
	    "\[" {
		# Keep stuff before subexpression
		append pline [string range $line $last [expr $length - 1]]
		# parse subexpression
		set temp [td_parseLine $m4 "\[" $name $lnum [expr $length + $start]]
		# add call to td_eval
		append pline "td_eval $name $lnum.[expr $length + $start]\
				$lnum.[expr $length + $start + [lindex $temp 0] -2]\
			    	\{[lindex $temp 1]\}\]"
		set m4 [string range $m4 [lindex $temp 0] end]
		incr length [lindex $temp 0]
		set last $length
	    }
	    "\]" {
		if {$state != "\["} {
		    error "Unmatched $state"
		} else {
		    append pline [string range $line $last [expr $length - 2]]
		    return [list $length $pline]
    }   }   }   }
    # No further subexpression. Append the rest and return.
    return [list $length [append pline [string range $line $last end]]]
}

# }}}
# {{{ td_eval

# This is the procedure that will be called when a procedure that
# has been prepared for debugging is being executed.
# The body of the procedure is displayed along with all
# available variables and their values.
# The current expression is highlighted and its result displayed.
#
# Args:
# name		The name of the procedure that's being debugged
# l1		Start index of current expession in text-widget format
# l2		End index of expression
# script	The expression that's evaluated
#
# Result: 	The result of the evaluation of script

proc td_eval {name l1 l2 script} {
    global td_priv td_Top td_ListName td_Listing td_Result td_Vars

    if {$td_priv(state) == "break"} {
	# Finish current procedure. uplevel {return} won't work, so use error.
	set td_priv(state) stop
	# Avoid standard tkerror procedure
	if {[info procs tkerror] == ""} {
	    set td_priv(error) ""
	} else {
	    set td_priv(error) "\{[info args tkerror]\} \{[info body tkerror]\}"
	}
	# Define new tk error handler that restores the old one when its called
	proc tkerror err {
	    global td_priv
	    if { $td_priv(error) == "" } {
		rename tkerror ""
	    } else {
		eval proc tkerror $td_priv(error)
	}   }
	$td_Listing tag remove active 1.0 end
	error "break"
    }
    scan $l1 %d lnum
    
    if {$script != ""} {
	# Evaluate script. Catch errors and notify the user.
	if {[catch {uplevel $script} td_priv(result.$name)]} {
	    set td_priv(result.$name) "error: $td_priv(result.$name)"
	    set td_priv(state) stop
    }   } else {
	set td_priv(result.$name) ""
    }
	
    # pop up debugger window
    if {[wm state $td_Top] != "normal"} {
	wm deiconify $td_Top
    }
    # Update display for current procedure
    if {$td_priv(current) != $name} {
	set td_priv(current) $name
	set td_priv(proc) "$name \{[info args $name]\}"
	$td_Listing configure -state normal
	$td_Listing delete 1.0 end
	$td_Listing insert 1.0 $td_priv(body.$name)
	foreach i $td_priv(break.$name) {
	    $td_Listing insert $i "B "
	    $td_Listing tag add break $i "$i + 1 chars"
	}
	$td_Listing configure -state disabled
	$td_Result configure -textvariable td_priv(result.$name)
	td_updateBacktrace
    }
    $td_Listing tag remove active 1.0 end
    if {$l1 == "end"} {
	if {$td_priv(state) == "end"} {
	    set td_priv(state) stop
	}
	$td_Listing tag remove active 1.0 end
	td_updateVars $name
	return
    }
    # Check for breakpoints
    if {$td_priv(state) != "nonstop" && \
	    [lsearch -exact $td_priv(break.$name) $l1] != -1} {
	set td_priv(state) stop
    }
    # Highlight current expression and try to display it in the center
    if {$td_priv(state) != "fast" && $td_priv(state) != "nonstop"} {
	td_realIndex $name l1 l2
	# try to display active line centered in Listing
	$td_Listing yview [expr $lnum - [lindex $td_priv(listheight) 1]/2]
	$td_Listing tag add active $l1 "$l2 + 1 chars"
	if {$td_priv(update) == "slow" || $td_priv(state) == "stop"} {
	    td_updateVars $name
	}
	update
    }
    if {$td_priv(state) == "stop"} {set td_priv(state) waiting}
    while {$td_priv(state) == "waiting"} {
	tkwait variable td_priv(state)
	switch -exact $td_priv(state) {
	    "eval" {
		if {[info complete $td_priv(eval)]} {
		    if {[catch {uplevel $td_priv(eval)} td_priv(result.$name)]} {
			set td_priv(result.$name) "error: $td_priv(result.$name)"
		    }   
		    td_updateVars $name
		}
		set td_priv(state) waiting
	    }
	    "stop" {set td_priv(state) waiting}
    }   }
    if {$td_priv(state) == "next"} {
	set td_priv(state) stop
    }
    if {$td_priv(state) == "slow"} {
	after $td_priv(delay)
    }
    return $td_priv(result.$name)
}

# }}}

# }}}
# {{{ interface procs

# {{{ td_configure

# Configure various Debugger options (right now only wrapping)
#
# Args:		As specified for `trace'

proc td_configure {name1 name2 op} {
    global td_priv td_Listing td_BTText td_WHText

    if {$name1 == "td_priv" && $op == "w"} {
	if {$name2 == "wrap"} {
	    $td_Listing configure -wrap $td_priv(wrap)
	} elseif {$name2 == "wrapback"} {
	    $td_BTText configure -wrap $td_priv(wrapback)
	} elseif {$name2 == "fullnames"} {
	    $td_WHText configure -state normal
	    $td_WHText delete 1.0 end
	    td_hierarchy $td_WHText
	    $td_WHText delete "end -1 chars" end
	    $td_WHText configure -state disabled
    }   }
}

# }}}
# {{{ td_setBreakpoint

# Set a breakpoint for the procedure currently being debugged.
# Use the innermost possible expression.
#
# Args:
# x,y		Coordinates of mouse click

proc td_setBreakpoint {x y} {
    global td_priv td_ListProc td_Listing

    set index [$td_Listing index @$x,$y]
    set proc $td_priv(current)
    
    if {[info proc $proc] == ""} {return}
    set body [info body $proc]
    if {! [string match #tdebug* $body]} {return}
    scan $index %d line
    set break ""
    set next [string first "td_eval $proc $line." $body]
    while {$next != -1} {
	set body [string range $body [expr $next + 1] end]
	set l1 [lindex $body 2]
	set l2 [lindex $body 3]
	set temp $l1
	td_realIndex $proc l1 l2
	if {[$td_Listing compare $l1 <= $index] && \
		[$td_Listing compare $l2 >= $index]} {
	    set break1 $temp
	    set break2 $l1
	}
	set next [string first "td_eval $proc $line." $body]
    }
    if {$break1 != ""} {
	set index [lsearch -exact $td_priv(break.$proc) $break1]
	if {$index == -1} {
	    lappend td_priv(break.$proc) $break1
	    $td_Listing configure -state normal
	    $td_Listing insert $break2 "B "
	    $td_Listing tag add break $break2 "$break2 + 1 chars"
	    $td_Listing configure -state disabled
	} else {
	    set td_priv(break.$proc) [lreplace $td_priv(break.$proc) $index $index]
	    $td_Listing configure -state normal
	    $td_Listing tag remove break $break2 "$break2 + 1 chars"
	    $td_Listing delete "$break2 - 2 chars" $break2
	    $td_Listing configure -state disabled
    }   }
}

# }}}
# {{{ td_realIndex

# Given an index into the body of the precedure being debugged,
# compute that index's position on the screen modified for
# display of breakpoints
#
# Args:
# proc		The name of the procedure
# i1		The name of the variable holding the start of the current expression
# i2		The name of the variable holding the end of the current expression
#
# Result:	None. The variables named by i1 and i2 are modified directly.

proc td_realIndex {proc i1 i2} {
    global td_priv td_Listing

    upvar $i1 l1
    upvar $i2 l2
    set add 0
    scan $l1 %d line
    foreach i $td_priv(break.$proc) {
	if {[string match $line.* $i] && [$td_Listing compare $i <= $l1] } {
	    incr add 2
    }	}
    if {$add != 0} {
	append l1 "+ $add chars"
	if {! [string match "*.end" $l2]} {
	    append l2 "+ $add chars"
}   }   }

# }}}
# {{{ td_updateVars

# Get the names of all variables accessible by the procedure being debugged
# and display them with the respective values.
#
# Args:
# proc		The name of the procedure.

proc td_updateVars proc {
    global td_Vars
    
    if {[catch {set vars [lsort -ascii [uplevel 2 "info vars"]]}]} {return}
    set view [$td_Vars nearest 0]
    $td_Vars delete 0 end
    foreach i $vars {
	upvar 2 $i temp
	if {[catch { $td_Vars insert end "$i: $temp" }]} {
	    if {![catch {lsort [array names temp]} names]} {
		foreach j $names {
		    catch {$td_Vars insert end "${i}($j): $temp($j)"}
    }   }   }   }
    $td_Vars yview $view
}

# }}}
# {{{ td_updateBacktrace

proc td_updateBacktrace {} {
    global td_BTText
    
    $td_BTText configure -state normal
    $td_BTText delete 1.0 end
    set level [info level]
    for {set i 1} {$i <= $level-2} {incr i} {
	$td_BTText insert end "[info level $i]\n\n"
    }
    if {$level >= 3} {
	$td_BTText delete "end -2 chars" end
    }
    $td_BTText configure -state disabled
}

# }}}
# {{{ td_debugFromSelection

# Get the current selection and pass it as a procedure name to
# td_prepareProc

proc td_debugFromSelection {} {
    global td_priv

    if {$td_priv(current) == ""} {return}
    if {[catch {selection get} sel]} {return}
    if {$sel == ""} {return}
    td_prepareProc $sel 
}

# }}}
# {{{ td_undebugFromSelection

# Get the current selection and pass it as a procedure name to
# td_restoreProc

proc td_undebugFromSelection {} {
    global td_priv

    if {$td_priv(current) == ""} {return}
    if {[catch {selection get} sel]} {return}
    if {$sel == ""} {return}
    td_restoreProc $sel 
}

# }}}
# {{{ td_evalFromSelection

# Get the current selection and evaluate it in the context of the procedure
# being debugged

proc td_evalFromSelection {} {
    global td_priv

    if {$td_priv(current) == "" || $td_priv(state) != "waiting"} {return}
    if {[catch {selection get} sel]} {return}
    if {$sel == ""} {return}
    set td_priv(eval) $sel
    set td_priv(state) eval
}

# }}}
# {{{ td_evalLine

# Evaluate the line in the eval widget in the context of the procedure
# being debugged.

proc td_evalLine {} {
    global td_priv

    if {$td_priv(current) == "" || $td_priv(state) != "waiting"} {return}
    if {$td_priv(eval) == ""} {return}
    set td_priv(state) eval
}

# }}}
# {{{ td_preparedProcs

proc td_preparedProcs {} {
    set names [info procs *]
    set procs ""
    foreach i $names {
	if {[string match #tdebug* [info body $i]]} {
	    lappend procs $i
	}
    }
    return [lsort $procs]
}

# }}}
# {{{ td_hierarchy

# Display a widget hierarchy in a text widget
# Args
# w		text widget to use
# start 	start of hierarchy (default .)
# level 	level of indentation (default 0)

proc td_hierarchy {w {start .} {level 0}} {
    global td_priv td_Top td_WH td_BT td_ET td_Choose

    set skip "$td_Top $td_WH $td_BT $td_ET"
    if {[info exists td_Choose]} {
	set skip "$skip $td_Choose"
    }
    set list  [winfo children $start]
    foreach i $list {
	# remove debugger toplevels
	if {$start != "." || [lsearch -exact $skip $i] == -1} {
	    $w insert end "[string range [format "%15s" \
		    [winfo class $i]] 0 14] "
	    for {set j 0} {$j < $level} {incr j} {
		$w insert end " "
	    }
	    if {! $td_priv(fullnames)} {
		set names [split $i .]
		$w insert end "[lindex $names [expr [llength $names] - 1]]\n"
	    } else {
		$w insert end "$i\n"
	    }
	    td_hierarchy $w $i [expr $level + 3]
    }   }
}

# }}}
# {{{ td_updateHierarchy

# Update Widget Hierarchy and make sure the toplevel is displayed

proc td_updateHierarchy {} {
    global td_WH td_WHText
    
    $td_WHText configure -state normal
    $td_WHText delete 1.0 end
    td_hierarchy $td_WHText
    $td_WHText delete "end -1 chars" end
    $td_WHText configure -state disabled
    wm deiconify $td_WH
}

# }}}
# {{{ td_updateErrorTrace

proc td_updateErrorTrace {} {
    global td_ET td_ETText errorInfo

    $td_ETText configure -state normal
    $td_ETText delete 1.0 end
    $td_ETText insert end $errorInfo
    $td_ETText configure -state disabled
    wm deiconify $td_ET
}

# }}}
# {{{ td_forceLoad

# This is just a dummy to force source-ing TdDebug.tcl via the
# auto-load mechanism
proc td_forceLoad {} {}

# }}}
# {{{ td_catchScroll

# Catch the yscrollcommand of td_Listing to fing out it's current
# height in characters Store the value in td_priv(listheight) and set
# the scrollbar
# Args:
# a b c d:		standard scrollbar settings

proc td_catchScroll {a b c d} {
    global td_priv td_ListScroll

    set td_priv(listheight) [list $a $b $c $d]
    $td_ListScroll set $a $b $c $d
}

# }}}
# {{{ td_constrainScroll

proc td_constrainScroll {start} {
    global td_priv td_Listing

    set total [lindex $td_priv(listheight) 0]
    set current [lindex $td_priv(listheight) 1]
    if {$total < $current} {
	$td_Listing yview 0
    } elseif {$total - $start < $current} {
	$td_Listing yview [expr $total - $current]
    } else {
	$td_Listing yview $start
}   }

# }}}
# {{{ td_catchVarScroll

proc td_catchVarScroll {a b c d} {
    global td_Vars td_VarScrollY
    
    if {$a < $b && $c > 0} {
	$td_Vars yview 0
	$td_VarScrollY set $a $b 0 [expr $b - 1]
    } elseif {$a -$c < $b} {
	$td_Vars yview [expr $a - $b]
	$td_VarScrollY set $a $b [expr $a - $b] [expr $a - 1]
    } else {
	$td_VarScrollY set $a $b $c $d
    }
}

# }}}
# {{{ dummies

# those two are just there to easily filter all procs belonging to tdebug
proc td_AAA {} {}
proc td_zzz {} {}

# }}}

# }}}
# {{{ interface

# {{{ setup symbolic widget names

set td_Top 		.tdTop

#set td_TopFrame		$td_Top.topFrame

set td_Menubar 		$td_Top.menubar
set td_MBDebug		$td_Menubar.mBDebug
set td_MenuDebug	$td_MBDebug.menuDebug
set td_MBOptions	$td_Menubar.mBOptions
set td_MenuOptions	$td_MBOptions.menuOptions
set td_MBHelp		$td_Menubar.mBHelp
set td_MenuHelp		$td_MBHelp.menuHelp
set td_MBSelection	$td_Menubar.mBSelection
set td_MenuSelection	$td_MBSelection.menuSelection

set td_MainRegion	$td_Top.mainRegion
set td_MainFrame	$td_MainRegion.mainFrame
set td_ListFrame	$td_MainFrame.listFrame
set td_ListNameFrame	$td_ListFrame.listNameFrame
set td_ListName		$td_ListNameFrame.listName
set td_ListProc		$td_ListNameFrame.listProc
set td_TextFrame	$td_ListFrame.textFrame
set td_Listing		$td_TextFrame.listing
set td_ListScroll	$td_TextFrame.listScroll

set td_VarFrame		$td_MainRegion.varFrame
set td_VarFrame1	$td_VarFrame.varFrame1
set td_VarName		$td_VarFrame1.varName
set td_VarFrame2	$td_VarFrame1.varFrame2
set td_VarFrame3	$td_VarFrame2.varFrame3
set td_Vars		$td_VarFrame3.vars
set td_VarScrollY	$td_VarFrame3.varScrollY
set td_VarFrame4 	$td_VarFrame2.varFrame4
set td_VarScrollX 	$td_VarFrame4.varScrollX
set td_VarFrame5	$td_VarFrame4.varFrame5

set td_ResultFrame	$td_ListFrame.resultFrame
set td_ResultName	$td_ResultFrame.resultName
set td_Result		$td_ResultFrame.result

set td_EvalFrame	$td_MainFrame.evalFrame
set td_EvalName		$td_EvalFrame.evalName
set td_Eval		$td_EvalFrame.eval

set td_Buttons		$td_Top.buttons
set td_BStop		$td_Buttons.bStop
set td_BNext		$td_Buttons.bNext
set td_BSlow		$td_Buttons.bSlow
set td_BFast		$td_Buttons.bFast
set td_BNonstop		$td_Buttons.bNonstop
set td_BBreak		$td_Buttons.bBreak

set td_DelayFrame	$td_VarFrame.delayFrame
set td_DelayLess	$td_DelayFrame.delayLess
set td_DelayFrame1	$td_DelayFrame.delayFrame1
set td_Delay		$td_DelayFrame1.delay
set td_DelayMore	$td_DelayFrame.delayMore

set td_BT		.td_BT
set td_BTMain		$td_BT.main
set td_BTText		$td_BTMain.text
set td_BTScroll		$td_BTMain.scroll
set td_BTClose		$td_BT.close

set td_WH		.td_WH
set td_WHMain		$td_WH.main
set td_WHText		$td_WHMain.text
set td_WHScroll		$td_WHMain.scroll
set td_WHClose		$td_WH.close

set td_ET		.td_ET
set td_ETMain		$td_ET.main
set td_ETText		$td_ETMain.text
set td_ETScroll		$td_ETMain.scroll
set td_ETClose		$td_ET.close

# }}}
# {{{ the toplevel

if [winfo exists $td_Top] {destroy $td_Top}

toplevel $td_Top -class TDebug -borderwidth 2 
wm title $td_Top "TDebug for [winfo name .]"
wm withdraw $td_Top

#frame $td_TopFrame
#pack $td_TopFrame -expand 1 -fill both ;# -ipadx 4 -ipady 8

# }}}
# {{{ the menubar

frame $td_Menubar -relief raised -borderwidth 2
pack $td_Menubar -side top -fill x -padx 2 -pady 2

menubutton $td_MBDebug -text "Debugger " -underline 0 -menu $td_MenuDebug -width 9
pack $td_MBDebug -side left
menu $td_MenuDebug
$td_MenuDebug add command -label "Backtrace  " -accelerator ^B \
	-underline 0 -command "wm deiconify $td_BT"
$td_MenuDebug add command -label "Widget Hierarchy  " -accelerator ^W \
	-underline 0 -command td_updateHierarchy
$td_MenuDebug add command -label "Error Trace  " -accelerator ^T \
	-underline 6 -command td_updateErrorTrace
$td_MenuDebug add separator
$td_MenuDebug add command -label "Close  " -accelerator ^C \
	-command "wm positionfrom $td_Top user ;\
	wm withdraw $td_Top" -underline 0

menubutton $td_MBOptions -text "Options " -underline 0 -menu $td_MenuOptions -width 8
pack $td_MBOptions -side left
menu $td_MenuOptions
$td_MenuOptions add checkbutton -label "Wrap Listing  " -accelerator ^L \
	-onvalue word -offvalue none -variable td_priv(wrap) \
	-underline 5
$td_MenuOptions add checkbutton -label "Wrap Backtrace  " \
	-onvalue word -offvalue none -variable td_priv(wrapback) \
	-underline 5
$td_MenuOptions add checkbutton -label "Full Widget Names  " -accelerator ^F \
	-onvalue 1 -offvalue 0 -variable td_priv(fullnames) \
	-underline 0
$td_MenuOptions add checkbutton -label "Slow Var Update  " -accelerator ^V \
	-onvalue slow -offvalue fast -variable td_priv(update) \
	-underline 5
$td_MenuOptions add checkbutton -label "High Detail  " -accelerator ^D \
	-onvalue high -offvalue low -variable td_priv(detail) \
	-underline 5

menubutton $td_MBSelection -text Selection -underline 0 -menu $td_MenuSelection \
	-width 10
pack $td_MBSelection -side left
menu $td_MenuSelection
$td_MenuSelection add command -label "Prepare Proc  " -accelerator ^P \
	-underline 0 -command td_debugFromSelection
$td_MenuSelection add command -label "Restore Proc  " -accelerator ^R\
	-underline 0 -command td_undebugFromSelection 
$td_MenuSelection add command -label "Eval  " -accelerator ^E \
	-underline 0 -command td_evalFromSelection

menubutton $td_MBHelp -text Help -underline 0 -menu $td_MenuHelp
pack $td_MBHelp -side right
menu $td_MenuHelp
$td_MenuHelp add separator

tk_menuBar $td_Menubar $td_MBDebug $td_MBOptions $td_MBSelection $td_MBHelp 
tk_bindForTraversal $td_Top

#We need the input focus when the cursor is inside!
if {"[focus default]" == "none"} {focus default .}
bind $td_Top <FocusIn> {
    if {"%d" == "NotifyVirtual"} {focus %W}
}
bind $td_Top <FocusOut> {
    if {"%d" == "NotifyVirtual"} {focus [focus default]}
}
# bind Accelerators
bind $td_Top <Control-b> {wm deiconify $td_BT}
bind $td_Top <Control-w> {td_updateHierarchy}
bind $td_Top <Control-t> {td_updateErrorTrace}
bind $td_Top <Control-c> {wm positionfrom $td_Top user ; wm withdraw $td_Top}
bind $td_Top <Control-l> {$td_MenuOptions invoke 0}
bind $td_Top <Control-f> {$td_MenuOptions invoke 2}
bind $td_Top <Control-v> {$td_MenuOptions invoke 3}
bind $td_Top <Control-d> {$td_MenuOptions invoke 4}
bind $td_Top <Control-p> {td_debugFromSelection}
bind $td_Top <Control-r> {td_undebugFromSelection}
bind $td_Top <Control-e> {td_evalFromSelection}


# }}}
# {{{ the main region

frame $td_MainRegion
pack $td_MainRegion -side top -expand 1 -fill both

# {{{ the listing

frame $td_MainFrame
pack $td_MainFrame -side left -expand 1 -fill both

frame $td_ListFrame -borderwidth 2 -relief raised
pack $td_ListFrame -side top -expand 1 -fill both -padx 2 -pady 2

frame $td_ListNameFrame -relief raised -borderwidth 0
pack $td_ListNameFrame -side top -fill x
label $td_ListName -text "Proc  :" -borderwidth 0 -width 8
pack $td_ListName -side left
entry $td_ListProc -textvariable td_priv(proc) -relief groove -state disabled
pack $td_ListProc -side left -expand 1 -fill x

frame $td_TextFrame -relief raised -borderwidth 0
pack $td_TextFrame -side top -expand 1 -fill both

if {$td_priv(constrainscroll)} {
    scrollbar $td_ListScroll -command "td_constrainScroll"
} else {
    scrollbar $td_ListScroll -command "$td_Listing yview"
}
pack $td_ListScroll -side $td_priv(scrollbarside) -fill y

text $td_Listing -width 27 -height 2 \
	-relief sunken -setgrid 1 -wrap none -state disabled \
	-borderwidth 2  -yscrollcommand td_catchScroll
pack $td_Listing -side $td_priv(scrollbarside) -expand 1 -fill both

$td_Listing tag configure break -foreground red -background gold \
	-relief raised -borderwidth 1
$td_Listing tag configure active -background orange -relief raised -borderwidth 1


bind $td_Listing <Double-Button-1> "td_setBreakpoint %x %y"

# }}}
# {{{ the variables

frame $td_VarFrame
pack $td_VarFrame -side left -fill both 

frame $td_VarFrame1 -borderwidth 2 -relief raised
pack $td_VarFrame1 -side top -fill both -expand 1 -padx 2 -pady 2

label $td_VarName -text "Variables: " -borderwidth 0
pack $td_VarName -side top -fill x

frame $td_VarFrame2 
pack $td_VarFrame2 -side top -expand 1 -fill both

frame $td_VarFrame3
pack $td_VarFrame3 -side top -expand 1 -fill both

scrollbar $td_VarScrollY -command "$td_Vars yview" -relief flat
pack $td_VarScrollY -side $td_priv(scrollbarside) -fill y

listbox $td_Vars -xscrollcommand "$td_VarScrollX set" \
	-relief sunken -geometry 15x2
if {$td_priv(constrainscroll)} {
    $td_Vars configure -yscrollcommand td_catchVarScroll
} else {
    $td_Vars configure -yscrollcommand "$td_VarScrollY set" 
}
pack $td_Vars -side $td_priv(scrollbarside) -expand 1 -fill both

frame $td_VarFrame4
pack $td_VarFrame4 -side top -fill x

scrollbar $td_VarScrollX -command "$td_Vars xview" -orient horiz -relief flat
frame $td_VarFrame5 -width [lindex [$td_VarScrollY configure -width] 4] \
	-height [lindex [$td_VarScrollX configure -width] 4]
pack $td_VarFrame5 -side $td_priv(scrollbarside)
pack $td_VarScrollX -side $td_priv(scrollbarside) -expand 1 -fill x

# }}}

# }}}
# {{{ the result

frame $td_ResultFrame -relief raised -borderwidth 0
pack $td_ResultFrame -side top -fill x 

label $td_ResultName -relief flat -text Result: -borderwidth 0 -width 8
pack $td_ResultName -side left

entry $td_Result -relief groove -state disabled
pack $td_Result -side left -expand 1 -fill x

# }}}
# {{{ the eval line

frame $td_EvalFrame -relief raised -borderwidth 2
pack $td_EvalFrame -side top -fill x -padx 2 -pady 2

label $td_EvalName -relief flat -text "Eval  :" -borderwidth 0 -width 8
pack $td_EvalName -side left

entry $td_Eval -relief groove -textvariable td_priv(eval)
pack $td_Eval -side left -expand 1 -fill x 

bind $td_Eval <Return> td_evalLine

# }}}
# {{{ the buttons

frame $td_Buttons
pack $td_Buttons -side top -fill x -padx 2 -pady 2

radiobutton $td_BStop -text Stop -width 4 -variable td_priv(state) -value stop
pack $td_BStop -side left -expand 1 -fill x
radiobutton $td_BNext -text Next -width 4 -variable td_priv(state) -value next
pack $td_BNext -side left -expand 1 -fill x
radiobutton $td_BSlow -text Slow -width 4 -variable td_priv(state) -value slow
pack $td_BSlow -side left -expand 1 -fill x
radiobutton $td_BFast -text Fast -width 4 -variable td_priv(state) -value fast
pack $td_BFast -side left -expand 1 -fill x
radiobutton $td_BNonstop -text Nonstop -width 4 -variable td_priv(state) -value nonstop
pack $td_BNonstop -side left -expand 1 -fill x
radiobutton $td_BBreak -text Break -width 4 -variable td_priv(state) -value break
pack $td_BBreak -side left -expand 1 -fill x

# }}}
# {{{ the delay

frame $td_DelayFrame
pack $td_DelayFrame -side top -fill x -padx 2 -pady 2

frame $td_DelayFrame1 -borderwidth 2 -relief raised
pack $td_DelayFrame1 -side left -expand 1 -fill x

label $td_Delay -text "Delay:  $td_priv(delay)" -width 11 -borderwidth 1
pack $td_Delay -side left -expand 1 -fill both

button $td_DelayLess -text "-" -width 2 -relief raised -command {
    global td_priv
    if {$td_priv(delay) >= 100} {
	incr td_priv(delay) -100
	$td_Delay configure -text "Delay: [format %4d $td_priv(delay)]"
    }
}
pack $td_DelayLess -side left -fill both

button $td_DelayMore -text "+" -width 2 -relief raised -command {
    global td_priv
    if {$td_priv(delay) < 1500} {
	incr td_priv(delay) 100
	$td_Delay configure -text "Delay: [format %4d $td_priv(delay)]"
    }
}
pack $td_DelayMore -side left -fill both

# }}}

wm geometry $td_Top 60x10

# {{{ The Backtrace toplevel

toplevel $td_BT -borderwidth 2
wm withdraw $td_BT
wm title $td_BT TDebug-Backtrace
frame $td_BTMain -relief raised -borderwidth 2
pack $td_BTMain -expand 1 -fill both -padx 2 -pady 2
scrollbar $td_BTScroll -command "$td_BTText yview" -relief flat
pack $td_BTScroll -side $td_priv(scrollbarside) -fill y
text $td_BTText -relief sunken -borderwidth 2 -width 30 -height 3 -setgrid 1 \
	-wrap none -state disabled
pack $td_BTText -side $td_priv(scrollbarside) -expand 1 -fill both
$td_BTText configure -yscrollcommand "$td_BTScroll set"
button $td_BTClose -text Close -relief raised \
	-command "wm positionfrom $td_BT user ;	wm withdraw $td_BT"
pack $td_BTClose -side top -fill x -padx 2 -pady 2

wm geometry $td_BT 80x10
bind $td_BT <Control-c> {wm positionfrom $td_BT user ; wm withdraw $td_BT}
bind $td_BT <Control-w> {$td_MenuOptions invoke 3}
#We need the input focus when the cursor is inside!
if {"[focus default]" == "none"} {focus default .}
bind $td_BT <FocusIn> {
    if {"%d" == "NotifyVirtual"} {focus %W}
}
bind $td_BT <FocusOut> {
    if {"%d" == "NotifyVirtual"} {focus [focus default]}
}

# }}}
# {{{ The Widget Hierarchy toplevel

toplevel $td_WH -borderwidth 2
wm withdraw $td_WH
wm title $td_WH TDebug-Widget-Hierarchy
frame $td_WHMain -relief raised -borderwidth 2
pack $td_WHMain -expand 1 -fill both -padx 2 -pady 2
scrollbar $td_WHScroll -command "$td_WHText yview" -relief flat
pack $td_WHScroll -side $td_priv(scrollbarside) -fill y
text $td_WHText -relief sunken -borderwidth 2 -width 30 -height 3 -setgrid 1 \
	-wrap none -state disabled 
pack $td_WHText -side $td_priv(scrollbarside) -expand 1 -fill both
$td_WHText configure -yscrollcommand "$td_WHScroll set"
button $td_WHClose -text Close -relief raised \
	-command "wm positionfrom $td_WH user ;	wm withdraw $td_WH"
pack $td_WHClose -side top -fill x -padx 2 -pady 2

wm geometry $td_WH 80x10
bind $td_WH <Control-c> {wm positionfrom $td_WH user ; wm withdraw $td_WH}
bind $td_WH <Control-f> {$td_MenuOptions invoke 4}
#We need the input focus when the cursor is inside!
if {"[focus default]" == "none"} {focus default .}
bind $td_WH <FocusIn> {
    if {"%d" == "NotifyVirtual"} {focus %W}
}
bind $td_WH <FocusOut> {
    if {"%d" == "NotifyVirtual"} {focus [focus default]}
}

# }}}
# {{{ The Error Trace toplevel

toplevel $td_ET -borderwidth 2
wm withdraw $td_ET
wm title $td_ET TDebug-ErrorTrace
frame $td_ETMain -relief raised -borderwidth 2
pack $td_ETMain -expand 1 -fill both -padx 2 -pady 2
scrollbar $td_ETScroll -command "$td_ETText yview" -relief flat
pack $td_ETScroll -side $td_priv(scrollbarside) -fill y
text $td_ETText -relief sunken -borderwidth 2 -width 30 -height 3 -setgrid 1 \
	-wrap none -state disabled
pack $td_ETText -side $td_priv(scrollbarside) -expand 1 -fill both
$td_ETText configure -yscrollcommand "$td_ETScroll set"
button $td_ETClose -text Close -relief raised \
	-command "wm positionfrom $td_ET user ;	wm withdraw $td_ET"
pack $td_ETClose -side top -fill x -padx 2 -pady 2

wm geometry $td_ET 80x10
bind $td_ET <Control-c> {wm positionfrom $td_ET user ; wm withdraw $td_ET}
#We need the input focus when the cursor is inside!
if {"[focus default]" == "none"} {focus default .}
bind $td_ET <FocusIn> {
    if {"%d" == "NotifyVirtual"} {focus %W}
}
bind $td_ET <FocusOut> {
    if {"%d" == "NotifyVirtual"} {focus [focus default]}
}

# }}}


# }}}

# {{{ Emacs Local Variables


# Local Variables:
# folded-file: t
# End:

# }}}


