#
# $Id: tkinspect.tcl,v 1.12 1993/07/20 03:25:41 sls Exp $
#
set release 4d
set releaseDate "July 20, 1993"
if {$argc != 1} {
    puts stderr "Ooops! tkinspect has not been installed correctly."
    puts stderr \
	"(You can try invoking tkinspect as: wish -f tkinspect.tcl .)"
    exit 1
}
scan $tkVersion %d.%d tkMajor tkMinor
if {$tkMajor != 3 || $tkMinor < 1} {
    puts stderr "Ooops! tkinspect $release requires Tk 3.x (x >= 1)"
    exit 1
}
set tkinspect_library $argv
source $tkinspect_library/class.tcl
source $tkinspect_library/bindings.tcl
source $tkinspect_library/filechsr.tcl
source $tkinspect_library/help.tcl

if {[set configFile [glob -nocomplain ~/.tkinspect]] == {}} {
    set configFile $tkinspect_library/dottkinspect
}
source $configFile

wm title . "Tk Inspect: "
wm iconname . "Tk Inspect"

#
# global variables
#
set target none

#
# Handle the about box.
#
class About {
    member w .about
    method run {} {
	set w [getmember w]
	catch {destroy $w}
	toplevel $w
	wm transient $w .
	global release releaseDate tkinspect_library
	label $w.title -text "tkinspect" \
	    -font -*-helvetica-bold-r-*-*-18-*
	label $w.ver -text "Release $release ($releaseDate)" \
	    -font -*-helvetica-medium-r-*-*-12-*
	label $w.com -text "Send comments, suggestions, bugs to:" \
	    -font -*-helvetica-medium-o-*-*-12-*
	frame $w.mug -bd 4
	label $w.mug.l -text "Sam Shen <sls@aero.org>"
	label $w.mug.bm -bitmap @$tkinspect_library/sls.xbm -bd 2 \
	    -relief sunken
	pack append $w.mug \
	    $w.mug.l "left expand fill" \
	    $w.mug.bm "left expand fill"
	button $w.ok -text "Ok" -command {destroy .about}
	pack append $w \
	    $w.title "top fillx" \
	    $w.ver "top fillx" $w.com "top fillx" $w.mug "top fillx" \
	    $w.ok "bottom pady 5"
	bind $w <Enter> "$w.ok activate"
	bind $w <Leave> "$w.ok deactivate"
	bind $w <Return> {destroy .about}
	focus $w
	centerWindow $w
	grab set $w
	tkwait window $w
    }
}

#
# Create the main menu.
#
class MainMenu {
    member w {}
    member filechooser {}
    member about {}
    member helpwin {}
    method create w {
	setmember w $w
	setmember filechooser [new FileChooser]
	setmember about [new About]
	setmember helpwin [new HelpWindow]
	frame $w -relief raised -bd 2
	menubutton $w.file -text "File" -menu $w.file.m -underline 0
	set m [menu $w.file.m]
	$m add cascade -label "Select Application" \
	    -command "MainMenu:fillApps $this" \
	    -menu $m.apps -underline 7 -accelerator =>
	$m add command -label "Save Value..." -underline 0 \
	    -command "MainMenu:saveValue $this"
	$m add command -label "Load Value..." -underline 0 \
	    -command "MainMenu:loadValue $this"
	$m add command -label "Update Lists" -underline 0 \
	    -command updateLists
	$m add separator
	$m add cascade -label "Delete Interpreter..." -underline 0 \
	    -command "MainMenu:fillDelInterps $this" -menu $m.delapps \
	    -accelerator =>
	$m add separator
	$m add command -label "Quit" -underline 0 \
	    -command "MainMenu:quit $this"
	menu $m.apps
	menu $m.delapps
	pack append $w $w.file left
	menubutton $w.options -text "Options" -menu $w.options.m -underline 0
	pack append $w $w.options left
	set m [menu $w.options.m]
	$m add radiobutton -variable windowInfoType -value config \
	    -label "Window Configuration" -underline 7 \
	    -command "MainMenu:changeWinInfo $this"
	$m add radiobutton -variable windowInfoType -value packing \
	    -label "Window Packing" -underline 7 \
	    -command "MainMenu:changeWinInfo $this"
	$m add radiobutton -variable windowInfoType -value parentpacking \
	    -label "Parent Window Packing" -underline 1 \
	    -command "MainMenu:changeWinInfo $this"
	$m add radiobutton -variable windowInfoType -value bindings \
	    -label "Window Bindings" -underline 7 \
	    -command "MainMenu:changeWinInfo $this"
	$m add radiobutton -variable windowInfoType -value classbindings \
	    -label "Window Class Bindings" -underline 8 \
	    -command "MainMenu:changeWinInfo $this"
	$m add separator
	$m add checkbutton -variable filterEmptyWindowConfigs \
	    -label "Filter Empty Window Options" -underline 0
	$m add checkbutton -variable getWindowInformation \
	    -label "Get Window Information" -underline 0
	menubutton $w.help -text "Help" -menu $w.help.m -underline 0
	pack append $w $w.help right
	set m [menu $w.help.m]
	$m add command -label "About..." -underline 0 \
	    -command "MainMenu:about $this"
	$m add command -label "Intro" -underline 0 \
	    -command "MainMenu:help $this Intro"
	$m add command -label "File menu" -underline 0 \
	    -command "MainMenu:help $this File"
	$m add command -label "Options menu" -underline 0 \
	    -command "MainMenu:help $this Options"
	$m add command -label "Patterns" -underline 0 \
	    -command "MainMenu:help $this Patterns"
	$m add command -label "What's new?" -underline 0 \
	    -command "MainMenu:help $this WhatsNew"
	$m add command -label "Notes" -underline 0 \
	    -command "MainMenu:help $this Notes"
	tk_menuBar $w $w.file $w.options $w.help
	return $w
    }
    method fillDelInterps {} {
	set w [getmember w]
	catch {$w.file.m.delapps delete 0 last}
	foreach i [lsort [winfo interps]] {
	    $w.file.m.delapps add command -label $i \
		-command [list delInterp $i]
	}
    }
    method fillApps {} {
	set w [getmember w]
	catch {$w.file.m.apps delete 0 last}
	foreach i [lsort [winfo interps]] {
	    $w.file.m.apps add command -label $i \
		-command [list setTarget $i]
	}
    }
    method saveValue {} {
	set fc [getmember filechooser]
	if {[set file [FileChooser:run $fc "Save value to:" 0 1]] != {}} {
	    set f [open $file w]
	    puts $f [.value.t get 1.0 end]
	    close $f
	}
    }
    method loadValue {} {
	set fc [getmember filechooser]
 	if {[set file [FileChooser:run $fc "Load value from:" 1 0]] != {}} {
	    .value.t delete 1.0 end
	    set f [open $file r]
	    .value.t insert end [read $f]
	    close $f
	}	
    }
    method changeWinInfo {} {
	global currentWindow valueType
	if [string match window.* $valueType] {
	    selectWindow $currentWindow
	}
    }
    method about {} {
	About:run [getmember about]
    }
    method help {topic} {
	HelpWindow:showTopic [getmember helpwin] $topic
    }
    method quit {} {
	destroy .
    }
}
pack append . [in [new MainMenu] { create .main }] "top fillx"

#
# center a toplevel window
#
proc centerWindow {win} {
    wm withdraw $win
    update idletasks
    set w [winfo reqwidth $win]
    set h [winfo reqheight $win]
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth $win]
    wm geometry $win +[expr {($sw-$w)/2}]+[expr {($sh-$h)/2}]
    wm deiconify $win
}

#
# yes no box
#
class YesNo {
    member w .yesno
    member yes 0
    method run {question} {
	set w [getmember w]
	catch {destroy $w}
	toplevel $w
	wm transient $w .
	frame $w.top
	message $w.top.msg -text $question -justify center \
	    -font -Adobe-helvetica-medium-r-normal--*-240* -aspect 200
	pack append $w.top $w.top.msg "fill padx 5 pady 5 expand"
	frame $w.bot
	button $w.yes -text "Yes" -command "YesNo:yes $this"
	pack append $w.bot $w.yes "left expand padx 20"
	frame $w.bot.no -relief sunken -bd 1
	pack append $w.bot $w.bot.no "left expand padx 20 pady 20"
	button $w.no -text "No" -command "YesNo:no $this"
	pack append $w.bot.no $w.no "expand padx 12 pady 12"
	pack append $w $w.top "top fill expand" $w.bot "bottom fill expand"
	bind $w.top <Enter> "$w.no activate"
	bind $w.top.msg <Enter> "$w.no activate"
	bind $w.bot <Enter> "$w.no activate"
	bind $w.top <Leave> "$w.no deactivate"
	bind $w.top.msg <Leave> "$w.no deactivate"
	bind $w.bot <Leave> "$w.no deactivate"
	centerWindow $w
	focus $w
	bind $w <Return> "YesNo:no $this"
	grab set $w
	tkwait window $w
	return [getmember yes]
    }
    method yes {} {
	setmember yes 1
	destroy [getmember w]
    }
    method no {} {
	setmember yes 0
	destroy [getmember w]
    }
}

#
# Ask if you really want to delete the interp.
#
set yesno [new YesNo]
proc delInterp {interp} {
    global yesno tkinspect_library
    set txt "Really delete interpreter \"$interp\"?"
    if {[catch {send $interp info tclversion}] == 0} {
	append txt "\n(It is still responding.)"
    }
    if [YesNo:run $yesno $txt] {
	exec $tkinspect_library/delinterp $interp
    }
}

#
# set the current target.
#
proc setTarget {t} {
    global target
    set target $t
    wm title . "Tk Inspect: $t"
    updateLists
}


#
# CommandEntry handles sending commands to the remote interpreter.  It
# also provides a history using ^n or Down and ^p or Up to scan through
# the history.
#
class CommandEntry {
    member w {}
    member history {}
    member ndx 0
    member n 0
    method create {w} {
	setmember w $w
	frame $w -borderwidth 2
	entry $w.e -relief sunken
	bindEntry $w.e
	bind $w.e <Return> "+CommandEntry:do $this"
	bind $w.e <Control-p> "CommandEntry:previous $this"
	bind $w.e <Control-n> "CommandEntry:next $this"
	bind $w.e <Up> "CommandEntry:previous $this"
	bind $w.e <Down> "CommandEntry:next $this"
	label $w.l -text "Command:"
	button $w.s -text "Send Command" -command "CommandEntry:do $this"
	pack append $w \
	    $w.l left \
	    $w.e "left fillx expand" \
	    $w.s "right padx .2c"
	return $w
    }
    method do {} {
	global target
	if {$target == "none"} {
	    statusMessage "no interpreter selected"
	    return
	}
	set cmd [[getmember w].e get]
	catch [list send $target $cmd] stuff
	insertValue $stuff command
	lappendmember history $cmd
	set n [getmember n]
	if {$n == 50} {
	    setmember history [lrange [getmember history] 1 end]
	} else {
	    setmember n [incr n]
	}
	setmember ndx [expr $n-1]
    }
    method previous {} {
	set ndx [getmember ndx]
	set w [getmember w]
	if {$ndx == 0} return
	incr ndx -1
	setmember ndx $ndx
	$w.e delete 0 end
	$w.e insert insert [lindex [getmember history] $ndx]
	tk_entrySeeCaret $w.e
    }
    method next {} {
	set ndx [getmember ndx]
	if {$ndx == [getmember n]} return
	set w [getmember w]
	incr ndx
	setmember ndx $ndx
	$w.e delete 0 end
	$w.e insert insert [lindex [getmember history] $ndx]
	tk_entrySeeCaret $w.e
    }
}
pack append . [CommandEntry:create [new CommandEntry] .cmd] "fillx expand top"

#
# PatternEditor sets up a dialog that edits include/exclude patterns.
#
class PatternEditor {
#|
# w is the name of the toplevel window.
# ok is 1 if ok was clicked, 0 if cancel was clicked.
# patterns is the list of patterns if ok was clicked.
#|
    member w .patedit
    member ok 0
    member patterns {}
#|
# run starts the dialog and waits until ok or cancel is clicked.  It
# returns 1 if ok, 0 if cancel.
#|
    method run {title patterns} {
	set w [getmember w]
	catch {destroy $w}
	toplevel $w
	wm title $w $title
	wm iconname $w $title
	frame $w.top
	label $w.l -text "Pattern:"
	entry $w.e -width 40 -relief sunken
	bindEntry $w.e
	pack append $w.top $w.l "left" $w.e "left fillx expand"
	pack append $w $w.top "top fillx pady .25c"
	frame $w.buttons -bd 3
	button $w.ok -text "Ok" -command "PatternEditor:ok $this"
	button $w.cancel -text "Cancel" -command "PatternEditor:cancel $this"
	button $w.add -text "Add Pattern" \
	    -command "PatternEditor:add $this"
	button $w.del -text "Delete Pattern(s)" \
	    -command "PatternEditor:delete $this"
	radiobutton $w.inc -variable patternType -value include \
	    -relief flat -text "Include Patterns"
	radiobutton $w.exc -variable patternType -value exclude \
	    -relief flat -text "Exclude Patterns"
	pack append $w.buttons \
	    $w.inc "top fillx pady .1c frame w" \
	    $w.exc "top fillx pady .1c frame w" \
	    $w.add "top fillx pady .1c" \
	    $w.del "top fillx pady .1c" \
	    $w.cancel "bottom fillx pady .1c" \
	    $w.ok "bottom fillx pady .1c"
	pack append $w $w.buttons "left filly"
	frame $w.lframe
	scrollbar $w.scroll -command "$w.list yview"
	listbox $w.list -yscroll "$w.scroll set" -relief raised \
	    -geometry 40x10 -exportselection false
	foreach pat $patterns {
	    $w.list insert end $pat
	}
	setmember patterns $patterns
	pack append $w.lframe \
	    $w.scroll "right filly" \
	    $w.list "left expand fill"
	pack append $w $w.lframe "right fill expand"
	grab set $w
	tkwait window $w
	return [getmember ok]
    }
    method getpatterns {} {
	getmember patterns
    }
    method cancel {} {
	setmember ok 0
	destroy [getmember w]
    }
    method ok {} {
	setmember ok 1
	setmember patterns {}
	set w [getmember w]
	for {set i 0} {$i < [$w.list size]} {incr i} {
	    lappendmember patterns [$w.list get $i]
	}
	destroy [getmember w]
    }
#|
# add adds a pattern.
#|
    method add {} {
	set w [getmember w]
	set thing [string trim [$w.e get]]
	if {$thing == {}} return
	$w.list insert end $thing
    }
#|
# delete deletes a pattern.
#|
    method delete {} {
	set w [getmember w]
	while {[set s [$w.list curselection]] != {}} {
	    $w.list delete [lindex $s 0]
	}
    }
}

#
# SelectiveList is a listbox that only displays things that either match
# or don't match a set of patterns.
#
class SelectiveList {
#|
# w is the name of the window
# command is the command to be invoked when a new selection is made.
# patternEditor holds the pattern editor.
# excludes is a list of patterns to exclude.
# contents stores all the contents.
#|
    member w {}
    member command {}
    member patternEditor {}
    member patternType exclude
    member patterns {}
    member contents {}
#|
# create creates the window.
#|
    method create {w title {command {}}} {
	setmember w $w
	setmember command $command
	frame $w -bd 2 -relief raised
	button $w.pats -text "Edit Patterns..." \
	    -command "SelectiveList:editPatterns $this"
	frame $w.lframe -bd 2 -relief sunken
	scrollbar $w.scroll -command "$w.list yview"
	listbox $w.list -yscroll "$w.scroll set" -relief raised \
	    -geometry 30x10 -exportselection false
	tk_listboxSingleSelect $w.list
	pack append $w.lframe \
	    $w.scroll "right filly" \
	    $w.list "left expand fill"
	label $w.title -text $title -anchor w
	pack append $w \
	    $w.title "top fillx" \
	    $w.lframe "top fill expand" \
	    $w.pats "padx 4 pady 4 bottom frame w"
	bind $w.list <1> [format {
	    %%W select from [%%W nearest %%y]
	    SelectiveList:select %s %%y
	} $this]
	setmember patternEditor [new PatternEditor]
    }
#|
# setPatterns sets the list of patterns.
#|
    method setPatterns {type pats} {
	setmember patternType $type
	setmember patterns $pats
    }
#|
# clear clears all the entries.
#|
    method clear {} {
	setmember contents {}
	updateList
    }
#|
# updateList goes through the current contents only inserting those
# items that should be included.
#|
    method updateList {} {
	set w [getmember w]
	$w.list delete 0 end
	foreach thing [getmember contents] {
	    if [include? $thing] {
		$w.list insert end $thing
	    }
	}
    }
#|
# include? checks to see if a thing should be included.
#|
    method include? {thing} {
	foreach re [set pats [getmember patterns]] {
	    if [regexp $re $thing] {
		if {[getmember patternType] == "include"} {
		    return 1
		} else {
		    return 0
		}
	    }
	}
	if {[getmember patternType] == "include" && [llength $pats] > 0} {
	    return 0
	} else {
	    return 1
	}
    }
#|
# add adds some stuff.
#|
    method add {stuff} {
	foreach thing $stuff {
	    lappendmember contents $thing
	    if [include? $thing] {
		[getmember w].list insert end $thing
	    }
	}
    }
#|
# editPatterns brings up the pattern editor.
#|
    method editPatterns {} {
	set p [getmember patternEditor]
	global patternType
	set patternType [getmember patternType]
	if [PatternEditor:run $p "Pattern Editor" [getmember patterns]] {
	    setmember patterns [PatternEditor:getpatterns $p]
	    setmember patternType $patternType
	}
	updateList
    }
#|
# select invokes command if there is one.
#|
    method select {y} {
	set command [getmember command]
	if {$command != {}} {
	    set w [getmember w]
	    set thing [$w.list get [$w.list nearest $y]]
	    if {$thing != {}} {
		append command " $thing"
		eval $command
	    }
	}
    }
#|
# setDefaultPatterns sets the default patterns for this list.
#|
    method setDefaultPatterns {defaultsName} {
	upvar #0 $defaultsName defaults
	if ![info exists defaults] return
	setPatterns [lindex $defaults 0] [lindex $defaults 1]
    }
}

#
# Feedback provides some feedback while a long operation is in
# progress
#
class Feedback {
    member w {}
    member steps {}
    member step {}
    method show {w title} {
	setmember w $w
	catch {destroy $w}
	toplevel $w
	label $w.title -width 50 -text $title
	frame $w.center
	frame $w.bar -bg DodgerBlue -relief raised -bd 2 -height 20 -width 1
	pack append $w.center \
	    [frame $w.spacer -width 5 -height 5] {left} \
	    $w.bar {frame w}
	label $w.percentage -text 0%
	pack append $w \
	    $w.title {top fill expand} \
	    $w.center {top pady 5 frame w} \
	    $w.percentage {top fill expand}
	wm transient $w .
	centerWindow $w
	setmember width [set width [expr [winfo width $w]-10]]
	$w.center config -width $width
	update
    }
    method setSteps steps {
	setmember steps [format %f $steps]
	setmember step 0
    }
    method step {} {
	set w [getmember w]
	set step [getmember step]
	set steps [getmember steps]
	set width [getmember width]
	incr step
	set fraction [expr $step/$steps]
	$w.percentage config -text [format %.0f%% [expr 100.0*$fraction]]
	$w.bar config -width [format %.0f [expr $width*$fraction]]
	setmember step $step
	update
    }
    method hide {} {
	set w [getmember w]
	catch {destroy $w}
    }
}

#
# Create the lists that hold the interpreter names, the procs, and the
# global variables.
#
frame .lists -borderwidth 2
in [set procList [new SelectiveList]] {
    create .lists.procs "Procs:" selectProc
    setDefaultPatterns defaultProcPatterns
}
in [set globalList [new SelectiveList]] {
    create .lists.globals "Globals:" selectGlobal
    setDefaultPatterns defaultGlobalPatterns
}
in [set winList [new SelectiveList]] {
    create .lists.windows "Windows:" selectWindow
    setDefaultPatterns defaultWindowPatterns
}
pack append .lists \
    .lists.procs "left fill expand" \
    .lists.globals "left fill expand" \
    .lists.windows "left fill expand"
pack append . .lists "fill expand"

#
# Create the text widget that holds the value.
#
frame .value -bd 2 -relief raised
label .value.l -text "Value:" -anchor w
frame .value.tframe -bd 2 -relief sunken
scrollbar .value.sb -command ".value.t yview"
text .value.t -relief raised -bd 2 -yscroll ".value.sb set" -width 80
bindText .value.t
pack append .value.tframe \
    .value.t "left fill expand" \
    .value.sb "right filly"
button .value.s -text "Send Value" -command sendValue
pack append .value \
    .value.l "top fillx" \
    .value.tframe "top fill expand" \
    .value.s "padx 4 pady 4 bottom frame w"
pack append . .value "fill expand"

#
# setup .value.t so that ^X^S sends the value.  This doesn't work quite
# right, since you can type ^X then, still holding down the control
# key use ^N, ^P etc to move around and ^S will still send.  Also,
# if you type ^X, then let up the control key, then type ^S \x13 will
# get inserted into .value.t instead of sending.  I can't think of
# any reasonable way to get ^X^S to work correctly, but I think this
# is better than nothing.
#
bind .value.t <Control-x> handleControlX
proc handleControlX {} {
    statusMessage ^X-
    bind .value.t <Control-s> {
	statusMessage "Value sent..."
	sendValue
	bind .value.t <Control-x> handleControlX
    }
    bind .value.t <Any-Key> {
	bind .value.t <Control-x> handleControlX
	bind .value.t <Control-s> {}
	bind .value.t <Any-Key> {}
	statusMessage ""
    }
}

#
# Create a status line.
#
frame .status -bd 2
label .status.l -relief sunken -anchor w
pack append .status .status.l "fill expand"
pack append . .status "fill expand"

#
# statusMessage sets the contents of the status line.
#
proc statusMessage msg {
    .status.l config -text $msg
}

#
# insertValue inserts some text into the value widget.  It stores
# what type the value is in valueType.
#
set valueType none
proc insertValue {thing type} {
    global valueType
    set valueType $type
    .value.t delete 1.0 end
    .value.t insert end $thing
}

#
# sendValue sends the currentValue back to the interpreter.
#
proc sendValue {} {
    global target valueType filterEmptyWindowConfigs
    if {$target == "none"} {
	statusMessage "no interpreter selected"
	return
    }
    set value [.value.t get 1.0 end]
    if {$valueType == "window.config" && $filterEmptyWindowConfigs} {
	regsub -all {[ \t]*-[^ \t]+[ \t]+{}([ \t]*\\?\n?)?} $value {\1} value
    }
    if {[catch {send $target $value} stuff] != 0} {
	insertValue $stuff error
    }
}

#
# updateLists updates the lists of globals, procs, and windows.
#
set feedback [new Feedback]
set getWindowInformation 1
proc updateLists {} {
    global target
    if {[catch {send $target info tclversion} result] != 0} {
	statusMessage $result
	return
    }
    global procList
    in $procList {
	clear
	add [lsort [send $target info procs]]
    }
    global globalList
    in $globalList {
	clear
	add [lsort [send $target info globals]]
    }
    global winList feedback getWindowInformation
    if $getWindowInformation {
	SelectiveList:clear $winList
	set children [send $target winfo children .]
	in $feedback {
	    setSteps [llength $children]
	    show .feedback "Getting window information...."
	}
	getWindows $target $children 1
	Feedback:hide $feedback
    }
    statusMessage "now sending to $target"
}

#
# getWindows recursively adds windows to the list of windows.
#
proc getWindows {target wins {stepFeedback 0}} {
    global winList feedback
    foreach win $wins {
	SelectiveList:add $winList $win
	getWindows $target [send $target winfo children $win]
	if $stepFeedback {
	    Feedback:step $feedback
	}
    }
}

#
# checkValid target checks to make sure target is o.k., if it is return 1
# otherwise put up a statusMessage and return 0.
#
proc checkValidTarget {} {
    global target
    if {$target == "none"} {
	statusMessage "no interpreter selected"
	return 0
    }
    if {[catch {send $target info tclversion} result] != 0} {
	statusMessage $result
	return 0
    }
    return 1
}

#
# selectProc handles getting the defn. of a proc in the remote interpreter.
#
proc selectProc {thing} {
    global target
    if ![checkValidTarget] return
    set args {}
    foreach arg [send $target info args $thing] {
	if [send $target info default $thing $arg __tkinspectDefaultArg__] {
	    lappend args [list $arg [send $target set __tkinspectDefaultArg__]]
	} else {
	    lappend args $arg
	}
    }
    catch {send $target unset __tkinspectDefaultArg__}
    insertValue [list proc $thing $args [send $target info body $thing]] proc
}

#
# selectGlobal handles getting the value of a global in the remote
# interpreter.
#
proc selectGlobal {thing} {
    global target
    if ![checkValidTarget] return
    if {[catch {send $target "set $thing"} result] == 0} {
	set stuff [list set $thing $result]
    } else {
	set stuff ""
	foreach name [lsort [send $target [list array names $thing]]] {
	    append stuff [list set [format %s(%s) $thing $name] \
			  [send $target [list set [format %s(%s) $thing \
						   $name]]]] \
		"\n"
	}
    }
    insertValue $stuff global
}

#
# selectWindow handles getting the configuration options for a window.
#
set flashColor "hot pink"
set currentWindow {}
set windowInfoType config
set filterEmptyWindowConfigs 1
proc selectWindow {win} {
    global target flashColor currentWindow windowInfoType
    if ![checkValidTarget] return
    set currentWindow $win
    if {$currentWindow == {}} return
    case $windowInfoType {
	config {
	    set class [send $target winfo class $win]
	    set stuff "# configuration for $win\n# class: $class\n$win config"
	    foreach item [send $target $win config] {
		if {[llength $item] != 5} continue
		append stuff " \\\n\t"
		lappend stuff [lindex $item 0] [lindex $item 4]
	    }
	}
	parentpacking {
	    set parent [send $target winfo parent $win]
	    set stuff "# packing info for $parent (parent of $win)\n"
	    set info [send $target pack info $parent]
	    set len [llength $info]
	    if {$len > 0} {
		for {set i 0} {$i < $len} {incr i 2} {
		    append stuff "pack unpack [lindex $info $i]\n"
		}
		append stuff "pack append $parent"
		for {set i 0} {$i < $len} {incr i 2} {
		    append stuff " \\\n\t[lindex $info $i] "
		    lappend stuff [lindex $info [expr $i+1]]
		}
	    }
	}
	packing {
	    set stuff "# packing info for $win\n"
	    set info [send $target pack info $win]
	    set len [llength $info]
	    if {$len > 0} {
		for {set i 0} {$i < $len} {incr i 2} {
		    append stuff "pack unpack [lindex $info $i]\n"
		}
		append stuff "pack append $win"
		for {set i 0} {$i < $len} {incr i 2} {
		    append stuff " \\\n\t[lindex $info $i] "
		    lappend stuff [lindex $info [expr $i+1]]
		}
	    }	    
	}
	bindings {
	    set stuff "# bindings for $win"
	    foreach sequence [send $target bind $win] {
		append stuff "\nbind $win $sequence "
		lappend stuff [send $target bind $win $sequence]
	    }
	}
	classbindings {
	    set class [send $target winfo class $win]
	    set stuff "# class bindings for $win\n# class: $class"
	    foreach sequence [send $target bind $class] {
		append stuff "\nbind $class $sequence "
		lappend stuff [send $target bind $class $sequence]
	    }
	}
    }
    insertValue $stuff window.$windowInfoType
    catch {
	if ![send $target info exists __tkinspectFlashing__] {
	    send $target set __tkinspectFlashing__ 1
	    set oldbg [lindex [send $target $win config -bg] 4]
	    send $target [list $win config -bg $flashColor]
	    send $target [list after 100 \
			  [format \
			   "%s config -bg %s; unset __tkinspectFlashing__" \
			       $win $oldbg]]
	}
    }
}

statusMessage "ready"
