##############################################################################
#
# config.tcl:
#       The configuration and initialization....
#
# ConfTkMH {}
#       Configuration window for TkMH
#
# ConfTkMH:Read {}
#       Read in the .tkmh configuration file in the users home
#
# ConfTkMH:Write {}
#       Adapt the .tkmh file in the users home directory. If it doesn't exist
#       make a copy of the global one and adapt that.
#
# ConfTkMH:Insert {}
#       Insert the values for the variables in the window
#
# ConfMH {}
#       Configuration window for MH
#
# ConfMH:Read {}
#       Read the MH configuration file: .mh_profile in the users home
#
# ConfMH:Write {}
#       Adapt the .mh_profile in the users home directory. If it doesn't 
#       exist, create one.
#
# ConfMH:Insert {}
#       Insert the variables in the window
#
# Init {}
#       Read in the global variables. Check for a .tkmh and a .mh_profile
#       if they're not there, create them. Then read in the .tkmh, .mh_profile
#       and build the folder and aliaslist. Also define some global bindings.
#
# BuildFolList {}
#       Fill the global var FolList with the foldernames
#
# BuildAliasList {}
#       Read in the aliasfile. Enter the aliases and their addresses in the 
#       global variables Aliases and Addresses.
#
# MakeAliMenu {}
#	Adapt the "Mail To" menu in the Control window to the new alias list
#
##############################################################################

proc Init {} {
    global FONT BM PREF MH cmd tkmh_lib env
    global VERSION DOTVERSION
    
    wm withdraw .
    source $tkmh_lib/tkmh.rc
    if {![file exists $env(HOME)/.mh_profile]} {
      if [Confirm "No .mh_profile!\nCan't run without one\n\nCreate it?\n"] {
        eval exec $cmd(cp) $tkmh_lib/dot_mh $env(HOME)/.mh_profile
        MsgWindow "Created $env(HOME)/.mh_profile\
                   \nYou need to configure it by choosing Configure \
                   MH from the main window.\n"
      } else {
        MsgWindow "As you want....\
                   \n\nI will abort now. If you want to run tkmh again, \
                   you need to create a .mh_profile in your home directory.\n"
        exit
      }
    }
    if {![file exists $env(HOME)/.tkmh]} {
        eval exec $cmd(cp) $tkmh_lib/dot_tkmh $env(HOME)/.tkmh
    }
    
    ConfTkMH:Read
    # Check to see if the version number of the .tkmh is the same as
    # that for the program.
    if [string compare $VERSION $DOTVERSION] {
        MsgWindow "The version of TkMH is different from the \
                   version of your .tkmh!\nThis could cause problems. \
                   \n\nExit TkMH, and remove the file .tkmh in your \
                   Home directory ($env(HOME)).\n"
    }

    ConfMH:Read
    BuildAliasList
    BuildFolList

    bind Text <2> {if {[catch {selection get} s]} {set s ""}
                   %W insert insert $s 
                   %W yview -pickplace insert}
    bind Text <Up> {%W mark set insert {insert - 1 line}
                    %W yview -pickplace insert}
    bind Text <Down> {%W mark set insert {insert + 1 line}
                      %W yview -pickplace insert}
    bind Text <Left> {%W mark set insert {insert - 1 char}
                      %W yview -pickplace insert}
    bind Text <Right> {%W mark set insert {insert + 1 char}
                       %W yview -pickplace insert}
    bind Entry <Left> {set x [%W index insert]
                       set x [expr $x - 1]
                       %W icursor $x
                       tk_entrySeeCaret %W}
    bind Entry <Right> {set x [%W index insert]
                        set x [expr $x + 1]
                        %W icursor $x
                        tk_entrySeeCaret %W}
}

proc BuildFolList {} {
    global MH FolList

    set FolList {}
    foreach Fol [exec $MH(bindir)/folders -fast -recurse] {
        lappend FolList $Fol
    }
}

proc BuildAliasList {} {
    global MH Aliases Addresses

    set Aliases {}; set Addresses {}
    if {![file exists [RF $MH(Path)/$MH(AliasFile)]]} return
    set Fhandle [open [RF $MH(Path)/$MH(AliasFile)] r]
    while {[gets $Fhandle Line] != -1} {
        # Add a check to the validity of the line
        regexp "\(.*\):\[ 	\]*\(.*\)" $Line dummy Alias Address
        lappend Aliases $Alias
        lappend Addresses $Address
    }
    close $Fhandle
}

proc MakeAliMenu {} {
    global Aliases

    if {[.tkmh.b.mail.m index last] >= 1} {.tkmh.b.mail.m delete 1 last}
    foreach A $Aliases {
       .tkmh.b.mail.m add command -label "$A" -command \
          "if \$PREF(ExpandAlias) {
             ComposeMsg \"\[SearchAlias $A 1\]\"
           } else {
             ComposeMsg $A
           }"
    }
}

proc ConfTkMH {} {
    global PREF FONT cmd

    set w .conftkmh
    toplevel $w
    wm title $w "Configure TkMH"

    frame $w.t

    frame $w.t.r -relief raised -borderwidth 2
    checkbutton $w.t.r.keepsent -variable PREF(KeepSent) -relief flat \
      -anchor w -text "Save messages after editing" -font $FONT(ProB)
    checkbutton $w.t.r.backup -variable PREF(BackupMsg) -relief flat \
      -anchor w -text "Backup when Deleting" -font $FONT(ProB)
    checkbutton $w.t.r.openinbox -variable PREF(OpenInbox) -relief flat \
      -anchor w -text "Open Inbox on Inc" -font $FONT(ProB)
    checkbutton $w.t.r.deicon -variable PREF(DeIcon) -relief flat \
      -anchor w -text "DeIconify when new mail" -font $FONT(ProB)
    checkbutton $w.t.r.expand -variable PREF(ExpandAlias) -relief flat \
      -anchor w -text "Use Address rather than Alias" -font $FONT(ProB)
    checkbutton $w.t.r.monoshow -variable PREF(ShowFont) \
      -onvalue $FONT(FixR) -offvalue $FONT(ProR) -font $FONT(ProB) \
      -relief flat -anchor w -text "Fixed Font Showing Message"
    checkbutton $w.t.r.monoedit -variable PREF(EditFont) \
      -onvalue $FONT(FixR) -offvalue $FONT(ProR) -font $FONT(ProB) \
      -relief flat -anchor w -text "Fixed Font Editing Message"
    checkbutton $w.t.r.wrap -variable PREF(Wrap) \
      -onvalue word -offvalue none -font $FONT(ProB) \
      -relief flat -anchor w -text "Wrap long lines"

    frame $w.t.b -relief raised -borderwidth 2
    button $w.t.b.done -text "Done" -height 2 -width 6 -command \
     "set cmd(print) \[$w.e.e get\]; destroy $w"
    button $w.t.b.help -text "Help" -height 2 -width 6 -command \
      "Help TkMH.Configuration"
    button $w.t.b.save -text "Save" -height 2 -width 6 -command \
      "$w.t.b.done invoke; ConfTkMH:Write"
    button $w.t.b.reset -text "Reset" -height 2 -width 6 -command \
      "ConfTkMH:Read; ConfTkMH:Insert"

    frame $w.s -relief raised -borderwidth 2
    scale $w.s.delay -orient horizontal -to 120 -font $FONT(ProB) \
      -label "Check for Mail every" -length 245 -command {set PREF(Delay)}

    frame $w.e -relief raised -borderwidth 2
    label $w.e.l -text "   Print Command:" -font $FONT(ProB) -anchor w
    entry $w.e.e -relief sunken -font $FONT(ProR) -width 35

    pack append $w.t.r \
      $w.t.r.keepsent {top fillx} \
      $w.t.r.backup {top fillx} \
      $w.t.r.openinbox {top fillx} \
      $w.t.r.deicon {top fillx} \
      $w.t.r.expand {top fillx} \
      $w.t.r.monoshow {top fillx} \
      $w.t.r.monoedit {top fillx} \
      $w.t.r.wrap {top fillx}

    pack append $w.t.b \
      $w.t.b.reset {pady 5 top padx 10} \
      $w.t.b.help {pady 5 top} \
      $w.t.b.done {pady 5 top} \
      $w.t.b.save {pady 5 top}

    pack append $w.t \
      $w.t.r {left expand fill} \
      $w.t.b {left filly}
   
    pack append $w.s \
      $w.s.delay {top pady 10}

    pack append $w.e \
      $w.e.l {top fillx} \
      $w.e.e {top pady 10}

    pack append $w \
      $w.t {top fillx} \
      $w.s {top fillx} \
      $w.e {top fillx}

    ConfTkMH:Insert
}

proc ConfTkMH:Insert {} {
    global PREF cmd
    set w .conftkmh
    $w.s.delay set $PREF(Delay)
    $w.e.e delete 0 end
    $w.e.e insert end $cmd(print)
}

proc ConfTkMH:Read {} {
    global FONT PREF BM cmd env tkmh_lib
    global DOTVERSION

    if {![file exists [RF .tkmh]]} {
        MsgWindow "I'm in trouble!\
                   \nNo $env(HOME)/.tkmh...\
                   \n\nI have to create a new one.\n"
        eval exec $cmd(cp) $tkmh_lib/dot_tkmh [RF .tkmh]
    }
    source [RF .tkmh]
}

proc ConfTkMH:Write {} {
    global env tkmh_lib PREF cmd FONT
    
    if [file exists [RF .tkmh]] {
      set CfgFile [RF .tkmh]
    } else {
      if [file exists $tkmh_lib/dot_tkmh] {
        set CfgFile $tkmh_lib/dot_tkmh
      } else {
        MsgWindow "TkMH not correctly installed!"
        exit
      }
    }
    
    set Cfg [eval exec $cmd(cat) $CfgFile]
    regsub {(set PREF\(KeepSent\)[ 	]+)([0-9]+)} "$Cfg" "\\1$PREF(KeepSent)" Cfg
    regsub {(set PREF\(BackupMsg\)[ 	]+)([0-9]+)} "$Cfg" "\\1$PREF(BackupMsg)" Cfg
    regsub {(set PREF\(OpenInbox\)[ 	]+)([0-9]+)} "$Cfg" "\\1$PREF(OpenInbox)" Cfg
    regsub {(set PREF\(ExpandAlias\)[ 	]+)([0-9]+)} "$Cfg" "\\1$PREF(ExpandAlias)" Cfg
    regsub {(set PREF\(Delay\)[ 	]+)([0-9]+)} "$Cfg" "\\1$PREF(Delay)" Cfg
    regsub {(set PREF\(DeIcon\)[ 	]+)([0-9]+)} "$Cfg" "\\1$PREF(DeIcon)" Cfg
    regsub {(set PREF\(Wrap\)[ 	]+)(word|none)} "$Cfg" "\\1$PREF(Wrap)" Cfg
    foreach font {ShowFont EditFont} {
      if {$PREF($font) == $FONT(ProR)} {set SubFont ProR} else {set SubFont FixR}
      regsub "\(set PREF\\($font\\)\[ 	\]+\\$\)\(FONT\)\(\\(ProR\\)|\\(FixR\\)\)" "$Cfg" "\\1\\2\($SubFont\)" Cfg
    }
    regsub {(set cmd\(print\)[ 	]+\{)(.*)(\})} "$Cfg" "\\1$cmd(print)\\3" Cfg
    set FHandle [open [RF .tkmh] w]
    puts $FHandle $Cfg
    close $FHandle
}

proc ConfMH {} {
    global MH FONT FolList

    set ew 40

    set w .confmh
    catch {destroy $w}
    toplevel $w
    wm title $w "Configure MH"

    frame $w.e -relief raised -borderwidth 2

    frame $w.e.l
    label $w.e.l.maildrop -text "Maildrop:" -anchor e -font $FONT(ProB)
    label $w.e.l.dir -text "Mail Dir:" -anchor e -font $FONT(ProB)
    label $w.e.l.drafts -text "Drafts:" -anchor e -font $FONT(ProB)
    label $w.e.l.inbox -text "Inbox:" -anchor e -font $FONT(ProB)
    label $w.e.l.alias -text "Alias file:" -anchor e -font $FONT(ProB)
    label $w.e.l.sig -text "Signature file:" -anchor e -font $FONT(ProB)

    frame $w.e.e
    entry $w.e.e.maildrop -relief sunken -width $ew -font $FONT(ProR) -borderwidth 3
    entry $w.e.e.dir -relief sunken -width $ew -font $FONT(ProR) -borderwidth 3
    entry $w.e.e.drafts -relief sunken -width $ew -font $FONT(ProR) -borderwidth 3
    entry $w.e.e.inbox -relief sunken -width $ew -font $FONT(ProR) -borderwidth 3
    entry $w.e.e.alias -relief sunken -width $ew -font $FONT(ProR) -borderwidth 3
    entry $w.e.e.sig -relief sunken -width $ew -font $FONT(ProR) -borderwidth 3

    frame $w.b -relief raise -borderwidth 2
    button $w.b.cancel -text "Cancel" -height 2 -width 6 -command "destroy $w"
    button $w.b.help -text "Help" -height 2 -width 6 -command \
      "Help MH.Configuration"
    button $w.b.save -text "Save" -height 2 -width 6 -command \
     "set MH(MailDrop) \[$w.e.e.maildrop get\]
      set MH(Path) \[$w.e.e.dir get\]
      set MH(Draft-Folder) \[$w.e.e.drafts get\]
      set MH(Inbox) \[$w.e.e.inbox get\]
      set MH(AliasFile) \[$w.e.e.alias get\]
      set MH(SigFile) \[$w.e.e.sig get\]
      ConfMH:Write
      ConfMH:Read
      BuildAliasList
      BuildFolList
      if [winfo exists .tkmh] {
          MakeAliMenu
          DisplayFols \$FolList
      }
      catch {destroy $w}"
    button $w.b.reset -text "Reset" -height 2 -width 6 -command \
      "ConfMH:Read; ConfMH:Insert"
    
    pack append $w.e.l \
      $w.e.l.maildrop {top fillx} \
      $w.e.l.dir {top fillx} \
      $w.e.l.drafts {top fillx} \
      $w.e.l.inbox {top fillx} \
      $w.e.l.alias {top fillx} \
      $w.e.l.sig {top fillx}

    pack append $w.e.e \
      $w.e.e.maildrop {top fillx} \
      $w.e.e.dir {top fillx} \
      $w.e.e.drafts {top fillx} \
      $w.e.e.inbox {top fillx} \
      $w.e.e.alias {top fillx} \
      $w.e.e.sig {top fillx}

    pack append $w.e \
      $w.e.l {left padx 10 pady 30} \
      $w.e.e {left padx 20 pady 30}

    pack append $w.b \
      $w.b.reset {pady 5 top padx 10} \
      $w.b.cancel {pady 5 top} \
      $w.b.help {pady 5 top} \
      $w.b.save {pady 5 top}

    pack append $w \
      $w.e {left filly} \
      $w.b {left filly}
      
      ConfMH:Insert
}

proc ConfMH:Read {} {
    global MH env cmd tkmh_lib

    if {![file exists $env(HOME)/.mh_profile]} {
        MsgWindow "I'm in trouble!\
                         \nNo $env(HOME)/.mh_profile...\
                         \n\nI have to create a new one\n"
        eval exec $cmd(cp) $tkmh_lib/dot_mh $env/(HOME)/.mh_profile
    }

    set FHandle [open $env(HOME)/.mh_profile r]
    while {[gets $FHandle Line] != -1} {
      regexp {^(.*):[   ](.*)[  ]*$} $Line Line Field Value
      switch -- $Field {
        Path           {set MH(Path) $Value}
        Aliasfile      {set MH(AliasFile) $Value}
        Draft-Folder   {set MH(Draft-Folder) [string trimleft $Value "+"]}
        Inbox          {set MH(Inbox) [string trimleft $Value "+"]}
        MailDrop       {set MH(MailDrop) $Value}
        Sigfile        {set MH(SigFile) $Value}
      }
    }
    close $FHandle

    # Check if Maildir exists, if not, Create it
    if {![file isdir [RF $MH(Path)]]} {
        if [file isfile [RF $MH(Path)]] {
            MsgWindow "$MH(Path) is not a directory"
            exit
        }
        if [Confirm "$MH(Path) does not exist. Create it?"] {
            eval exec $cmd(mkdir) [RF $MH(Path)]
            eval exec $cmd(mkdir) [RF $MH(Path)/$MH(Inbox)]
            eval exec $cmd(mkdir) [RF $MH(Path)/$MH(Draft-Folder)]
        } else {
            MsgWindow "Cannot run without a Mailing dir:\
                 \nChange your settings and try again."
            ConfMH
            tkwait window .confmh
        }
    }
    # If it still doesn't exist, exit
    if {![file isdir [RF $MH(Path)]]} {
        MsgWindow "Exiting...."
        exit
    }
    # Check if inbox and drafts exist, and create them
    if {![file isdir [RF $MH(Path)/$MH(Inbox)]]} {
        if {![file isfile [RF $MH(Path)/$MH(Inbox)]]} {
            eval exec $cmd(mkdir) [RF $MH(Path)/$MH(Inbox)]
        } else {
            eval exec $cmd(mv) [RF $MH(Path)/$MH(Inbox)] \
                               [RF $MH(Path)/$MH(Inbox)].file
            eval exec $cmd(mkdir) [RF $MH(Path)/$MH(Inbox)]
        }
    }
    if {![file isdir [RF $MH(Path)/$MH(Draft-Folder)]]} {
        if {![file isfile [RF $MH(Path)/$MH(Draft-Folder)]]} {
            eval exec $cmd(mkdir) [RF $MH(Path)/$MH(Draft-Folder)]
        } else {
            eval exec $cmd(mv) [RF $MH(Path)/$MH(Draft-Folder)] \
                               [RF $MH(Path)/$MH(Draft-Folder)].file
            eval exec $cmd(mkdir) [RF $MH(Path)/$MH(Draft-Folder)]
        }
    }
}

proc ConfMH:Insert {} {
    global MH
    set w .confmh
    $w.e.e.maildrop delete 0 end
    $w.e.e.dir delete 0 end
    $w.e.e.drafts delete 0 end
    $w.e.e.inbox delete 0 end
    $w.e.e.alias delete 0 end
    $w.e.e.sig delete 0 end
    $w.e.e.maildrop insert end $MH(MailDrop)
    $w.e.e.dir insert end $MH(Path)
    $w.e.e.drafts insert end $MH(Draft-Folder)
    $w.e.e.inbox insert end $MH(Inbox)
    $w.e.e.alias insert end $MH(AliasFile)
    $w.e.e.sig insert end $MH(SigFile)
}

proc ConfMH:Write {} {
    global env tkmh_lib MH cmd
    
    if {![file exists $env(HOME)/.mh_profile]} {
      if {![Confirm "No .mh_profile in your directory!\
                     \nI can't run without it.\
                     \nCreate it?"]} {
        exit
      } else {
        if {![file exists $tkmh_lib/dot_mh]} {
          MsgWindow "TkMH not correctly installed!"
        } else {
          eval exec $cmd(cp) $tkmh_lib/dot_mh $env(HOME)/.mh_profile
        }
      }
    }
        
    # Put the options that control TkMH's behaviour in a list
    # Be intelligent about foldernames (prepend a + if in Mail directory)
    # and the filenames for .Signature-file and AliasFile
    lappend Cfg "Path: $MH(Path)"
    lappend Cfg "Inbox: +$MH(Inbox)"
    lappend Cfg "Draft-Folder: +$MH(Draft-Folder)"
    lappend Cfg "MailDrop: $MH(MailDrop)"
    lappend Cfg "Aliasfile: $MH(AliasFile)"
    lappend Cfg "Sigfile: $MH(SigFile)"
    
    # Append the other settings from the .mh-profile
    set FHandle [open [RF .mh_profile] r]
    while {[gets $FHandle Line] != -1} {
      regexp {^(.*):[ 	]*(.*)[         ]*$} $Line Line Field Value
      switch -- $Field {
          Path           {-}
          Aliasfile      {-}
          Draft-Folder   {-}
          Inbox          {-}
          MailDrop       {-}
          Sigfile {continue}
          default        {}
      }
      lappend Cfg "$Field: $Value"
    }
    close $FHandle
    set FHandle [open [RF .mh_profile] w]
    foreach Line "$Cfg" {
      puts $FHandle $Line
    }
    close $FHandle
}
