/*****************************
 * File : eif_tcl.c
 * 
 * Purpose : low level support for Eiffel class TCL_INTERPRETER and descendants.
 *
 * Author : stephan.cs.tu-berlin.de (Stephan Herrmann)
 * 
 * Date : 02.03.1994
 *****************************/


#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "tcl.h"
#include "tk.h"
#include "cecil.h"
#include "hector.h"
#include "plug.h"

/* Completely dummy, just to please libtcl.a: */
int Tcl_AppInit(interp)Tcl_Interp * interp; {};

/* The following serves to please ld. Don't ask me who needs it ! */
void __main () {};


/* 
 * interpreter handel to be kept by Eiffel object: 
 */
typedef struct {
        Tcl_Interp *interp;  /* handle for tcl-calls           */
	Tk_Window   w;       /* main window, if tk-interpreter */
	int result;          /* result of current callback     */
      } TCL_STRUCT;      

/* 
 * ClientData for tcl commands, pointing to all information needed to 
 * execute a call_back:
 */
typedef struct{
	EIF_OBJ   target;  /* target object for callback               */
	EIF_PROC callback; /* callback routine                         */
	EIF_PROC set_argc, /* callback for initializing argument array */
	         set_arg;  /* callback to set an argument              */
        TCL_STRUCT *interp;/* struct for the associated interpreter    */
      } CALL_BACK_STRUCT; 

/* --------------------------------------------------------------------------
 * exported routine: c_create_tcl_interpreter
 *
 *    name         C-Type        Eiffel-type      Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 * arguments :
 *    none
 * return :        TCL_STRUCT *  POINTER          Handle for this interpreter 
 *
 * effect : 
 *    Create and initialize a tcl interpreter
 * --------------------------------------------------------------------------
 */

void *c_create_tcl_interpreter () 
{

	TCL_STRUCT *tcl_struct;
	char *args, buf[130];
	int argc;
	char ** argv;

	tcl_struct =(TCL_STRUCT *)malloc (sizeof(TCL_STRUCT));
    	tcl_struct->interp = Tcl_CreateInterp();

    /*
     *  Set the "tcl_interactive" variable.
     */

        Tcl_SetVar(tcl_struct->interp, "tcl_interactive", "0", TCL_GLOBAL_ONLY);

    /*
     * Make dummy command-line arguments available in the Tcl variables 
     * "argc" and "argv".
     */
	
	argc=1;
        argv=(char **)malloc(sizeof(char *));
        argv[0]=strdup("EIFTCL");;
        args = Tcl_Merge(argc-1, argv+1);
        Tcl_SetVar(tcl_struct->interp, "argv", args, TCL_GLOBAL_ONLY);
        ckfree(args);
        sprintf(buf, "%d", argc-1);
        Tcl_SetVar(tcl_struct->interp, "argc", buf, TCL_GLOBAL_ONLY);

    /*
     *  Execute Tcl's initialization script
     */

	if (Tcl_Init(tcl_struct->interp) == TCL_ERROR) {
	  fprintf(stderr, "Tcl_AppInit failed: %s\n", tcl_struct->interp->result);
	  return;
	}
	return ((void *)tcl_struct);
}

/* --------------------------------------------------------------------------
 * exported routine: c_tcl_result_string
 *
 * arguments :
 *    name         C-Type        Eiffel-type     Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *    tcl_struct   TCL_STRUCT *  POINTER         Handle for my interpreter
 *    
 * return:         char *        ANY (C-string)  return code from interpreter
 *
 * effect    : 
 *    return the current value of tcl-result
 * --------------------------------------------------------------------------
 */
char *c_tcl_result_string (tcl_struct) 
EIF_POINTER tcl_struct;
{
  return(((TCL_STRUCT *)tcl_struct)->interp->result );
}


/* --------------------------------------------------------------------------
 * exported routine: c_set_tcl_result
 *
 * arguments :
 *    name         C-Type        Eiffel-type     Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *    tcl_struct   TCL_STRUCT *  POINTER         Handle for my interpreter
 *    code         int           INTEGER         Result code
 *    result       char *        ANY             Result (as string)
 * effect    : 
 *    set value of tcl-result to be reported when current callback returns
 * --------------------------------------------------------------------------
 */
char *c_set_tcl_result (tcl_struct, code, result) 
EIF_POINTER tcl_struct;
int code;
char *result;
{
  ((TCL_STRUCT *)tcl_struct)->result = code;
  Tcl_SetResult(((TCL_STRUCT *)tcl_struct)->interp, result, TCL_VOLATILE );
}


/* --------------------------------------------------------------------------
 * exported routine: c_tcl_eval
 *
 * arguments :
 *    name         C-Type        Eiffel-type     Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *    tcl_struct   TCL_STRUCT *  POINTER         Handle for my interpreter
 *    cmd          char *        ANY             commandline
 *    
 * return:         int           INTEGER         return code from interpreter
 *
 * effect    : 
 *    execute a tcl command.
 * --------------------------------------------------------------------------
 */
int c_tcl_eval (tcl_struct, cmd) 
EIF_POINTER tcl_struct;
char *cmd;
{
  return(Tcl_Eval(((TCL_STRUCT *)tcl_struct)->interp,
		  cmd, 
		  0, 
		  (char **) NULL)
	 );
}


/* --------------------------------------------------------------------------
 * exported routine: c_tcl_eval_file  
 *
 * arguments :
 *    name         C-Type        Eiffel-type     Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *    tcl_struct   TCL_STRUCT *  POINTER         Handle for my interpreter
 *    filename     char *        ANY             tcl-file to be executed
 *
 * return: INTEGER;
 *    return code from interpreter.
 *
 * effect    : 
 *    execute the commands from the given tcl-file.
 * --------------------------------------------------------------------------
 */

int c_tcl_eval_file (tcl_struct, filename)
EIF_POINTER tcl_struct;
char *filename;
{       
    return( Tcl_EvalFile (((TCL_STRUCT *)tcl_struct)->interp, filename));
};

/* --------------------------------------------------------------------------
 * exported routine: c_create_tcl_command
 *
 * arguments :
 *    name         C-Type        Eiffel-type  Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *    target       EIF_OBJ       TCL_COMMAND  Target on which to execute the 
 *                                            callback
 *    tcl_struct   TCL_STRUCT *  POINTER      Handle for my interpreter
 *    
 *    callback     EIF_PROC      POINTER      callback-pointer
 *    
 *    tcl_name     char *        ANY (see STRING.to_c)
 *        name under which this routine can be called from within tcl.
 *    set_argc     EIF_PROC      POINTER      callback to initialize an
 *        Eiffel ARRAY[STRING] for argument passing. Eiffel declaration is:
 *        init_args (count : INTEGER) is ...
 *    set_arg      EIF_PROC      POINTER      callback to pass one argument
 *        to Eiffel. Eiffel declaration is:
 *        set_arg (arg : ANY; index : INTEGER) is ...
 *
 * return:  --
 *
 * effect    : 
 *    Install an Eiffel routine so that it can be called from within tcl.
 * --------------------------------------------------------------------------
 */

/* defined later in this file:  */
extern int c_dispatch();
extern void c_delete_cmd();

void c_create_tcl_command (target, tcl_struct, callback, tcl_name, 
			   set_argc, set_arg)
EIF_OBJ target; 
EIF_POINTER tcl_struct;
EIF_PROC callback;
char *tcl_name; 
EIF_PROC set_argc, set_arg;
{
  CALL_BACK_STRUCT 
  *cb_str  = (CALL_BACK_STRUCT *)malloc(sizeof(CALL_BACK_STRUCT));

  cb_str->target = eif_adopt(target);
  cb_str->interp = (TCL_STRUCT *)tcl_struct;
  cb_str->callback = callback;
  cb_str->set_argc = set_argc;
  cb_str->set_arg = set_arg;
  Tcl_CreateCommand (((TCL_STRUCT *)tcl_struct)->interp, 
		     tcl_name, 
		     c_dispatch,
		     (ClientData)cb_str, 
		     c_delete_cmd);
}; 

int c_dispatch(clientData, interp, argc, argv )
ClientData clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{              
   int i;
   CALL_BACK_STRUCT *cb_str=(CALL_BACK_STRUCT *)clientData;
   EIF_OBJ target = eif_access(cb_str->target);

   cb_str->interp->result = TCL_OK;       /* initialize result to OK */

   (* (cb_str->set_argc))(target, argc);  /* init argument passing   */
   for (i=0;i<argc;i++) {                 /* pass all args to Eiffel */
      (* (cb_str->set_arg))(target, argv[i], i+1);
   };
   (* (cb_str->callback))(target);        /* execute the callback       */
   return (cb_str->interp->result);       /* return stored result       */
                                          /* may be set by c_set_result */
}

void c_delete_cmd (clientData)
ClientData clientData;
{
   CALL_BACK_STRUCT 
   *cb_str=(CALL_BACK_STRUCT *)clientData;

   eif_wean(cb_str->target);
   free (cb_str);
}

/* --------------------------------------------------------------------------
 * exported routine: c_set_tcl_var
 *
 * arguments :
 *    name         C-Type        Eiffel-type  Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *    tcl_struct   TCL_STRUCT *  POINTER      Handle for my interpreter
 *    name         char *        ANY          variable name
 *    value        char *        ANY          variable value
 *    
 *
 * return:  ---
 *
 * effect    : 
 *    set a tcl variable.
 * --------------------------------------------------------------------------
 */
void c_set_tcl_var (tcl_struct, name, value)
EIF_POINTER tcl_struct;
char *name;
char *value;
{
  Tcl_SetVar(((TCL_STRUCT *)tcl_struct)->interp, name, value, TCL_GLOBAL_ONLY);
}

/* --------------------------------------------------------------------------
 * exported routine: c_get_tcl_var
 *
 * arguments :
 *    name         C-Type        Eiffel-type  Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *    tcl_struct   TCL_STRUCT *  POINTER      Handle for my interpreter
 *    name         char *        ANY          variable name
 *    
 * return:         char *        ANY          variable value
 *
 * effect    : 
 *    get the value of a tcl variable.
 * --------------------------------------------------------------------------
 */
EIF_OBJ c_get_tcl_var (tcl_struct, name, voidref)
EIF_POINTER tcl_struct;
char *name;
EIF_OBJ voidref;
{
  char *value;

  value = Tcl_GetVar(((TCL_STRUCT *)tcl_struct)->interp, 
				 name, TCL_LEAVE_ERR_MSG);
  return (value?(EIF_OBJ)(value):voidref);
}


/******************************************************************************
        TK
******************************************************************************/


/* --------------------------------------------------------------------------
 * exported routine: c_init_tk
 *
 * arguments :
 *    name         C-Type        Eiffel-type  Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *    tcl          TCL_STRUCT *  POINTER      struct of the interpreter
 *    title        char *        INTEGER      window title
 *
 * return : -
 *
 * effect    : 
 *    Create and initialize a tk main window
 * --------------------------------------------------------------------------
 */

void c_init_tk (tcl, title) 
EIF_POINTER tcl;
char *title;
{

	TCL_STRUCT *tcl_struct = (TCL_STRUCT *)tcl;

    	tcl_struct->w = Tk_CreateMainWindow(tcl_struct->interp, 
					   NULL,
					   title,
					   "Tk");
    	if ((tcl_struct->w == NULL) || 
	    (Tk_Init(tcl_struct->interp) == TCL_ERROR)) {
		fprintf(stderr, "%s\n", tcl_struct->interp->result);
		exit(1);
    	}
}


/* --------------------------------------------------------------------------
 * exported routine: c_tk_show
 *
 * arguments :
 *    name         C-Type        Eiffel-type  Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *    tcl          TCL_STRUCT *  POINTER      Handle for my interpreter
 *    modal_flag   int
 *
 * return:
 *    none
 *
 * effect    : 
 *    show the tk main window.
 *    depending on modal_flag either enter tk_main_loop or ...
 * --------------------------------------------------------------------------
 */

static int main_loop_active = 0; 
                            /* make sure, Tk_MainLoop is only entered once */

void c_tk_show (tcl, modal_flag)
EIF_POINTER *tcl;
int modal_flag;
{
	int rc;
	char *wait_args[3];
	TCL_STRUCT *tcl_struct = (TCL_STRUCT *)tcl;

 	(void) Tcl_Eval(tcl_struct->interp, "update", 0, (char **) NULL);
	switch(modal_flag){
	case (0):
	  if (!main_loop_active){
	    main_loop_active=1;
	    Tk_MainLoop();
	  };
	  break;
	case (1):
	  wait_args[0] = "tkwait";
	  wait_args[1] = "window";
	  wait_args[2] = ".";
	  Tk_TkwaitCmd((ClientData)(tcl_struct->w), tcl_struct->interp, 
		       3, wait_args);
	  break;
	case (2):
	  Tk_Grab(tcl_struct->interp, tcl_struct->w, 1 /*GRAB_GLOBAL*/);
	  while (tcl_struct->w){
	    Tk_DoOneEvent(0);
	  };
	};
};

/* --------------------------------------------------------------------------
 * exported routine: c_close_win
 *
 * arguments :
 *    name         C-Type        Eiffel-type  Description
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *    tcl_struct   TCL_STRUCT *  INTEGER      Handle for my interpreter
 *
 * return: none
 *
 * effect    : 
 *    close this Interpreters main window
 * --------------------------------------------------------------------------
 */

void c_close_win (tcl_struct)
TCL_STRUCT *tcl_struct;
{
/*
  int rc=Tcl_Eval(tcl_struct->interp, "destroy .", 0, (char **)NULL);
*/
  Tk_DestroyWindow(tcl_struct->w);
  tcl_struct->w = 0;
};



