#---------------------------------------------------------------------------
#
#	Implementation of an association list form, i.e. a widget with
#	two columns: a column of labels, and a column of associated
#	values. The value fields are mouse-sensitive and may produce a
#	menu or other selection mechanism when clicked on with the
#	left mouse button.
#
#	The 'alist' parameter of 'new(alist)' is a list of triples:
#	a label, an initial value, and a menu action.
#
#---------------------------------------------------------------------------

defwidget Alist _Filter

defmethod Alist new {name args} {

  args	text alist actions buttons {label1 Attributes} {label2 Values} \
	inputlabel layout embedded wait

  if { $buttons == {} } {
    if { [assoc Ok+Save $actions] == {} } {
      set buttons { Ok }
    } {
      set buttons { Ok Ok+Save }
    }
  }

  if { [assoc Help $actions] == {} } {
    lappend actions {Help {} Alistbox/Help}
  }

  if { $embedded == "true" } {
    Frame new $name -relief flat
  } {
    Toplevel new $name -title $text \
	-buttons $buttons \
	-handler [list $name _button] \
	-actions $actions
  }
  defsuper $name Alist

  set bm_menu [the(image) menu]
  set bm_empty [the(image) empty]
  set bm_input [the(image) input]
  set bm_action [the(image) action]

  Frame new $name.head \
	-layout {top padx 5 pady 5 fillx} \
	-relief raised
  Label new $name.head.a \
	-layout {left frame nw padx 5} \
	-relief flat -text $label1 -width 20
  Bitmap new $name.head.m \
	-layout {left frame nw padx 5} \
	-relief flat -bitmap @$bm_empty
  Label new $name.head.v \
	-layout {left frame nw padx 5} \
	-relief flat -text $label2 -width 32

  if { $inputlabel != {} && [assoc input $alist] == "" } {
    set inputlabel {}
  }

  set start_a 5
  set start_m [expr 10+$start_a+[winfo reqwidth $name.head.a]]
  set start_v [expr 10+$start_m+[winfo reqwidth $name.head.m]]
  set start_x [expr 10+$start_v+[winfo reqwidth $name.head.v]]
  set size_y [expr [winfo reqheight $name.head.a]/2*2]

  set f $name.avm
  Frame new $f -layout {top padx 5 pady 5 expand fill} -relief raised
  set pos 0
  set ypos 5
  foreach spec $alist {
    set need_a 1; set need_v 1; set need_m 1; set bonus 0.5

    case [lindex $spec 0] in {
    {menu} {
	# Fixed menu of options
	set m [Menubutton new $f.m$pos -bitmap @$bm_menu]
	foreach item [lrange $spec 3 end] {
	  $m addItem -text $item -action [list $f.v$pos set $item]
	}
      }
    {input} {
	# Input from input line - display in label
	if { $inputlabel == {} } {
	  Bitmap new $f.m$pos -bitmap @$bm_empty
	} {
	  Button new $f.m$pos -bitmap @$bm_input -action \
	    [list $name _input $f.v$pos [lrange $spec 3 end]]
	}
      }
    {text} {
	# Input from input line - display in text item
	if { $inputlabel == {} } {
	  Bitmap new $f.m$pos -bitmap @$bm_empty
	} {
	  Button new $f.m$pos -bitmap @$bm_input -action \
	    [list $name _input $f.v$pos [lrange $spec 3 end]]
	}
	set need_v 2
      }
    {consttext} {
	# Constant line of text
	set need_m 0; set need_v 2
      }
    {command} {
	# Special command activation - allows dynamic recomputation of options
	Button new $f.m$pos -bitmap @$bm_action \
	  -action [concat [lrange $spec 3 end] $f.v$pos]
      }
    {const} {
	# Constant label
	set need_m 0
      }
    {*} {
	# Space? The final frontier?
	set need_a 0; set need_m 0; set need_v 0
	set bonus 0
      }
    }

    if { $need_a == 1 } {
      Label new $f.a$pos -relief flat -text [lindex $spec 1] -width 20
      place $f.a$pos -in $f -anchor nw -y $ypos -x $start_a
    }
    if { $need_m == 1 } {
      place $f.m$pos -in $f -anchor nw -y $ypos -x $start_m
    }
    if { $need_v > 0 } {
      if { $need_v == 1 } {
	Label new $f.v$pos -relief sunken -text [lindex $spec 2] \
		-width 32 -textfont text
      } {
	Label new $f.v$pos -relief sunken -text [lindex $spec 2] \
		-width 32 -textfont text -edittext true
	set bonus -1
      }
      if { $inputlabel != {} } {
        bind $f.v$pos <Button-1> [list $name _setinput $f.v$pos]
      }
      place $f.v$pos -in $f -anchor nw -y $ypos -x $start_v
    }

    if { $bonus == -1 } {
      incr ypos [expr [winfo reqheight $f.v$pos]+$size_y/2]
    } {
      incr ypos [expr $size_y*(1+$bonus)]
    }
    incr pos
  }

  # The maximum number of items in this alist.
  $name slot _max $pos

  $f configure -width $start_x -height $ypos
  if { $inputlabel != {} } {
    Inputline new $name.input -layout {bottom padx 5 pady 5 fillx} \
	-edittext true -width 32 -label $inputlabel -input any
    $name.input resize $start_x
  }

  $name layout $layout

  if { $wait == "true" } {
    tkwait window $name
  }

  return $name
}

#---------------------------------------------------------------------------

Window addDemo Alist

defmethod Alist demo {} {

  set name [anon]

  Alist new $name -alist [list \
	{const {Activity Name} {Assemble e-box}} \
	{space} \
	{menu {Start Earliest} 0 0 10 20 30 40 50 60 70 80} \
	{command {Start Latest} * puts stdout} \
	{const {Duration} 10} \
	{space} \
	[list input {Activity Id} 8732 $name chooseNumber] \
	[list input {Another Id} 8732 $name chooseNumberX] \
	{text {Description} "Get parts and assemble electronics box."} \
	] \
	-text		{Attribute Definition Form} \
	-label1		ATTRIBUTES \
	-label2		VALUES \
	-inputlabel	{New Value} \
	-buttons	{Constraints Effects Resources {} Ok Ok+Save} \
	-actions	{ \
	{Ok		Ok action Ok} \
	{Ok+Save	{Ok/Speichern} action Ok+Save} \
	{Constraints	{} action Constraints} \
	{Effects	{} action Effects} \
	{Resources	{} action Resources} \
	}
}

#---------------------------------------------------------------------------

defmethod Alist clear {{value {}}} {

  $self.input set $value
}

defmethod Alist items {} {

  set f $self.avm
  set list {}
  for {set i 0} {$i < [$self slot _max]} {incr i} {
    if { ! [catch {$f.a$i get}] } {
      set label [$f.a$i get]
      set value [$f.v$i get]
      lappend list [concat [list $label] $value]
    }
  }
  return $list
}

defmethod Alist _button {button action} {

  case $button in {

  {Help Dismiss} {
	$self $button $action
    }
  default {
	if { $action != {} } {
	  set res [eval [concat $action $box]]
	  if { $res != {} } {
	    new(problem) Prestige/failed $res
	  }
	}
    }
  }
}

defmethod Alist _setinput {value} {

  $self.input set [$value get]
}

defmethod Alist _input {value form} {

  set new [$self.input get]
  set old [$value get]

  if { $form != {} } {
    lappend form $old $new
    set new [uplevel #0 [list eval $form]]
  }
  if { $new != $old } {
    $value set $new
  }
}
