#---------------------------------------------------------------------------
#
#	Canvas Creation
#
#---------------------------------------------------------------------------

defwidget Canvas
defwidget General-Canvas Canvas

defmethod Canvas new {name args} {

  args	background width height scroll yscroll xscroll \
	title {titlefont bold} {relief sunken} layout

  if { $background == {} } {
    set background [Color slot bg]
  }

  set options [list -relief $relief -borderwidth 3 -background $background]
  if { $width != {} } {
    lappend options -width $width
  }
  if { $height != {} } {
    lappend options -height $height
  }

  if { $scroll != {} } {
    set canvas $name.c
    Frame new ${name} -layout $layout -width ${width} -height ${height}
    defsuper $name General-Canvas

    Label newnonempty $name.title -text $title -textfont $titlefont

    if { $xscroll == {} } {
      set xscroll [list $canvas! xview]
    }
    Scrollbar new ${name}.horiz -layout {bottom fillx} \
	-dir horizontal -action $xscroll
    if { $yscroll == {} } {
      Scrollbar new ${name}.vert -layout {right filly} \
	-dir vertical -action [list $canvas! yview]
      set yscroll [list $name.vert! set]
    }

    if { [llength $scroll] == 4 } {
      set o_x [expr [lindex $scroll 0]-5]
      set o_y [expr [lindex $scroll 1]-5]
      set s_x [expr [lindex $scroll 2]+5]
      set s_y [expr [lindex $scroll 3]+5]
    } {
      set o_x -5
      set o_y -5
      set s_x [expr [lindex $scroll 0]+5]
      set s_y [expr [lindex $scroll 1]+5]
    }
    lappend options \
	-scrollregion [list $o_x $o_y $s_x $s_y] \
	-xscroll [list $name.horiz! set] \
	-yscroll $yscroll

    eval [concat canvas $canvas $options]

    bind $canvas <Button-2> "$canvas! scan mark %x %y"
    bind $canvas <Button2-Motion> "$canvas! scan dragto %x %y"

    Canvas instantiate $canvas {expand fill}
    $canvas! create rectangle 0 0 [expr $s_x-6] [expr $s_y-6] \
	-width 1 -outline black -fill $background -tags MARGIN_RECT

    return $name

  } {

    eval [concat [list canvas $name] $options]
    Canvas instantiate $name $layout

  }
}

defmethod Canvas adjust {scroll} {

  if { [llength $scroll] == 4 } {
    set o_x [expr [lindex $scroll 0]-5]
    set o_y [expr [lindex $scroll 1]-5]
    set s_x [expr [lindex $scroll 2]+5]
    set s_y [expr [lindex $scroll 3]+5]
  } {
    set o_x -5
    set o_y -5
    set s_x [expr [lindex $scroll 0]+5]
    set s_y [expr [lindex $scroll 1]+5]
  }

  if { [is $self General-Canvas] } {
    set canvas $self.c
  } {
    set canvas $self
  }

  $canvas! configure -scrollregion [list $o_x $o_y $s_x $s_y]
  $canvas! coords MARGIN_RECT \
	[expr $o_x+5] [expr $o_y+5] [expr $s_x-5] [expr $s_y-5]

  return $self
}

defmethod Canvas print {args} {
  global system

  args	{file /tmp/canvas.ps} {colormode mono} printer action layout

  if { [is $self General-Canvas] } {
    set canvas $self.c
  } {
    set canvas $self
  }

  set scroll [lindex [$canvas! configure -scrollregion] 4]

  if { $printer != {} && $file == {} } {
    if { $system(has_defobject) } {
      set tmp "/tmp/Canvas-[getpid]-[date].ps"
    } {
      set tmp "/tmp/Canvas-[date].ps"
    }
  } {
    set tmp $file
  }

  set x [lindex $scroll 0]
  set y [lindex $scroll 1]
  set width [expr [lindex $scroll 2]-[lindex $scroll 0]]
  set height [expr [lindex $scroll 3]-[lindex $scroll 1]]

  set fit {}
  case $layout {
  {fit} {
	  set size [assoc $system(papersize) $system(papersizes)]
	  if { $width > $height } {
	    set rotate "true"
	    set size [list [lindex $size 1] [lindex $size 0]]
	  } {
	    set rotate "false"
	  }
	  set factx [expr [lindex $size 0]/$width]
	  set facty [expr [lindex $size 1]/$height]
	  if { $factx <= $facty } {
	    set fit [list -pagewidth [lindex $size 0]c]
	  } {
	    set fit [list -pageheight [lindex $size 1]c]
	  }
	}
  {best} {
	  set rotate [expr {($width <= $height) ? "false" : "true"}]
	}
  {portrait} {
	  set rotate "false"
	}
  {landscape} {
	  set rotate "true"
	}
  default {
	  set rotate "false"
	}
  }

  eval [concat [list $canvas! postscript \
	-colormode $colormode \
	-rotate $rotate \
	-file $tmp \
	-pageanchor center \
	-x $x -y $y -width $width -height $height] \
	$fit]

  if { $printer != {} } {
    set problem 0
puts stdout "*** lpr -P$printer $tmp"
    if { [catch {exec lpr "-P$printer" $tmp}] && $action != "" } {
      set problem [catch {uplevel #0 [concat $action [list $self $tmp]]}]
    }
    if { $file != {} } {
      exec /bin/rm -f $tmp
    }
    if { $system(verbose) && !$problem } {
      puts stdout "** Printed canvas $self on printer $printer"
    }
    return
  }

  if { $system(verbose) } {
    puts stdout "** Printed canvas $self to file $tmp"
  }
  return $tmp
}

defmethod General-Canvas canvas {} {
  return $self.c
}

defmethod Canvas canvas {} {
  return $self
}

defmethod Canvas overlap {x1 y1 x2 y2} {

  if { $x1 > $x2 } {
    set z $x1; set x1 $x2; set x2 $z
  }
  if { $y1 > $y2 } {
    set z $y1; set y1 $y2; set y2 $z
  }
  [$self canvas] find overlap $x1 $y1 $x2 $y2
}

Window addDemo Canvas

defmethod Canvas demo {} {

  set t [Toplevel new * -title "New Toplevel" -info true -resizable true]
  Bitmap new *$t -bitmap @[the(image) icon-Prestige] -layout top
  set c [Canvas new *$t -layout {top expand fill} -scroll {-10 -10 510 510}]
  for {set i 10} {$i < 500} {incr i 50} {
    for {set j 10} {$j < 500} {incr j 50} {
      $c newRect3D $i $j [expr $i+40] [expr $j+40] \
	-fill green -fill2 red
      $c create text [expr $i+25] [expr $j+25] -text "$i\n$j" -anchor c
    }
  }
  set p [Popup new * -parent [$c canvas]]
  foreach i {foo bar baz} {
    $p addItem -text "Command $i" -action [list $self showAction $i]
  }
  $p addOptions -list {Foo Bar Baz} -action {- showAction option}

  Button new *$t -layout {padx 20 pady 20 bottom} \
	-textfont largebold -text "Dismiss" \
	-action [list $t Dismiss]
  $t layout center
}

#---------------------------------------------------------------------------

defmethod Canvas newRect3D {x y xe ye args} {

  args	{delta 4} tags2 tags noitem fill fill2

  if { [is $self General-Canvas] } {
    set canvas $self.c
  } {
    set canvas $self
  }

  if { $fill2 == {} } {
    set fill2 [Color slot bg,dark]
  }
  if { $fill == {} } {
    set fill [Color slot bg,button]
  }

  $canvas! create polygon \
	$x $y [expr $x+$delta] [expr $y-$delta] \
	[expr $xe+$delta] [expr $y-$delta] \
	[expr $xe+$delta] [expr $ye-$delta] $xe $ye \
	-tags $tags2 -fill $fill2
  $canvas! create line \
	$x $y [expr $x+$delta] [expr $y-$delta] \
	[expr $xe+$delta] [expr $y-$delta] \
	[expr $xe+$delta] [expr $ye-$delta-1] $xe [expr $ye-1] \
	-tags $tags2 -fill black
  $canvas! create line \
	$xe $y [expr $xe+$delta] [expr $y-$delta] \
	-tags $tags2 -fill black

  if { $noitem != "true" } {
    $canvas! create rectangle $x $y $xe $ye \
	-outline [Color slot fg] -fill $fill -width 1 \
	-tags $tags
  } {
    return
  }
}

defmethod Canvas newRectShadow {x y xe ye args} {

  args	{delta 3} tags tags2 noitem fill fill2

  if { [is $self General-Canvas] } {
    set canvas $self.c
  } {
    set canvas $self
  }

  if { $fill == {} } {
    set fill [Color slot bg,button]
  }
  if { $fill2 == {} } {
    set fill2 [Color slot bg,dark]
  }

  $canvas! create rectangle \
	[expr $x+$delta] [expr $y+$delta] [expr $xe+$delta] [expr $ye+$delta] \
	-width 1 -outline black -fill $fill2 \
	-tags $tags2

  if { $noitem != "true" } {
    $canvas! create rectangle $x $y $xe $ye \
	-outline [Color slot fg] -fill $fill -width 1 \
	-tags $tags
  } {
    return
  }
}

defmethod Canvas newTextElement {posx posy sizex sizey args} {

  args	{type Label} text textfont width tags

  if { [is $self General-Canvas] } {
    set canvas $self.c
  } {
    set canvas $self
  }

  set cmd [lindex $type 0]
  set window [anon $canvas]
  eval [concat [list $cmd new $window -width $width \
			-text $text -textfont $textfont] \
	       [lrange $type 1 end]]
  $canvas create window $posx $posy \
	-anchor nw -width $sizex -height $sizey \
	-tags [concat $tags [list [list window $window]]] \
	-window $window
  return $window
}

#---------------------------------------------------------------------------

defmethod General-Canvas DEFAULT {method args} {

  eval [concat [list $self.c $method] $args]
}

#---------------------------------------------------------------------------

defobject TextItem

defmethod TextItem get {} {

  lindex [[$self slot canvas] itemconfigure [$self slot id] -text] 4
}

defmethod TextItem set {text} {

  [$self slot canvas] itemconfigure [$self slot id] -text $text
}
