#---------------------------------------------------------------------------
#
#	Dynamically Determine Font Settings
#
#	Juergen Wagner, Aug 1992
#	J_Wagner@iao.fhg.de
#

catch "canvas ._int(font)"
._int(font) delete all
set _fontid [._int(font) create text 0 0 -text ""]

#	ARGS is a list of font names. The first available font from ARGS is
#	assigned to the global variable 'font(NAME)'.
#
defmethod Font use {name args} {

  set font [$self slot $name]
  if { $font != "" } {
    return $font
  }

  foreach fname $args {
    if { [$self exists $fname] } then {
      return [$self set $name $fname]
    }
  }

  puts stdout "** Can't find appropriate $name font."
  return fixed
}

#	FMT is a format string with one format control. The format string
#	is used to successively build font names from ARGS. The first font
#	name denoting an available font will be assigned to the global variable
#	'font(NAME)'.
#
defmethod Font usef {name fmt args} {

  set font [$self slot $name]
  if { $font != "" } {
    return $font
  }

  foreach farg $args {
    set fname [format $fmt $farg $farg]
    if { [$self exists $fname] } then {
      return [$self set $name $fname]
    }
  }

  puts stdout "** Can't find appropriate $name font."
  return fixed
}

#	Copy all font information from NAME to NEWNAME.
#
defmethod Font copy {name1 name2} {
  global system

  $self slot $name2 [$self slot $name1]
  $self slot $name2,w [$self slot $name1,w]
  $self slot $name2,h [$self slot $name1,h]
  if { $system(verbose) } {
    puts stdout "** Using $name2 font [$self slot $name2]"
  }
  return $name2
}

defmethod Font exists {fname} {
  global _fontid

  expr ![catch "._int(font) itemconfig $_fontid -font {$fname}"]
}

defmethod Font set {name fname} {
  global system

  if { $system(verbose) } {
    puts stdout "** Using $name font $fname"
  }
  $self slot $name $fname

  set info [textsize $fname X]

  $self slot $name,w [lindex $info 0]
  $self slot $name,h [lindex $info 1]

  return $fname
}

#	Return a list of two integers giving the width and height of the
#	bounding box of TEXT in FONT. FONT may be any font name.
#
proc textsize {font {text "X"}} {
  global _fontid

  set text "$text$text$text$text$text$text$text$text$text$text"

  ._int(font) itemconfigure $_fontid -text $text -font $font
  set info [._int(font) bbox $_fontid]

  return [list	[expr ([lindex $info 2]-[lindex $info 0]+9)/10] \
		[expr [lindex $info 3]-[lindex $info 1]+1] ]
}

# fonts
#	The following fonts are defined: text, small, large, huge, each in
#	a normal style "font(NAME)", a bold style "font(NAMEbold)", and some
#	in italics style "font(NAMEitalics)". In addition to these fonts, the
#	font "symbol" is defined. The font "menu" denotes a font used for
#	menu entries. "italics" and "bold" are nicknames for the respective
#	text font styles.
#

#--	Normal, italics, and bold fonts for text
#
Font usef text \
	"-*-helvetica-medium-r-normal--*-120-%s-*" 75 100
Font usef textitalics \
	"-*-helvetica-medium-o-normal--*-120-%s-*" 75 100
Font usef textbold \
	"-*-helvetica-bold-r-normal--*-120-%s-*" 75 100

Font copy textitalics italics
Font copy textbold bold

#--	A very small font
#
Font usef small	\
	"-*-helvetica-medium-r-normal--*-100-%s-%s-*-*-*-*" 75 100
Font usef smallbold \
	"-*-helvetica-bold-r-normal--*-100-%s-%s-*-*-*-*" 75 100

#--	A very large font
#
Font usef large	\
	"-*-helvetica-medium-r-normal--*-180-%s-%s-*-*-*-*" 75 100
Font usef largebold \
	"-*-helvetica-bold-r-normal--*-180-%s-%s-*-*-*-*" 75 100

#--	An even larger font
#
Font usef huge	\
	"-*-helvetica-medium-r-normal--*-240-%s-%s-*-*-*-*" 75 100
Font usef hugebold \
	"-*-helvetica-bold-r-normal--*-240-%s-%s-*-*-*-*" 75 100

#--	The font used for menu items
#
Font copy textbold menu

#--	A fixed width font
Font usef fixed	\
	"-*-courier-medium-r-normal--*-120-%s-%s-*-*-*-*" 75 100
Font usef fixedbold \
	"-*-courier-bold-r-normal--*-120-%s-%s-*-*-*-*" 75 100

#--	A font containing special symbols
Font usef symbol \
	"-*-symbol-medium-r-normal--*-120-%s-%s-*-*-*-*" 75 100
