#---------------------------------------------------------------------------
#
#	Indented List Widget
#	(c) Copyright Juergen Wagner, 1993
#
#	This widget can be used to display and edit hierarchical
#	structures (single inheritance).
#
# Operations:
#	Help => Invoke respective help message.
#	Select => Returns list of selected text items.
#	New Child => Create a new child under the selected item.
#	New Sibling => Create a new sibling after the selected item.
#	Rename => Change the name of the selected node.
#	Remove => Remove the selected node.
#	Edit => Invoke editor on selected node.
#	Ok => Confirm the current tree. Send it to the action handler.
#	Ok+Save => Same as Ok but also save data.
#		{New Print}
#
# Options:
#	Selection [single|multiple] => Single or multiple selections.
#	Names [unique|multiple] => Unique node names?
#	InitMethod, Init => Specification of source of input data.
#	Font => Font name to use for text.
#
#---------------------------------------------------------------------------

defwidget Ilist Window {
	{alignpos 24}
	{minx 32}
	{miny 10}
	{ind 3}
	{cw 320}
	{ch 320}
	{r 1.3}
	{v 1.3}
	{h 0.8}
}

defmethod Ilist new {name args} {

  args	{text {Overview}} title title2 {titlefont bold} {aligned} \
	{layout choose} actions {unique false} copyroot \
	initmethod init textfont buttons {inputlabel {New Name}} \
	{default DEFAULT} defaultCreator positions \
	{viewhelp Library/ilist-view} embedded

  set actions [$self buildActions $actions $name]

  # Add default help action
  if { [assoc Help $actions] == {} } {
    if { $positions == {} } {
      lappend actions [list Help {} Library/ilist-full]
    } {
      lappend actions [list Help {} Library/ilist-partial]
    }
  }

  # Create toplevel window
  if { $embedded == "true" } {
    Frame new $name -relief flat
    set needname 0
    set copyroot 0
  } {
    Toplevel new $name \
	-title $text \
	-resizable true \
	-buttons $buttons \
	-handler [list $name _do] \
	-actions $actions

    set needname 0
    foreach i $actions {
      if { [position [lindex $i 0] {New New_Child New_Sibling Rename}] >= 0 } {
        set needname 1
        break
      }
    }
    if { $needname } {
      Inputline new $name.name -layout {bottom padx 20 pady 20} \
	-label "$inputlabel: " -width 24
      set copyroot [expr {$copyroot=="true"}]
    } {
      set copyroot 0
    }
  }
  defsuper $name Ilist

  $name slot _default	$default
  $name slot _defaultCreator [$self buildAction $defaultCreator $name]
  $name slot _viewhelp	$viewhelp
  $name slot _aligned	[expr {$aligned == "true"}]

  if { $textfont == {} } {
    set textfont text
  }

  $name slot _font	$textfont
  $name slot _unique	$unique

  $name slot _copyroot	$copyroot
  $name slot _changed	{}

  set cw [$self slot cw]
  set ch [$self slot ch]

  if { $positions != {} } {

    $name slot _selector 1

    Canvas new $name.c -layout {right expand fill} \
	-width $cw -height $ch \
	-scroll [list $cw $ch] \
	-title $title2 -titlefont $titlefont
    Listbox new $name.l -layout {left padx 10 pady 10 filly} \
	-textfont $textfont -title $title -titlefont $titlefont \
	-action [list $name _exhibit]

    set items [app(get) $initmethod $init]
    $name slot _positions $positions
    set goodlength [expr 1+[llength $positions]]

    set empty [assertlength {} [expr [llength [lindex $items 0]]+1]]
    $name slot _empty $empty

    set descrs {}
    foreach item $items {
      if { $item == {} } {
	continue
      }
      set diff [expr $goodlength-[llength $item]]
      if { $diff < 0 } {
        set item [lrange $item 0 [expr $goodlength-1]]
      } elseif { $diff > 0 } {
        set item [concat $item [lrange $empty 0 [expr $diff-1]]]
      }
      lappend descrs $item
    }

    $name slot _items $descrs
    $name _items 0

  } {

    $name slot _selector 0

    Canvas new $name.c -layout {right padx 10 pady 10 expand fill} \
	-width $cw -height $ch -scroll [list $cw $ch] \
	-title $title -titlefont $titlefont

    set items [app(get) $initmethod $init]
    $name slot _items -
    $name _display $items

  }

  set canvas [$name.c canvas]

  $canvas bind item(rect) <Any-Enter> \
	[list $name _event enter]
  $canvas bind item(rect) <Any-Leave> \
	[list $name _event leave]
  $canvas bind item(rect) <1> \
	[list $name _event select]
  $canvas bind item(rect) <Shift-1> \
	[list $name _event deselect]
  $canvas bind item(rect) <Double-1> \
	[list $name _event double]

  set double [concat Select [assoc Select $actions]]
  if { [lindex $double 2] == {} } {
    set double [concat Edit [assoc Edit $actions]]
    if { [lindex $double 2] == {} } {
      set double {}
    }
  }
  $name slot _double $double

  $name layout $layout
}

#---------------------------------------------------------------------------

Window addDemo Ilist

defmethod Ilist demo {} {

  Ilist new * \
	-layout +100+100 \
	-text "Product Tree" \
	-buttons {Print Select New_Child New_Sibling Rename Remove Edit Ok} \
	-initmethod	list \
	-init	{ \
		{0 Product} {1 System 1} {2 Subsystem 1.1} \
		{3 Assembly X} {3 Assembly Y} {4 Subassembly Z} \
		{2 Subsystem 1.2} {3 Antennas Panel} {4 Antennas} \
		{4 Panel} {4 Fastening} {1 System 2} {2 Subsystem 2.1} \
		{0 Product} \
		{1 {Satellite} {Satellite} {Engineering Model}} \
	} \
	-actions {
		{Print} \
		{Select		{} - showAction Select} \
		{New_Child	{} - showAction New Child} \
		{New_Sibling	{} - showAction New Sibling} \
		{Rename		{} - showAction Rename} \
		{Remove		{} - showAction Remove} \
		{Edit		{} - showAction Edit} \
		{Ok		{} - showAction Ok} \
	}

  Ilist new * \
	-layout -100-100 \
	-text "Product Tree" \
	-title "Item Types" \
	-title2 "Item Structure" \
	-titlefont bold \
	-buttons {New Remove {Edit Change} Rename {} View {} Ok} \
	-positions { \
			{STANDARD}
			{Engineering Model}
			{Flight Spare}
			{Flight Model}
		} \
	-initmethod list \
	-init	{ \
		{{Product} {
			{{System 1} {} SAME}
			{{System 2} {} SAME}
			{{System 3} {} SAME}
			}}
		{{System 1}}
		{{System 2}}
		{{System 3}}
		} \
	-actions {
		{New		{} - showAction New} \
		{Rename		{} - showAction Rename} \
		{Remove		{} - showAction Remove} \
		{Edit		{} - showAction Edit} \
		{View		{} - showAction View} \
		{Ok		{} - showAction Ok} \
		}
}

#---------------------------------------------------------------------------

defmethod Ilist _display {items} {

  $self slot _list $items

  set fnt [$self slot _font]

  $self.c.c delete item(rect)
  $self.c.c delete line

  set count 0
  set pos -1
  set font [Font slot $fnt]
  set fontw [Font slot $fnt,w]
  set fonth [Font slot $fnt,h]
  set max [expr [Ilist slot minx]*$fontw]
  set aligned [$self slot _aligned]
  set xp0 [expr [Ilist slot alignpos]*$fontw]

  foreach i $items {
    incr pos
    set indent [lindex $i 0]
    if { $indent == {} } {
      set count [expr 0.5+$count]
      continue
    }
    if { $indent == 0 } {
      set count [expr 0.5+$count]
    }

    set text [lindex $i 1]
    set annotation [lindex $i 2]
    if { [lindex $i 3] != {} } {
      append annotation " ([lindex $i 3])"
    }

    set x [expr $fontw*(1+[Ilist slot ind]*$indent)]
    set y [expr $fonth*(0.5+$count*[Ilist slot v]*[Ilist slot r])]

    set size [textsize $font $text]
    set xe [expr $x+[lindex $size 0]+2*$fontw]
    set ye [expr $y+[lindex $size 1]*[Ilist slot r]]

    if { ! $aligned || $xp0 < $xe } {
      set xp $xe
    } {
      set xp $xp0
    }

    set size [textsize $font $annotation]
    set xa [expr $xp+2*$fontw+[lindex $size 0]]

    if { $max < $xa } {
      set max $xa
    }

    $self.c newRect3D $x $y $xe $ye \
	-tags [list item(rect) rect [list pos $pos] [concat info $text]] \
	-tags2 {item(rect)}

    $self.c.c create text [expr ($x+$xe)/2] [expr ($y+$ye)/2] \
	-anchor center -text $text -font $font \
	-tags {item(rect) text}

    set lastx($indent) [expr $x+$fontw]
    set lasty($indent) $ye
    if { $indent > 0 } {
      set ex $lastx([expr $indent-1])
      set ey [expr ($y+$ye)/2]
      $self.c.c create line $ex $lasty([expr $indent-1]) $ex $ey $x $ey \
	-arrow last -arrowshape {4 4 2} -width 1 -fill [Color slot fg] \
	-tags line
      set lasty([expr $indent-1]) $ey
    }

    if { $annotation != {} } {
      $self.c.c create text [expr $xp+$fontw] [expr ($y+$ye)/2] \
	-anchor w -text $annotation -font $font -tags {item(rect) item(ann)}
    }

    set count [expr 1+$count]
  }

  if { $count < [Ilist slot miny] } {
    set count [Ilist slot miny]
  }

  $self slot _sx [expr $max+$fontw]
  $self slot _sy [expr $fonth*(1+$count*[Ilist slot v]*[Ilist slot r])]

  $self.c adjust [list [$self slot _sx] [$self slot _sy]]

  $self slot _id {}

  return $self
}

#---------------------------------------------------------------------------

defmethod Ilist _items {pos} {

  # name of item
  set sel [lindex [lindex [$self slot _items] $pos] 0]

  set items {}
  foreach item [$self slot _items] {
    lappend items [lindex $item 0]
  }
  set items [lsort $items]

  $self.l.list delete 0 end
  foreach item $items {
    $self.l.list insert end $item
  }

  if { $sel == {} } {
    $self.c.c create text [expr [Ilist slot cw]/2] [expr [Ilist slot ch]/2] \
	-anchor c -font [Font slot largebold] -tags {item(rect)} \
	-fill black -text "Make a selection!"
  } {
    set pos [lsearch $items $sel]
    if { $pos < 0 } {
      set pos 0
    }
    $self.l.list select from $pos
    $self _exhibit {} {} $sel
    if { $pos > 7 } {
      incr pos -7
    } {
      set pos 0
    }
    $self.l.list yview $pos
  }
}

defmethod Ilist _exhibit {ignore1 ignore2 sel} {

  $self slot _index [associ $sel [$self slot _items]]
  set def [assoc $sel [$self slot _items]]

  set list {}
  set pos 0
  foreach elem [$self slot _positions] {
    lappend list [list 0 $elem]
    foreach slot [lindex $def $pos] {
      if { [lindex $slot 1] == {} } {
        set slot [lreplace $slot 1 1 [lindex $slot 0]]
      }
      lappend list [concat 1 $slot]
    }
    incr pos
  }
  $self _display $list
}

#---------------------------------------------------------------------------

defmethod Ilist _event {event} {

  set id [$self.c.c find withtag current]
  set tags [$self.c.c gettags $id]

  foreach tag $tags {
    if { $tag == "text" } {
      return [$self _handleEvent $event [expr $id-1]]
    }
    if { $tag == "rect" } {
      return [$self _handleEvent $event $id]
    }
  }
}

defmethod Ilist _handleEvent {event id} {

  case $event in {
  {enter} {
      $self.c.c itemconfigure $id -fill [Color slot bg,active]
    }
  {leave} {
      $self.c.c itemconfigure $id -fill [Color slot bg,button]
    }
  {double select} {
      if { [$self slot _id] != {} } {
	$self _handleEvent deselect [lindex [$self slot _id] 0]
      }
      set tags [$self.c.c gettags $id]
      set pos [assoc pos $tags]

      $self slot _id [list $id $pos]
      $self.c.c itemconfigure $id -width 3

      if { [$self slot _name] == 1 } {
	set selected [lindex [$self slot _name] $pos]
	if { [$self slot _copyall] || [lindex $selected 0] != 0 } {
	  $self.name set [lindex $selected 1]
	}
      }
    }
  {deselect} {
      $self.c.c itemconfigure $id -width 1
      $self slot _id {}
    }
  }

  incr id

  case $event in {
  {enter} {
      $self.c.c itemconfig $id -fill [Color slot fg,active]
    }
  {leave} {
      $self.c.c itemconfig $id -fill [Color slot fg]
    }
  {double} {
      set double [$self slot _double]
      if { $double != {} } {
	uplevel #0 [list $self [lindex $double 0] [lrange $double 2 end]]
      }
    }
  }

  return $self
}

defmethod Ilist _do {button action} {

  case $button {
  {Help Dismiss Select Rename Remove View Edit Ok Ok+Save Switch Print} {
	$self $button $action
    }
  {New_Child New_Sibling New} {
	$self Create $action $button \
	  [assoc $button {{New_Child 1} {New_Sibling 0}}]
    }
  default {
	uplevel #0 $action
    }
  }
}

#---------------------------------------------------------------------------

defmethod Ilist _sel {button} {
  upvar sel sel

  set sel [$self slot _id]
  if { ${sel} == "" } {
    Dialog new * -help [list Library/ilist-complain $button]
    return 1
  }

  set node [lindex [$self slot _list] [lindex $sel 1]]
  set sel [concat $sel $node]

  return 0
}

defmethod Ilist _eval {action form} {

  set res [uplevel #0 [concat $action $form]]
  if { $res != {} } {
    if { $res != "ignore" } {
      Dialog new * -help [list Library/failed $res]
    }
    return 1
  }
  return 0
}

defmethod Ilist _new {empty} {
  upvar new new

  set new [$self.name get]
  if { $new == {} } {
    Dialog new * -help [list Library/failed $empty]
    return 1
  }

  if { [$self slot _unique] == "true" } {
    foreach i [$self slot _list] {
      if { [lindex $i 1] == $new } {
	Dialog new * -help Library/ilist-duplicate
	return 1
      }
    }
  }
  return 0
}

defmethod Ilist _changed {} {

  if { ! [$self slot _selector] } {
    return
  }
  set changed [lindex [lindex [$self slot _items] [$self slot _index]] 0]
  if { $changed == {} } {
    return
  }
  if { [lsearch [$self slot _changed] $changed] < 0 } {
    $self slotappend _changed $changed
  }
  return
}

#-- check whether inserting 'new' at 'pos' with level 'level' would cause
#   any name clashes (offset is 0 or 1) with other siblings. If offset is
#   0, all existing nodes are checked. With offset = 1, the node 'pos' is
#   ignored.
#
defmethod Ilist _clash {pos level new {offset 0}} {

  set item [lindex [$self slot _list] $pos]

  for {set i [expr $pos-$offset]} {$i >= 0} {incr i -1} {
    set node [lindex [$self slot _list] $i]
    if { $node == {} } {
      continue
    }
    if { [lindex $node 0] < $level } {
      break;
    }
    if { [lindex $node 0] == $level && [lindex $node 1] == $new } {
      Dialog new * -help Library/ilist-clash
      return 1
    }
  }

  set total [llength [$self slot _list]]
  for {set i [expr $pos+1]} {$i < $total} {incr i} {
    set node [lindex [$self slot _list] $i]
    if { $node == {} } {
      continue
    }
    if { [lindex $node 0] < $level } {
      break;
    }
    if { [lindex $node 0] == $level && [lindex $node 1] == $new } {
      Dialog new * -help Library/ilist-clash
      return 1
    }
  }
  return 0
}

defmethod Ilist _struct {node key indent list} {

  set descr [assoc $node [$self slot _items]]

  if { $key == 0 } {
    set slots {}
  } {
    set slots [lindex $descr $key]
  }

  foreach slot [lindex $descr 0] {
    if { [assoc [lindex $slot 0] $slots] == {} } {
      set type [lindex $slot 1]
      if { $type == {} } {
	set type [lindex $slot 0]
      }
      lappend list [list $indent [lindex $slot 0] $type [lindex $slot 2]]
      set list [$self _struct $type $key [expr 1+$indent] $list]
    }
  }

  foreach slot $slots {
    set type [lindex $slot 1]
    if { $type == {} } {
      set type [lindex $slot 0]
    }
    lappend list [list $indent [lindex $slot 0] $type [lindex $slot 2]]
    set list [$self _struct $type $key [expr 1+$indent] $list]
  }

  return $list
}

defmethod Ilist _parent {index} {

  set items [$self slot _list]

  set myindex [lindex [lindex $items $index] 0]
  if { $myindex == {} || $myindex == 0 } {
    return
  }
  incr index -1

  for {} {$index >= 0} {incr index -1} {

    set node [lindex $items $index]
    if { $node != {} && [lindex $node 0] < $myindex } {
      return [concat $index $node]
    }
  }

  return
}

defmethod Ilist _parents {index} {

  set parents {}

  while { [set next [$self _parent $index]] != {} } {
    lappend parents $next
    set index [lindex $next 0]
  }

  return $parents
}

#---------------------------------------------------------------------------

defmethod Ilist Select {action} {

  if { [$self _sel Select] } {
    return
  }

  set pos [lindex $sel 1]
  set item [concat $pos [lindex [$self slot _list] $pos]]

  if { [$self _eval $action [list $item]] } {
    return
  }

  $self Dismiss
}

defmethod Ilist Create {action button level} {

  if { [$self _sel $button] } {
    return
  }

  if { [$self _new "create items with empty names"] } {
    return
  }

  if { $level == {} } {
    if { [lindex $sel 2] == 0 } {
      set level 1
    } {
      set level 0
    }
  }

  set def [list [$self slot _default] {}]
  set newitem [concat [list [expr $level+[lindex $sel 2]] $new] $def]
  set index [expr $level+[lindex $sel 1]]

  if { [$self _clash [lindex $sel 1] [lindex $newitem 0] $new] } {
    return
  }

  if { [$self _eval $action [list [concat $index $newitem]]] } {
    return
  }

  $self _changed

  if { [$self slot _selector] } {
    if { [lindex $sel 2] == 0 } {
      set key [lindex $sel 3]
    } {
      set key [lindex [$self _parent $index] 2]
    }
    set keypos [expr 1+[lsearch [$self slot _positions] $key]]
    set descrpos [$self slot _index]
    set descr [lindex [$self slot _items] $descrpos]
    set subdescr [lindex $descr $keypos]
    
    # new subdescr with new slot
    set subdescr [concat $subdescr [list [lrange $newitem 1 end]]]
    # replace subdescr in descr
    set descr [lreplace $descr $keypos $keypos $subdescr]
    # replace descr in list of all descrs
    $self slot _items \
	[lreplace [$self slot _items] $descrpos $descrpos $descr]

    $self _exhibit {} {} [lindex $descr 0]
  } {
    $self _display [linsert [$self slot _list] $index $newitem]
  }

  set defh [$self slot _defaultCreator]
  if { $defh != {} } {
    uplevel #0 [concat $defh [list [concat [list $index] $newitem]]]
  }

  $self View2 {} 0
}

defmethod Ilist Rename {action} {

  if { [$self _sel Rename] } {
    return
  }

  if { [lindex $sel 2] == 0 } {
    Dialog new * -help [list Library/failed "I can't rename root items."]
    return
  }

  if { [$self _new "use empty names"] } {
    return
  }

  if { [lindex $sel 3] == $new } {
    Dialog new * -help [list Library/ilist-no-change $new]
    return
  }

  if { [$self _clash [lindex $sel 1] $new 1] } {
    return
  }

  if { [$self _eval $action [list [lrange $sel 1 end] $new]] } {
    return
  }

  set index [lindex $sel 1]
  set newitem [concat [list $new] [lrange $sel 4 end]]

  $self _changed

  if { [$self slot _selector] } {
    set key [lindex [$self _parent $index] 2]
    set keypos [expr 1+[lsearch [$self slot _positions] $key]]
    set descrpos [$self slot _index]
    set descr [lindex [$self slot _items] $descrpos]
    set subdescr [lindex $descr $keypos]
    set slotpos [associ [lindex $sel 3] $subdescr]

    # update slot information
    set subdescr [lreplace $subdescr $slotpos $slotpos $newitem]
    # replace subdescr in descr
    set descr [lreplace $descr $keypos $keypos $subdescr]
    # replace descr in list of all descrs
    $self slot _items \
	[lreplace [$self slot _items] $descrpos $descrpos $descr]

    $self _exhibit {} {} [lindex $descr 0]
  } {
    $self _display [lreplace [$self slot _list] $index $index $newitem]
  }

  $self View2 {} 0
}

defmethod Ilist Remove {action} {

  if { [$self _sel Remove] } {
    return
  }

  if { [lindex $sel 2] == 0 } {
    Dialog new * -help [list Library/failed "I can't remove root items."]
    return
  }

  if { [Yesno new $self.d \
	-help [list Library/ilist-delete [lindex $sel 3]]] != "Yes" } {
    return
  }

  if { [$self _eval $action [list [lrange $sel 1 end]]] } {
    return
  }

  set index [lindex $sel 1]

  if { [$self slot _selector] } {
    set key [lindex [$self _parent $index] 2]
    set keypos [expr 1+[lsearch [$self slot _positions] $key]]
    set descrpos [$self slot _index]
    set descr [lindex [$self slot _items] $descrpos]
    set subdescr [lindex $descr $keypos]
    set slotpos [associ [lindex $sel 3] $subdescr]

    # update slot information
    set subdescr [lreplace $subdescr $slotpos $slotpos]
    # replace subdescr in descr
    set descr [lreplace $descr $keypos $keypos $subdescr]
    # replace descr in list of all descrs
    $self slot _items \
	[lreplace [$self slot _items] $descrpos $descrpos $descr]

    $self _exhibit {} {} [lindex $descr 0]
  } {
    set level [lindex $sel 2]
    set index2 $index
    foreach i [lrange [$self slot _list] [expr 1+$index] end] {
      if { $level >= [lindex $i 0] } {
	break;
      }
      incr index2
    }
    $self _display [lreplace [$self slot _list] $index $index2]
  }

  $self View2 {} 0
}

defmethod Ilist Edit {action} {

  if { [$self _sel Edit] } {
    return
  }

  if { [$self _eval $action [list [lrange $sel 1 end]]] } {
    return
  }
}

defmethod Ilist Print {action} {
  global system

  $self.c print -printer $system(printer) -colormode gray
  return
}

defmethod Ilist View {action} {

  if { [winfo exists $self.view] } {
    $self.view Dismiss
  } {
    $self View2 $action
  }
}

defmethod Ilist View2 {action {always 1}} {

  if { [winfo exists $self.view] } {
    set key [$self slot _new]
  } {
    set key 0
  }

  set node [lindex [lindex [$self slot _items] 0] 0]
  set struct [$self _struct $node $key 1 [list [list 0 $node]]]

  if { [winfo exists $self.view] } {

    $self.view.c.title configure \
	-text "-- [lindex [$self slot _positions] $key] --"
    $self.view _display $struct

  } elseif { $always } {

    Ilist new $self.view \
	-layout -0-0 \
	-text "Hierarchical Structure" \
	-buttons {{} Switch {} Print} \
	-title "-- [lindex [$self slot _positions] $key] --" \
	-initmethod list \
	-init $struct \
	-actions [list	[list Help {} [$self slot _viewhelp]] \
			[list Switch {} $self Switch $node] \
			[list Print {}]]
    $self slot _new 0

  }

  return
}

defmethod Ilist Switch {action} {

  set parent [lindex $action 0]
  set node [lindex $action 1]

  puts stdout "Switch $parent $action"
  if { [$parent slot _items] == "-" } {
    return
  }

  set result [Selector new *$parent.view \
	-layout center \
	-title {Select Structure} \
	-initmethod list \
	-init [$parent slot _positions] \
	-wait true \
	-actions [list \
		[list Help {} Library/ilist-switch] \
		[list Select {} $parent SwitchView]] \
	]

  if { $result == "Dismiss" } {
    return
  }

  $parent View2 {} 0
}

defmethod Ilist SwitchView {sel} {

  $self slot _new [lsearch [$self slot _positions] $sel]
  return
}

defmethod Ilist Ok+Save {action} {

  $self Ok $action
}

defmethod Ilist Ok {action} {

  if { [Yesno new * -help Library/confirm] != "Yes" } {
    return
  }

  if { [$self slot _selector] } {
    set args [list [$self slot _items] [$self slot _changed]]
    if { [$self _eval $action $args] } {
      return
    }
  } {
    if { [$self _eval $action [list [$self slot _list]]] } {
      return
    }
  }

  $self Dismiss
}

#---------------------------------------------------------------------------

defmethod Ilist Get {what} {
  global info

  case $what {
  {items} {
	return [$self slot _items]
    }
  {itemnames} {
	set items {}
	foreach i [$self slot _items] {
	  lappend items [lindex $i 0]
	}
	return $items
    }
  {positions} {
	return [$self slot _positions]
    }
  }
}

#-- set annotation from new1/new2
#
defmethod Ilist Set {item {new1 {}} {new2 {}}} {

  set index [lindex $item 0]
  set newitem [list [lindex $item 1] [lindex $item 2] $new1 $new2]

  $self _changed

  if { [$self slot _selector] } {

    if { $new1 == {} } {
      set new1 [lindex $item 2]
    }

    if { $new1 != [$self slot _default] &&
	 [assoc $new1 [$self slot _items]] == {} } {
      $self slotappend _items [concat [list $new1] [$self slot _empty]]
      $self slotappend _changed $new1
      $self _items [$self slot _index]
    }

    set key [lindex [$self _parent $index] 2]
    set keypos [expr 1+[lsearch [$self slot _positions] $key]]
    set descrpos [$self slot _index]
    set descr [lindex [$self slot _items] $descrpos]
    set subdescr [lindex $descr $keypos]
    set slotpos [associ [lindex $newitem 1] $subdescr]

    # update slot information
    set subdescr [lreplace $subdescr $slotpos $slotpos [lrange $newitem 1 end]]
    # replace subdescr in descr
    set descr [lreplace $descr $keypos $keypos $subdescr]
    # replace descr in list of all descrs
    $self slot _items \
	[lreplace [$self slot _items] $descrpos $descrpos $descr]

    $self _exhibit {} {} [lindex $descr 0]
  } {
    $self _display [lreplace [$self slot _list] $index $index $newitem]
  }

  $self View2 {} 0

  return $self
}
