# USAGE:	DLG:<type> parent name message button_1 button_2 ...
#
# parent	The name of the window to parent from
# name		The name to use for the toplevel of this dialog
# message	Message to print in the message area.
# button_{1,2,3} Text to put on each button.
#
#
#
# This starts a dialog of the type indicated in <type>.  It has [1..3]
# buttons, depending on how many (non-null) strings are passed.  It does
# a passive grab, to force the user to respond.  It waits until one of
# the buttons are pressed, then kills off the dialog and returns.  The
# return value is the number for the button which was pressed {1,2,3}.
#
# THIS FILE REQUIRES tk2.2 or greater.

set DLGBitmapPath $tkW3SourcePath/bitmap

proc DLG:information {parent w msg args} {
    global DLGBitmapPath
    DLG:build "$parent" "$w" "$msg" @$DLGBitmapPath/information "$args"
}

proc DLG:warning {parent w msg args} {
    global DLGBitmapPath
    DLG:build "$parent" "$w" "$msg" @$DLGBitmapPath/warning "$args"
}

proc DLG:error {parent w msg args} {
    global DLGBitmapPath
    DLG:build "$parent" "$w" "$msg" @$DLGBitmapPath/error "$args"
}

proc DLG:question {parent w msg args} {
    global DLGBitmapPath
    DLG:build "$parent" "$w" "$msg" @$DLGBitmapPath/question "$args"
}

proc DLG:working {parent w msg args} {
    global DLGBitmapPath
    DLG:build "$parent" "$w" "$msg" @$DLGBitmapPath/working "$args"
}

# Create an entry dialog
# Calling arguments are
# parent - dialog to be centered on
# w - name of dialog
# message - a list of entry labels
# command_output - what to put in the command button
# command - command to execute if command button is pressed

proc DLG:entry {parent w title message command_output command} {
    DLG:toplevel $parent $w
    DLG:draw_entries $w $message
    DLG:draw_buttons $w [list "$command_output" "Dismiss" "Help"]

    DLG:bind_entry $w [llength $message] "DLG:invoke_button $w 1"
    DLG:bind_button $w 1 "DLG:hide $w
eval $command \[DLG:get_entry_values $w [llength $message]\]"
    DLG:bind_button $w 2 "DLG:hide $w"
    DLG:bind_button $w 3 "tkWWWHelpNoHelp"
    return $w
}

proc DLG:listbox {parent w title list_items command_output command} {
    DLG:toplevel $parent $w
    DLG:draw_listbox $w $list_items
    DLG:draw_buttons $w [list "$command_output" "Dismiss" "Help"]

    DLG:bind_button $w 1 "DLG:hide $w
eval $command \[DLG:get_listbox_highlighted $w\]"
    DLG:bind_button $w 2 "DLG:hide $w"
    DLG:bind_button $w 3 "tkWWWHelpNoHelp"
    return $w
}

# The next procedures manage the dialog boxes

proc DLG:show {parent dialog} {
    # First update to make sure the boxes are the right size
    update idletask

    # Then we set the position and update
    DLG:position $parent $dialog
    update idletask

    # and now make it visible. Viola!  Centered over parent.
    wm deiconify $dialog
}

proc DLG:position {parent dialog} {
    # Tell the WM that we'll do this ourselves.
    wm sizefrom $dialog user
    wm positionfrom $dialog user

    # Where is my parent and what are it's dimensions
    set pargeo [split [wm geometry $parent] "+x"]
    set parwidth [lindex $pargeo 0]
    set parheight [lindex $pargeo 1]
    set parx [lindex $pargeo 2]
    set pary [lindex $pargeo 3]

    # What are my dimensions ?
    set dialogwidth [winfo reqwidth $dialog]
    set dialogheight  [winfo reqheight $dialog]

    # What are is the offset of the virtual window
    set vrootx [winfo vrootx $parent]
    set vrooty [winfo vrooty $parent]

    set dialogx [expr $parx+($parwidth-$dialogwidth)/2+$vrootx]
    set dialogy [expr $pary+($parheight-$dialogheight)/2+$vrooty]

    set maxx [expr "[winfo screenwidth $parent] - $dialogwidth"]
    set maxy [expr "[winfo screenheight $parent] - $dialogheight"]

# Make sure it doesn't go off screen
    if {$dialogx < 0} {
	set dialogx 0
    } {
	if {$dialogx > $maxx} {
	    set dialogx $maxx
	}
    }
    if {$dialogy < 0} {
	set dialogy 0
    } {
	if {$dialogy > $maxy} {
	    set dialogy $maxy
	}
    }

    set dialogx [expr $dialogx-$vrootx]
    set dialogy [expr $dialogy-$vrooty]

    # Build my new position (and dimensions)
    wm geometry $dialog [ format "%dx%d+%d+%d" $dialogwidth $dialogheight \
			 $dialogx $dialogy]

}

proc DLG:hide {w} {
    wm withdraw $w
}

proc DLG:destroy {w} {
    destroy $w
}

# These are procedures for building the dialog boxes
# You can use them to create new types of boxes

# Create a toplevel window with dialog box characteristics

proc DLG:toplevel {parent w {title ""}} {
    catch { destroy $w }    
    toplevel $w -class Dialog

    # Lets the dialog window be handled like Motif dialogs by the WM
    wm group $w $parent
    wm transient $w $parent
    wm minsize $w 0 0

    if {$title != ""} {
	wm title $w title
    }
    wm withdraw $w

    return $w
}

# Draw a frame with an icon and a message
proc DLG:draw_icon_msg {w iconfile msg} {
    pack append $w \
	[frame $w.msg]  { top fill expand } 

    pack append $w.msg \
	[label $w.msg.icon -bitmap $iconfile] { left } \
	[message $w.msg.msg  -text "$msg" -justify left -aspect 1500] \
	{ left fill expand }
}

# Build a modal dialog box
proc DLG:build {parent w msg iconfile blist} {
    set done "DLG[set w]done"
    global $done
    set $done 0

    DLG:toplevel $parent $w
    DLG:draw_icon_msg $w $iconfile $msg
    DLG:draw_buttons $w $blist
    for {set i 1} {$i <= [llength $blist]} {incr i 1} {
	DLG:bind_button $w $i "global $done; set $done $i"
    }

    DLG:show $parent $w
    grab $parent
    tkwait variable $done
    grab release $parent
    DLG:destroy $w
    return "[set $done]"
}

# Draw some entries in your box
proc DLG:draw_entries {w llist} {
    set index 1
    pack append $w \
	[frame $w.entry_frame -borderwidth 2 -relief raised] {top fill expand}
    foreach label $llist {
	set name ef$index
	pack append $w.entry_frame \
	    [frame $w.$name -borderwidth 5 -relief flat] {top fill expand}
 
	# Add the entry widget
	pack append $w.$name \
	    [label $w.$name.label -text "$label" ] {left fillx} \
	    [entry $w.$name.entry -relief sunken] { right expand fillx}
	incr index 1
    }
}

# Add bindings to the entries
proc DLG:bind_entry {w index binding} {
    bind $w.ef$index.entry <Return> $binding
}

# Get values from entries
proc DLG:get_entry_values {w number} {
    set return_string ""
    for {set i 1} {$i <= $number} {incr i 1} {
	lappend return_string [$w.ef$i.entry get]
    }
    return $return_string
}

proc DLG:get_entry_value {w i} {
    $w.ef$i.entry get
}

# Set entry values
proc DLG:set_entry_value {w index message} {
    $w.ef$index.entry delete 0 end
    $w.ef$index.entry insert 0 $message
}

# create a listbox

proc DLG:draw_listbox {w elements} {
    pack append $w \
	[frame $w.frame -borderwidth 5 -relief flat] {top fill expand}
    pack append $w.frame \
	[listbox $w.list -relief sunken ] {top fill expand}
    foreach elem $elements {
	$w.list insert end $elem
    }
}

proc DLG:get_listbox_highlighted {w} {
    $w.list curselection
}

# Draw the buttons at the bottom
proc DLG:draw_buttons  { w blist } {
    pack append $w \
	[frame $w.cmds] { top fill expand }

    # The extra frames below are to indicate current focus.  But we don't
    # have anything to handle keyboard focus.  Besides, the Motif
    # implementation uses a `knurled' look to the frame around buttons
    # in their standard dialogs and we must use two layers of frame to
    # implement that (see above).
    #
    # Also tried to set up stuff for keyboard focus, but it did a couple of
    # weird things and there isn't time to fix them now.  Keyboard focus
    # remained wherever the last `focus' command had gone, regardless of
    # where the pointer focus was.  This is a bug with the regular FOCUS
    # module as well.  The cure will be to watch Enter and Leave events
    # to know which window the pointer is in.  But this seems to be wrong
    # as well: What if the users configures their window manager for
    # a click-to-focus mode.  Won't we still get Enter&Leave events
    # regardless of where they've clicked-to-focus?

    set index 1
    foreach btn $blist {
	if {"$btn" != ""} {
	    pack append $w.cmds  \
		[ frame $w.cmds.f$index -borderwidth 5 -relief flat ] \
		{ left fill expand }
	    pack append $w.cmds.f$index \
		[ label $w.cmds.f$index.left -text "" ] {left expand} \
		[ label $w.cmds.f$index.right -text "" ] {right expand} \
		[  button $w.cmds.f$index.btn -text "$btn" -anchor center] \
{ left fill}

	    # DLG:keybindings $w.cmds.f$index.btn$index \
		"[lindex $btn$index 1]"
	}
	incr index 1
    }
}    

proc DLG:bind_button {w index command} {
    $w.cmds.f$index.btn config -command $command
}

proc DLG:invoke_button {w index} {
    $w.cmds.f$index.btn invoke
}

# Future expansion: For when we are able to do keyboard focus and
# traversal in these things.
#proc DLG:keybindings { w cmd } {
#	bind $w <Key-space> "$cmd"
#	bind $w <Key-Return> "$cmd"
#}

