 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File: debug.tcl

# Description:
#	Auxiliary procedures for debugging Tcl programs.

 # $Id: debug.tcl,v 1.9 1993/11/01 18:20:46 kennykb Exp $
 # $Source: /homedisk/julius/u0/kennykb/src/tkauxlib_ship/RCS/debug.tcl,v $
 # $Log: debug.tcl,v $
 # Revision 1.9  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.8  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.7  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.6  1993/10/20  18:41:01  kennykb
 # Changed so that tk 3.3 traverses windows in stacking order.  This has
 # the side effect that tk 3.2 will no longer traverse them in creation
 # order, but using the order of `winfo children' probably makes more sense
 # in any case.
 #
 # Fixed copyright notice so that it doesn't look like structured commentary.
 #
 # Revision 1.5  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.4  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.3  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.2  1993/07/20  13:12:03  kennykb
 # Made `choicebox', `collapsible', and `debug' conform with naming and
 # commentary conventions
 #
 # Revision 1.1  1993/06/03  15:26:32  kennykb
 # Initial revision
 #

# Procedure:	debug_traces
#
# Synopsis:
#	Determine the traces that are active on a set of variables.
#
# Usage:
#c 	debug_traces ?pattern?
#
# Parameters:
#c	pattern
#		Pattern for variables to match.  The pattern is matched
#		according to the rules for the `string match' command.  The
#		default pattern is "*", which matches all variables.
#
# Return value:
#	List of trace commands active on all variables matching the specified
#	pattern.
#
# Description:
#	debug_traces is used to determine the set of traces active on a given
#	variable or set of variables.  It returns the set of traces that are
#	are active on all variables in the current scope that match the
#	given pattern.  The pattern is matched according to the rules for
#	`info vars.'

proc debug_traces {{pattern *}} {
	set result ""
	foreach v [lsort [uplevel 1 info vars $pattern]] {
		set status [catch {uplevel 1 trace vinfo $v} traces]
		if {$status == 0} {
			foreach t $traces {
				lappend result [concat trace variable \
							[list $v] $t]
			}
		}
	}
	return $result
}

# Procedure:	debug_dump
#
# Synopsis:
#	Produce a memory dump of a Tk application.
#
# Usage:
#c	debug_dump ?fileId?
#
# Parameters:
#c	fileId
#		File where the dump should be written.  `fileId' may be the
#		result of a call to `open', or it may be `stdout' or `stderr'
#		to designate one of the standard I/O channels.
#
# Return value:
#	None.
#
# Description:
#	debug_dump writes a description of the state of a Tk application to
#	a given file; it is usually used just prior to destroying the
#	application main window after a fatal error.  The dump is written
#	as ASCII text; it includes the following information.
#	- The date and time of the dump.
#	- The name of the application.
#	- The interpreter's command count.
#	- The library and script path names.
#	- The version of the Tcl interpreter in use.
#	- The error code and error information.
#
#	- A backtrace of the call stack including not only the
#	command invocations, but the value of all local 
#	variables.
#
#	- The values of all global variables.
#
#	- The traces active on all variables.
#
#	- Any procedures that were not obtained through autoload.
#	(The assumption is that autoloaded procedures can
#	be examined by examining the library files)
#
#	- The complete widget tree of the application, including
#	types, configuration, packing, placement, window
#	information, window manager hints, protocol handlers,
#	and bindings for all widgets.
#
#	- Class and `all' bindings for widgets.

proc debug_dump file {
	global errorInfo
	global errorCode
	global TCLENV
	global auto_index
	global debug_priv

	puts $file ------------------------------------------------------------
	puts $file [exec /bin/date]
	puts $file "Application: [winfo name .]"

	puts $file "Command count: [info cmdcount]"
	puts $file "Library: [info library]"
	puts $file "Script: [info script]"
	puts $file "Tcl version: [info tclversion]"

	puts $file "Error code: $errorCode"
	puts $file "Error info:\n$errorInfo"

	catch {
		foreach item [array names debug_priv] {
			unset $debug_priv($item)
		}
	}

	for {set level [info level]} {$level >= 0} {incr level -1} {
		if {$level == 0} {
			puts $file "\nGlobal variables:"
		} else {
			puts $file \
				"\nCall at level #$level:\n[info level $level]"
		}
		if {$level < [info level]} {
			foreach var [lsort [uplevel #$level info vars]] {
				debug:dumpVar $file $var $level
			}
		}
	}

	puts $file "\nNon-autoloaded procedures:"
	foreach pname [lsort [info procs]] {
		if {![info exists auto_index($pname)]
		    && ![info exists TCLENV(PROC:$pname)]} {
			debug:dumpProc $file $pname
		}
	}
	debug:dumpWinTree $file .

	puts $file "\nClass bindings:"
	catch {
		foreach item [lsort [array names debug_priv]] {
			set b [bind $item]
			if {$b != ""} {
				puts $file \t[list $item]:
				foreach event $b {
					puts $file \t\t[list $event \
							[bind $item $event]]
				}
			}
			unset $debug_priv($item)
		}
	}

	set b [bind all]
	if {$b != ""} {
		puts $file "\nGlobal bindings:"
		foreach event $b {
			puts $file \t[list $event [bind all $event]]
		}
	}
}

# Procedure:	debug:dumpVar
#
# Synopsis:
#	Internal procedure to write a variable to a memory dump.
#
# Usage:
#c	debug:dumpVar fileId v level
#
# Parameters:
#c	fileId
#		fileId of the dump file
#c	v
#		Name of the variable to dump
#c	level
#		Stack level at which the variable appears.
#
# Result:	None
#
# Description:
#	The specified variable and its associated traces are
#	written to the dump file.

proc debug:dumpVar {file v level} {
	upvar #$level $v var
	if {[catch {set var} content] == 0} {
		puts $file "\t$v = [list $content]"
	}
	foreach item [uplevel #$level trace vinfo [list $v]] {
		puts $file "\t\ttrace variable $v $item"
	}
	set status [catch {array names var} names] 
	if {$status == 0} {
		foreach name [lsort $names] {
			puts $file "\t${v}($name) = [list $var($name)]"
			foreach item [uplevel #$level \
					trace vinfo [list ${v}($name)]] {
				puts $file "\
\t\ttrace variable ${v}($name) $item"
			}
		}
	}
}

# Procedure:	debug:dumpProc
#
# Synopsis:
#	Internal procedure to copy a procedure definition to a dump file.
#
# Usage:
#c	debug:dumpProc file procName
#
# Parameters:
#c	fileId
#		Id of the dump file.
#c	procName
#		Name of the procedure to dump
#
# Return value:
#	None.
#
# Description:
#	debug:dumpProc writes the definition of a procedure to the dump file.

proc debug:dumpProc {file pname} {
	set params {}
	foreach v [info args $pname] {
		if {[info default $pname $v dval]} {
			lappend params [list $v $dval]
		} else {
			lappend params $v
		}
	}
	puts $file "\nproc $pname [list $params] [list [info body $pname]]"
}

# Procedure:	debug:dumpWinTree
#
# Synopsis:
#	Internal procedure to dump part ot the widget tree to a dump file.
#
# Usage:
#c	debug:dumpWinTree fileId pathName
#
# Parameters:
#c	fileId
#		The dump file
#c	pathName
#		Path name of the root of the widget tree being dumped
#
# Description:
#	debug:dumpWinTree dumps a portion of the widget tree rooted at pathName
#	to the specified file.

proc debug:dumpWinTree {file w} {
	global debug_priv
	puts $file "\nWindow: $w"
	puts $file "\tClass: [winfo class $w]"
	set debug_priv([winfo class $w]) 1
	puts $file "\tConfiguration:"
	if {[info commands $w] != $w} {
		puts $file "\t\tUNAVAILABLE"
	} else {
		foreach item [$w config] {
			if {[lindex $item 3] != [lindex $item 4]} {
				puts $file \t\t[list [lindex $item 0] \
						[lindex $item 4]]
			}
		}
	}

	set p [pack info $w]
	if {$p != ""} {
		puts $file "\tPacking:"
		while {$p != ""} {
			puts $file \t\t[list [lindex $p 0] [lindex $p 1]]
			set p [lrange $p 2 end]
		}
	}

	set p [place info $w]
	if {$p != ""} {
		puts $file "\tPlacement:"
		while {$p != ""} {
			puts $file \t\t[list [lindex $p 0] [lindex $p 1]]
			set p [lrange $p 2 end]
		}
	}

	puts $file "\tInfo:"
	foreach item {
		cells children depth geometry height id ismapped name
		parent reqheight reqwidth rootx rooty screen screencells
		screendepth screenheight screenmmheight screenmmwidth
		screenvisual screenwidth toplevel visual vrootheight
		vrootwidth vrootx vrooty width x y} {
			if {$w == "." \
			    || [winfo $item $w] \
				!= [winfo $item [winfo parent $w]]} {
				puts $file \t\t[list $item = [winfo $item $w]]
			}
	}

	if {$w == [winfo toplevel $w]} {
		puts $file "\tWindow management:"
		foreach item {
			aspect client command focusmodel geometry grid
			group iconbitmap iconmask iconname iconposition
			iconwindow maxsize minsize overrideredirect
			positionfrom sizefrom title transient} {
			set v [wm $item $w]
			if {$v != ""} {
				puts $file \t\t[list $item = $v]
			}
		}
		foreach item [wm protocol $w] {
			set v [wm protocol $w $item]
			puts $file \t\t[list protocol $item = $v]
		}
	}

	set bindings [bind $w]
	if {$bindings != ""} {
		puts $file "\tBindings:"
		foreach item [bind $w] {
			puts $file \t\t[list $item [bind $w $item]]
		}
	}

	if {[winfo class $w] == "Canvas"} {
		set ctags(all) 1
		set ctags(current) 1
		puts $file "\tItems:"
		foreach item [$w find all] {
			puts $file \
				"\t\t$item: [$w type $item] [$w coords $item]"
			foreach c [$w itemconfig $item] {
				if {[lindex $c 3] != [lindex $c 4]} {
					puts $file \
						\t\t\t\t[list [lindex $c 0] \
								[lindex $c 4]]
				}
			}
			set bindings [$w bind $item]
			if {$bindings != ""} {
				puts $file \t\t\tBindings:
				foreach event $bindings {
					puts $file \
						\t\t\t\t[list $event \
							[$w bind \
								$item \
								$event]]
				}
			}
			foreach tag [lindex [$w itemconfig $item -tags] 4] {
				set ctags($tag) 1
			}
		}

		if [info exists ctags] {
			puts $file "\tTag bindings:"
			foreach tag [lsort [array names ctags]] {
				unset ctags($tag)
				set b [$w bind $tag]
				if {$b != ""} {
					puts $file \t\t[list $tag]:
					foreach event $b {
						puts $file \t\t\t[list \
							$event \
							[$w bind $tag $event]]
					}
				}
			}
		}
				
	}

	foreach kid [winfo children $w] {
		debug:dumpWinTree $file $kid
	}
}
