/*
 * mxInt.h --
 *
 *	This file defines structures and procedures shared by
 *	the top-level routines that implement Mx.
 *
 * Copyright (C) 1986 Regents of the University of California
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 * $Header: /sprite/src/lib/mx/RCS/mxInt.h,v 1.27 90/03/25 15:17:09 ouster Exp $ SPRITE (Berkeley)
 */

#ifndef _MXINT
#define _MXINT

#include <sx.h>
#include <tcl.h>
#include "cmd.h"
#include "mx.h"
#include <sys/types.h>

/*
 * For each file that's open, regardless of how many windows in
 * which it's visible, there's one structure of the following
 * type with overall information about the file.  This same structure
 * type is used both by Mx and by Tx (each program doesn't necessarily
 * use all of the fields, though).
 */

typedef struct MxFileInfo {
    Mx_File file;			/* File storage. */
    char *name;				/* Name of file (NULL for typescripts
					 * or empty if this is a scratch
					 * window). */
    time_t lastMod;			/* Time file was last modified. */
    Undo_Log log;			/* Token for undo/recovery info, or
					 * NULL if no undo-ing. */
    struct MxHighlight *hlPtr;		/* First in list of highlights
					 * associated with this file.  Each
					 * highlight gets displayed in all
					 * the windows that view the file. */
    struct MxWindow *mxwPtr;		/* First in list of windows that view
					 * this file. */
    struct MxFileInfo *nextPtr;		/* Next in list of MxFileInfo structs
					 * (one for Tx, one for Mx). */
    int flags;				/* Miscellaneous flags (see below). */

    /*
     * Information about the caret:
     */

    Mx_Position caretFirst;		/* Location of character that caret's
					 * just to left of. */
    Mx_Position caretLast;		/* Identical to caretFirst (needed for
					 * caretFloater, below. */
    struct MxWindow *caretWindow;	/* Window in which caret is being
					 * displayed.  If caret isn't
					 * currently displayed in a window
					 * in this file, then this is NULL. */

    /*
     * Information used only by Mx:
     */
     
    struct MxHighlight *openParenPtr;	/* Open parenthesis being highlighted
					 * in this file (NULL if none). */
    struct MxHighlight *closeParenPtr;	/* Close parenthesis being highlighted
					 * in this file (NULL if none). */

    /*
     * Information used only by Tx.
     */

    void (*inputProc)();	/* Procedure to invoke to process input from
				 * keyboard. */
    void (*sizeChangeProc)();	/* Procedure to invoke when size of Tx window
				 * changes (to reset termcap, for example). */
    void (*closeProc)();	/* Procedure to invoke when last window on
				 * this typescript is closed. */
    ClientData clientData;	/* Additional stuff to pass to inputProc
				 * and closeProc. */
    int viLines;		/* If not in vi mode, this is < 0.  Otherwise,
    				 * it indicates the number of lines being used
				 * at the top of the typescript for vi-style
				 * terminal emulation. */
} MxFileInfo;

/* Flag values for MxFileInfo structures:
 * 
 * DELETE:		1 means delete file on last window close.
 * MODIFIED:		1 means this file had been modified as of the last
 *			it was checked.  This is used only to tell when the
 *			file becomes modified or unmodified again.
 * NO_ESCAPES:		(Tx only) 1 means ignore escape sequences in output
 *			destined for typescript;  0 means interpret them as
 *			Tx commands.
 */

#define DELETE		1
#define MODIFIED	4
#define NO_ESCAPES	8

/*
 * The data structure below keeps track of one line on the display,
 * which corresponds to a line or a piece of a line from a file.
 * When the end of the file is above the bottom of the window, the
 * lines that are past the last one in the file have positions that
 * continue to be numbered up sequentially, as if the lines existed
 * after all and contained a single newline character (this is needed
 * to allow the screen to be scrolled).
 */

typedef struct {
    Mx_Position position;	/* Position within the file of the first
				 * character to be displayed on this line
				 * of the display.  The character index is
				 * 0 except for the trailing parts of lines
				 * that wrap around. */
    int length;			/* How many characters to display on this
				 * line.  Includes the newline character,
				 * if this display line is the last for
				 * the file line. */
    int redisplayFirst;		/* Index (within file line, not display line)
				 * of first character that needs to be
				 * redisplayed, if any.  -1 means line is
				 * up-to-date:  nothing to redisplay. */
    int redisplayLast;		/* Index within file of last char that needs
				 * to be redisplayed, if any. */
} MxDisplayLine;
    
/*
 * For each Mx or Tx window, there is one structure of the following
 * type, which keeps track of all the information about that window's
 * display.
 */

typedef struct MxWindow {
    MxFileInfo *fileInfoPtr;	/* Information about file being displayed
				 * in window. */

    /*
     * Window and subwindow information:
     */

    Display *display;		/* Connection to X server. */
    Window w;			/* The X window that is the top-level
				 * container for the Mx stuff.   This window
				 * is also used as the focus group id for
				 * the subwindows. */
    Window title;		/* Title window at top of w.  NULL
				 * means no title for window (window
				 * mgr is probably providing it). */
    Window menuBar;		/* Strip window in which to allocate menus;
				 * just underneath title. */
    Window msgWindow;		/* One-line information window, used in Mx
				 * to output information such as current
				 * line number.   Appears just under
				 * menu bar.*/
    Window searchContainer;	/* Container for entry windows for search and
				 * replacement strings.  NULL means no entry
				 * windows are on the screen.  Appears just
				 * under info window. */
    Window searchWindow;	/* Entry window for search string.  NULL means
				 * it isn't displayed right now.  When it is
				 * displayed, it's just underneath border2. */
    Window replaceWindow;	/* Entry window for replacement string.  When
				 * it's displayed, it's just under
				 * searchWindow. */
    Window cmdWindow;		/* Entry window where user can type commands,
				 * at bottom of w.  NULL means no command
				 * entry window exists. */
    Window scrollbar;		/* Scrollbar at right side of w. */
    Window fileWindow;		/* Where file appears.  Fills remainder of
				 * window. */
    /*
     * Fonts, colors, and other displaying information:
     */

    XFontStruct *fontPtr;	/* Information about font used for display. */
    unsigned long foreground, background;
				/* Colors for display. */
    unsigned long border;	/* How to display borders. */
    unsigned long titleForeground, titleBackground, titleStripe;
				/* How to display title. */
    unsigned long sbForeground, sbBackground, sbElevator;	
				/* How to display scrollbar. */
    Cursor cursor;		/* Cursor for use in fileWindow. */
    GC textGc;			/* Graphics context for displaying text,
				 * underlines, boxes, and the caret. */
    GC grayGc;			/* Used for drawing MX_GRAY highlights.  Starts
				 * off as None, gets created when first
				 * needed. */
    GC reverseGc;		/* Used for drawing MX_REVERSE highlights.
				 * Starts off as None, gets created when first
				 * needed. */
    Pixmap iconPixmap;		/* Id for pixmap to be displayed in window's
				 * icon, or NULL if none. */
    int cursorMode;		/* Type of cursor to use. */

    /*
     * Redisplay information:  managed by mxDisplay.c.  The charWidths
     * information is provided to augment what's in *fontPtr:  it is
     * complete (i.e. there's a value for every single ASCII character),
     * and it's also got "correct" values for control characters (i.e.
     * the width of the two-character sequence "^X").  Tabs and newlines
     * are still special;  they have charWidths < 0.
     */

    int fontHeight;		/* Height of each line, in pixels. */
    unsigned char *charWidths;	/* Width of each character, in pixels.
				 * See note above. */
    int width, height;		/* Dimensions of fileWindow, in pixels. */
    int heightLines;		/* Height of fileWindow in lines of text. */
    MxDisplayLine *linePtr;	/* Pointer to array of line descriptors. */
    int lineArraySize;		/* Size of linePtr array (>= heightLines). */
    int tabWidth;		/* Width of a tab in this font. */
    Mx_Position keyPosition;	/* When the view changes size, try to
				 * keep this position centered in window. */
    Mx_Spy spy;			/* Used to monitor changes in the file. */
    Pixmap eofPixmap;		/* Pattern to use for areas off the end of
    				 * the file. */
    int shiftFirst;		/* First display line to be shifted during
				 * next display update. */
    int shiftNumLines;		/* How many lines to shift during update.
				 * 0 means there's no shift pending. */
    int shiftDistance;		/* Amount by which to shift: may be positive
				 * or negative. */
    GC eofGc;			/* For the stippled end-of-file area. */
    int eofYOrigin;		/* Current origin for drawing eof stipple;
				 * must be adjusted as window scrolls to keep
				 * alignment. */

    /*
     * Information for command interpretation:
     */

    Cmd_Table cmdTable;		/* Keystroke bindings for this window. */
    Tcl_Interp *interp;		/* Tcl interpreter containing command,
				 * variable, and procedure information. */
    char *cmdString;		/* Command that's been typed in cmdWindow. */
    char *searchString;		/* Search string, from search window. */
    char *replaceString;	/* Replacement string, from search window. */
    char *msgString;		/* String to display in message window. */
    int quoteFlag;		/* 1 if next keypress is quoted. */

    /*
     * Information for history recording.  Managed by mxHistory.c.
     */

    int histEnabled;		/* > 0 means history recording is on;  < 0
				 * means it's disabled. */
    char *histBuffer;		/* String containing current history info
				 * (dynamically allocated). */
    int histSpace;		/* Total size of histBuffer, in bytes. */
    int histCurSize;		/* Current length of string in histBuffer,
				 * not including terminating null. */
    Tcl_Trace histTrace;	/* Token for history tracer. */

    int flags;			/* Various flags:  see below. */
    struct MxWindow *nextPtr;	/* Next in list of MxWindows for this file.
				 * NULL means end of list. */
} MxWindow;

/*
 * The flag bits for MxWindows are:
 *
 * FOCUS_WINDOW:	1 means this window has the focus (i.e., caret
 *			should be displayed, except when it's temporarily
 *			turned off to process commands).
 * NEEDS_UPDATE:	1 means that changes have been made to the window's
 *			contents but the screen hasn't been redisplayed to
 *			reflect all the changes.
 * TX_WINDOW:		1 means this is a Tx window.  0 means Mx.
 * DELAY_REDISPLAY:	1 means a ConfigureNotify event has been received
 *			but the corresponding Expose event has not been
 *			received.  In this case, redisplay is delayed until
 *			the Expose event arrives (in order to avoid double-
 *			redisplay).
 * REDRAW_SLIVER:	1 means there's a small strip at the bottom of the
 *			window that's too small to hold a whole line but
 *			needs to be redisplayed in the appropriate
 *			background.  This bit gets set in MxRedisplayRegion.
 * SLIVER_EOF:		1 means that the last time the sliver was displayed,
 *			the end-of-file stipple was used.  0 means the
 *			last display was in the background color.
 * DESTROYED:		1 means that the window has been destroyed, but hasn't
 *			been cleaned up yet.  This flag is needed to avoid
 *			accidentally touching the window and thereby causing
 *			an X error.
 * MAPPED:		1 means the window is mapped;  0 means it hasn't
 *			yet been mapped.
 * CARET_VISIBLE:	1 means the caret was visible in this window before
 *			characters were output to the typescript, so the window
 *			should be scrolled after the output in order to keep
 *			the caret visible.  This flag bit is only used by Tx.
 * TX_MAIN_WINDOW:	1 means this is the "main" window on a typescript: the
 *			one whose size is reflected in termcap and winsize
 *			information.  This flag bit is only used by Tx.
 */

#define TX_WINDOW	1
#define FOCUS_WINDOW	4
#define NEEDS_UPDATE	0x10
#define DELAY_REDISPLAY	0x20
#define REDRAW_SLIVER	0x40
#define SLIVER_EOF	0x80
#define DESTROYED	0x100
#define MAPPED		0x200
#define CARET_VISIBLE	0x400
#define TX_MAIN_WINDOW	0x800

/*
 * Modes for the cursor.
 */
#define OFF 0
#define CARET 1
#define BLOCK 2
#define VIBLOCK 3

/*
 * Maximum length of various strings (search, replace, msg, etc.)
 */

#define MAX_STRING_LENGTH 100

/*
 * For each highlighted area of a file, there's a structure of the
 * following shape that describes the highlight.
 */

typedef struct MxHighlight {
    MxFileInfo *fileInfoPtr;	/* Information about file containing
				 * highlight. */
    Mx_Position first;		/* First byte of highlighted information. */
    Mx_Position last;		/* Last byte of highlighted information. */
    Mx_Floater floater;		/* Keeps first and last up-to-date when
				 * file is modified. */
    int style;			/* If < 0, gives style in which to highlight.
				 * If >= 0, gives background color; text is
				 * simply redrawn in different colors. */
    unsigned long foreground;	/* Foreground color for highlighting. */
    struct MxHighlight *nextPtr;/* Next in list of highlights associated with
				 * this file. */
} MxHighlight;

/*
 * For each entry in each menu there's a structure of the following
 * type that is passed to the menu callback procedure:
 */

typedef struct MxMenuInfo {
    MxWindow *mxwPtr;		/* Window to which menu belongs. */
    char command[4];		/* A command to invoke in mxwPtr when this
				 * menu entry is invoked.  The actual size
				 * of this area may be greater than 4: it
				 * will be whatever it takes to hold the
				 * command.  BEWARE:  this must be the last
				 * entry in the record! */
} MxMenuInfo;

/*
 * Top-level command procedures:
 */

extern int Mx_BindCmd(), Mx_CaretCmd(), Mx_CleanCmd(), Mx_ColumnCmd();
extern int Mx_ControlCmd(), Mx_DeleteCmd(), Mx_ExtractCmd(), Mx_FocusCmd();
extern int Mx_GeometryCmd(), Mx_HistoryCmd();
extern int Mx_IndentCmd(), Mx_InsertCmd(), Mx_MarkCmd();
extern int Mx_MenuCmd(), Mx_MessageCmd(), Mx_NewlineCmd(), Mx_OpenCmd();
extern int Mx_QuitCmd(), Mx_QuoteCmd(), Mx_ReadCmd();
extern int Mx_ReplaceCmd(), Mx_ResetCmd();
extern int Mx_SearchCmd(), Mx_SeeCmd(), Mx_SelectionCmd();
extern int Mx_SendCmd(), Mx_SwitchCmd(), Mx_TaginfoCmd();
extern int Mx_UndoCmd(), Mx_UpdateCmd(), Mx_WriteCmd();

extern int Tx_EscapeCmd(), Tx_InsertCmd(), Tx_OpenCmd();
extern int Tx_OutputCmd(), Tx_QuitCmd(), Tx_SendCmd();
extern int Tx_TitleCmd(), Tx_UpdateCmd(), Tx_ViCmd();

/*
 * Procedures exported by mxCmdUtils.c for use by the rest of Mx:
 */

extern char *		(mxOpenParens[]);
extern char *		(mxCloseParens[]);
extern char		mxWordMask[16];
extern void		MxAdvanceHistory();
extern void		MxBadArgs();
extern int		MxCheckWritten();
extern Window		MxCreateWindow();
extern void		MxCvtToPrintable();
extern int		MxDoCmd();
extern void		MxFocusProc();
extern void		MxError();
extern void		MxGetColor();
extern int		MxGetMark();
extern void		MxKeyProc();
extern time_t		MxLastMod();
extern void		MxMakeMenu();
extern void		MxMarkParens();
extern void		MxMenuEntryProc();
extern void		MxMenuProc();
extern void		MxMouseProc();
extern void		MxSetMark();
extern void		MxSetWMHints();

/*
 * Procedures exported by mxHighlight.c for use by the rest of Mx:
 */

extern void		MxCaretComputeLoc();
extern void		MxCaretRedisplayText();
extern void		MxCaretSetPosition();
extern void		MxDisplayCaret();
extern void		MxDisplayHighlights();
extern MxHighlight *	MxHighlightCreate();
extern void		MxHighlightDelete();
extern void		MxHighlightGetRange();
extern void		MxHighlightSetRange();

/*
 * Procedures and constants exported by mxDisplay.c for use by the rest of Mx:
 */

extern int		MxCharToPoint();
extern void		MxCleanupRedisplay();
extern void		MxGetInWindow();
extern int		MxFindPosition();
extern int		MxMeasureChars();
extern void		MxRedisplayRange();
extern void		MxRedisplayRegion();
extern void		MxScroll();
extern void		MxSetupRedisplay();
extern void		MxUpdateWindow();

/*
 * Procedures exported by mxEntries.c for use by the rest of Mx:
 */

extern void		MxMakeCmdWindow();
extern void		MxMakeSearchWindow();
extern void		MxOutputMsg();

/*
 * Procedures exported by mxHistory.c for use by the rest of Mx:
 */

extern void		MxHistoryCleanup();
extern void		MxHistoryInit();
extern void		MxHistoryNext();
extern void		MxHistoryOff();
extern void		MxHistoryOn();

/*
 * Procedures exported by mxIndent.c for use by the rest of Mx:
 */

extern void		MxCleanIndent();
extern int		MxGetIndent();
extern void		MxSetIndent();

/*
 * Procedures exported by mxSelect.c for use by the rest of Mx:
 */

extern MxHighlight *	MxSelection;
extern MxFileInfo *	MxSelectedFile;
extern void		MxSelectionClear();
extern void		MxSelectionSet();
extern int		MxGetSelRange();

/*
 * Procedures exported by mxWindow.c for use by the rest of Mx:
 */

extern MxWindow *	MxGetMxWindow();
extern int		MxSwitchFile();

/*
 * Procedures exported by txCmdAZ.c for use by the rest of Tx:
 */

extern void		TxReplaceLines();

/*
 * Procedure exported by txUtmp.c for use by the rest of TRx:
 */

extern int		TxUtmpEntry();

/*
 * Procedures exported by txWindow.c for use by the rest of Tx:
 */

extern MxWindow *	TxGetMxWindow();
extern void		TxMakePosExist();

#endif _MXINT
