/*
 * mt.c  - send control commands to a magtape device, like the BSD command.
 *		  by James da Silva <jds@cs.umd.edu>
 */
#include <stdio.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/mtio.h>

#define BOGUS_IOCTL	1	/* must be 1 for Minix 1.5 */

struct optable {
  char *cmdname;
  int unique;
  int opcode;
  char *desc; 
};

struct optable ops[] = {
  { "rewind",	3, MTREW,	"rewind tape"  },
  { "fsf",	3, MTFSF,	"forward space <count> files"  },
  { "fsr",	3, MTFSR,	"forward space <count> records" },
  { "bsf",	3, MTBSF,	"back space <count> files" },
  { "bsr",	3, MTBSR,	"back space <count> records" },
#ifdef notyet
  { "eof",	2, MTWEOF,	"write <count> EOF marks" },
  { "weof",	2, MTWEOF,	"write <count> EOF marks" },
#endif
  { "offline",	3, MTOFFL,	"rewind and go offline" },
  { "reoffl",	3, MTOFFL,	"rewind and go offline" },
  { "retension",3, MTRETEN,	"retension" },
#ifdef notyet
  { "status",	2, MTNOP,	"report status" },
#endif
  { "erase",	3, MTERASE,	"erase tape" },
  { NULL }
};

void usage()
{
  struct optable *op;

  fprintf(stderr, "Usage: mt [-f <tapedevice>] <command> [<count>]\n");
  fprintf(stderr, "	Default <tapedevice> is %s, default <count> is 1\n",
		DEFTAPE);
  fprintf(stderr, "	Valid commands are:\n"); 
  for(op=ops; op->cmdname != NULL; op++) {
	fprintf(stderr, "\t\t%-10s - %s\n", op->cmdname, op->desc);
  }
  exit(1);
}


void main(argc, argv)
int argc;
char **argv;
{
  int fd, rc;
  long count;
  char *tapefile, *cmd;
  struct mtop tape_op;
  struct optable *op;

  if(argc < 2) usage();
  if(strcmp(argv[1], "-f") == 0) {
	if(argc < 4) usage();
	tapefile = argv[2];
	argv += 2;
	argc -= 2;
  }
  else tapefile = DEFTAPE;
  
  cmd = argv[1];
  if(argc < 3) count = 1;
  else sscanf(argv[2], "%l", &count);

  for(op = ops; op->cmdname != NULL; op++)
	if(!strncmp(op->cmdname, cmd, op->unique))  break;

  if(op->cmdname == NULL) {
	fprintf(stderr, "mt: bad command `%s'\n", cmd);
	usage();
  }

  if((fd = open(tapefile, (op->opcode == MTWEOF || op->opcode == MTERASE)?
		O_WRONLY : O_RDONLY)) < 0) {
	perror(tapefile);
	usage();
  }

  tape_op.mt_count = count;
  tape_op.mt_op = op->opcode;
  rc = ioctl(fd, MTIOCTOP, &tape_op);
  if(rc) {
	perror(tapefile);
	exit(2);
  }
  exit(0);	
}

#if BOGUS_IOCTL
/*
 * The Minix 1.5 ioctl() library wrapper is TTY specific.  Rather than
 * just passing the parameters through, the library routine does some of
 * the ioctl parm processing that should be in the TTY task.  Here is my
 * vote for a future ioctl() library routine.
 */
#include <lib.h>
#include <minix/com.h>

#define DEVICE		m2_i1
#define IOCTL_REQUEST	m2_i3	/* COUNT */
#define IOCTL_ARGP	m2_p1	/* ADDRESS */

PUBLIC int ioctl(fd, request, argp)
int fd;
int request;
void *argp;
{
  int n;

  _M.DEVICE = fd;
  _M.IOCTL_REQUEST = request;
  _M.IOCTL_ARGP = (char *)argp;

  n = callx(FS, IOCTL);
  return(n);
}
#endif /* BOGUS_IOCTL */
