/*
 * WMAIL -	MicroWalt Extended Mail Agent.
 *		This is the MicroWalt Mail Agent; which is derived
 *		from  the  "Mini-Mail" written by Peter S. Housel.
 *		This version  has a better  user-interface, and it
 *		also "knows" about things like forwarding, replies
 *		etc. Overall, it looks like the Mail (uppercase M)
 *		on our local DYNIX(tm) (==BSD) system...
 *		The paging-code (for paging letters on the screen)
 *		was taken from "more.c", by Brandon Allbery.
 *
 * Author:	Fred van Kempen, MicroWalt Corporation
 *
 * Revisions:
 *		11/07/89 FvK	Edited a little for the new MSS.
 *				Allocate the "From_" path dynamically;
 *				it uses too much space now.
 *				Fixed the "reply" bug that was caused
 *				by a badly-decoded "From_" line.
 *		11/10/89 FvK	Fixed the overall security-bug.
 *		12/04/89 FvK	Fixed the "-i filename" security bug.
 *				Fixed the 'adressee' typo.
 *				Fixed return() bugs.
 *		12/16/89 FvK	Fixed 'reply' troubles.
 *				Cleanup.
 *		02/17/90 Fvk	Cleaned for release.
 *		04/10/90 FvK	Added support for RFC-generated headers,
 *				which tend to contain () and <> marks.
 *		04/28/90 FvK	Modified allowed() for ROOT (==UID 0)
 *				Added "<address>" literal support in the
 *				"From_ " line (RFC-badness!).
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <unistd.h>
#include <stdio.h>
#include "wmail.h"


char *Version = VERSION;
int old_uid, old_gid;			/* "real" ID's of caller */
int remote = FALSE;			/* use RMAIL to deliver (if any) */
int loclink = FALSE;			/* LMAIL: local delivery only! */
int printmode = FALSE;			/* print-and-exit mode */
int immediate = FALSE;			/* send remote immediately! */
int quitmode = FALSE;			/* take interrupts */
int usedrop = TRUE;			/* read the maildrop (no -f given) */
int verbose = FALSE;			/* pass "-v" flag on to mailer */
int needupdate = FALSE;			/* need to update mailbox */
int sayall = FALSE;			/* include line with all recipients */
int checkonly = FALSE;			/* only chack if there is mail */
char sender[PATHLEN];			/* who sent the message? */
char forward[PATHLEN];			/* who is the message forwarded to? */
char addressee[PATHLEN];		/* current recipient */
char recipients[PATHLEN];		/* also to... users */
char mailbox[PATHLEN];			/* user's mailbox/maildrop */
char subject[PATHLEN];			/* subject of message */
char msg_temp[PATHLEN];			/* temporary file */
FILE *infp = (FILE *)NULL;		/* current message input file */
FILE *boxfp = (FILE *)NULL;		/* mailbox file */
char *progname;				/* program name */
jmp_buf printjump;			/* for quitting out of letters */
LETTER *firstlet, *lastlet;		/* message pointers for readbox() */
int numlet, seqno;			/* number of active letters */
unsigned oldmask;			/* saved umask() */ 


extern int getopt(), optind, opterr;	/* from the GETOPT(3) package */
extern char *optarg;


void onint()
{
  longjmp(printjump, TRUE);
}


/*
 * Chop off the last (file) part of a filename.
 */
char *basename(name)
register char *name;
{
  register char *p;

  p = strrchr(name, '/');
  if (p == (char *)NULL) return(name);	/* no pathname */
    else return(p + 1);
}


/*
 * Chop off the last (user) part of a UUCP path-name.
 * This is needed for the summary() routine.
 */
char *basepath(name)
register char *name;
{
  register char *p;

  p = strrchr(name, '!');
  if (p == (char *)NULL) return(name);	/* no pathname */
    else return(p + 1);
}


/*
 * return ASCII text of our login-name.
 */
char *whoami()
{
  register struct passwd *pw;
  extern struct passwd *getpwuid();

  if ((pw = getpwuid(getuid())) != (struct passwd *)NULL)
						return(pw->pw_name);
    else return("nobody");
}


/*
 * Check if we may perform operation 'mode' on
 * file 'name'. System Security!
 * If the error is 'ENOENT', then test the parent
 * directory for the desired access.
 */
int allowed(name, mode)
char *name;			/* name of file to be checked */
unsigned short mode;		/* mode to check (R=4, W=2, X=1) */
{
  char abuf[1024];		/* temp. buf for filename */
  struct stat stb;
  char *p;

  /* Is this 'The Master' calling? */
  if (old_uid == 0) return(TRUE);

  if (stat(name, &stb) < 0) {
	if (errno == ENOENT) {			/* file does not exist */
		strcpy(abuf, name);		/* so check its parent dir */
		p = strrchr(abuf, '/');	
		if (p == (char *)NULL)		/* plain filename, */
			getcwd(abuf, 1023);	/* get current dir */
		  else *p = '\0';		/* strip 'file' part */
		if (stat(abuf, &stb) < 0) return(FALSE);	/* error? */
	} else return(FALSE);			/* it exists, other error! */
  }

  /* We now have the status of the file or its parent dir. */
  if (stb.st_uid == old_uid) {			/* we are owner! */
	if ((stb.st_mode >> 6) & mode)
				 return(TRUE);	/* OK, we may do it. */
  	  else return(FALSE);			/* Alas... */
  } else if (stb.st_uid == old_gid) {		/* are we the same group? */
	if ((stb.st_mode >>3) & mode)
				 return(TRUE);	/* OK, we may do it. */
  	  else return(FALSE);			/* Alas... */
  } else if (stb.st_mode & mode)		/* we are 'others' */
				 return(TRUE);	/* OK, we may do it. */
  return(FALSE);				/* Alas... */
}


/*
 * Find the given entry in the mail-header
 * Search for the first occurence of string 'text' in the header.
 * Copy the text following it into the 'let' structure.
 * Return buffer if found, else NULL.
 */
char *find_string(let, text)
LETTER *let;
char *text;
{
  static char findbuff[128];
  static char inbuff[512];
  off_t curr, limit;
  register char *sp;
  int all;

  fseek(boxfp, let->location, SEEK_SET);
  limit = (off_t) -1L;
  if (let->next != NIL_LET) limit = let->next->location;

  all = FALSE;
  curr = let->location;
  while (curr != limit && all==FALSE) {
	if (fgets(inbuff, sizeof(inbuff), boxfp) == (char *)NULL) all = TRUE;
      	if (inbuff[0] == '\0') all = TRUE; /* end-of-header */

      	if (!strncmp(inbuff, text, strlen(text))) {
		sp = &inbuff[0];		/* remove '\n' */
      		while (*sp && *sp!='\n') sp++;
      		*sp = '\0';
		sp = &inbuff[0] + strlen(text);	/* copy to static buff */
		strcpy(findbuff, sp);
		return(findbuff);		/* return address of buff */
       	}

	curr += (off_t) strlen(inbuff);		/* update message offset */

	if (all==FALSE && limit > 0L)		/* quit if past message */
		if (curr >= limit) all = TRUE;
  }
  return((char *)NULL);
}


/*
 * Check is the first line of the mailbox contains a line like
 *
 *	Forward to XXXX
 *
 * then all mail for the calling user is being forwarded
 * to user XXXX. Return a 1 value if this is the case.
 * Otherwise, return 0 (or -1 for error).
 */
int chk_box()
{
  char xbuf[128];
  FILE *fp;
  char *bp;

  if (access(mailbox, 4) < 0 || 
    (fp = fopen(mailbox, "r")) == (FILE *)NULL) {
	if (usedrop && errno==ENOENT) return(-1);
     	fprintf(stderr, "%s: cannot access mailbox ", progname);
      	perror(mailbox);
      	exit(1);
  }

  bp = fgets(xbuf, sizeof(xbuf), fp);
  fclose(fp);

  if (bp!=(char *)NULL && !strncmp(xbuf, "Forward to ", 11)) {
	strcpy(forward, strrchr(xbuf, ' ') + 1);	/* get username */
	forward[strlen(forward)-1] = '\0';		/* remove \n */
	return(1);
  }
  return(0);
}


/*
 * Clear out any in the given address line.
 * Some mailers add text which shouldn't be in a V6/V7
 * "From user date" address line.  This routine removes
 * all that junk, and returns the address of the new
 * string.  Actually, this means conversion of:
 *
 *	From waltje (Fred van Kempen) Tue, 10 Apr 90 20:30:00
 * to:
 *	From waltje Tue, 10 Apr 90 20:30:00
 */
char *clr_hdr(old)
char *old;
{
  char *buf;
  register char *bp, *sp;
  int needpar, needlit;

  /* First of all, clear out any junk. */
  buf = (char *) malloc(strlen(old) + 2);
  if (buf == (char *)NULL) {
	fprintf(stderr, "%s: out of memory: \"%s\"\n", progname, old);
	return((char *)NULL);
  }
  bp = buf; sp = old;
  needpar = needlit = FALSE;
  while (*sp) switch(*sp) {
	case '\t':
	case ' ':
		if (*(bp - 1) != ' ') *bp++ = ' ';
		sp++;
		while (*sp == ' ' || *sp == '\t') sp++;
		break;
	case '(':	/* start comment */
		needpar = TRUE;			/* we need the closer! */
		sp++;
		break;
	case ')':	/* end comment */
		if (needpar == TRUE) {
			needpar = FALSE;
			sp++;
		} else {
			fprintf(stderr, "%s: bad header: \"%s\"\n",
							progname, old);
			return((char *)NULL);
		}
		break;
	case '<':	/* start literal */
		needlit = TRUE;
		sp++;
		break;
	case '>':	/* end literal */
		if (needlit == TRUE) {
			needlit = FALSE;
			sp++;
		} else {
			fprintf(stderr, "%s: bad header: \"%s\"\n",
							progname, old);
			return((char *)NULL);
		}
		break;
	default:
		if (needpar == FALSE && needlit == FALSE) *bp++ = *sp;
		sp++;
  }
  *bp = '\0';
  return(buf);
}


/*
 * Decode an old-style (V6/V7) mail header.
 * This is a line like:
 *
 *	From <user> <date> [remote from <host>]
 *
 * We want to find out the <user>, <date> and possible <remote> fields.
 * Warning: some mailers (especially some configurations of the
 *	    SendMail program) allow comments (like (RealName)) to be
 *	    placed in this field.  These comments are removed by the
 *	    clr_hdr() routine.
 */
static void old_hdr(let, text)
LETTER *let;			/* which letter? */
char *text;			/* message header text */
{
  register char *bp, *sp;
  char *buf, *cp;
  int i;

  /* First of all, clear out any junk. */
  buf = clr_hdr(text);
  sp = buf;
  if (sp == (char *)NULL) return;

  /*
   * Then, mark the end of the 'user' field.
   * Skip until <date> field.
   */
  while (*sp && *sp != ' ' && *sp != '\t') sp++;
  *sp++ = '\0';			/* mark end, 'text' is now <user> */

  /*
   * SP now contains <date> and (possible) <remote> fields.
   * Parse line to seek out "remote from".
   */
  cp = sp;			/* save the Date-pointer */
  while (TRUE) {
	bp = strchr(sp++, 'r');
	if (bp != (char *)NULL) {	/* we found an 'r' */
		if (!strncmp(bp, "remote from ", 12)) break;
        } else break;
  }

  if (bp != (char *)NULL) {		/* host found --> remote mail */
	sp = strrchr(bp, ' ');		/* start of "remote from" text */
	*(bp - 1) = '\0';		/* mark end-of-date */
	strcpy(let->date, cp);		/* set date */
	i = strlen(++sp);		/* set length of hostname */
  } else {
	  strcpy(let->date, cp);	/* set date */
	  i = 0;			/* no hostname! */
  }
  let->sender = (char *)malloc(strlen(buf) + i + 4);
  if (let->sender == (char *)NULL) {	/* no memory left! */
	fprintf(stderr, "%s: out of memory.\n", progname);
	exit(1);
  }

  /* Create the return-address. */
  if (i > 0) sprintf(let->sender, "%s!%s", sp, buf);
    else strcpy(let->sender, buf);

  /* Release the allocated memory. */
  free(buf);
}


/* 
 * Read the contents of the Mail-Box into memory.
 */
static int readbox()
{
  static char lbuff[512];
  register LETTER *let;
  register char *sp;
  off_t current;
 
  firstlet = lastlet = NIL_LET;
  numlet = 0;
  seqno = 1;

  if (chk_box() == 1) return(FALSE);	/* mail is being forwarded... */

  if ((boxfp = fopen(mailbox, "r")) == (FILE *)NULL) {
	if (usedrop && errno==ENOENT) return(-1);
      	fprintf(stderr, "%s: cannot access mailbox ", progname);
      	perror(mailbox);
      	exit(1);
  }

  /*
   * Determine where all messages start.
   * This should be done with an index file in the future!
   */
  current = (off_t) 0;
  while(fgets(lbuff, sizeof(lbuff), boxfp) != (char *)NULL) {
	current = ftell(boxfp);
      	if (!strncmp(lbuff, "From ", 5)) {
		if ((let = (LETTER *)malloc(sizeof(LETTER))) == NIL_LET) {
			fprintf(stderr, "%s: out of memory.\n", progname);
			exit(1);
	 	}
        	if (lastlet == NIL_LET) {
			firstlet = let;
			let->prev = NIL_LET;
	 	} else {
	 		let->prev = lastlet;
	     	 	lastlet->next = let;
	    	}
		lastlet = let;
		let->next = NIL_LET;

		let->status = UNREAD;
		let->location = current - (off_t) strlen(lbuff);
		let->seqno = seqno++;
		numlet++;
	}
  }

  /*
   * We now know where the messages are, read message headers.
   */
  let = firstlet;
  while (let != NIL_LET) {
	sp = find_string(let, "From ");		/* Find the "From_" field. */
	if (sp == (char *)NULL) {
		fprintf(stderr, "%s: no \"From\"-line.\n", progname);
		exit(-1);
	}

	old_hdr(let, sp);			/* decode it */

	/* Find the "Subject" field, if any... */
	sp = find_string(let, "Subject: ");
	if (sp == (char *)NULL) sp = "<none>";

	/* ..and stuff it in memory. */
	let->subject = (char *)malloc(strlen(sp) + 1);
	if (let->subject == (char *)NULL) {	/* no memory left! */
		fprintf(stderr, "%s: out of memory.\n", progname);
		exit(1);
	}
	strcpy(let->subject, sp);

	let = let->next;
  }
  return(TRUE);
}


/*
 * Check if there is any mail for the calling user.
 * Return 0 if there is mail, or 1 for NO MAIL.
 */
static int chk_mail()
{
  FILE *fp;
  char temp[512];

  if ((fp = fopen(mailbox, "r")) != (FILE *)NULL) {
	if (fgets(temp, sizeof(temp), fp) == (char *)NULL) {
		fclose(fp);	/* empty mailbox, no mail! */
		return(1);
	}
      	if (!strncmp(temp, "Forward to ", 11)) {
		fclose(fp);	/* FORWARD line in mailbox */
		return(2);	/* so no mail. */
	}
	fclose(fp);	/* another line, so we have mail! */
	return(0);
  }
  return(1);
}


static void usage()
{
  fprintf(stderr, "Usage:\n");
  fprintf(stderr, "\t%s [-epqrv] [-f file]\n", progname);
  fprintf(stderr, "\t%s [-dtv] [-i file] [-s subject] user ...\n", progname);
  fprintf(stderr, "\t%s [-lv] [-i file] user\n\n", progname);
}


int main(argc, argv)
int argc; char *argv[];
{
  int c, st;

  strcpy(sender, whoami());		/* get our real name */

  old_uid = getuid();			/* This is dangerous, but */
  old_gid = getgid();			/* is is necessary for the */
  setuid(geteuid());			/* message delivery! */
  setgid(getegid());

  progname = basename(argv[0]);		/* how are we called? */
  if (*progname == 'l') {
	remote = FALSE;			/* 'lmail' link? */
	loclink = TRUE;
  }
  strcpy(msg_temp, MAILTEMP);
  mktemp(msg_temp);			/* name the temp file */
  oldmask = umask(077);			/* change umask for security */
  infp = stdin;				/* set INPUT to stdin */
  
  while ((c = getopt(argc, argv, "def:i:lpqrs:tv")) != EOF) switch(c) {
	case 'd':	/* Deliver immediately. */
		immediate++;
		break;
	case 'e':	/* Only check for mail, do not read it. */
		checkonly++;
		break;
	case 'f':	/* use another mailbox. */
		usedrop = FALSE;
		setuid(old_uid);	/* security! */
		setgid(old_gid);
		strncpy(mailbox, optarg, PATHLEN - 1);
		break;
	case 'i':	/* Use another input-file. */
		if (allowed(optarg, 04) == TRUE) {
			infp = fopen(optarg, "r");
		} else infp = (FILE *)NULL;
		if (infp == (FILE *)NULL) {
			fprintf(stderr, "%s: cannot open %s\n",
						progname, optarg);
			exit(-1);
		}
		break;
	case 'l':	/* Specify 'local' mail; same as 'lmail'. */
		loclink = TRUE;
		break;
	case 'p':	/* Print all messages and exit. */
		printmode++;
		break;
	case 'q':	/* Abort if SIGINT received. */
		quitmode++;
		break;
	case 'r':	/* Show messages in reverse order. */
		break;	/* This is only present for comp. with binmail! */
	case 's':	/* Specify "subject" line. */
		strcpy(subject, optarg);
		break;
	case 't':	/* Show all addressees in "To:" line. */
		sayall++;
		break;
	case 'v':	/* Turn on debugging. */
		verbose++;
		break;
	default:
		usage();
		exit(2);
  }

  if (optind >= argc) {
	if (usedrop) sprintf(mailbox, DROPNAME, sender);

	if (checkonly) {
		st = chk_mail();
		exit(st);
	}

	if (readbox() == FALSE) {
		fprintf(stderr, "Your mail is being forwarded to %s.\n",
								forward);
		exit(1);
	 } else {
		 st = 0;
	 	 if (printmode) printall();
		   else interact();

		 if (needupdate) updatebox();
	}
  } else {
	  st = deliver(argc - optind, argv + optind);

	  if (st != 0) dead_letter();	/* something went wrong... */
  }
  unlink(msg_temp);
  exit(st);    
}
