/* Copyright (c) 1989, 1991 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  std.t   - standard default adventure definitions
  Version 1.2
  
  This file is part of TADS:  The Text Adventure Development System.
  Please see the file LICENSE.DOC (which should be part of the TADS
  distribution) for information on using this file, and for information
  on reaching High Energy Software, the developers of TADS.

  This file provides some simple definitions for objects and functions
  that are required by TADS, but not defined in the file "adv.t".
  The definitions in std.t are suitable for use while a game is
  being written, but you will probably find that you will want to
  customize the definitions in this file for your game when the
  game is nearing completion.  This file is intended to help you
  get started more quickly by providing basic definitions for these
  functions and objects.

  When you decide to customize these functions and objects for
  your game, be sure to remove the inclusion of std.t to avoid
  duplicate definitions.
*/

/*
 *   Pre-declare all functions, so the compiler knows they are functions.
 *   (This is only really necessary when a function will be referenced
 *   as a daemon or fuse before it is defined; however, it doesn't hurt
 *   anything to pre-declare all of them.)
 */
die: function;
scoreRank: function;
preinit:function;
init: function;
terminate: function;
pardon: function;
eatDaemon: function;
darkTravel: function;

/*
 *   The die() function is called when the player dies.  It tells the
 *   player how well he has done (with his score), and asks if he'd
 *   like to start over (the alternative being quitting the game).
 */
die: function
{
    "\b*** You have died ***\b";
    scoreRank();
    "\bYou may restore a saved game, start over, take back
       the last move, or quit.\n";
    while ( 1 )
    {
        local resp;

        "\nPlease enter RESTORE, RESTART, UNDO, or QUIT: >";
        resp := upper(input());
        if ( resp = 'RESTORE' )
        {
            resp := askfile( 'File to restore' );
            if ( resp = nil ) "Restore failed. ";
            else if ( restore( resp )) "Restore failed. ";
            else
            {
                setscore( global.score, global.turnsofar );
                abort;
            }
        }
        else if ( resp = 'RESTART' )
        {
            setscore( 0, 0 );
            restart();
        }
        else if ( resp = 'UNDO' )
        {
            if (undo())
            {
                "(Undoing one command)\b";
                Me.location.lookAround(true);
                setscore(global.score, global.turnsofar);
                abort;
            }
            else
                "Sorry, no undo information is available.";
        }
        else if ( resp = 'QUIT' )
        {
            terminate();
            quit();
            abort;
        }
    }
}

/*
 *   The score() function displays how well the player is doing.
 *   This default definition doesn't do anything aside from displaying
 *   the current and maximum scores.  Some game designers like to
 *   provide a ranking that goes with various scores ("Novice Adventurer,"
 *   "Expert," and so forth); this is the place to do so if desired.
 *
 *   Note that "global.maxscore" defines the maximum number of points
 *   possible in the game; change the property in the "global" object
 *   if necessary.
 */
scoreRank: function
{
    "In a total of "; say( global.turnsofar );
    " turns, you have achieved a score of ";
    say( global.score ); " points out of a possible ";
    say( global.maxscore ); ".\n";
		"This gives you the rank ";
		if (global.score <=25) {"of being lost, hopelessly lost.\b";}
		if (global.score <=50 and global.score > 25) {"of lost without a clue.\b";}
		if (global.score <=75 and global.score > 50) {"of a lost suburban with a compass.\b"; }
		if (global.score <=100 and global.score > 75) {"of being lost up the proverbial river without the 
					proverbial paddle.\b"; }
		if (global.score <=150 and global.score > 100) {"of being lost in the dark while 
								wearing sunglasses.\b";}
		if (global.score <=200 and global.score > 150) {"of being lost.  Just lost.\b"; }
		if (global.score <=250 and global.score > 250) {"of being lost with some resemblance of 
							a clue.\b";}
		if (global.score <=300 and global.score > 250) {"of being lost with a comapss.\b";}
		if (global.score <=350 and global.score > 300) {"of being lost with a good compass and a 
							good map.\b";}
		if (global.score < 400 and global.score > 350 ) {"of being lost, but having a 
					really good idea of where you are.\b";}
		if (global.score = 400) {"of \"Found.\"\b";}
		if (global.score > 400) {"of an impossible score, since there are only 400 points in 
						the game. (How did you do that????)\b";}
}

/*
 *   The init() function is run at the very beginning of the game.
 *   It should display the introductory text for the game, start
 *   any needed daemons and fuses, and move the player's actor ("Me")
 *   to the initial room, which defaults here to "startroom".
 */
init: function
{

	"\b\b\(Earth.  1995 AD\)\b\b
	\t\"Wow cool!\" you thought as you looked out your bedroom window and saw that meteor 
			crash.\n
	\t\"It must have landed only a few miles from here,\" you thought as you stuffed your flashlight 
			and a peanut butter sandwich into you backpack.\n
	\t\"If I find that meteor I will be famous!\" you gleefully told yourself as you bounded out of 
			your house.\n
	\tWell it is a few hours later and all that you have managed to do is get yourself ...\b\b\b";
   "LOST\nVersion 2.0.2\nA TADS Adventure
      	\nDeveloped with TADS, the Text Adventure Development System.\n
		 \nWritten by Jeffrey Hersh.\b
			Big thanks to all of those who beta tested Lost.\b
			This game now Shareware.  If you like it, register by sending $10, 
			along with your name, mail address, E-mail address and the type of system 
				you are running Lost on to\b
			\tJeffrey Hersh\n\t30 Renfrew Road\n\tNew City, NY 10956\n\tRE Lost\b
			\tRegistered users will receive complete maps and hints to the game.
			\n\tIf you have any questions or comments about
			the game or wish a copy of the source code write to the above 
			address or through e-mail to 
			Frumple@earthlink.com.
			\n\tGeneral instructions for Lost can be accessed by typing INSTRUCTIONS at the 
			prompt.\b\b\b\b";
     setdaemon( turncount, nil );               // start the turn counter daemon
    setdaemon( eatDaemon, nil );                     // start the hunger daemon
    Me.location := startroom;                // move player to initial location
    startroom.lookAround( true );                    // show player where he is
}

/*
 *   preinit() is called after compiling the game, before it is written
 *   to the binary game file.  It performs all the initialization that can
 *   be done statically before storing the game in the file, which speeds
 *   loading the game, since all this work has been done ahead of time.
 *
 *   This routine puts all lamp objects (those objects with islamp = true) into
 *   the list global.lamplist.  This list is consulted when determining whether
 *   a dark room contains any light sources.
 */
preinit: function
{
    local o;
    
    global.lamplist := [];
    o := firstobj();
    while( o <> nil )
    {
        if ( o.islamp ) global.lamplist := global.lamplist + o;
        o := nextobj( o );
    }
    initSearch();
}

/*
 *   The terminate() function is called just before the game ends.  It
 *   generally displays a good-bye message.  The default version does
 *   nothing.  Note that this function is called only when the game is
 *   about to exit, NOT after dying, before a restart, or anywhere else.
 */
terminate: function
{
}

/*
 *   The pardon() function is called any time the player enters a blank
 *   line.  The function generally just prints a message ("Speak up" or
 *   some such).  This default version just says "I beg your pardon?"
 */
pardon: function
{
    "I beg your pardon? ";
}


/*
 *   This function is a daemon, set running by init(), which monitors how
 *   long it has been since the player has had anything to eat.  It will
 *   provide warnings for some time prior to the player's expiring from
 *   hunger, and will kill the player if he should go too long without
 *   heeding these warnings.
 */
eatDaemon: function( parm )
{
    local e, l;

    global.lastMealTime := global.lastMealTime + 1;
    e := global.eatTime;
    l := global.lastMealTime;

    if ( l = e or l = e+5 or l = e+10 )
        "\bYou're feeling a bit peckish. Perhaps it would be a good
        time to find something to eat. ";
    else if ( l = e+15 or l = e+20 or l = e+25 )
        "\bYou're feeling really hungry. You should find some food
        soon or you'll pass out from lack of nutrition. ";
    else if ( l=e+30 or l = e+35 )
        "\bYou can't go much longer without food. ";
    else if ( l >= e+40 )
    {
        "\bYou simply can't go on any longer without food. You perish from
        lack of nutrition. ";
        die();
    }
}

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj;  // use default definition from adv.t

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */
strObj: basicStrObj;     // use default definition from adv.t

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.  The properties of this
 *   object that are particularly important to the objects and functions
 *   are defined here; if you replace this object, but keep other parts
 *   of this file, be sure to include the properties defined here.
 *
 *   Note that awakeTime is set to zero; if you wish the player to start
 *   out tired, just move it up around the sleepTime value (which specifies
 *   the interval between sleeping).  The same goes for lastMealTime; move
 *   it up to around eatTime if you want the player to start out hungry.
 *   With both of these values, the player only starts getting warnings
 *   when the elapsed time (awakeTime, lastMealTime) reaches the interval
 *   (sleepTime, eatTime); the player isn't actually required to eat or
 *   sleep until several warnings have been issued.  Look at the eatDaemon
 *   and sleepDaemon functions for details of the timing.
 */
global: object
    turnsofar = 0                            // no turns have transpired so far
    score = 0                            // no points have been accumulated yet
    maxscore = 400                                    // maximum possible score
    verbose = nil                             // we are currently in TERSE mode
	ver = nil														// verbosity tag
	suitw = nil													//suit worn tag
	darkcount = nil											//darktravel tag
    awakeTime = 0               // time that has elapsed since the player slept
    lastMealTime = 0              // time that has elapsed since the player ate
    eatTime = 200         // interval between meals (longest time without food)
    lamplist = []              // list of all known light providers in the game
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */
version: object
    sdesc = "LOST\nVersion 2.0.2\nA TADS Adventure"
;

/*
 *   "Me" is the player's actor.  Pick up the default definition, basicMe,
 *   from "adv.t".
 */
Me: basicMe
;

/*
 *   darkTravel() is called whenever the player attempts to move from a dark
 *   location into another dark location.  By default, it just says "You
 *   stumble around in the dark," but it could certainly cast the player into
 *   the jaws of a grue, whatever that is...
 */
darkitem: item
	darkcount = 0
;

darkTravel: function
{
darkitem.darkcount := darkitem.darkcount + 1;
    if(darkitem.darkcount < 3)
		{
		"Wandering around in the dark is a bad idea.  I suggest you find some light.";
		exit;
		}
	if(darkitem.darkcount = 3)
		{
		"Well now you did it.  By wandering around in the dark you manage to trip over 
		something, fall and break your neck.";
		die();
		}
}


;
