// Created with Squiffy 5.1.0
// https://github.com/textadventures/squiffy

(function(){
/* jshint quotmark: single */
/* jshint evil: true */

var squiffy = {};

(function () {
    'use strict';

    squiffy.story = {};

    var initLinkHandler = function () {
        var handleLink = function (link) {
            if (link.hasClass('disabled')) return;
            var passage = link.data('passage');
            var section = link.data('section');
            var rotateAttr = link.attr('data-rotate');
            var sequenceAttr = link.attr('data-sequence');
            if (passage) {
                disableLink(link);
                squiffy.set('_turncount', squiffy.get('_turncount') + 1);
                passage = processLink(passage);
                if (passage) {
                    squiffy.story.passage(passage);
                }
                var turnPassage = '@' + squiffy.get('_turncount');
                if (turnPassage in squiffy.story.section.passages) {
                    squiffy.story.passage(turnPassage);
                }
                if ('@last' in squiffy.story.section.passages && squiffy.get('_turncount')>= squiffy.story.section.passageCount) {
                    squiffy.story.passage('@last');
                }
            }
            else if (section) {
                disableLink(link);
                section = processLink(section);
                squiffy.story.go(section);
            }
            else if (rotateAttr || sequenceAttr) {
                var result = rotate(rotateAttr || sequenceAttr, rotateAttr ? link.text() : '');
                link.html(result[0].replace(/&quot;/g, '"').replace(/&#39;/g, '\''));
                var dataAttribute = rotateAttr ? 'data-rotate' : 'data-sequence';
                link.attr(dataAttribute, result[1]);
                if (!result[1]) {
                    disableLink(link);
                }
                if (link.attr('data-attribute')) {
                    squiffy.set(link.attr('data-attribute'), result[0]);
                }
                squiffy.story.save();
            }
        };

        squiffy.ui.output.on('click', 'a.squiffy-link', function () {
            handleLink(jQuery(this));
        });

        squiffy.ui.output.on('keypress', 'a.squiffy-link', function (e) {
            if (e.which !== 13) return;
            handleLink(jQuery(this));
        });

        squiffy.ui.output.on('mousedown', 'a.squiffy-link', function (event) {
            event.preventDefault();
        });
    };

    var disableLink = function (link) {
        link.addClass('disabled');
        link.attr('tabindex', -1);
    }

    squiffy.story.begin = function () {
        if (!squiffy.story.load()) {
            squiffy.story.go(squiffy.story.start);
        }
    };

    var processLink = function(link) {
        var sections = link.split(',');
        var first = true;
        var target = null;
        sections.forEach(function (section) {
            section = section.trim();
            if (startsWith(section, '@replace ')) {
                replaceLabel(section.substring(9));
            }
            else {
                if (first) {
                    target = section;
                }
                else {
                    setAttribute(section);
                }
            }
            first = false;
        });
        return target;
    };

    var setAttribute = function(expr) {
        var lhs, rhs, op, value;
        var setRegex = /^([\w]*)\s*=\s*(.*)$/;
        var setMatch = setRegex.exec(expr);
        if (setMatch) {
            lhs = setMatch[1];
            rhs = setMatch[2];
            if (isNaN(rhs)) {
                squiffy.set(lhs, rhs);
            }
            else {
                squiffy.set(lhs, parseFloat(rhs));
            }
        }
        else {
            var incDecRegex = /^([\w]*)\s*([\+\-])=\s*(.*)$/;
            var incDecMatch = incDecRegex.exec(expr);
            if (incDecMatch) {
                lhs = incDecMatch[1];
                op = incDecMatch[2];
                rhs = parseFloat(incDecMatch[3]);
                value = squiffy.get(lhs);
                if (value === null) value = 0;
                if (op == '+') {
                    value += rhs;
                }
                if (op == '-') {
                    value -= rhs;
                }
                squiffy.set(lhs, value);
            }
            else {
                value = true;
                if (startsWith(expr, 'not ')) {
                    expr = expr.substring(4);
                    value = false;
                }
                squiffy.set(expr, value);
            }
        }
    };

    var replaceLabel = function(expr) {
        var regex = /^([\w]*)\s*=\s*(.*)$/;
        var match = regex.exec(expr);
        if (!match) return;
        var label = match[1];
        var text = match[2];
        if (text in squiffy.story.section.passages) {
            text = squiffy.story.section.passages[text].text;
        }
        else if (text in squiffy.story.sections) {
            text = squiffy.story.sections[text].text;
        }
        var stripParags = /^<p>(.*)<\/p>$/;
        var stripParagsMatch = stripParags.exec(text);
        if (stripParagsMatch) {
            text = stripParagsMatch[1];
        }
        var $labels = squiffy.ui.output.find('.squiffy-label-' + label);
        $labels.fadeOut(1000, function() {
            $labels.html(squiffy.ui.processText(text));
            $labels.fadeIn(1000, function() {
                squiffy.story.save();
            });
        });
    };

    squiffy.story.go = function(section) {
        squiffy.set('_transition', null);
        newSection();
        squiffy.story.section = squiffy.story.sections[section];
        if (!squiffy.story.section) return;
        squiffy.set('_section', section);
        setSeen(section);
        var master = squiffy.story.sections[''];
        if (master) {
            squiffy.story.run(master);
            squiffy.ui.write(master.text);
        }
        squiffy.story.run(squiffy.story.section);
        // The JS might have changed which section we're in
        if (squiffy.get('_section') == section) {
            squiffy.set('_turncount', 0);
            squiffy.ui.write(squiffy.story.section.text);
            squiffy.story.save();
        }
    };

    squiffy.story.run = function(section) {
        if (section.clear) {
            squiffy.ui.clearScreen();
        }
        if (section.attributes) {
            processAttributes(section.attributes);
        }
        if (section.js) {
            section.js();
        }
    };

    squiffy.story.passage = function(passageName) {
        var passage = squiffy.story.section.passages[passageName];
        if (!passage) return;
        setSeen(passageName);
        var masterSection = squiffy.story.sections[''];
        if (masterSection) {
            var masterPassage = masterSection.passages[''];
            if (masterPassage) {
                squiffy.story.run(masterPassage);
                squiffy.ui.write(masterPassage.text);
            }
        }
        var master = squiffy.story.section.passages[''];
        if (master) {
            squiffy.story.run(master);
            squiffy.ui.write(master.text);
        }
        squiffy.story.run(passage);
        squiffy.ui.write(passage.text);
        squiffy.story.save();
    };

    var processAttributes = function(attributes) {
        attributes.forEach(function (attribute) {
            if (startsWith(attribute, '@replace ')) {
                replaceLabel(attribute.substring(9));
            }
            else {
                setAttribute(attribute);
            }
        });
    };

    squiffy.story.restart = function() {
        if (squiffy.ui.settings.persist && window.localStorage) {
            var keys = Object.keys(localStorage);
            jQuery.each(keys, function (idx, key) {
                if (startsWith(key, squiffy.story.id)) {
                    localStorage.removeItem(key);
                }
            });
        }
        else {
            squiffy.storageFallback = {};
        }
        if (squiffy.ui.settings.scroll === 'element') {
            squiffy.ui.output.html('');
            squiffy.story.begin();
        }
        else {
            location.reload();
        }
    };

    squiffy.story.save = function() {
        squiffy.set('_output', squiffy.ui.output.html());
    };

    squiffy.story.load = function() {
        var output = squiffy.get('_output');
        if (!output) return false;
        squiffy.ui.output.html(output);
        currentSection = jQuery('#' + squiffy.get('_output-section'));
        squiffy.story.section = squiffy.story.sections[squiffy.get('_section')];
        var transition = squiffy.get('_transition');
        if (transition) {
            eval('(' + transition + ')()');
        }
        return true;
    };

    var setSeen = function(sectionName) {
        var seenSections = squiffy.get('_seen_sections');
        if (!seenSections) seenSections = [];
        if (seenSections.indexOf(sectionName) == -1) {
            seenSections.push(sectionName);
            squiffy.set('_seen_sections', seenSections);
        }
    };

    squiffy.story.seen = function(sectionName) {
        var seenSections = squiffy.get('_seen_sections');
        if (!seenSections) return false;
        return (seenSections.indexOf(sectionName) > -1);
    };

    squiffy.ui = {};

    var currentSection = null;
    var screenIsClear = true;
    var scrollPosition = 0;

    var newSection = function() {
        if (currentSection) {
            disableLink(jQuery('.squiffy-link', currentSection));
        }
        var sectionCount = squiffy.get('_section-count') + 1;
        squiffy.set('_section-count', sectionCount);
        var id = 'squiffy-section-' + sectionCount;
        currentSection = jQuery('<div/>', {
            id: id,
        }).appendTo(squiffy.ui.output);
        squiffy.set('_output-section', id);
    };

    squiffy.ui.write = function(text) {
        screenIsClear = false;
        scrollPosition = squiffy.ui.output.height();
        currentSection.append(jQuery('<div/>').html(squiffy.ui.processText(text)));
        squiffy.ui.scrollToEnd();
    };

    squiffy.ui.clearScreen = function() {
        squiffy.ui.output.html('');
        screenIsClear = true;
        newSection();
    };

    squiffy.ui.scrollToEnd = function() {
        var scrollTo, currentScrollTop, distance, duration;
        if (squiffy.ui.settings.scroll === 'element') {
            scrollTo = squiffy.ui.output[0].scrollHeight - squiffy.ui.output.height();
            currentScrollTop = squiffy.ui.output.scrollTop();
            if (scrollTo > currentScrollTop) {
                distance = scrollTo - currentScrollTop;
                duration = distance / 0.4;
                squiffy.ui.output.stop().animate({ scrollTop: scrollTo }, duration);
            }
        }
        else {
            scrollTo = scrollPosition;
            currentScrollTop = Math.max(jQuery('body').scrollTop(), jQuery('html').scrollTop());
            if (scrollTo > currentScrollTop) {
                var maxScrollTop = jQuery(document).height() - jQuery(window).height();
                if (scrollTo > maxScrollTop) scrollTo = maxScrollTop;
                distance = scrollTo - currentScrollTop;
                duration = distance / 0.5;
                jQuery('body,html').stop().animate({ scrollTop: scrollTo }, duration);
            }
        }
    };

    squiffy.ui.processText = function(text) {
        function process(text, data) {
            var containsUnprocessedSection = false;
            var open = text.indexOf('{');
            var close;

            if (open > -1) {
                var nestCount = 1;
                var searchStart = open + 1;
                var finished = false;

                while (!finished) {
                    var nextOpen = text.indexOf('{', searchStart);
                    var nextClose = text.indexOf('}', searchStart);

                    if (nextClose > -1) {
                        if (nextOpen > -1 && nextOpen < nextClose) {
                            nestCount++;
                            searchStart = nextOpen + 1;
                        }
                        else {
                            nestCount--;
                            searchStart = nextClose + 1;
                            if (nestCount === 0) {
                                close = nextClose;
                                containsUnprocessedSection = true;
                                finished = true;
                            }
                        }
                    }
                    else {
                        finished = true;
                    }
                }
            }

            if (containsUnprocessedSection) {
                var section = text.substring(open + 1, close);
                var value = processTextCommand(section, data);
                text = text.substring(0, open) + value + process(text.substring(close + 1), data);
            }

            return (text);
        }

        function processTextCommand(text, data) {
            if (startsWith(text, 'if ')) {
                return processTextCommand_If(text, data);
            }
            else if (startsWith(text, 'else:')) {
                return processTextCommand_Else(text, data);
            }
            else if (startsWith(text, 'label:')) {
                return processTextCommand_Label(text, data);
            }
            else if (/^rotate[: ]/.test(text)) {
                return processTextCommand_Rotate('rotate', text, data);
            }
            else if (/^sequence[: ]/.test(text)) {
                return processTextCommand_Rotate('sequence', text, data);
            }
            else if (text in squiffy.story.section.passages) {
                return process(squiffy.story.section.passages[text].text, data);
            }
            else if (text in squiffy.story.sections) {
                return process(squiffy.story.sections[text].text, data);
            }
            return squiffy.get(text);
        }

        function processTextCommand_If(section, data) {
            var command = section.substring(3);
            var colon = command.indexOf(':');
            if (colon == -1) {
                return ('{if ' + command + '}');
            }

            var text = command.substring(colon + 1);
            var condition = command.substring(0, colon);

            var operatorRegex = /([\w ]*)(=|&lt;=|&gt;=|&lt;&gt;|&lt;|&gt;)(.*)/;
            var match = operatorRegex.exec(condition);

            var result = false;

            if (match) {
                var lhs = squiffy.get(match[1]);
                var op = match[2];
                var rhs = match[3];

                if (op == '=' && lhs == rhs) result = true;
                if (op == '&lt;&gt;' && lhs != rhs) result = true;
                if (op == '&gt;' && lhs > rhs) result = true;
                if (op == '&lt;' && lhs < rhs) result = true;
                if (op == '&gt;=' && lhs >= rhs) result = true;
                if (op == '&lt;=' && lhs <= rhs) result = true;
            }
            else {
                var checkValue = true;
                if (startsWith(condition, 'not ')) {
                    condition = condition.substring(4);
                    checkValue = false;
                }

                if (startsWith(condition, 'seen ')) {
                    result = (squiffy.story.seen(condition.substring(5)) == checkValue);
                }
                else {
                    var value = squiffy.get(condition);
                    if (value === null) value = false;
                    result = (value == checkValue);
                }
            }

            var textResult = result ? process(text, data) : '';

            data.lastIf = result;
            return textResult;
        }

        function processTextCommand_Else(section, data) {
            if (!('lastIf' in data) || data.lastIf) return '';
            var text = section.substring(5);
            return process(text, data);
        }

        function processTextCommand_Label(section, data) {
            var command = section.substring(6);
            var eq = command.indexOf('=');
            if (eq == -1) {
                return ('{label:' + command + '}');
            }

            var text = command.substring(eq + 1);
            var label = command.substring(0, eq);

            return '<span class="squiffy-label-' + label + '">' + process(text, data) + '</span>';
        }

        function processTextCommand_Rotate(type, section, data) {
            var options;
            var attribute = '';
            if (section.substring(type.length, type.length + 1) == ' ') {
                var colon = section.indexOf(':');
                if (colon == -1) {
                    return '{' + section + '}';
                }
                options = section.substring(colon + 1);
                attribute = section.substring(type.length + 1, colon);
            }
            else {
                options = section.substring(type.length + 1);
            }
            var rotation = rotate(options.replace(/"/g, '&quot;').replace(/'/g, '&#39;'));
            if (attribute) {
                squiffy.set(attribute, rotation[0]);
            }
            return '<a class="squiffy-link" data-' + type + '="' + rotation[1] + '" data-attribute="' + attribute + '" role="link">' + rotation[0] + '</a>';
        }

        var data = {
            fulltext: text
        };
        return process(text, data);
    };

    squiffy.ui.transition = function(f) {
        squiffy.set('_transition', f.toString());
        f();
    };

    squiffy.storageFallback = {};

    squiffy.set = function(attribute, value) {
        if (typeof value === 'undefined') value = true;
        if (squiffy.ui.settings.persist && window.localStorage) {
            localStorage[squiffy.story.id + '-' + attribute] = JSON.stringify(value);
        }
        else {
            squiffy.storageFallback[attribute] = JSON.stringify(value);
        }
        squiffy.ui.settings.onSet(attribute, value);
    };

    squiffy.get = function(attribute) {
        var result;
        if (squiffy.ui.settings.persist && window.localStorage) {
            result = localStorage[squiffy.story.id + '-' + attribute];
        }
        else {
            result = squiffy.storageFallback[attribute];
        }
        if (!result) return null;
        return JSON.parse(result);
    };

    var startsWith = function(string, prefix) {
        return string.substring(0, prefix.length) === prefix;
    };

    var rotate = function(options, current) {
        var colon = options.indexOf(':');
        if (colon == -1) {
            return [options, current];
        }
        var next = options.substring(0, colon);
        var remaining = options.substring(colon + 1);
        if (current) remaining += ':' + current;
        return [next, remaining];
    };

    var methods = {
        init: function (options) {
            var settings = jQuery.extend({
                scroll: 'body',
                persist: true,
                restartPrompt: true,
                onSet: function (attribute, value) {}
            }, options);

            squiffy.ui.output = this;
            squiffy.ui.restart = jQuery(settings.restart);
            squiffy.ui.settings = settings;

            if (settings.scroll === 'element') {
                squiffy.ui.output.css('overflow-y', 'auto');
            }

            initLinkHandler();
            squiffy.story.begin();

            return this;
        },
        get: function (attribute) {
            return squiffy.get(attribute);
        },
        set: function (attribute, value) {
            squiffy.set(attribute, value);
        },
        restart: function () {
            if (!squiffy.ui.settings.restartPrompt || confirm('Are you sure you want to restart?')) {
                squiffy.story.restart();
            }
        }
    };

    jQuery.fn.squiffy = function (methodOrOptions) {
        if (methods[methodOrOptions]) {
            return methods[methodOrOptions]
                .apply(this, Array.prototype.slice.call(arguments, 1));
        }
        else if (typeof methodOrOptions === 'object' || ! methodOrOptions) {
            return methods.init.apply(this, arguments);
        } else {
            jQuery.error('Method ' +  methodOrOptions + ' does not exist');
        }
    };
})();

var get = squiffy.get;
var set = squiffy.set;


squiffy.story.start = 'introduction';
squiffy.story.id = 'fe6fd45a08';
squiffy.story.sections = {
	'introduction': {
		'text': "<p>{game intro}</p>\n<p>{label:81=<br><br><em>A Fly on the Wall, or An Appositional Eye</em> Interactive apposition by Nigel Jayne. Version 1. Copyright 2017 by Nigel Jayne. IFID: 47V21736-6DM9-42CZ-8QD5-BC29184J621Z<br><br>{game options}}</p>",
		'attributes': ["screentime=0","hauntedManifestations=0","storyline=threads"],
		'js': function() {
			$("#soundWrapper").html("<audio id='theme' autoplay loop controls>");
		},
		'passages': {
			'@1': {
				'text': "",
				'attributes': ["@replace 81="],
			},
			'Visit': {
				'text': "<p><hr>In <i>A Fly on the Wall, or An Appositional Eye</i>, you are represented by &quot;Goldie,&quot; a Fortean Society member tasked with watching five (very high-definition) monitors that receive feeds from five paranormal hotspots in the Harrison Mansion. You&#39;re likely to hear sounds throughout your vigil, if your browser supports the audio.<br><br>There are three kinds of links in the game. Clicking the name of someone or something displays a description. Clicking links that suggest an action will perform that action or cycle through options. Once Goldie settles In the Fly Eye, you will also be able to choose a room to watch by clicking its orange-colored name. Note for screen reader and voice over users: Links are always de-activated after selecting them, though your software may suggest otherwise. ostensibly important events continue to play out in the other rooms as you focus on the feed you selected, and you won&#39;t be able to see them. Fortunately, nothing of interest occurs on the monitors when you choose a different action, such as talking to a character. Your progress is automatically saved, so you need to click the restart button whenever you want to start a new game, even if you closed the browser.<br><br>This game is available in <a href=\"http://www.nigeljayne.ca/njgames.html\">a parser format</a>, including its original zcode release under the title <i>A Fly on the Wall,</i> in which commands are typed rather than clicked.<br><br>Consult the Help Desk if you want some advice on how best to sort out the events or reveal the combinations to the individual stories. The game is short, so you should try playing it at least once before deciding whether or not to seek help. The staff here at the mansion wishes you a frightfully good evening. {game options}</p>",
			},
			'Unmask': {
				'text': "<p><hr>{credits}{game options}</p>",
			},
			'game options': {
				'text': "<p><a class=\"squiffy-link link-section\" data-section=\"Ballroom 0\" role=\"link\" tabindex=\"0\">Start</a> the game and join the party.\n{if not seen Visit:<a class=\"squiffy-link link-passage\" data-passage=\"Visit\" role=\"link\" tabindex=\"0\">Visit</a> the Info Desk, which is recommended for anyone who hasn&#39;t played the game.}\n{if not seen Unmask:<a class=\"squiffy-link link-passage\" data-passage=\"Unmask\" role=\"link\" tabindex=\"0\">Unmask</a> the people behind the scenes.}</p>",
			},
		},
	},
	'game intro': {
		'text': "<p>A sasquatch bends down before a leprechaun, who pulls a gold coin from behind the beast&#39;s ear. Chuckling, the sasquatch takes the treasure and places it onto a table as the leprechaun speaks to him. With a shake of the head, he follows his fellow conspirator through a tangle of costumed guests and out of the ballroom.</p>\n<p>A pirate wobbles his way to the gold, a wooden prosthesis attached to one knee, a plush toucan on his shoulder bobbing as if it were keeping time with the music. the pirate retrieves the coin, scowls, and peels away the bright gold foil. Eating the chocolate and blinking his unpatched eye in the dim light, he watches the other members of the Fortean Society  &mdash; a collection of caricatures, characters, monsters, and myths &mdash; mingle in the Soul Dancers ballroom.</p>",
		'passages': {
		},
	},
	'Ballroom 0': {
		'clear': true,
		'text': "<p>{game intro}\n{label:80=<hr>}</p>\n<p>A green-skinned man with a stitch across his forehead lumbers toward you, two jumper cables swinging from his chest. \nFrankie bumps into Cleopatra standing next to the stage, sending her to the floor and knocking the headdress from her head. He extends his hand down to her, but she swats it away and glowers. Black Widow, a woman dressed in black with an hourglass symbol on her back, picks up the headdress. Cleopatra snatches it back, stands, and returns to her spot beside the stage.</p>\n<p>Frankie stops in front of you. <i>Are those batteries attached to opposite sides of his neck? </i> you wonder. As Frankie speaks, his chin taps at the antenna protruding up from your forehead, and his breath suffocates your nose with alcohol. &quot;Our whorey sister over there has sent me.&quot; He gestures erratically behind him. A nun with a hunched back bends over the punchbowl by the stage. You think he must have said &quot;holy&quot; sister, not &quot;whorey.&quot;</p>\n<p>The nun straightens and turns around to face you. She rests her cup of punch on her pregnant belly.</p>\n<p>&quot;She told me it&#39;s your turn in the Fly Eye, Goldie,&quot; Frankie says.</p>\n<p><i>Goldie? </i>you wonder. <i>My name isn&#39;t Goldie.</i> Then you realize that, despite the alcoholic soup bathing his brain, Frankie connected your costume with the eponymous horror film, Goldie being a contraction of its star&#39;s surname, Goldblum.</p>\n<p>Frankie retreats toward Cleopatra. Arms crossed, she stands firm. Frankie stumbles away, passing Black Widow who, head cocked, watches Cleopatra.</p>\n<p>From the stage, the Pied Piper sways on his         chair, playing enchantments from his laptop and tapping his fingers next to a bag surrounded by candy wrappers. Most of the guests have congregated into small groups in which someone &mdash; or something &mdash; tries to talk over the music. Holographic dancers materialize momentarily to dip, spin, and swing their partners next to the corporeal partygoers before disappearing to continue their dance in the ethereal. </p>\n<p>{label:1=The <a class=\"squiffy-link link-passage\" data-passage=\"nun\" role=\"link\" tabindex=\"0\">nun</a> smiles at you and raises her cup in salute. <i>Does she want to <a class=\"squiffy-link link-passage\" data-passage=\"nun discussion\" role=\"link\" tabindex=\"0\">talk</a>,</i>\nyou wonder. <i>Maybe she&#39;s saying goodbye and wants me to <a class=\"squiffy-link link-section\" data-section=\"take my turn\" role=\"link\" tabindex=\"0\">take my turn</a> in the Fly Eye.</i>}\n    {label:2=}\n    {label:3=}\n    {label:4=}</p>\n<p>{label:5=}\n{label:900=}</p>",
		'attributes': ["characterInteractions=0","noFirstGlance","BlackWidowCostume=Black Widow appears ready to seduce and slay. Dressed all in black, she wears a short skirt, pantyhose, and a curve-hugging shirt. Her hair is also black, and it falls just short of the red hourglass on her back. As if a mate needed a stronger warning, fangs poke from between blood-red lips. Those floppy leg-things dangling from her sides, though, are likely to provoke a giggle rather than a scream from potential prey, and that black purse doesn't match the killer look.","CleopatraCostume=Befitting of Cleopatra's mystique and majestic caste are her vibrant multi-colored gown laced in gold and bejeweled headdress fronted by a fanged snake ready to strike. But, like her namesake, this Cleopatra's features are decidedly bourgeois: long nose, thin lips, and a knobby chin. And, though the hairstyle seems right,  that blond color isn't.","FrankieCostume=Reanimation, apparently achieved by shocking one's heart through jumper cables attached to the chest, has turned Frankie's face green and tore his shirt and jeans, and the brain transplant has left a nasty black stitch across his forehead. Given the way Frankie has been stumbling around the party, maybe it's time to hook up those batteries attached to his neck because he definitely isn't fully charged.","ScientistCostume=isn't doing anything to dispel the nerd cliche. His white lab coat is clean, and it even has a pocket protector full of pens. His hair is white and spiked, as if electrically-charged from the sparking of overworked neurons."],
		'js': function() {
			soundEffect = document.getElementById("theme");
			soundEffect.volume=0.2;
			soundEffect.innerHTML = "<source src='additional_files/Techno-Celebration.mp3' type='audio/mp3' /> \ <source src='additional_files/Techno-Celebration.ogg' type='audio/ogg' />";
			
		},
		'passages': {
			'@1': {
				'text': "",
				'attributes': ["@replace 1=","@replace 80="],
			},
			'@2': {
				'text': "",
				'attributes': ["not noFirstGlance"],
			},
			'character situations': {
				'text': "<p>{if seenBlackWidow:Black Widow and Cleopatra}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Black Widow and Cleopatra\" role=\"link\" tabindex=\"0\">Black Widow and Cleopatra</a>} watch each other, the latter standing guard over a shadowed area next to the stage and uncomfortably close to a pair of stacked speakers. Cutlass in one wobbly hand, {if seenFrankie:Frankie}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Frankie\" role=\"link\" tabindex=\"0\">Frankie</a>} spars with an invisible foe as the pirate tries to get his weapon back. The {if seenNun:nun}{else:<a class=\"squiffy-link link-passage\" data-passage=\"nun\" role=\"link\" tabindex=\"0\">nun</a>} {if seen nun discussion:preaches to the {if seenScientist:scientist}{else:<a class=\"squiffy-link link-passage\" data-passage=\"scientist\" role=\"link\" tabindex=\"0\">scientist</a>} whose glances at the double doors suggest he is planning an escape. You suppose it may be time to <a class=\"squiffy-link link-section\" data-section=\"take my turn\" role=\"link\" tabindex=\"0\">make your exit</a> as well, and take your shift in the Fly Eye}{else:smiles at you from the refreshment table along one wall. She may want you to <a class=\"squiffy-link link-passage\" data-passage=\"nun discussion\" role=\"link\" tabindex=\"0\">have a few words with her</a> before you <a class=\"squiffy-link link-section\" data-section=\"take my turn\" role=\"link\" tabindex=\"0\">start your shift</a> in the Fly Eye}.</p>",
			},
			'Black Widow and Cleopatra': {
				'text': "",
				'attributes': ["seenBlackWidow","seenCleopatra","@replace 81=","@replace 2=","@replace 4=<p>{character situations}</p>","@replace 5=<p><hr>{BlackWidowCostume}<br><br>With slight smiles, Black Widow occasionally acknowledges other costumed guests before returning her attention to Cleopatra in front of a darkened section beside the stage. {CleopatraCostume} With arms crossed and a scowl, she stares down anyone who passes her by.</p>"],
			},
			'Frankie': {
				'text': "",
				'attributes': ["seenFrankie","@replace 81=","@replace 2=","@replace 4=<p>{character situations}</p>","@replace 5=<p><hr>Parry. Thrust. Swipe. Frankie uses the pirate&#39;s cutlass to battle enemies that exist only in his mind. He also takes a swipe at nearby soul dancers when they appear, his blade slicing through their bodies to no effect. {FrankieCostume}<br><br>The pirate tries to take the cutlass from Frankie, but an eyepatch over the buccaneer&#39;s left eye and the wooden leg protruding from the knee of his left pantleg impede his efforts. The toucan on his shoulder swings stiffly with every wobbly step and gyration. It&#39;s difficult to tell if the prosthesis is real; he could be bending his leg and hiding it in his baggy pants.</p>"],
			},
			'nun intro': {
				'text': "<p>{if seen nun discussion:The nun has the scientist cornered away from the crowd. She preaches to him with her head bobbing and her arms flailing, like a woman possessed. She}{else:The attractive nun sips from her plastic cup of punch. She occasionally rubs her stomach, or maybe she is just making sure it stays in position. She stands head-and-shoulders taller than you, and she\n} wears the customary apostolnik and habit, which sits atop red hair. A shawl covers her shoulders and hump, which has slid down her spine to relocate more toward the center of her back. The veracity of the scoliosis is questionable, but the pregnancy may very well be real{if seen nun discussion:: The mass in her belly shifts heavily with her every movement}{else:. She is, indeed, drinking non-alcoholic punch}.</p>",
			},
			'nun': {
				'text': "",
				'attributes': ["seenNun","@replace 81=","@replace 2=","@replace 4=<p>{if noFirstGlance:<hr>{nun intro}}{else:{character situations}}</p>","@replace 5=<p>{if noFirstGlance:{character situations}}{else:<hr>{nun intro}}</p>"],
			},
			'scientist': {
				'text': "",
				'attributes': ["seenScientist","@replace 81=","@replace 2=","@replace 4=<p>{character situations}</p>","@replace 5=<p><hr>He looks like he would like a distraction from the nun, but you dare not try to get in a word when she is preaching the Word. He {ScientistCostume}</p>"],
			},
			'nun discussion': {
				'text': "",
				'attributes': ["@replace 2=<p>{if noFirstGlance:}{else:{character situations}}</p>","@replace 3=<p>{label:81=<hr>}The nun sips from her punch as you approach.<br><br>&quot;So, Congratulations?&quot; you say, gesturing at the nun&#39;s large belly.<br><br>The nun rests her drink on her belly. &quot;It is a good disguise, isn&#39;t it? One could say even divine.&quot; She winks.<br><br>&quot;What do you think about the turnout?&quot;<br><br>&quot;It looks like it is a success. And everyone sure has gotten in the spirit.&quot;<br><br>&quot;Did you see anything good on your watch?&quot;<br><br>&quot;I did hear some odd sounds out in the hall, but that&#39;s about it. But there&#39;s always something going on with this bunch.&quot; She gestures her cup at the guests. &quot;I&#39;d keep an eye on them, if I were you.&quot; She laughs.<br><br>&quot;Do you know many of the . . . characters here?&quot;<br><br>She gestures toward Black Widow. &#39;Well, she is my sister, and I can tell you that costume is very appropriate.&quot;<br><br>&quot;Cleopatra needs to mellow out, huh?&quot;<br><br>The nun scowls. &quot;She said I was uglier than that guy in Notre Dame. Can you believe it? I&#39;m no Vanessa Redgrave, I know, but  . . . &quot; She frowns at Cleopatra. &quot;What a &mdash; &quot; Her voice lowers, and you only hear &quot;itch.&quot; It could have been &quot;witch.&quot;<br><br>The nun tosses her cup into a trash can. &quot;Well, I guess it&#39;s time for round two.&quot; Stern faced, she strides toward a white-haired man in a lab coat. The scientist&#39;s shoulders sag, and he frowns.</p>","@replace 4=","@replace 5=<p>{if noFirstGlance:{character situations}}{else:}</p>"],
			},
		},
	},
	'take my turn': {
		'text': "<p>{if not seen nun discussion:The nun waves to you as you begin toward the hall outside. She strides off, targeting a man with spiky white hair and wearing a lab coat. The scientist takes a few steps back and stares at the floor.</p>\n<p>}\nYou head out the double doors, descend the stairs, and <a class=\"squiffy-link link-section\" data-section=\"Fly Eye\" role=\"link\" tabindex=\"0\">enter the small room</a> nicknamed, for tonight, the Fly Eye.</p>",
		'attributes': ["@replace 5=","@replace 80=","@replace 81=","@replace 900=<p><br><br><hr></p>"],
		'passages': {
		},
	},
	'Fly Eye': {
		'text': "<hr>\n\n<p>The wire frames of your wings bend as you gently lean back into the only chair in the room. The two floppy legs attached to your belt rise as you rest your hands on a table in front of a bank of screens on the wall. The five screens receive signals from cameras in five different rooms in the mansion. Below them, a laptop plays the song &quot;{if songNumber=1:Superstitious}{if songNumber=2:Devil Woman}{if songNumber=3:Frankenstein}{if songNumber=4:Monster Mash}{label:6=&quot; and shows an open <a class=\"squiffy-link link-passage\" data-passage=\"web browser\" role=\"link\" tabindex=\"0\">web browser</a>.}</p>\n<p>Normally, the monitors would be used for security, but tonight the Society is trying to record footage of the supernatural that will raise the eyebrows of even the most stone-faced skeptics. While you&#39;re hoping to witness such phenomena, following the exploits of any one of the costumed characters would probably be more interesting. </p>\n<p><a class=\"squiffy-link link-passage\" data-passage=\"You\" role=\"link\" tabindex=\"0\">You</a> get comfortable and <a class=\"squiffy-link link-section\" data-section=\"Fly Eye 0\" role=\"link\" tabindex=\"0\">watch the screens</a>.\n{label:901=<br>&mdash;&mdash;<br><em>Set Goldie&#39;s thoughts to {rotate thoughts:hidden:explicit}. Hiding Goldie&#39;s thoughts conceals Goldie&#39;s comments about the events you watch. Explicit shows these comments whenever Goldie witnesses an event that is part of one of the storylines.</em><br>&mdash;&mdash;}</p>\n<p>{label:8=}</p>",
		'attributes': ["thoughts=hidden"],
		'js': function() {
			$("#soundWrapper").html("<audio id='sound effects' autoplay>");
			squiffy.set("songNumber", randomizedNumber(1, 4));
			
		},
		'passages': {
			'web browser': {
				'text': "",
				'attributes': ["seenBrowser","@replace 6=<p>.&quot;</p>","@replace 8=<p><hr>{Browser description}</p>"],
				'js': function() {
					
				},
			},
			'You': {
				'text': "",
				'attributes': ["seenPlayer","@replace 8=<p><hr>{Goldie description}</p>"],
			},
		},
	},
	'Fly Eye 0': {
		'text': "<p>{room monitors}\n{label:10=}</p>",
		'attributes': ["butcherInApron","notExplainedChildren","leprechaunCostume=He wears black pants and buckled shoes, and the pockets of his green jacket are bulging. A giant shamrock dangles from his green hat like Halloween's version of a mistletoe. You can't imagine the trick inspiring many kisses, though, not with that full red beard, ruddy complexion, and mischievous grin.","RasputinCostume=Greg is well-disguised in uncharacteristic long black hair and bushy beard. His shirt is plastered with silvery arcane symbols, and his cape falls in thick folds down to his knees. Most of his fingers sport rings.","sasquatchCostume=If this sasquatch were photographed in a forest, the image could fool more people than did the Patterson footage. Casts of his feet, though, would be unconvincing; they wouldn't justify the moniker Bigfoot. At least he's unashamedly barefoot.","PlayroomSequence1Dialogue1=I'd be more scared if you showed up in my house","PlayroomSequence1Dialogue2=Guess you should have taken it easy on the cabbage and beans at dinner","PlayroomSequence1Dialogue3=No way, man. Didn't you smell that when we came in? It's all over this room","PlayroomSequence1Dialogue4=Someone's coming. Go out there and bring our friend in","@replace 901=","@replace 8=<hr>"],
		'js': function() {
			window.realizations = [];
			
				window.curseStoryline = ["Ballroom","Communications","Laboratory","Ballroom","Communications","Ballroom","Ballroom"];
				window.frightStoryline = ["Playroom","Playroom","Alien Zoo","Communications","Communications","Communications","Ballroom"];
				window.greedStoryline = ["Laboratory","Playroom","Communications","Communications","Communications","Ballroom","Ballroom"];
		},
		'passages': {
			'': {
				'text': "",
				'attributes': ["@replace 11=<p>fiddles with her headdress as {if seenBlackWidow:Black Widow}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Black Widow\" role=\"link\" tabindex=\"0\">Black Widow</a>} crouches beside the stage.</p>"],
			},
			'@1': {
				'text': "",
				'attributes': ["@replace 12 ="],
			},
			'Black Widow': {
				'text': "",
				'attributes': ["seenBlackWidow","@replace 10=<p><hr>{BlackWidowCostume}</p>"],
			},
			'Cleopatra': {
				'text': "",
				'attributes': ["seenCleopatra","@replace 10=<p><hr>{CleopatraCostume}</p>"],
			},
			'medicine man': {
				'text': "",
				'attributes': ["seenMedicineMan","@replace 10=<p><hr>{Medicine Man Costume}</p>"],
			},
			'Rasputin': {
				'text': "",
				'attributes': ["seenRasputin","@replace 10=<p><hr>{RasputinCostume}</p>"],
			},
			'sasquatch watches the leprechaun': {
				'text': "",
				'attributes': ["seenLeprechaun","seenSasquatch","@replace 10=<p><hr>{sasquatchCostume}<br><br>The leprechaun is so short that he may just fit the definition of &quot;little person.&quot; {leprechaunCostume}</p>"],
			},
		},
	},
	'Fly Eye 1': {
		'text': "<hr>\n\n<p>The door behind you creaks. You whirl around. The door is still, still in the position you left it. You think it is, anyway. You stand and swing the door, but it doesn&#39;t make a sound. Hearing &quot;{if songNumber=1    :Magic}{if songNumber=2:Spirit in the Sky}{if songNumber=3:Calling Occupants of Interplanetary Craft}{if songNumber=4:Superstitious Feeling}&quot; playing from the {if seenBrowser:laptop}{else:<a class=\"squiffy-link link-passage\" data-passage=\"web browser\" role=\"link\" tabindex=\"0\">laptop</a>}, {if seenPlayer:you}{else:<a class=\"squiffy-link link-passage\" data-passage=\"You\" role=\"link\" tabindex=\"0\">you</a>} resume your surveillance.</p>\n<p>{room monitors}\n{label:20=}</p>",
		'attributes': ["PlayroomSequence2Dialogue1=Wayne Harrison was having a birthday party for his twin daughters in this room. They had cake and drinks.\" His voice lowers. \"Most of the kids died. Right here. The twins, too. Poisoned. People suspected Wayne's chauffeur, but he was never arrested. He was going to be fired for gambling and . . . doing inappropriate things in front of the kids. His son didn't get sick. That's why . . .","PlayroomSequence2Dialogue2=This place is haunted","PlayroomSequence2Dialogue3=We can have some fun with this"],
		'js': function() {
			squiffy.set("songNumber",randomizedNumber(1,4));
			soundEffect = document.getElementById("sound effects");
			soundEffect.volume=0.5;
			soundEffect.innerHTML = "<source src='additional_files/door-creak.mp3' type='audio/mp3' /> \ <source src='additional_files/door-creak.ogg' type='audio/ogg' />";
			
		},
		'passages': {
			'': {
				'text': "",
				'attributes': ["@replace 23=<p>Wide-eyed and in the clutches of a smiling {if seenSasquatch:sasquatch}{else:<a class=\"squiffy-link link-passage\" data-passage=\"sasquatch\" role=\"link\" tabindex=\"0\">sasquatch</a>}, {if seenRasputin:Rasputin}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Rasputin\" role=\"link\" tabindex=\"0\">Rasputin</a>} looks around</p>"],
			},
			'@1': {
				'text': "",
				'attributes': ["@replace 21=<p>stares at the far corner of the room</p>","@replace 22 =<p>Vacant</p>"],
			},
			'Cleopatra': {
				'text': "",
				'attributes': ["seenCleopatra","@replace 20=<p><hr>With a bag over her shoulder, Cleopatra smiles at a corner of the room. {CleopatraCostume}</p>"],
			},
			'medicine man': {
				'text': "",
				'attributes': ["seenMedicineMan","@replace 20=<p><hr>{Medicine Man Costume}</p>"],
			},
			'Rasputin': {
				'text': "",
				'attributes': ["seenRasputin","@replace 20=<p><hr>{RasputinCostume}</p>"],
			},
			'sasquatch': {
				'text': "",
				'attributes': ["seenSasquatch","@replace 20=<p><hr>{sasquatchCostume}</p>"],
			},
			'web browser': {
				'text': "",
				'attributes': ["seenBrowser","@replace 20=<p><hr>{Browser description}</p>"],
			},
			'You': {
				'text': "",
				'attributes': ["seenPlayer","@replace 20=<p><hr>{Goldie description}</p>"],
			},
		},
	},
	'Fly Eye 2': {
		'text': "<hr>\n\n<p>Something thumps onto the hallway floor. On instinct, you pull your phone from your pocket and look out the door. The guestbook has fallen from the table in the lobby. {if seenPlayer:You}{else:<a class=\"squiffy-link link-passage\" data-passage=\"You\" role=\"link\" tabindex=\"0\">You</a>} wait but don&#39;t detect anyone, so you return to your chair and tuck away the phone as the {if seenBrowser:laptop}{else:<a class=\"squiffy-link link-passage\" data-passage=\"web browser\" role=\"link\" tabindex=\"0\">laptop</a>} plays &quot;I don&#39;t like spiders and snakes, and that ain&#39;t what it takes to love me.&quot; </p>\n<p>{room monitors}\n{label:30=}</p>",
		'attributes': ["devilCostume=You're guessing that this guy is supposed to be a devil because he has a red face and horns on his head. In fact, he is probably the Jersey Devil, given the cardboard wings and not-so-subtle jersey with an \"N\" and horned \"J.\" You don't think the real demon's pitchfork is just two upside-down hockey sticks that have been tied together, though.","RedHairedDollDescription=The red-haired doll dangles at the leprechaun's side, her face smothered under the leprechaun's arm, her legs dragging along the floor."],
		'js': function() {
			$("#soundWrapper").html("<audio id='sound effects' autoplay>");
			
				if (JSroomProxy == "SDBR")
					squiffy.set ("butcherInApron", false);
			soundEffect = document.getElementById("sound effects");
			soundEffect.volume=0.7;
			soundEffect.innerHTML = "<source src='additional_files/book-falling.mp3' type='audio/mp3' /> \ <source src='additional_files/book-falling.ogg' type='audio/ogg' />";
		},
		'passages': {
			'': {
				'text': "",
				'attributes': ["@replace 33=<p>The {if seenSasquatch:sasquatch}{else:<a class=\"squiffy-link link-passage\" data-passage=\"sasquatch\" role=\"link\" tabindex=\"0\">sasquatch</a>} is standing in the doorway staring into the hall while the {if seenLeprechaun:leprechaun}{else:<a class=\"squiffy-link link-passage\" data-passage=\"leprechaun\" role=\"link\" tabindex=\"0\">leprechaun</a>} studies the exhibits with the <a class=\"squiffy-link link-passage\" data-passage=\"redhaired doll\" role=\"link\" tabindex=\"0\">red-haired doll</a> trapped under one arm</p>"],
			},
			'@1': {
				'text': "",
				'attributes': ["@replace 31=<p>{if seenMedicineMan:medicine man}{else:<a class=\"squiffy-link link-passage\" data-passage=\"medicine man\" role=\"link\" tabindex=\"0\">medicine man</a>} makes faces at his distorted reflection in the mirror</p>","@replace 32=<p>continues to transfer weapons from her bag to the chest</p>"],
			},
			'Cleopatra': {
				'text': "",
				'attributes': ["seenCleopatra","@replace 30=<p><hr>{CleopatraCostume}</p>"],
			},
			'leprechaun and sasquatch': {
				'text': "",
				'attributes': ["seenLeprechaun","seenSasquatch","@replace 30=<p><hr>The leprechaun is so short that he may just fit the definition of &quot;little person,&quot; the doll deceptively large under his arm. {leprechaunCostume} {sasquatchCostume}</p>"],
			},
			'leprechaun': {
				'text': "",
				'attributes': ["seenLeprechaun","@replace 30=<p><hr>The leprechaun is so short that he may just fit the definition of &quot;little person,&quot; the doll deceptively large under his arm. {leprechaunCostume}</p>"],
			},
			'medicine man': {
				'text': "",
				'attributes': ["seenMedicineMan","seenTotems","@replace 30=<p><hr>{Medicine Man Costume}</p>"],
			},
			'redhaired doll': {
				'text': "",
				'attributes': ["seenRedHairedDoll","@replace 30=<p><hr>{RedHairedDollDescription}</p>"],
			},
			'sasquatch': {
				'text': "",
				'attributes': ["seenSasquatch","@replace 30=<p><hr>{sasquatchCostume}</p>"],
			},
			'web browser': {
				'text': "",
				'attributes': ["seenBrowser","@replace 30=<p><hr>{Browser description}</p>"],
				'js': function() {
					
				},
			},
			'You': {
				'text': "",
				'attributes': ["seenPlayer","@replace 30=<p><hr>{Goldie description}</p>"],
			},
		},
	},
	'Fly Eye 3': {
		'text': "<hr>\n\n<p>{room monitors}</p>\n<p>{label:46=The laptop plays &quot;{if songNumber=1:Strange Magic}{if songNumber=2:Werewolves of London}{if songNumber=3:Magic Man}{if songNumber=4:Abracadabra}.&quot; }{label:44=But you hear something else. Crunching. It isn&#39;t coming from the monitors or computer. It&#39;s behind you.<br><br>You spin around. A black cat with a white stripe down its back tears at a rat on the floor. The cat discards its prey, leaps onto your lap, and settles. It purrs loudly, inviting you to <a class=\"squiffy-link link-passage\" data-passage=\"pet\" role=\"link\" tabindex=\"0\">pet it</a>.} {label:45=}</p>\n<p>{label:40=}</p>",
		'js': function() {
			squiffy.set("songNumber",randomizedNumber(1,4));	
		},
		'passages': {
			'@1': {
				'text': "",
				'attributes': ["@replace 41=<p> scans the crowd from the stage</p>","@replace 42=<p>The {if seenTricksters:sasquatch and leprechaun}{else:<a class=\"squiffy-link link-passage\" data-passage=\"sasquatch and leprechaun\" role=\"link\" tabindex=\"0\">sasquatch and leprechaun</a>}{if seenRedHairedDoll:}{else:, <a class=\"squiffy-link link-passage\" data-passage=\"redhaired doll\" role=\"link\" tabindex=\"0\">red-haired doll</a> in tow,} stare at the chandelier as the {if seenTotems:medicine man}{else:<a class=\"squiffy-link link-passage\" data-passage=\"medicine man\" role=\"link\" tabindex=\"0\">medicine man</a>} jabs it with his staff</p>","@replace 43=<p>The {if seenMIB:man in black}{else:<a class=\"squiffy-link link-passage\" data-passage=\"man in black\" role=\"link\" tabindex=\"0\">man in black</a>} continues to stare down at the {if seenDevil:devil}{else:<a class=\"squiffy-link link-passage\" data-passage=\"devil\" role=\"link\" tabindex=\"0\">devil</a>}, who tries to look beyond the imposing figure and out the door</p>","@replace 44=<p>The laptop plays music as the cat purrs on your lap{if seenPet:.}</p>","@replace 45=<p>{if seenPet:}{else:, an <a class=\"squiffy-link link-passage\" data-passage=\"pet\" role=\"link\" tabindex=\"0\">odd stripe</a> down its back.}</p>","@replace 46="],
				'js': function() {
							
				},
			},
			'Cleopatra': {
				'text': "",
				'attributes': ["seenCleopatra","@replace 40=<p><hr>{CleopatraCostume}</p>"],
			},
			'devil': {
				'text': "",
				'attributes': ["seenDevil","@replace 40=<p><hr>{devilCostume}</p>"],
			},
			'man in black': {
				'text': "",
				'attributes': ["seenMIB","@replace 40=<p><hr>{MIBCostume}</p>"],
			},
			'medicine man': {
				'text': "",
				'attributes': ["seenMedicineMan","seenTotems","@replace 40=<p><hr>{Medicine Man Costume}</p>"],
			},
			'pet': {
				'text': "",
				'attributes': ["seenPet","@replace 40 =<p><hr>{petting the cat}</p>"],
			},
			'redhaired doll': {
				'text': "",
				'attributes': ["seenRedHairedDoll","@replace 40=<p><hr>{RedHairedDollDescription}</p>"],
			},
			'sasquatch and leprechaun': {
				'text': "",
				'attributes': ["seenLeprechaun","seenSasquatch","seenTricksters","@replace 40=<p><hr>{sasquatchCostume}<br><br>The leprechaun is so short that he may just fit the definition of &quot;little person,&quot; The doll deceptively large under his arm. {leprechaunCostume}</p>"],
			},
		},
	},
	'Fly Eye 4': {
		'text': "<hr>\n\n<p>A shadow is cast over you. You swivel around. </p>\n<p>{if seenFrankie:Frankie }{else:<a class=\"squiffy-link link-passage\" data-passage=\"Frankie\" role=\"link\" tabindex=\"0\">Frankie</a> }crowds the tiny room. He considers the cat on your lap then looks up to you. His grin is crooked. &quot;What&#39;s new, pussycat?&quot; he says. He laughs then starts humming the eponymous Tom Jones song before turning to the monitors.{if storyline=curse:&quot;Even the cameras are afraid of Cleopatra,&quot; he says. &quot;Stay away from her tonight.&quot;}{else:&quot;Nervous guy, that Rasputin,&quot; he says. &quot;Bit of a wimp, if you ask me. I&#39;m surprised he came out from hiding under his bed tonight.&quot;}</p>\n<p>{label:55=You wonder whether you could <a class=\"squiffy-link link-passage\" data-passage=\"talk to Frankie\" role=\"link\" tabindex=\"0\">get some useful information</a> from this drunken monster.} </p>\n<p>{room monitors}</p>\n<p>{label:53=The cat has turned to watch Frankie&#39;s bobbing head as the monster tries to follow the peripatetic society members on the monitors.{label:56={if seenPet:}{else: Perhaps the wary feline needs to be <a class=\"squiffy-link link-passage\" data-passage=\"soothed\" role=\"link\" tabindex=\"0\">soothed</a>.}}}</p>\n<p>{label:50=}</p>",
		'attributes': ["nunCostume=The nun wears the customary apostolnik and habit, which sits atop her red hair. A shawl covers her shoulders and hump, which seems to have spread out over her shoulders, giving her an NFL linebacker look.","not positionedFrankieDialogue"],
		'js': function() {
			if (realizations[3]=="Alien Zoo") 
			squiffy.set ("describedGray");
		},
		'passages': {
			'': {
				'text': "",
				'attributes': ["@replace 51=<p>talks to {if seenRasputin:Rasputin}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Rasputin\" role=\"link\" tabindex=\"0\">Rasputin</a>} as he warily scans the room</p>"],
			},
			'@1': {
				'text': "",
				'attributes': ["@replace 52=<p>Vacant</p>"],
			},
			'Black Widow': {
				'text': "",
				'attributes': ["seenBlackWidow","@replace 50=<p><hr>{BlackWidowCostume}</p>"],
			},
			'Frankie': {
				'text': "",
				'attributes': ["seenFrankie","@replace 50=<p><hr>{FrankieCostume}</p>"],
			},
			'Rasputin': {
				'text': "",
				'attributes': ["seenRasputin","@replace 50=<p><hr>{RasputinCostume}</p>"],
			},
			'soothed': {
				'text': "",
				'attributes': ["seenPet","@replace 56=","@replace 50 =<p><hr>{petting the cat}</p>"],
			},
			'talk to Frankie': {
				'text': "",
				'attributes': ["talkedToFrankie","@replace 50=","@replace 53=","@replace 55=<p>{Frankie conversation}</p>"],
			},
		},
	},
	'Fly Eye 5': {
		'text': "<hr>\n\n<p>Bobbing his head, {if seenFrankie:Frankie}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Frankie\" role=\"link\" tabindex=\"0\">Frankie</a>} watches the monitors, although you aren&#39;t sure that he&#39;s keeping time with &quot;{if songNumber=1:Spooky}{if songNumber=2:Clap for the Wolfman}{if songNumber=3:Black Magic Woman}&quot; playing on the {if seenBrowser:laptop}{else:<a class=\"squiffy-link link-passage\" data-passage=\"web browser\" role=\"link\" tabindex=\"0\">laptop</a>}. He {if storyline=curse:seems to be searching the ballroom for Cleopatra}{else:rubs his chin, {if followingFOrG:presumably searching for Rasputin}{if storyline=threads:trying to follow the action}}. {label:65={if talkedToFrankie:You could <a class=\"squiffy-link link-passage\" data-passage=\"question him further\" role=\"link\" tabindex=\"0\">question him further</a>, but you don&#39;t want to encourage any more interaction with this monster}{else:maybe by <a class=\"squiffy-link link-passage\" data-passage=\"talking to him\" role=\"link\" tabindex=\"0\">talking to him</a> you can get him to share his thoughts}.} </p>\n<p>{room monitors}\n{label:66={if talkedToFrankie:}{else:The {if seenPet:cat}{else:<a class=\"squiffy-link link-passage\" data-passage=\"pet\" role=\"link\" tabindex=\"0\">cat</a>} on your lap remains wary of Frankie&#39;s erratic             movements.}}</p>\n<p>{label:60=}</p>",
		'js': function() {
			squiffy.set("songNumber",randomizedNumber(1,3));
		},
		'passages': {
			'': {
				'text': "",
				'attributes': ["@replace 61=<p>chats with the {if seenNun:nun}{else:<a class=\"squiffy-link link-passage\" data-passage=\"nun\" role=\"link\" tabindex=\"0\">nun</a>} near the stage</p>"],
			},
			'@1': {
				'text': "",
				'attributes': ["@replace 62=<p>scans the shelves</p>"],
			},
			'Black Widow': {
				'text': "",
				'attributes': ["seenBlackWidow","@replace 60=<p><hr>{BlackWidowCostume}</p>"],
			},
			'Frankie': {
				'text': "",
				'attributes': ["seenFrankie","@replace 60=<p><hr>{FrankieCostume}</p>"],
			},
			'leprechaun': {
				'text': "",
				'attributes': ["seenLeprechaun","seenSasquatch","@replace 60=<p><hr>The leprechaun is so short that he may just fit the definition of &quot;little person.&quot; {leprechaunCostume} {sasquatchCostume}</p>"],
			},
			'man in black': {
				'text': "",
				'attributes': ["seenMIB","@replace 60=<p><hr>{MIBCostume}</p>"],
			},
			'nun': {
				'text': "",
				'attributes': ["seenNun","@replace 60=<p><hr>{NunCostume}</p>"],
			},
			'pet': {
				'text': "",
				'attributes': ["seenPet","@replace 60 =<p><hr>{petting the cat}</p>"],
			},
			'question him further': {
				'text': "",
				'attributes': ["@replace 65=<p>{second conversation}</p>"],
			},
			'talking to him': {
				'text': "",
				'attributes': ["talkedToFrankie","@replace 66=","@replace 65=<p>{Frankie conversation}</p>"],
			},
			'web browser': {
				'text': "",
				'attributes': ["seenBrowser","@replace 60=<p><hr>{Browser description}</p>"],
			},
			'second conversation': {
				'text': "<p>&quot;Have you been a member of the society for long?&quot; you say. </p>\n<p>Frankie turns to you. He narrows his bleary eyes. &quot;Who wants to know?&quot;</p>\n<p>&quot;It&#39;s just that I&#39;ve been to a lot of these things, and I haven&#39;t met you.&quot;</p>\n<p>Frankie&#39;s head wobbles. &quot;I just joined. Normally I like to keep a low profile, but every now and then you gotta surface and sniff &#39;em out.&quot; He sniffs at you and smiles.</p>\n<p>You frown. &quot;Do you know any of them?&quot;</p>\n<p>&quot;{if storyline=curse:}{else:Cleopatra is Fay Clarkson. She&#39;s usually such a nice girl, but tonight . . . I don&#39;t know what&#39;s gotten into her. }{if storyline=greed:}{else:I ran into the medicine man in the exhibit on Mesoamerica. Lots of good stuff in there. Even blowguns and darts and poisonous plants. They&#39;re auctioning all that stuff away. I bet it will make someone . . . &quot; he focuses on you and winks. &quot;A pretty penny. }{if storyline=fright:}{else:I certainly wouldn&#39;t trust that leprechaun. Running around with that doll. Is anyone really afraid of that? }So I would keep careful watch tonight, my friend. But then everyone has a target on them, don&#39;t they? Thanks to the Greada Treaty that Eisenhower signed with the grays. Human experiments for alien technology. I&#39;ve seen them all. The Nordics. Arcturians. Reptilians.&quot;</p>\n<p>&quot;How do you know I&#39;m not one of them?&quot; </p>\n<p>He smiles. They&#39;re cold blooded. You&#39;re hot. Black Widow is tempting, though, isn&#39;t she? Maybe it isn&#39;t too late to introduce myself. We have something in common.&quot; He nods. &quot;I bite, too.&quot;</p>",
			},
		},
	},
	'screentime 7 sound': {
		'text': "<hr>\n\n<p>A high-pitched scream pierces the Fly Eye. That was from somewhere upstairs. </p>\n<p>Static bursts from the playroom. Are children giggling?{if talkedToFrankie:}{else: The cat leaps from your lap and scrambles out the door.}</p>\n<p>&quot;Gotta check that out,&quot; Frankie says before lumbering away.</p>\n<p>{label:994=<a class=\"squiffy-link link-section\" data-section=\"Fly Eye 6\" role=\"link\" tabindex=\"0\">Search</a> the screens         for the cause of the commotion.}</p>\n<p>{label:900=}</p>",
		'js': function() {
			$("#soundWrapper").html("<audio id='sound effects' autoplay controls loop>");
			
			soundEffect = document.getElementById("sound effects");
			soundEffect.volume=0.1;
			soundEffect.innerHTML = "<source src='additional_files/evp.mp3' type='audio/mp3' /> \ <source src='additional_files/evp.ogg' type='audio/ogg' />";
			
		},
		'passages': {
		},
	},
	'Fly Eye 6': {
		'text': "<p>{room monitors}</p>\n<p>{label:70=}</p>",
		'attributes': ["@replace 994=","@replace 900=<hr>"],
		'passages': {
			'': {
				'text': "",
				'attributes': ["@replace 71=<p>Guests have gathered around {if seenGreg:Greg}{else:the <a class=\"squiffy-link link-passage\" data-passage=\"man\" role=\"link\" tabindex=\"0\">man</a>}, who remains motionless on the floor</p>"],
			},
			'@1': {
				'text': "",
				'attributes': ["@replace 72=<p>continues to talk to someone through the doorway</p>"],
			},
			'man': {
				'text': "",
				'attributes': ["seenGreg","@replace 70=<p><hr>You study the fallen man. His fingers are covered in something. Rings. Like those that Greg wore tonight. This must be Greg, but his Rasputin-black hair is now white and spiky. You don&#39;t remember his cape having two red holes in it, but they could have been hidden within the fabric&#39;s folds.</p>"],
			},
			'medicine man': {
				'text': "",
				'attributes': ["seenTotems","@replace 70=<p><hr>{Medicine Man Costume}</p>"],
			},
		},
	},
	'Soul Dancers Ballroom': {
		'text': "<p>{if screentime&lt;3:{alert to other monitors}}</p>\n<h3 class=\"class2\">Soul Dancers Ballroom (second floor, east wing)</h3>\n\n<p>{if screentime=1:\nCleopatra slaps the headdress back onto her head and, noticing Black Widow, strides over to the crouching woman. Jerking her head up, Black Widow teeters on her feet and almost falls over. She rises and backs away from the other woman. </p>\n<p>Several of the guests watch as the spider and queen exchange words. Black widow shakes her head and holds up her empty hands. Cleopatra&#39;s shoulders lower, and her hands fall to her sides. Black Widow walks away, shaking her head. </p>\n<p>Cleopatra pulls a bulky bag from the shadow beside the stage. She looks inside the bag, hefts it over her shoulder, and leaves the room. {if thoughts=explicit:{label:902=<br><br><i>Cleopatra didn&#39;t want Black Widow snooping around in that bag,</i> you think.}}\n{label:903=<br><br>You recognize <a class=\"squiffy-link link-passage\" data-passage=\"listen\" role=\"link\" tabindex=\"0\">the voice of the leprechaun</a> playing from the playroom feed.}\n}\n{if stageInShot:\nThe monitor shows about a third of the room, including downstage, which is flanked by stacked speakers. Most of the guests have drifted toward the refreshment table lining the far wall{if ballroomNumber=1:. A couple of {if seenSpiritedDancers:spirited dancers}{else:<a class=\"squiffy-link link-passage\" data-passage=\"spirited dancers\" role=\"link\" tabindex=\"0\">spirited dancers</a>} materializes in front of the stage. The woman, dressed in a purple ball gown, twirls as her partner holds her hand over her head. They embrace then disappear}{if ballroomNumber=2:. A monocled, mustachioed man appears at the periphery of the corporeal guests and {if seenSpiritedDancers:ethereal dancers}{else:<a class=\"squiffy-link link-passage\" data-passage=\"spirited dancers\" role=\"link\" tabindex=\"0\">ethereal dancers</a>}  . Motionless, he watches, nods, and disappears}{if ballroomNumber=3:. A man in a white tuxedo waltzes a few steps with his elegant partner before they dip into oblivion, and another pair of {if seenSpiritedDancers:soul dancers}{else:<a class=\"squiffy-link link-passage\" data-passage=\"spirited dancers\" role=\"link\" tabindex=\"0\">soul dancers</a>} attempt a tango}{if ballroomNumber=4:. None of the souls, otherworldly or otherwise, are dancing}.}{if butcherIsVisible:<br><br>{if butcherInApron:Wearing a bloody apron and carrying a cleaver, a}{else:The} {if seenButcher:butcher}{else:<a class=\"squiffy-link link-passage\" data-passage=\"butcher\" role=\"link\" tabindex=\"0\">butcher</a>} {if screentime=2: {label:105=makes his way to the refreshment table. He glances at the selection of cold cuts and sausage on a tray and places some fruit and vegetables on a paper plate in his hand.}}{else: {butcherSnacking}}\n}\n{if screentime=2:<br><br>{label:904=A woman screams from monitor 5, which watches the playroom. <a class=\"squiffy-link link-passage\" data-passage=\"listen again\" role=\"link\" tabindex=\"0\">A trembling voice</a> follows, and you recognize it as the somewhat nasal tones of Rasputin.}}\n{if screentime=5:{zombie description}}\n{if screentime=4:Rasputin rushes past Cleopatra. He grabs a drink &mdash; from the alcoholic punch bowl &mdash; just as Cleopatra catches up to him. He gulps down his drink as Cleopatra talks. </p>\n<p>Black Widow emerges from a crowd and approaches Rasputin. Smiling, she places a hand on his shoulder. She speaks, silencing Cleopatra. Rasputin replies to Black Widow, who nods. Arms crossed, Cleopatra gets in a few words, but Rasputin only shrugs and finishes his drink. He hands the cup to Cleopatra then allows himself to be lead out of the room arm-in-arm with Black Widow. </p>\n<p>Cleopatra crushes the cup, tosses it onto the table. Staring at the doors, she taps her foot. She walks to the exit and disappears off-screen. {if thoughts=explicit:{label:902=<br><br><i>It looks like Cleopatra had her own plans for Rasputin,</i> you think. <i>She certainly didn&#39;t want him going off with Black Widow{if seen CleoInLab:. Was she setting up something for him in the lab first?}{else:.}}}\n}\n{if screentime=6:{Ballroom 6}}\n{if screentime=7:{if seenGreg:Guests gather around Greg, who lies prone with the cape flattened on his back}{else:Cleopatra steps back. The arms of the fallen man spasm, the fingers covered in rings. It&#39;s Greg in his Rasputin costume. Except for that spiked white hair. He lies prone with his cape flat on his back. Two red holes are punched into the black fabric}. </p>\n<p>Guests gather round your fallen friend. Several grab their cell phones. Some take pictures while others make calls. The nun bends down and touches the holes in Greg&#39;s cape. She rubs her fingers together and shakes her head. Black Widow helps Frankie turn Greg onto his back. His face is almost as white as his hair. The nun starts the chest compressions for cardiopulmonary resuscitation. Cleopatra watches with a concerned look. </p>\n<p>A tall woman wearing a white dress and tiara takes the hand of a suited man in a top hat. They swirl and step around the crowd. {if thoughts=explicit:{label:902=<br><br><i>{if seenRasputinScream:That scream sounded like Greg. }{else:Was that scream from Greg? }What happened . . . </i>}}{Frankie arrives}</p>\n<p>{label:900=}\n}{else:{if screentime=5:}{else:{return to all perspectives}}\n}\n{room line labels}</p>",
		'attributes': ["@replace 8="],
		'js': function() {
			squiffy.set("ballroomNumber", randomizedNumber(1, 4));
			if (JSscreentime == 1)
				squiffy.set("seenCleoWithBag");
			if (JSscreentime == 4)
				squiffy.set("seenRaspInBallroom");
			
			if (realizations[3] == "Ballroom" || realizations[4] == "Communications")
			squiffy.set ("seenRaspInteraction");
			if (JSscreentime == 2 || JSscreentime == 3 || JSscreentime == 5) {
			squiffy.set ("stageInShot");
			}
			else
			squiffy.set ("stageInShot", false);
			if (JSscreentime == 2 || JSscreentime == 3) {
			squiffy.set ("butcherIsVisible");
			snack = "some kind of fruit";
			if (squiffy.get ("ballroomNumber") == 2)
					snack = "a carrot";
							if (squiffy.get ("ballroomNumber") == 3)
					snack = "something green";
							if (squiffy.get ("ballroomNumber") == 4)
					snack = "his treats";
			butcherSnackSummary = "munches on " + snack + " as he watches the colorful guests.";
			squiffy.set ("butcherSnacking", butcherSnackSummary);
			}
			if (JSscreentime != 2 && JSscreentime != 3) 
			squiffy.set ("butcherIsVisible", false);
			if (JSscreentime == 1)
			squiffy.set ("storyline", "curse");
		},
		'passages': {
			'zombie description': {
				'text': "<p>{label:108=A zombie lurches toward a dish on the refreshment table. She reaches for a slice of meat. Suddenly, she stops. She swats her hand in the air over the dish. She turns away and ambles out of view. On the other side of the screen, t}he {if seenPirate:pirate}{else:<a class=\"squiffy-link link-passage\" data-passage=\"pirate\" role=\"link\" tabindex=\"0\">pirate</a>} dances and pounds his wooden leg into the floor as the {if seenNun:nun}{else:<a class=\"squiffy-link link-passage\" data-passage=\"nun\" role=\"link\" tabindex=\"0\">nun</a>}, smiling, watches the bobbing toucan on his shoulder. {label:103=Maybe he&#39;s explaining how he got his cutlass back.<br><br>A white orb appears at the bottom of the screen. It rises slowly then disappears. {trackedHauntings}.} {return to all perspectives} </p>",
			},
			'Ballroom 6': {
				'text': "<p>Cleopatra appears beside the doors and rushes over to Black Widow, interrupting her conversation with the nun. Cleopatra speaks to Black Widow, who responds with a few words and a pat on the woman&#39;s arm. Cleopatra heads to the exit, and {if seenBlackWidow:Black Widow}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Black Widow\" role=\"link\" tabindex=\"0\">Black Widow</a>} shrugs at the {if seenNun:nun}{else:<a class=\"squiffy-link link-passage\" data-passage=\"nun\" role=\"link\" tabindex=\"0\">nun</a>}. </p>\n<p>{if seenCleopatra:Cleopatra}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Cleopatra\" role=\"link\" tabindex=\"0\">Cleopatra</a>} stops at the doors and stares out into the hall.\n{if thoughts=explicit:{label:902=<br><br><i>Cleopatra must be looking for someone,</i> you think. {if seenRaspInteraction:<i>{if seenRaspInBallroom:Rasputin? What&#39;s holding him up?}{if seenRaspAndBWInComm:{if seenRaspInBallroom: He}{else: Rasputin} and Black Widow must have         split up after leaving Communications. {if followingFOrG:Greg is so shaken up, I wouldn&#39;t want to leave him alone}{else:Greg was really distressed; I hope the widow left him with someone}.}</i>}}}</p>",
			},
			'@1': {
				'text': "",
				'attributes': ["@replace 103=","@replace 105=<p>{butcherSnacking}</p>","@replace 106=","@replace 108=<p>T</p>","@replace 902=","@replace 904=<p>{if seen listen again:{Playroom dialogue 2}}{else:The <a class=\"squiffy-link link-passage\" data-passage=\"listen again\" role=\"link\" tabindex=\"0\">voices</a> continue to sound from the playroom.}</p>"],
			},
			'Black Widow': {
				'text': "",
				'attributes': ["seenBlackWidow","@replace 200=<p><hr>{BlackWidowCostume}</p>"],
			},
			'butcher': {
				'text': "",
				'attributes': ["seenButcher","@replace 200=<p><hr>A link of intestines falls over the butcher&#39;s shoulder, and some kind of animal head hangs around his thick neck. The intestines resemble sausage, but they are too shiny. Plastic, perhaps. You can make out long ears on the head around the butcher&#39;s neck, and you think there&#39;s a pair of buck teeth in its mouth. A rabbit. It has cartoonish, oversized features, and it has clearly been ripped from a stuffed toy. He carries a stained cleaver covered with chunks of . . . chunks. The cleaver points out from his apron, which is covered in bright red blood.</p>"],
			},
			'Cleopatra': {
				'text': "",
				'attributes': ["seenCleopatra","@replace 200=<p><hr>{CleopatraCostume}</p>"],
			},
			'listen': {
				'text': "",
				'attributes': ["@replace 200=","@replace 903=<p>{Playroom dialogue 1}</p>"],
			},
			'listen again': {
				'text': "",
				'attributes': ["@replace 200=","@replace 904=<p>{Playroom dialogue 2}</p>"],
			},
			'nun': {
				'text': "",
				'attributes': ["seenNun","@replace 200=<p><hr>{nunCostume}</p>"],
			},
			'pirate': {
				'text': "",
				'attributes': ["seenPirate","@replace 200=<p><hr>Except for the stuffed toucan on his shoulder, this pirate could have stepped out of a Pixar cartoon, with that eye patch and wooden leg. The gleam off that cutlass, though, makes you wonder if the blade isn&#39;t for children.</p>"],
			},
			'spirited dancers': {
				'text': "",
				'attributes': ["seenSpiritedDancers","@replace 200=<p><hr>The Forteans share the dance floor with flickering holograms of dancers dressed in 19th century garments. This is odd because the house wasn&#39;t built until early in the next century. Maybe the design of the clothes was meant to clash with that of the spirits who supposedly really do haunt this mansion, you think.</p>"],
			},
		},
	},
	'Communications': {
		'text': "<p>{if screentime&lt;3:{alert to other monitors}}</p>\n<h3 class=\"class2\">Communications (second floor, west wing)</h3>\n\n<p>{if noteDimmedLights:Something has changed in this room . . . The light is dimmer, shrouding the room&#39;s corners in shadow.</p>\n<p>}\n{if stuffedCommunications:\n{if screentime=3:The medicine man touches the bumpy surface of the <a class=\"squiffy-link link-passage\" data-passage=\"mirror\" role=\"link\" tabindex=\"0\">mirror</a>, tracing the contours of a visage that distorts his own. The chandelier above him begins to shake{label:201=. {trackedHauntings}}.</p>\n<p>Staring at the fixture, the medicine man backpedals toward the door and collides with Rasputin, who has just backed into the room. The staff falls from the medicine man&#39;s hand, spilling the frog totem onto the floor.</p>\n<p>Rasputin spins around, sees the totem, and picks it up. Frowning, he rolls the frog over in his hands. The medicine man snatches it away and hangs it back onto his staff. Rasputin looks at the collection of totems then at the medicine man, who scowls back at him. He retreats out the door.</p>\n<p>The {if seenTotems:medicine man}{else:<a class=\"squiffy-link link-passage\" data-passage=\"medicine man\" role=\"link\" tabindex=\"0\">medicine man</a>} studies the <a class=\"squiffy-link link-passage\" data-passage=\"chandelier\" role=\"link\" tabindex=\"0\">chandelier</a>, which remains motionless.}\n{if screentime=4:The sasquatch and leprechaun turn to each other, and the sasquatch shrugs. The leprechaun speaks to the medicine man, who glances at the red-haired doll in the leprechaun&#39;s hand as he listens. He points at the door. Smiling, the leprechaun leads the sasquatch out of the room.</p>\n<p>The medicine man touches his staff to the bottom of the {if seenChandelier:chandelier}{else:<a class=\"squiffy-link link-passage\" data-passage=\"chandelier\" role=\"link\" tabindex=\"0\">chandelier</a>} as the white-haired scientist arrives. The scientist smiles and speaks. The medicine man nods then lowers his staff. He speaks. The scientist nods then begins to pull a liquor bottle with a ribbon around its neck out of his pocket, but the medicine man stops him. The scientist glances at the camera, and the medicine man gestures at the door with his staff. They leave together.}\n{if screentime=5:\nBlack Widow notices a curtained off area at the far end of the room. Passing between tables, she approaches it and pulls back the curtains, revealing a set of audio equipment. Fiddling with the stereo, she says something and gestures for Rasputin to join her.</p>\n<p>Rasputin takes a deep breath, rubs his palms on his pants, and makes his way over to Black Widow. She points at a knot of cables snaking from the <a class=\"squiffy-link link-passage\" data-passage=\"stereo\" role=\"link\" tabindex=\"0\">equipment</a> to a dark corner behind some speakers. Rasputin hesitates then reaches into the darkness. Suddenly, he flings his arm back and stumbles backward. With wide eyes, he rubs his hand and speaks.</p>\n<p>Black Widow frowns and inspects Rasputin&#39;s hand. She shrugs. She attempts to look behind the speaker, but Rasputin stops her with a hand to her shoulder. Smiling, she shakes her head then bends over into the darkness.</p>\n<p>The leprechaun, without the doll, steps through the doorway, the sasquatch behind him. They move aside as Black Widow, head shaking, leads the trembling Rasputin toward the door. As she exits, the scientist appears in the doorway with a bottle. Rasputin snatches it from his hand and takes a swig. He pauses and does it again. Nodding, he returns the bottle, and the two men leave.</p>\n<p>The sasquatch follows the grinning leprechaun to the doorway where {if seenTricksters:the two}{else:<a class=\"squiffy-link link-passage\" data-passage=\"they\" role=\"link\" tabindex=\"0\">the two</a>} stop and look into the hall.\n} \n}\n{else:\n{if screentime=2:\nThe bag over Cleopatra&#39;s shoulder swings as she finds a switch and dims the lighting. Before leaving, she smiles at the far corner, which is now hidden by shadow. {if thoughts=explicit:{label:902=<br><br><i>Is Cleopatra trying to hide something by dimming the lights?</i> you wonder.}}    </p>\n<p>}\n        {if screentime=6:\nThe grin falters on the leprechaun&#39;s face as he peers out the door. He scowls and leaves the room, the sasquatch trailing. {if thoughts=explicit:{label:902=<br><br><i>{if storyline=fright:\nSo Rasputin didn&#39;t see that doll,</i> you think. <i>but the leprechaun isn&#39;t giving up.</i>\n}{else:\n<i>{if seenRaspAndBWInComm:The leprechaun&#39;s plans didn&#39;t work out}{else:Whatever their plan was, it doesn&#39;t seem to have worked},</i> you think. <i>I doubt they will give up, though.</i>}}\n}</p>\n<p>The {if seenChandelier:chandelier}{else:<a class=\"squiffy-link link-passage\" data-passage=\"chandelier\" role=\"link\" tabindex=\"0\">chandelier</a>} shakes. {if notSeenShakyChandelier:        {label:202={trackedHauntings}.}}\n}\n{if notExplainedChildren:{if seeingShakingChandelier:<br><br>}{label:25=You recall what you learned about this room from a speech Cleopatra gave at dinner earlier this evening. }Wayne Harrison found his wife, Gabrielle, slumped in a chair with an open book at her feet, an aneurysm ending her life after twenty-something years, when her twin daughters were just a year old. Enter Esmeralda Runesine. The medium facilitated spiritual communication between the distraught and the deceased for almost a year, ending when Wayne remarried, apparently with Gabrielle&#39;s blessing. Esmeralda returned, however, a few years later when Wayne&#39;s twins died. Sessions between her, Wayne, and the children were unconvincing, and Wayne moved out of the mansion months later.</p>\n<p>}{if screentime=7:The exhibit seems to be undisturbed, and nothing, living or dead, lurks in the shadows.}{else:{if notExplainedChildren:Since Wayne&#39;s departure, t}{else:{if seeingShakingChandelier:<br><br>}T}he museum&#39;s curator has filled the room with not only the {if seenTools:tools}{else:<a class=\"squiffy-link link-passage\" data-passage=\"tools\" role=\"link\" tabindex=\"0\">tools</a>} of transcommunication, such as ouija boards and sound trumpets, but also the {if seenTricks:tricks}{else:<a class=\"squiffy-link link-passage\" data-passage=\"trickery\" role=\"link\" tabindex=\"0\">tricks</a>} of the trade, such as ectoplasmic cheesecloth and wobbly furniture.\n}<br>}\n{if screentime=2:{label:904=<br><br>A woman screams from monitor 5, which              watches the playroom. <a class=\"squiffy-link link-passage\" data-passage=\"listen again\" role=\"link\" tabindex=\"0\">A trembling voice</a> follows, and you recognize it as the somewhat nasal tones of Rasputin.}}\n{if stuffedCommunications:{if thoughts=explicit:{label:902=<br><br>{if screentime=3:<i>{if seenRasputinScream:It&#39;s a good thing Greg didn&#39;t see that, </i>you think. <i>He&#39;s already freaked out. He}{else:Rasputin} was interested in those totems{if seenRasputinScream:, though. }{else:, </i> you think. <i>}Especially the frog. </i>{if seenFrog:<i>Maybe he thinks it is somehow different than the one he picked up in the lab earlier.</i>}} \n{if screentime=4:<br><br>You think. <i>{if seenRasputinScream:The medicine man probably told the leprechaun where to find Rasputin}{else:The leprechaun is searching for someone, and the medicine man wants to talk to the scientist privately about the liquor bottle in his pocket.} {if seenFrog:{if seenNewFrog:And maybe Rasputin and the totems as well.}}</i>}\n{if screentime=5:<br><br><i>Something stung Rasputin when he reached behind the speakers, </i> You think. <i> A shock? Bite? Black Widow doesn&#39;t seem concerned, but Greg sure is, drawing fortitude from the scientist&#39;s liquor bottle. {if seenBottle:Is it just a coincidence the scientist showed up to offer it to him after talking with the medicine man? }</i>{if seenCleoInLab:<i>Cleopatra could have dropped something{if seenCleoWithBag:from that suspicious bag} through a hole in the Laboratory floor, and it would have landed behind the speaker.</i> }{if seenRaspInBallroom:<i>Is this why Cleopatra was so upset in the ballroom? She has to get Greg here, but Black Widow whisks him away instead, and he ends up here anyway. </i> }\n{if seenRasputinScream:<i>{if seenLepInZoo:The leprechaun has finally found Rasputin, too. As if the guy hasn&#39;t been through enough already.}{else:Has the leprechaun been chasing Rasputin with that doll all this time? Can Greg handle another shock?}</i> }{else:You consider the pair in the doorway: Are they scheming against Rasputin or Black Widow?}\n}\n}}}</p>\n<p>{if screentime=1:{label:903=You recognize <a class=\"squiffy-link link-passage\" data-passage=\"listen\" role=\"link\" tabindex=\"0\">the voice of the leprechaun</a> playing from the playroom feed.}}\n{if screentime=7:{Frankie arrives}}\n{else:{return to all perspectives}}</p>\n<p>{room line labels}</p>",
		'attributes': ["@replace 8="],
		'js': function() {
			if (JSscreentime > 2 && JSscreentime < 6) { 
					squiffy.set ("stuffedCommunications");
			}
				else
					squiffy.set ("stuffedCommunications", false);
			if (JSscreentime == 3 || (realizations[2] != "Communications" && JSscreentime == 6))
			squiffy.set ("seeingShakingChandelier");
			if (JSscreentime == 3) 
			squiffy.set("seenNewFrog");
			if (JSscreentime == 4 || JSscreentime == 5) 
			squiffy.set("seenBottle");
			if (JSscreentime == 5)
			squiffy.set("seenRaspAndBWInComm");
			if (JSscreentime == 6 && realizations[2] != "Communications") 
			squiffy.set ("notSeenShakyChandelier");
		},
		'passages': {
			'@1': {
				'text': "",
				'attributes': ["@replace 25=","@replace 201=","@replace 202=","@replace 902=","@replace 904=<p>{if seen listen again:{Playroom dialogue 2}}{else:<br>The <a class=\"squiffy-link link-passage\" data-passage=\"listen again\" role=\"link\" tabindex=\"0\">voices</a> continue to sound from the playroom.}</p>"],
			},
			'chandelier': {
				'text': "",
				'attributes': ["seenChandelier","@replace 200=<p><hr>The chandelier has lots of dangly glass bits for maximum jangling and clattering.</p>"],
			},
			'listen': {
				'text': "",
				'attributes': ["@replace 200=","@replace 903=<p>{Playroom dialogue 1}</p>"],
			},
			'listen again': {
				'text': "",
				'attributes': ["@replace 200=","@replace 904=<p>{Playroom dialogue 2}</p>"],
			},
			'medicine man': {
				'text': "",
				'attributes': ["seenMedicineMan","seenTotems","@replace 200=<p><hr>{Medicine Man Costume}</p>"],
			},
			'mirror': {
				'text': "",
				'attributes': ["seenMirror","@replace 200=<p><hr>Bumps distort the mirror&#39;s glass; it looks as if a head had tried to emerge from the Other Side and left an impression. The surface bends, compresses, and exaggerates the facial features of anyone gazing upon it.</p>"],
			},
			'stereo': {
				'text': "",
				'attributes': ["@replace 200=<p><hr>You scrutinize the speakers where Rasputin had attempted to revive the stereo. Nothing but a tangle of cables and shadow. You guess the setup provides ambient sounds, whether they be stormy weather, discordant music, or the whisperings of the deceased.</p>"],
			},
			'they': {
				'text': "",
				'attributes': ["seenLeprechaun","seenSasquatch","@replace 200=<p><hr>The leprechaun is so short that he may just fit the definition of &quot;little person.&quot; {leprechaunCostume}<br><br>{sasquatchCostume}</p>"],
			},
			'tools': {
				'text': "",
				'attributes': ["seenTools","@replace 200=<p><hr>On exhibit are several versions of Hasbro&#39;s Ouija boards, which were intended only as a parlor game before they became associated with the psychography initiated by spirits. Whether or not a planchette and lettered board is used, messages for loved ones and the curious are recorded using some kind of writing implement and paper or slate, and the psychographic texts gathered here are protected in glass displays. Some spirits announce their presence audibly, and these are amplified with spirit trumpets like those on a shelf on the back wall. You guess the rocks on the shelf above are apports, gifts teleported from the spirit world.</p>"],
			},
			'trickery': {
				'text': "",
				'attributes': ["seenTricks","@replace 200=<p><hr>Hanging from the walls is cheesecloth which, to the undiscerning eye and in the gloom of a poorly-lighted room, can pass as the ectoplasm secreted from a medium, and which is often interpreted as a manifestation from the spirit world. One table has been upturned to show hidden compartments. The wardrobe, curtains, and most of the mirrors are unremarkable by themselves, but you know that with strategically-placed conspirators and candlelight, a medium can create a credible illusion for the credulous and disillusioned.</p>"],
			},
		},
	},
	'Laboratory': {
		'text': "<p>{if screentime&lt;3:{alert to other monitors}}</p>\n<h3 class=\"class2\">Laboratory (third floor, west wing)</h3>\n\n<p>{if screentime=1:Axe in hand, Rasputin continues to root around in the {if seenPropChest:chest}{else:<a class=\"squiffy-link link-passage\" data-passage=\"chest\" role=\"link\" tabindex=\"0\">chest</a>} at a dark corner of the room. The medicine man leans his totem-laden staff onto the wall between the two of them. He taps a couple of dark bulbs surrounding the vanity&#39;s mirror. When they don&#39;t light, he tries tightening them into their sockets, but that doesn&#39;t work either. He gives up and studies himself in the mirror&#39;s reflection. He reaches into a pouch on his belt and applies more black makeup to his face.</p>\n<p>Rasputin stands, turns, and kicks over the staff. Animal figurines spill onto the floor. He drops the axe and begins to gather them up.</p>\n<p>Watching Rasputin, the medicine man retrieves his staff. He scowls as Rasputin, on his knees and talking constantly, polishes each totem before looping them onto their former holder. A raven settles down the staff first, followed by an owl, frog, and wolf.</p>\n<p>Rasputin stands, and the {if seenMedicineMan:medicine man}{else:<a class=\"squiffy-link link-passage\" data-passage=\"medicine man\" role=\"link\" tabindex=\"0\">medicine man</a>} glares at him as Greg slinks out of the room. {label:902={if thoughts=explicit:<br><br><i>No harm done, </i>you think. <i>Or maybe you&#39;re just-plain ornery anyway?</i> You study the medicine man&#39;s staff and its animal figures aligned as wolf, frog, owl, and raven from the top to bottom. <i>They look like wood carvings. They don&#39;t appear fragile.</i>}}\n}\n{if screentime=3:Cleopatra bends over her bag in the dark corner of the room. Her back faces the camera, and she needs to contort her body as she places a skull and knife in the chest next to her. She bolts upright and spins around as {label:205=you glimpse someone enter} from the opposite side of the room.</p>\n<p>The scientist walks toward Cleopatra but stops when she speaks. She points at the vanity table, full-length mirror, and wardrobe. The scientist glances at them, but he seems more interested in the grills set into the floor. Cleopatra watches him as he inspects one then pulls it free. He drops to his knees and stares into the hole. He replaces the grill and steps over to the {if seen pole:pole}{else:<a class=\"squiffy-link link-passage\" data-passage=\"pole\" role=\"link\" tabindex=\"0\">pole</a>} sticking from the floor beside the door. He shakes it vigorously and looks back at Cleopatra, who smiles . She nods in response to something the scientist says and waves goodbye as he leaves.</p>\n<p>Cleopatra puts the ax into the chest before turning back to her bag. She moves her arms and shifts around. You wait. And wait. Finally, she begins to fill the chest again with an odd assortment of weapons and limbs. She knocks the chest&#39;s lid closed with her elbows and places the candles in her hands on the table next to the {if seenPropChest:chest}{else:<a class=\"squiffy-link link-passage\" data-passage=\"chest\" role=\"link\" tabindex=\"0\">chest</a>}. Empty-handed, she leaves the room. {if thoughts=explicit:{label:902=<br><br><i>Cleopatra seemed to want the scientist to stay away from her side of the room,</i> you think. <i>Did she put her bag there to hide it from the camera? She seemed to be doing something more than just returning props{if seenCleoWithBag:. Maybe Black Widow was right to be curious about the bag}.</i>}}\n}\n{if screentime=7:You wait, but nobody appears, and you don&#39;t see anything unexpected.}\n{if emptyLab:The lab is usually off-limits to guests, but the society has been granted access tonight because the staff has reported strange goings-on that were not of their own creation.\n{if screentime=2: \nAn axe lies next to a {if seenPropChest:chest}{else:<a class=\"squiffy-link link-passage\" data-passage=\"chest\" role=\"link\" tabindex=\"0\">chest</a>} sitting in the far dark corner, and g}{else:A {if seenPropChest:chest}{else:<a class=\"squiffy-link link-passage\" data-passage=\"chest\" role=\"link\" tabindex=\"0\">chest</a>}, probably full of props, sits in the far dark corner of the room}{if screentime&gt;2:. You remember the macabre candles on the small table beside it as those lighting the society&#39;s dinner party held at a restaurant a couple of hours ago. G}rills are embedded in the floor at various spots, a tilted {if seenPole:pole}{else:<a class=\"squiffy-link link-passage\" data-passage=\"pole\" role=\"link\" tabindex=\"0\">pole</a>} protruding from an uncovered hole beside the door. \n{if screentime=6:</p>\n<p>The devil enters and circles the room, passing the mirror, vanity table, wardrobe, and chest. He shakes the {if seenPole:pole}{else:<a class=\"squiffy-link link-passage\" data-passage=\"pole\" role=\"link\" tabindex=\"0\">pole</a>} before leaving}{else:{if seenPlayer:You}{else:<a class=\"squiffy-link link-passage\" data-passage=\"You\" role=\"link\" tabindex=\"0\">You</a>} imagine a perfectly-ordinary someone grabbing an outfit from the wardrobe, sitting at the vanity table, applying makeup, and finally checking the results in the full-length mirror before heading back out as a werewolf, zombie, vampire, or some other ungodly creation}.\n}\n{if screentime=7:{Frankie arrives}</p>\n<p>{label:900=}\n}{else:{if screentime=1:{label:903=<br><br>You recognize <a class=\"squiffy-link link-passage\" data-passage=\"listen\" role=\"link\" tabindex=\"0\">the voice of the leprechaun</a> playing from the playroom feed.}}{if screentime=2:{label:904=<br><br>A woman screams from monitor 5, which watches the playroom. <a class=\"squiffy-link link-passage\" data-passage=\"listen again\" role=\"link\" tabindex=\"0\">A trembling voice</a> follows, and you recognize it as the somewhat nasal tones of Rasputin.}}{return to all perspectives}\n}</p>\n<p>{room line labels}        </p>",
		'attributes': ["@replace 8="],
		'js': function() {
			if (JSscreentime == 1) {
				squiffy.set ("seenFrog");
			squiffy.set ("storyline", "greed");
				}
			if (JSscreentime == 3)
			squiffy.set ("seenCleoInLab");	
			if (JSscreentime != 1 && JSscreentime !=3 && JSscreentime !=7) {
			squiffy.set ("emptyLab");
			}
			else
			squiffy.set ("emptyLab", false);
						
		},
		'passages': {
			'@1': {
				'text': "",
				'attributes': ["@replace 205=<p>the scientist enters</p>","@replace 902=","@replace 904=<p>{if seen listen again:{Playroom dialogue 2}}{else:<br>The <a class=\"squiffy-link link-passage\" data-passage=\"listen again\" role=\"link\" tabindex=\"0\">voices</a> continue to sound from the playroom.}</p>"],
			},
			'listen': {
				'text': "",
				'attributes': ["@replace 903=<p>{Playroom dialogue 1}</p>"],
			},
			'listen again': {
				'text': "",
				'attributes': ["@replace 200=","@replace 904=<p>{Playroom dialogue 2}</p>"],
			},
			'medicine man': {
				'text': "",
				'attributes': ["@replace 200=","seenTotems","seenMedicineMan","@replace 200=<p><hr>{Medicine Man Costume}</p>"],
			},
			'pole': {
				'text': "",
				'attributes': ["seenPole","@replace 200=<p><hr>A pole about shoulder height leans against the side of a hole and disappears within. Grills protect visitors from twisting an ankle in any of the other holes.</p>"],
			},
			'chest': {
				'text': "",
				'attributes': ["seenPropChest","@replace 200=<p><hr>The {if screentime<3:open}{else:closed} chest sits in a dark corner of the room, next to a small table{if screentime>2:with a few candles on its top}. You imagine two pirates hefting the ornate chest up the stairs but, instead of ill-gotten spoils, this booty consists of ghoulish decorations and grisly props{if screentime&lt;3:, like the chainsaw and axe on the floor. Both weapons are cartoonish plastic exaggerations, the blood on the axe bright red}.</p>"],
			},
			'You': {
				'text': "",
				'attributes': ["seenPlayer","@replace 200=<p><hr>{Goldie description}</p>"],
			},
		},
	},
	'Alien Zoo': {
		'text': "<p>{if screentime&lt;3:{alert to other monitors}}</p>\n<h3 class=\"class2\">Alien Zoo (second floor, north wing)</h3>\n\n<p>{if screentime=4:As the man in black watches, the devil hooks the rope in his hand onto a stanchion. The man in black signals at the door. An alien of the gray-type, or a petit female <i>Homo sapiens,</i> enters and walks to a two-headed goat as the devil scurries around her and out the door. The gray begins to pet the goat then shivers. {label:401={trackedHauntings}} </p>\n<p>The alien makes several hand gestures to the man in black. He removes his jacket and drapes it over her shoulders. She wanders away to examine the {if seenMutants:other oddities}{else:<a class=\"squiffy-link link-passage\" data-passage=\"mutants\" role=\"link\" tabindex=\"0\">other oddities</a>} while he checks a brochure. Having finished reading, he watches the <a class=\"squiffy-link link-passage\" data-passage=\"gray\" role=\"link\" tabindex=\"0\">gray</a> standing before a mass of tentacles. When she turns to him, he gestures with his hands. She nods, and he leaves.\n}\n{else:\n{if screentime=3:The leprechaun stops in front of some rope hanging between two stanchions, which have been pushed up against the far wall to allow greater access to the collections. He unhinges one end of the rope and begins to wrap it around the doll&#39;s neck. </p>\n<p>The sasquatch steps back from the door, nodding at a <a class=\"squiffy-link link-passage\" data-passage=\"devil\" role=\"link\" tabindex=\"0\">devil</a> in a hockey jersey entering the room. The two speak. The sasquatch addresses the leprechaun. Scowling, the leprechaun unwinds the rope from the doll. He slaps the freed end into the devil&#39;s hand then rushes out the door, his hairy friend following behind. {if thoughts=explicit:{label:902=<br><br><i>{if seenRasputinScream:The sasquatch must have told the leprechaun that Greg is going elsewhere, so the leprechaun will need to set up his scare tactic somewhere else}{else:The leprechaun is out to try to scare someone who isn&#39;t headed to the zoo}, </i>you think.}}</p>\n<p>}\n{if screentime=7:The scientist steps into the room. You recognize him from the lab coat he wears; his white hair is gone, replaced by a horseshoe of thinning brown strands. He speaks and gestures at the bottle in his pocket. The medicine man nods before the two men split up, each attracted to a different display. Nothing seems out of place among the strange species and mutants in the collection.<br><br>Someone stomps toward the door to the Fly Eye. You <a class=\"squiffy-link link-section\" data-section=\"game over\" role=\"link\" tabindex=\"0\">whirl around</a>.}{else:\nTrack lighting spotlights creatures that would make even Barnum and Bailey shudder. Specimens have been collected from the land and sea, and others have been torn from the womb. Some are {if seenMutants:mutants}{else:<a class=\"squiffy-link link-passage\" data-passage=\"mutants\" role=\"link\" tabindex=\"0\">mutants</a>}, but {if seenBizarre:others}{else:<a class=\"squiffy-link link-passage\" data-passage=\"bizarre\" role=\"link\" tabindex=\"0\">others</a>} are just bizarre common variations.}} \n{if screentime=7:{label:900=}\n}{else:{if screentime=1:{label:903=<br><br>You recognize <a class=\"squiffy-link link-passage\" data-passage=\"listen\" role=\"link\" tabindex=\"0\">the voice of the leprechaun</a> playing from the playroom feed.}}{if screentime=2:{label:904=<br><br>A woman screams from monitor 5, which watches the playroom. <a class=\"squiffy-link link-passage\" data-passage=\"listen again\" role=\"link\" tabindex=\"0\">A trembling voice</a> follows, and you recognize it as the somewhat nasal tones of Rasputin.}}{return to all perspectives}\n}\n{room line labels}</p>",
		'attributes': ["@replace 8="],
		'js': function() {
			if (JSscreentime == 3)
					squiffy.set("seenLepInZoo");
			if (JSscreentime ==4) 
			squiffy.set("seenJacketTransfer");
		},
		'passages': {
			'@1': {
				'text': "",
				'attributes': ["@replace 401=","@replace 902=","@replace 904=<p>{if seen listen again:{Playroom dialogue 2}}{else:<br>The <a class=\"squiffy-link link-passage\" data-passage=\"listen again\" role=\"link\" tabindex=\"0\">voices</a> continue to sound from the playroom.}</p>"],
			},
			'bizarre': {
				'text': "",
				'attributes': ["seenBizarre","@replace 200=<p><hr>You recognize the terrier-sized thing that looks part weasel and part bear as the ferocious Honey badger, although this one has a white head, which you&#39;ve never seen on them before. The crustacean with the hairy front arms is the Yeti crab, and that blue arachnid is almost certainly a variation of the wolf spider. The poster on the far wall shows a Goblin shark with its blade-like horn protruding from its snout. It glows distinctly pink because its blood circulates under semitransparent skin.</p>"],
			},
			'devil': {
				'text': "",
				'attributes': ["seenDevil","@replace 200=<p><hr>{devilCostume}</p>"],
				'js': function() {
					
				},
			},
			'gray': {
				'text': "",
				'attributes': ["seenGray","@replace 200=<p><hr>{grayCostume}</p>"],
				'js': function() {
					
				},
			},
			'listen': {
				'text': "",
				'attributes': ["@replace 200=","@replace 903=<p>{Playroom dialogue 1}</p>"],
			},
			'listen again': {
				'text': "",
				'attributes': ["@replace 200=","@replace 904=<p>{Playroom dialogue 2}</p>"],
			},
			'mutants': {
				'text': "",
				'attributes': ["seenMutants","@replace 200=<p><hr>Some of the mutants in the glass display cases, jars, and tanks are sufficiently lighted by spotlight that macabre details resolve through the murky preservation fluid. In one tank, you can discern a shark with two large heads and a shriveled body. In another, a fish with two                                 mouths. The fetus in one of the jars appears to have one leg, but it &mdash; her? &mdash; may be a mermaid. The head in another jar is so malformed that you can&#39;t determine a species, but one single detail is glaring: a large eye centered in its forehead.</p>"],
			},
		},
	},
	'Children\'s Possessions Playroom': {
		'text': "<p>{if screentime&lt;3:{alert to other monitors}}</p>\n<h3 class=\"class2\">Children&#39;s Possessions Playroom (main floor, west wing)</h3>\n\n<p>{if screentime=1:\nThe leprechaun releases the gold plate and tries to lift a cup from the <a class=\"squiffy-link link-passage\" data-passage=\"table with chairs\" role=\"link\" tabindex=\"0\">table</a>. </p>\n<p>&quot;It&#39;s pretty creepy in here,&quot; the sasquatch says, &quot;All these <a class=\"squiffy-link link-passage\" data-passage=\"dolls and puppets\" role=\"link\" tabindex=\"0\">dolls</a> staring at us.&quot;</p>\n<p>The leprechaun looks around the room and shrugs. &quot;{PlayroomSequence1Dialogue1}.&quot; He walks over to a toybox and peers inside. He turns away then waves his hand in front of his face. &quot;{PlayroomSequence1Dialogue2}.&quot; </p>\n<p>The sasquatch chuckles. &quot;{PlayroomSequence1Dialogue3}.&quot;{label:501=<br><br>{trackedHauntings}.}    </p>\n<p>The leprechaun turns to the door. &quot;{PlayroomSequence1Dialogue4}.&quot; He leaps into the giant <a class=\"squiffy-link link-passage\" data-passage=\"toybox\" role=\"link\" tabindex=\"0\">toybox</a> and disappears from view.</p>\n<p>The sasquatch shakes his head then goes out the door. {if thoughts=explicit:{label:902=<br><br><i>I&#39;m sure the staff haunting this place uses the toybox to hide as well, </i>you think.}}\n}\n{if screentime=2:Suddenly, a black-and-white blur darts from behind the <a class=\"squiffy-link link-passage\" data-passage=\"clown\" role=\"link\" tabindex=\"0\">clown</a> and out the door.</p>\n<p>Rasputin pulls away from the sasquatch. He takes in the shelves of staring {if seenDollsAndPuppets:dolls and puppets}{else:<a class=\"squiffy-link link-passage\" data-passage=\"dolls and puppets\" role=\"link\" tabindex=\"0\">dolls and puppets</a>}. He chokes, and his face blanches.</p>\n<p>The sasquatch doesn&#39;t seem to notice Rasputin&#39;s distress. &quot;Pretty impressive collection, huh? You should see what&#39;s in that toybox.&quot;</p>\n<p>Rasputin glances at the toybox just as the leprechaun pops out of it. Rasputin screams and falls back into the sasquatch&#39;s arms{label:502=.<br><br>You&#39;re surprised the shriek didn&#39;t shatter the monitor&#39;s screen}.</p>\n<p>The leprechaun climbs out of the {if seenToyBox:toybox}{else:<a class=\"squiffy-link link-passage\" data-passage=\"toybox\" role=\"link\" tabindex=\"0\">toybox</a>}. &quot;You seem a bit on edge.&quot; He laughs.</p>\n<p>&quot;Do you guys know what happened in here?&quot; Rasputin says.</p>\n<p>&quot;Why don&#39;t you tell us what you&#39;ve heard,&quot; the sasquatch says as he steadies Rasputin&#39;s wobbly stance.</p>\n<p>Rasputin watches the dolls as he speaks. &quot;{PlayroomSequence2Dialogue1}</p>\n<p>&quot;{PlayroomSequence2Dialogue2},&quot; the sasquatch says.</p>\n<p>The leprechaun picks up a child-sized doll with red hair and a white dress. Wailing like a ghost, he shakes it at Rasputin. The mystic rushes out the door.</p>\n<p>The leprechaun grins broadly. &quot;{PlayroomSequence2Dialogue3}.&quot; He carries the doll out of the room with him, and the sasquatch follows, shaking his head. {if thoughts=hidden:<br>{label:994=<a class=\"squiffy-link link-section\" data-section=\"transfer to Fly Eye\" role=\"link\" tabindex=\"0\">Return</a> to all five perspectives.}}{else:{label:28={label:902=<br><i>Greg sure seems to think this place is haunted,</i> you think. <i>I never would have guessed he&#39;d have such sensitive nerves, though, or such a high-pitched scream.</i>}<br><br>{label:994=<a class=\"squiffy-link link-section\" data-section=\"transfer to Fly Eye\" role=\"link\" tabindex=\"0\">Return</a> to all five perspectives.}}}\n}\n{if emptyPlayroom:\n    {if seenDollsAndPuppets:Dolls and puppets}{else:<a class=\"squiffy-link link-passage\" data-passage=\"dolls and puppets\" role=\"link\" tabindex=\"0\">Dolls and puppets</a>} sit upon and dangle from shelves lining all four walls. Those with the sturdiest legs and largest sizes stand on the floor. The {if seenClown:clown}{else:<a class=\"squiffy-link link-passage\" data-passage=\"clown\" role=\"link\" tabindex=\"0\">clown</a>} rests alone near the door. Across from it, more toys are positioned around the single {if seenToyBox:toybox}{else:<a class=\"squiffy-link link-passage\" data-passage=\"toybox\" role=\"link\" tabindex=\"0\">toybox</a>}. A table with chairs suitable for pint-sized hostesses have been set for dinner.\n    }\n{if screentime=6:\n    The man in black checks inside the toybox. He makes a hand signal at the door. The gray enters with {if seenjacketTransfer:her partner&#39;s}{else:a} black jacket wrapped around her shoulders. While her guardian watches the dolls, she walks across the room to the table, where she tries to sit on a small chair.</p>\n<p>The picture distorts. Stabilizes.</p>\n<p>The gray bolts from the chair and rushes over to join the man in black in the center of the room. The image begins to break up, but {label:503=you can just }make out the pair holding hands as they look around, fixed to the spot.\n}\n{if screentime=7:You see short humanoid figures in the distortion. The gray and man in black move toward the door before the feed goes black. The static from the speaker persists, scrambling what may be giggling{label:504=. {trackedHauntings}}. {Frankie arrives}</p>\n<p>{label:900=}\n}\n{else:\n{if screentime=2:}{else:{label:994={return to all perspectives}}}}\n{room line labels}</p>",
		'attributes': ["@replace 8="],
		'js': function() {
			if (JSscreentime == 1)
			squiffy.set ("storyline", "fright");
			
			if (JSscreentime == 2)
			squiffy.set ("seenRasputinScream");
			if (JSscreentime == 3 || JSscreentime == 4 || JSscreentime ==5) {
				squiffy.set ("emptyPlayroom");
					}
						else
				squiffy.set ("emptyPlayroom", false);
		},
		'passages': {
			'@1': {
				'text': "",
				'attributes': ["@replace 28=<p><br>{label:994=<a class=\"squiffy-link link-section\" data-section=\"transfer to Fly Eye\" role=\"link\" tabindex=\"0\">Return</a> to all five perspectives.}</p>","@replace 501=","@replace 502=","@replace 503=","@replace 504=","@replace 902="],
			},
			'clown': {
				'text': "",
				'attributes': ["seenClown","@replace 200=<p><hr>The clown rests his head face-down on his spindly outstretched legs. He is the only doll of its kind in the collection, his singular presence targeting coulrophobic and stalwart visitors alike, an outcast turned menace.</p>"],
			},
			'dolls and puppets': {
				'text': "",
				'attributes': ["seenDollsAndPuppets","@replace 200=<p><hr>Many of these toys belonged to Wayne Harrison&#39;s daughters, Sarah and Jessica, but others have been collected from around the world. Some of the dolls are as small as your hand and others as tall as their 4-year-old owners had been. The imported dolls are generally more crudely-crafted than their mass-produced counterparts. You remember reports from the mansion staff that the deceased sisters are still playing with their dolls and not putting them back where they belong. You can also imagine the multi-jointed puppets poking people as they pass and dancing with delight to their invisible puppetmasters. A colorful collection of stuffed animals and other toys, such as balls and jumping ropes, have also been collected.</p>"],
			},
			'table with chairs': {
				'text': "",
				'attributes': ["seenTableWithChairs","@replace 200=<p><hr>A small table with chairs must have been where Sarah and Jessica held their dinner parties for their dolls and stuffed animals. The glasses, cups, bowls, and plates are not a matching set, and many are likely plastic. The centerpiece is a small collection of ornate dinnerware within a glass dome, shielded from the carefree hands of children and sticky fingers of visitors.</p>"],
			},
			'toybox': {
				'text': "",
				'attributes': ["seenToyBox","@replace 200=<p><hr>Animals and children play and dance together in the cartoons painted on the toybox in the far corner of the room. It is open, but you can&#39;t see inside. You assume it has been emptied of any valuables.</p>"],
			},
		},
	},
	'room monitors': {
		'text': "<p><a class=\"squiffy-link link-section\" data-section=\"set ballroom proxy\" role=\"link\" tabindex=\"0\"><span class=\"class1\">Soul Dancers Ballroom</span></a>: \n{if screentime=0:\n{if seenCleopatra:Cleopatra}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Cleopatra\" role=\"link\" tabindex=\"0\">Cleopatra</a>} {label:11=stomps over to where her headdress had lain, picks something up, and tries to re-affix it to her headgear. {if seenBlackWidow:Black Widow}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Black Widow\" role=\"link\" tabindex=\"0\">Black Widow</a>} rushes over to the side of the stage and bends into the shadows. <i>Was Cleopatra hiding something? </i> you wonder.}\n}\n{if screentime=1:Guests talk in small groups.}\n{if screentime=2:Some guests mingle at the refreshment table.}\n{if screentime=3:\n{if seenCleopatra:Cleopatra}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Cleopatra\" role=\"link\" tabindex=\"0\">Cleopatra</a>}{label:41={if notSeenCleoDropBag:, without her bag,} bursts through the double doors. She makes her way to the front of the stage and searches the crowd}.\n}\n{if screentime=4:A couple of dancers materializes, jitterbugs, and disappears.}\n{if screentime=5:{if seen Black Widow:Black Widow}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Black Widow\" role=\"link\" tabindex=\"0\">Black Widow</a>} {label:61=strolls into the room and begins to chat with the nun}.}\n{if screentime=6:{label:71=Cleopatra rushes out the door, returning with a man in her arms. He has white hair and a cape. The two stumble forward, and the <a class=\"squiffy-link link-passage\" data-passage=\"man\" role=\"link\" tabindex=\"0\">man</a> slumps to the floor    .}}</p>\n<p><a class=\"squiffy-link link-section\" data-section=\"set Communications proxy\" role=\"link\" tabindex=\"0\"><span class=\"class1\">Communications</span></a>: \n{if screentime=0:Vacant.}\n{if screentime=1:\n{if seenCleopatra:Cleopatra}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Cleopatra\" role=\"link\" tabindex=\"0\">Cleopatra</a>} {label:21=appears in the doorway with a bag over one shoulder}.\n}\n{if screentime=2:\n{label:31=The <a class=\"squiffy-link link-passage\" data-passage=\"medicine man\" role=\"link\" tabindex=\"0\">medicine man</a> enters, stops at a mirror, and stares into his contorted reflection}.\n}\n{if screentime=3:\n{label:42=The {if seenTotems:medicine man}{else:<a class=\"squiffy-link link-passage\" data-passage=\"medicine man\" role=\"link\" tabindex=\"0\">medicine man</a>} pokes at a chandelier with his staff. The {if seenTricksters:sasquatch and leprechaun}{else:<a class=\"squiffy-link link-passage\" data-passage=\"sasquatch and leprechaun\" role=\"link\" tabindex=\"0\">sasquatch and leprechaun</a>} arrive, the latter with the doll in his hand. They look at the witch doctor then up at the fixture}.<br>}\n{if screentime=4:\n{if seenBlackWidow:Black Widow}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Black Widow\" role=\"link\" tabindex=\"0\">Black Widow</a>} {label:51=appears and pulls {if seenRasputin:Rasputin}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Rasputin\" role=\"link\" tabindex=\"0\">Rasputin</a>} through the doorway}.\n}\n{if screentime=5:From the doorway, the {if seenTricksters:leprechaun and sasquatch}{else:<a class=\"squiffy-link link-passage\" data-passage=\"leprechaun\" role=\"link\" tabindex=\"0\">leprechaun and sasquatch</a>} look into the hall.}\n{if screentime=6:Vacant.}</p>\n<p><a class=\"squiffy-link link-section\" data-section=\"set Laboratory proxy\" role=\"link\" tabindex=\"0\"><span class=\"class1\">Laboratory</span></a>:\n{if screentime=0:<a class=\"squiffy-link link-passage\" data-passage=\"Rasputin\" role=\"link\" tabindex=\"0\">Rasputin</a>, bloody axe in one hand, rummages through a chest as a <a class=\"squiffy-link link-passage\" data-passage=\"medicine man\" role=\"link\" tabindex=\"0\">medicine man</a> studies his reflection in a mirror. {label:12= You know the Russian mystic and healer Grigori Rasputin as Gregory Hoblett, a member of your local chapter of the Society. Everybody likes Hoblett, unlike Grigori who survived a stabbing, possible cyanide poisoning, and two point blank gunshots, only to succumb after being shot twice in the back and beaten. The medicine man, however, is a mystery.}}\n{if screentime=1:{label:22=The {if seenMedicineMan:medicine man}{else:<a class=\"squiffy-link link-passage\" data-passage=\"medicine man\" role=\"link\" tabindex=\"0\">medicine man</a>} looks out the door before exiting, leaving the room vacant}.}\n{if screentime=2:{if seenCleopatra:Cleopatra}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Cleopatra\" role=\"link\" tabindex=\"0\">Cleopatra</a>} {label:32=strides to the far end of the room where she drops her bag next to a chest and unpacks some whips}.}\n{if screentime&gt;2:Vacant.}</p>\n<p><a class=\"squiffy-link link-section\" data-section=\"set Alien Zoo proxy\" role=\"link\" tabindex=\"0\"><span class=\"class1\">Alien Zoo</span></a>: \n{if screentime=0:Vacant.}\n{if screentime=1:Vacant.}\n{if screentime=5:Vacant.}\n{if screentime=2:\n{label:33=The {if seenTricksters:leprechaun and sasquatch}{else:<a class=\"squiffy-link link-passage\" data-passage=\"leprechaun and sasquatch\" role=\"link\" tabindex=\"0\">leprechaun and sasquatch</a>} appear in the entrance. The big guy positions himself in the doorway while the short one searches the exhibits with {if notedRedHairedDoll:the}{else:a} {if seenRedHairedDoll:red-haired doll}{else:<a class=\"squiffy-link link-passage\" data-passage=\"redhaired doll\" role=\"link\" tabindex=\"0\">redhaired doll</a>} tucked under his arm}.\n}\n{if screentime=3:\n{label:43=A man dressed completely in black and wearing a sports jacket enters. The <a class=\"squiffy-link link-passage\" data-passage=\"man in black\" role=\"link\" tabindex=\"0\">man in black</a> stares down through sunglasses at {if seenLepInZoo:the}{else:a} {if seenDevil:devil}{else:<a class=\"squiffy-link link-passage\" data-passage=\"devil\" role=\"link\" tabindex=\"0\">devil</a>}{if seenLepInZoo:}{else: in a hockey jersey}, who retreats with rope dangling from his hand}.\n}\n {if screentime=4:\n {label:52={if describedGray:The gray}{else:An alien of the gray-type, or a petit female <i>Homo sapiens,</i>} waits inside the doorway then steps out}.\n }\n{if screentime=6:The {if seenTotems:medicine man}{else:<a class=\"squiffy-link link-passage\" data-passage=\"medicine man\" role=\"link\" tabindex=\"0\">medicine man</a>} {label:72=has entered, and he turns around to talk to someone through the doorway}.}</p>\n<p><a class=\"squiffy-link link-section\" data-section=\"set playroom proxy\" role=\"link\" tabindex=\"0\"><span class=\"class1\">Children&#39;s Possessions Playroom</span></a>: \n{if screentime=0:The <a class=\"squiffy-link link-passage\" data-passage=\"sasquatch watches the leprechaun\" role=\"link\" tabindex=\"0\">sasquatch watches the leprechaun</a> struggle to lift a gold dinner plate from a table. You can hear the sasquatch chuckling softly because the room is wired to record the electronic voice phenomena of ghostly children. Is the leprechaun pulling any more pranks tonight?}\n{else:\n{if screentime=1:{label:23=The {if seenSasquatch:sasquatch}{else:<a class=\"squiffy-link link-passage\" data-passage=\"sasquatch\" role=\"link\" tabindex=\"0\">sasquatch</a>} enters with one hairy arm wrapped around Rasputin&#39;s shoulders, pulling {if seenRasputin:his reluctant captive}{else:<a class=\"squiffy-link link-passage\" data-passage=\"Rasputin\" role=\"link\" tabindex=\"0\">his reluctant captive</a>} inside. A toy clown sitting on the floor behind them leans over}.}\n{else:\n{if screentime=5:The {if seenMIB:man in black}{else:<a class=\"squiffy-link link-passage\" data-passage=\"man in black\" role=\"link\" tabindex=\"0\">man in black</a>}{label:62=, without his jacket, steps into the room, one hand on the device in his belt}.}\n{else:\n{if screentime=6:The footage is a blur of color. Children seem to giggle in the static of the tiny speaker.}\n{else:Vacant.}\n}}}</p>",
		'passages': {
			'Fillers': {
				'text': "<p>These prevent warning messages (no such link errors) during compilation.</p>",
			},
			'Black Widow': {
				'text': "<p>Filler.</p>",
			},
			'Cleopatra': {
				'text': "<p>Filler.</p>",
			},
			'devil': {
				'text': "<p>Filler.</p>",
			},
			'leprechaun': {
				'text': "<p>Filler.</p>",
			},
			'leprechaun and sasquatch': {
				'text': "<p>filler.</p>",
			},
			'man': {
				'text': "<p>Filler.</p>",
			},
			'man in black': {
				'text': "<p>Filler.</p>",
			},
			'medicine man': {
				'text': "<p>Filler.</p>",
			},
			'nun': {
				'text': "<p>Filler.</p>",
			},
			'Rasputin': {
				'text': "<p>Filler.</p>",
			},
			'redhaired doll': {
				'text': "<p>Filler.</p>",
			},
			'sasquatch': {
				'text': "<p>Filler.</p>",
			},
			'sasquatch and leprechaun': {
				'text': "<p>Filler.</p>",
			},
			'sasquatch watches the leprechaun': {
				'text': "<p>Filler.</p>",
			},
			'web browser': {
				'text': "<p>Filler.</p>",
			},
			'You': {
				'text': "<p>Filler.</p>",
			},
		},
	},
	'return to all perspectives': {
		'text': "<p>{if screentime=6:{label:994=<a class=\"squiffy-link link-section\" data-section=\"transfer to Fly Eye\" role=\"link\" tabindex=\"0\">Take a wider view</a> of the night&#39;s events.}}\n{if screentime=5:{label:994=With on-screen activity at a lull, <a class=\"squiffy-link link-section\" data-section=\"transfer to Fly Eye\" role=\"link\" tabindex=\"0\">check out</a> all the monitors.}}\n{if screentime=4:{label:994=<a class=\"squiffy-link link-section\" data-section=\"transfer to Fly Eye\" role=\"link\" tabindex=\"0\">Scan</a> all the screens.}}\n{if screentime=3:{label:994=<a class=\"squiffy-link link-section\" data-section=\"transfer to Fly Eye\" role=\"link\" tabindex=\"0\">Consider</a> the events playing out on all of the monitors.}}\n{if screentime=2:{label:994=<a class=\"squiffy-link link-section\" data-section=\"transfer to Fly Eye\" role=\"link\" tabindex=\"0\">Return</a> to all five perspectives.}}\n{if screentime=1:{label:994=<a class=\"squiffy-link link-section\" data-section=\"transfer to Fly Eye\" role=\"link\" tabindex=\"0\">Watch</a> all the screens.}}</p>",
		'passages': {
		},
	},
	'transfer to Fly Eye': {
		'text': "",
		'attributes': ["@replace 100=","@replace 994="],
		'js': function() {
			if (squiffy.get ("seenMedicineMan") && JSscreentime < 2)
			squiffy.set ("seenMMOnce");
			if (JSscreentime == 2 && realizations[1] != "Communications")
			squiffy.set ("noteDimmedLights");
			if (JSroomProxy == "CR") {
			if (JSscreentime > 2 && squiffy.get ("noteDimmedLights", true))
			squiffy.set ("noteDimmedLights", false);
			if (squiffy.get ("notExplainedChildren") && JSscreentime != 3 &&  JSscreentime != 4 && JSscreentime !=5)
			squiffy.set ("notExplainedChildren", false)	; 
			}
			if (JSscreentime == 3 && JSroomProxy != "LR")
				squiffy.set ("notSeenCleoDropBag");
					
			if (JSscreentime == 1)
			squiffy.story.go ("Fly Eye 1");
			if (JSscreentime == 2)
			squiffy.story.go ("Fly Eye 2");
			if (JSscreentime == 3)
			squiffy.story.go ("Fly Eye 3");
			if (JSscreentime == 4)
			squiffy.story.go ("Fly Eye 4");
			if (JSscreentime == 5)
			squiffy.story.go ("Fly Eye 5");
				if (JSscreentime == 6)
			squiffy.story.go ("screentime 7 sound");
		},
		'passages': {
		},
	},
	'Frankie arrives': {
		'text': "<p>Someone stomps toward the door to the Fly Eye. You <a class=\"squiffy-link link-section\" data-section=\"game over\" role=\"link\" tabindex=\"0\">whirl around</a>.</p>",
		'passages': {
		},
	},
	'game over': {
		'text': "<p>A doll the size of a small girl is sitting on the floor next to the door and with the black{if seenPet:}{else:-and-white} cat covering her lap. She smiles a black line under red freckles. She shakes as the cat raises its head and Frankie stomps into the room. The doll begins saying &quot;Mama . . . mama . . . mama . . .&quot; </p>\n<p>Frankie&#39;s voice booms. &quot;We have a problem. Everyone&#39;s meeting in the ballroom.&quot;</p>\n<p>Follow Frankie <a class=\"squiffy-link link-section\" data-section=\"leave\" role=\"link\" tabindex=\"0\">out</a>.</p>",
		'attributes': ["@replace 900=<hr>"],
		'js': function() {
			$("#soundWrapper").html("<audio id='sound effects' autoplay controls loop>");
			soundEffect = document.getElementById("sound effects");
			soundEffect.volume=0.2;
			soundEffect.innerHTML = "<source src='additional_files/dolly.mp3' type='audio/mp3' /> \ <source src='additional_files/dolly.ogg' type='audio/ogg' />";
			
		},
		'passages': {
		},
	},
	'leave': {
		'text': "<hr>\n\n<p>Outside the mansion, two paramedics lift Greg into an ambulance. {if roomProxy=SDBR:The resuscitation attempts by the nun may have saved his life}{else:In the ballroom, the nun had been performing CPR on Greg. He was deathly pale, and his hair was white. The nun may have saved his life}, but from what? You know Greg wasn&#39;t shot. The bullet holes in the cape had been part of the Rasputin costume, and Greg had no visible wounds.</p>\n<p>Everyone seems to have gathered around the lights cast by the mansion and its exterior lamps. Black Widow and Cleopatra, both crying, tell their stories to a police officer. The medicine man places his staff into the trunk of a car before finding the scientist. They talk and wait. The leprechaun watches the paramedics, but the sasquatch stares at the ground. </p>\n<p>Someone is missing, though . . . You don&#39;t see the man in black or the gray.</p>\n<p>The ambulance drives away. Cleopatra points at you, and the officer she was talking to nods. It&#39;s time to tell your story{if storyline=threads:, but you can&#39;t make any reasonable connections between what happened to Greg and what you saw on the monitors. Maybe the events would have come together more clearly had you watched from a different perspective. {endgame}}{else:. The monitors gave you enough information to <a class=\"squiffy-link link-passage\" data-passage=\"conceive a narrative\" role=\"link\" tabindex=\"0\">conceive a narrative</a> explaining Greg&#39;s fate, but some questions remain. Would your story be different had you watched from another perspective? Satisfied with your own imaginings, you can just say <a class=\"squiffy-link link-section\" data-section=\"endgame\" role=\"link\" tabindex=\"0\">goodnight</a>.}</p>",
		'js': function() {
			$("#soundWrapper").html("");
			previousRealizations = [];
			toldStories = 0;
			if ($.cookie('CStoryline') == "curse") {
			previousRealizations.push("curse");
			toldStories++;
			}
				if ($.cookie('FStoryline') == "fright") {
			previousRealizations.push("fright");
			toldStories++;
			}
			if ($.cookie('GStoryline') == "greed") { 
			previousRealizations.push("greed");
			toldStories++;
			}
			if (JSstoryline == "curse")
			$.cookie('CStoryline', 'curse');
			if (JSstoryline == "fright")
			$.cookie("FStoryline", "fright");
			if (JSstoryline == "greed")
			$.cookie("GStoryline", "greed")
			if (JSstoryline != "threads" && toldStories<3) {
				toldStories++;	
			squiffy.set ("seenStory");
				}	
			$.cookie('cookieTest', 'successful');
			
				if ($.cookie('cookieTest') != "successful")
					squiffy.set ("testCookie", "failed");
						
			squiffy.set ("recalledStories", toldStories);
			if (previousRealizations[2] == undefined) {
			storySummary = "none";
			if (previousRealizations[0] != undefined) { 
			storySummary = "\"" + previousRealizations[0] + "\"";
			if (previousRealizations[0] != JSstoryline)
			squiffy.set ("otherStory", storySummary);
				}
					if (previousRealizations[1] != undefined) { 
				storySummary = storySummary + " and " + "\"" + previousRealizations[1] + "\"";
			if (previousRealizations[1] != JSstoryline) {
			secondStory = "\"" + previousRealizations[1] + "\"";
			squiffy.set ("otherStory", secondStory);
			}
			}
			squiffy.set ("otherStories", storySummary);
					}
		},
		'passages': {
			'conceive a narrative': {
				'text': "",
				'attributes': ["playerViewsSuggestion"],
				'js': function() {
					squiffy.story.go ("endgame");
				},
			},
		},
	},
	'endgame': {
		'text': "<p>{if playerViewsSuggestion:\n{if storyline=curse:Cleopatra didn&#39;t want anyone to open the bag she was carrying, because she was hiding something in it. Something she could drop through a hole in the Laboratory&#39;s floor. She dimmed the lights in Communications to help conceal it. It could have poisoned Greg. Cleopatra needed to ensure Greg would reach behind those speakers, so she planned to take him to Communications, but he left the ballroom with Black Widow instead. She waited for him to return, not knowing that her scheme had worked, Black Widow an unwitting accomplice. {if talkedToFrankie:Maybe she blames Greg&#39;s father for the death of her family. Or maybe she was acting out the curse, either intentionally or unintentionally}{else:Nothing you&#39;ve seen suggests a motive, though}.}\n{if storyline=greed:The medicine man was angry with Greg for bumping into him and touching the totems. Greg seemed puzzled by them when he picked up the frog in Communications. Maybe he noticed they had been re-arranged along the staff, as if the medicine man had removed some and hadn&#39;t replace them in the same order. Or maybe there was something different about the frog totem itself, and it was worth killing for. The medicine man may have also conspired with the scientist. {if talkedToFrankie:Frankie mentioned an exhibit on Mesoamerica. The medicine man could have switched one of his totems for one that had been on display. He was afraid that Greg noticed the discrepancy, and he enlisted the scientist to help him steal the goods. They could have poisoned Greg by putting curare in the bottle}{else:You can&#39;t figure out how they may have done it, though, or why the totems are so important}. }\n{if storyline=fright:Greg certainly seemed anxious about the mansion and its reputation. Maybe the leprechaun decided to have some fun with the idea. After two unsuccessful attempts, one inside the zoo and one outside Communications, maybe the leprechaun did finally scare Greg with the doll he had taken from the playroom. Or maybe Harrison&#39;s daughters really were haunting the mansion tonight, and they frightened Greg{if roomProxy=AZR:. The leprechaun could have put the scientist&#39;s wig on Greg . . . or maybe the ghosts did}. }{if talkedToFrankie:}{else:{if storyline=fright:}{else:You wonder how much Frankie knew about what was going on. }}\n}</p>\n<hr>\n\n<p>There are three different interpretations of Greg&#39;s fate, each dependent upon which character you decide to follow when you settle in the Fly Eye. {if storyline=threads:You didn&#39;t follow any of them during your vigil}{if storyline=curse:You witnessed Cleopatra&#39;s vengeful spirit enact the &quot;curse&quot; storyline}{if storyline=fright:The leprechaun&#39;s tricks pulled you into the &quot;fright&quot; storyline}{if storyline=greed:You found some value in watching the medicine man&#39;s scheming in the &quot;greed&quot; storyline}. {if testCookie=failed:}{else:{if recalledStories=3:You&#39;ve {if seenStory:now }{else:already }discovered all of the perspectives (&quot;curse,&quot; &quot;fright,&quot; and &quot;greed&quot;).}{if recalledStories=2:You{if seenStory: previously watched the {otherStory} perspective}{else:&#39;ve seen two perspectives, {otherStories}}, so there is one last to discover.}{if recalledStories=1:{if seenStory:There}{else:You&#39;ve seen the {otherStory} perspective, so there} are two more for you to discover.}} You {if hauntedManifestations=0:{if storyline=threads: also }didn&#39;t witness any phenomena that are usually reported at haunted sites}{else:{if seenStory:also} witnessed {if hauntedManifestations=5:all of the}{else:{if hauntedManifestations=1:one}{if hauntedManifestations=2:two}{if hauntedManifestations=3:three}{if hauntedManifestations=4:four} of the five} phenomena that are typically reported as hauntings. You decide how much influence, if any, {if hauntedManifestations=1:it        }{else:they} had on tonight&#39;s events}.</p>\n<p><a class=\"squiffy-link link-section\" data-section=\"Read the credits\" role=\"link\" tabindex=\"0\">Read the credits</a> or play again by clicking the restart button at the top of the screen.</p>",
		'passages': {
		},
	},
	'Read the credits': {
		'text': "<hr>\n\n<p>{credits}</p>",
		'passages': {
		},
	},
	'': {
		'text': "",
		'attributes': ["grayCostume={if screentime>3:A large black jacket engulfs the tiny frame of the MIB's charge. She}{else:The man in black's spindly-armed charge} is undoubtedly human, unless the grays wear wetsuits and shower caps, both differing shades of gray than that of their faces, lips, and hands. More convincing are those big black eyes, which could have been plucked from an alien's orbitals at autopsy.","@replace 900=","@replace 902="],
		'js': function() {
				window.randomizedNumber = function randomRng(min, max) {
			return Math.floor(Math.random() * (max - min + 1)) + min;
			    	}
			window.JSscreentime = squiffy.get ("screentime");
			window.JSstoryline = squiffy.get ("storyline");
			window.JSroomProxy = squiffy.get ("roomProxy");
			if (JSstoryline == "fright" || JSstoryline == "greed") {
			squiffy.set ("followingFOrG");
				}
					else
			squiffy.set ("followingFOrG", false);
			if (squiffy.get("seenLeprechaun") && squiffy.get("seenSasquatch"))
				squiffy.set("seenTricksters");
		},
		'passages': {
		},
	},
	'room line labels': {
		'text': "<p>{label:100={label:200=}}</p>",
		'passages': {
		},
	},
	'set ballroom proxy': {
		'text': "",
		'attributes': ["roomProxy=SDBR"],
		'js': function() {
			realizations.push("Ballroom");
			squiffy.story.go ("text deletions")
		},
		'passages': {
		},
	},
	'set Communications proxy': {
		'text': "",
		'attributes': ["roomProxy=CR"],
		'js': function() {
			realizations.push("Communications");
			squiffy.story.go ("text deletions")
		},
		'passages': {
		},
	},
	'set Laboratory proxy': {
		'text': "",
		'attributes': ["roomProxy=LR"],
		'js': function() {
			realizations.push("Laboratory");
			
			squiffy.story.go ("text deletions")
		},
		'passages': {
		},
	},
	'set Alien Zoo proxy': {
		'text': "",
		'attributes': ["roomProxy=AZR"],
		'js': function() {
						realizations.push("Alien Zoo");
			squiffy.story.go ("text deletions")
		},
		'passages': {
		},
	},
	'set playroom proxy': {
		'text': "",
		'attributes': ["roomProxy=CPPR"],
		'js': function() {
			realizations.push("Playroom");
			squiffy.story.go ("text deletions")
		},
		'passages': {
		},
	},
	'text deletions': {
		'text': "",
		'attributes': ["@replace 10=","@replace 20=","@replace 30=","@replace 40=","@replace 50=","@replace 60=","@replace 70=","@replace 901=","screentime+=1"],
		'js': function() {
			if (squiffy.get ("talkedToFrankie") && squiffy.get ("positionedFrankieDialogue", false)) {
			if (JSscreentime == 4) {
			squiffy.set ("positionedFrankieDialogue");
				squiffy.story.go ("insert Frankie dialogue");
			}
			if (JSscreentime == 5)
			squiffy.story.go ("insert Frankie dialogue 2");
			}
			else
				squiffy.story.go ("go to room");
		},
		'passages': {
		},
	},
	'insert Frankie dialogue': {
		'text': "",
		'attributes': ["@replace 55=<p>{if talkedToFrankie:{Frankie conversation}}{else:You wonder whether you could <a class=\"squiffy-link link-passage\" data-passage=\"talk to Frankie\" role=\"link\" tabindex=\"0\">get some useful information</a> from this drunken monster.}</p>"],
		'js': function() {
			squiffy.story.go ("go to room");
		},
		'passages': {
		},
	},
	'insert Frankie dialogue 2': {
		'text': "",
		'attributes': ["@replace 65=<p>{if talkedToFrankie:. <br><br>{Frankie conversation}}</p>"],
		'js': function() {
			squiffy.story.go ("go to room");
			
		},
		'passages': {
		},
	},
	'go to room': {
		'text': "",
		'js': function() {
			function assessHaunting() {
			newHaunting = squiffy.get("hauntedManifestations");
			newHaunting++;
			squiffy.set ("hauntedManifestations", newHaunting);
			hauntingNumber = "a ";
			hauntingType = "unmistakable EVP";
			if (squiffy.get ("hauntedManifestations") == 2)
					hauntingNumber = "a second ";
							if (squiffy.get ("hauntedManifestations") == 3)
								hauntingNumber = "a third ";
							if (squiffy.get ("hauntedManifestations") == 4)
								hauntingNumber = "a fourth ";
							if (squiffy.get ("hauntedManifestations") == 5)
								hauntingNumber = "a fifth ";
			hauntingList = "\"hauntings\" list";
							if (squiffy.get ("hauntedManifestations") == 1)
											hauntingList = "\"haunted house\" checklist";
			if (JSroomProxy == "SDBR")
					hauntingType = "ghostly orb";
			if (JSroomProxy == "CR")
					hauntingType = "shaking chandelier";
							if (JSroomProxy == "AZR")
								hauntingType = "cold spots";
							if (JSroomProxy == "CPPR" && JSscreentime == 1)
			hauntingType = "odd odors";
				hauntingSummary = "<i>Tick " + hauntingNumber + "box on the " + hauntingList + ": " + hauntingType + "</i>, you think";
			squiffy.set ("trackedHauntings", hauntingSummary);
			}
			if (JSscreentime != 2 && JSroomProxy != "LR") {
			if (JSroomProxy == "SDBR" && JSscreentime == 5)
				assessHaunting();
					if (JSroomProxy == "CR" && (JSscreentime == 3 || (JSscreentime == 6 && realizations[2] != "Communications")))
							assessHaunting();
			if (JSroomProxy == "AZR" && JSscreentime == 4)
					assessHaunting();
						if (JSroomProxy == "CPPR" 	&& (JSscreentime ==1 || JSscreentime == 7)) 
			assessHaunting();
			}
			currentNarrative = JSstoryline; 
			previousScreentime = JSscreentime --- 1;
			plotline = curseStoryline;
				if (currentNarrative == "fright")
					plotline = frightStoryline;
			if (currentNarrative == "greed")
			plotline = greedStoryline;
			if (JSscreentime < 6 && currentNarrative != "threads" && plotline[previousScreentime] != realizations[previousScreentime])
			squiffy.set ("storyline", "threads");
			if (currentNarrative == "greed" && JSscreentime < 2)	
			squiffy.set ("storyline", "greed");
			if (JSroomProxy == "SDBR")
				squiffy.story.go ("Soul Dancers Ballroom");
			if (JSroomProxy == "CR") 
			squiffy.story.go ("Communications");
			if (JSroomProxy == "LR") 
			squiffy.story.go ("Laboratory");
			if (JSroomProxy == "AZR") 
			squiffy.story.go ("Alien Zoo");
							if (JSroomProxy == "CPPR") 
			squiffy.story.go ("Children's Possessions Playroom");
		},
		'passages': {
		},
	},
	'Links used as variables': {
		'text': "",
		'js': function() {
					
		},
		'passages': {
		},
	},
	'alert to other monitors': {
		'text': "<p>{if screentime=1:As events on the other monitors unfold, y}{if screentime=2:Remembering that something could be happening on the other screens, y}ou watch the monitor.</p>",
		'passages': {
		},
	},
	'Goldie description': {
		'text': "<p>You&#39;re dressed completely in black, including leather gloves and a hoody, its hood hiding your hair but not your forehead, from which a paper towel roll wrapped in a black sock points like a unicorn horn. Two more of these appendages are stuck to your belt, each attached by a length of string to the arm above. Oval wire scaffolds covered in plastic wrap protrude from your back.</p>",
		'passages': {
		},
	},
	'Medicine Man Costume': {
		'text': "<p>{if seenMMOnce:He hasn&#39;t changed since you saw him in the Laboratory . . . but his staff has}{else:At odds with the medicine man&#39;s contemporary shirt and pants are the mystical nature of his work. His cheeks are smeared with black markings, and he wears a necklace of beads and small bones. Pouches are strung along his belt. He holds a staff from which four totems swing as it moves}. {if screentime&lt;2:The animal representatives are aligned vertically with a wolf on top, followed by a frog, an owl, and a raven}{else:{if seenFrog:Now t}{else:T}he frog is at the top, followed by the wolf, the owl, and the raven}.</p>",
		'passages': {
		},
	},
	'MIBCostume': {
		'text': "<p>{if screentime&lt;6:The bulky man in black has a no-nonsense look about him, except for that egg beater in his belt. His black attire really brings out the gleam off his bald head}{else:The bulky man in black has a no-nonsense look about him, from the shaved head, sunglasses, and wicked sneer to his black shirt and pants. Still, he was\nwilling to stuff an egg beater in his belt}.</p>",
		'passages': {
		},
	},
	'Frankie conversation': {
		'text': "<p>{if storyline=curse:&quot;Do you know her,&quot; you say. &quot;Cleopatra.&quot;</p>\n<p>Frankie nods. &quot;That&#39;s Fay Clarkson. We are in the Boston chapter. She&#39;s usually such a nice girl, but tonight . . . I don&#39;t know what&#39;s gotten into her. Maybe it&#39;s her time of the &mdash; &quot;</p>\n<p>The cat hisses, leaps from your lap, and scrambles out the door. Frankie just smiles. &quot;Fay&#39;s father was Arthur Clarkson. An archaeologist. He would have been another Indiana Jones. But he and his wife and his son died in a boating accident while excavating a temple in the Amazon. Fay survived. Another archaeologist was supposed to be on that boat too. Greg&#39;s father, in fact. You know Greg, right? Rasputin. But Dr. Hoblett claimed to be sick that day. He finishes the excavation. He finds a tomb, takes a lot of stuff, and becomes famous. and Fay comes back with nothing. An orphan. Over the years, other people who were there have died.&quot; He makes spooky noises. &quot;Some say it&#39;s a curse. Such nonsense.&quot;}\n{if storyline=greed:\n&quot;Do you know our medicine man?&quot; you say. &quot;He&#39;s had a couple of run-ins with my friend Greg. Rasputin.&quot;</p>\n<p>Frankie shrugs. &quot;I ran into him in the exhibit on Native American culture. Lots of good stuff in there. Even blowguns and darts and poisonous plants.&quot;</p>\n<p>&quot;Curare,&quot; you say. &quot;The stuff they put on the darts.&quot;</p>\n<p>Frankie&#39;s head bobs. &quot;Dangerous, huh? Well, they&#39;ve got it all locked up, but there&#39;s some smart people here tonight. Don&#39;t trust anyone. Especially with those trinkets and totems. You know they&#39;re auctioning all that stuff off? I bet it will make someone . . . ,&quot; he winks at you. &quot;A pretty penny.&quot;</p>\n<p>Frankie attempts to pet the cat, but it hisses, leaps from your lap, and runs out the door.}\n{if storyline=fright:\n&quot;It looks like we need to keep an eye on the leprechaun,&quot; you say. &quot;He&#39;s already spooked Rasputin.&quot;</p>\n<p>&quot; I wouldn&#39;t trust Lucky Charms as far as I could toss &mdash;&quot;</p>\n<p>The cat hisses, leaps from your lap, and scrambles out the door.</p>\n<p>Frankie wobbles on his feet. &quot;He&#39;s trying to scare us with that stupid doll, but I&#39;m too smart for that.&quot;}\n{if storyline=threads:\n&quot;So, have you seen anything unusual tonight?&quot; you say.</p>\n<p>&quot;You mean besides the fact that Bigfoot is wandering the halls?&quot; Frankie regards the cat. &quot;And there&#39;s a giant skunk sitting on a fly&#39;s lap?&quot;</p>\n<p>The cat hisses, jumps to the floor, and runs out the door.</p>\n<p>&quot;Sorry, pussy,&quot; Frankie says. &quot;You don&#39;t smell as bad!&quot; His voice booms in the small room. &quot;What was your question again?&quot;</p>\n<p>&quot;Have you seen anything creepy tonight. Supernatural.&quot;</p>\n<p>Like ghosts? That&#39;s crap. I&#39;ll tell you what you gotta look out for. Government types. The man in black. The grays. Those costumes don&#39;t fool me. They&#39;re hiding in plain sight. You can never know whether it is them.&quot; He pauses. &quot;Well, I can tell. Did you see the way they&#39;ve been signaling each other tonight? That&#39;s not sign language. At least none that we know.&quot;}</p>",
		'passages': {
		},
	},
	'petting the cat': {
		'text': "<p>You run your hand down the cat&#39;s back, and the cat raises its head. There&#39;s something wrong with its fur. The white color is rough, unlike the soft black. The stripe is dry paint. It runs from neck to tail, and the intensity of the white color is uniform along its length, so the cat probably didn&#39;t struggle when it was being marked &mdash; or disguised &mdash; for Halloween.</p>",
		'passages': {
		},
	},
	'Playroom dialogue 1': {
		'text': "<p>The leprechaun says, &quot;{PlayroomSequence1Dialogue1}.&quot; A pause and he continues, &quot;{PlayroomSequence1Dialogue2}.&quot;</p>\n<p>&quot;{PlayroomSequence1Dialogue3}.&quot; That sounded like the sasquatch. </p>\n<p>The leprechaun says, &quot;{PlayroomSequence1Dialogue4}.&quot;</p>",
		'passages': {
		},
	},
	'Playroom dialogue 2': {
		'text': "<p>Rasputin says, &quot;{PlayroomSequence2Dialogue1}&quot; </p>\n<p>&quot;{PlayroomSequence2Dialogue2}.&quot;  That was the baritone of the sasquatch. He chuckles as a ghost moans. The gravelly-voiced leprechaun interrupts, &quot;{PlayroomSequence2Dialogue3}.&quot;</p>",
		'passages': {
		},
	},
	'Browser description': {
		'text': "<p>You read through the webpage: The Harrison Museum is the former home of the archaeologist Wayne Harrison, whose work was pivotal in our understanding of early Native American culture. His personal life, however, was marked by tragedy. First, his wife, Gabrielle, died while reading to their twin daughters. The girls, along with many of their friends, died years later in the same room as did their mother, with their deaths deemed mysterious by some. Conflicted and distraught, Wayne converted the mansion into a museum filled with collections of the bizarre and occult, daring you to believe.<br><br>Guests be wary. Ghoulish monsters roam the halls, and ghostly dancers have been spotted in the ballroom. And just maybe you&#39;ll encounter Wayne&#39;s daughters and their friends as they continue to play in the house long after their deaths.</p>",
		'passages': {
		},
	},
	'credits': {
		'text': "<p><i>A Fly on the Wall, or An Appositional Eye</i> was staged by Nigel Jayne, <a href=\"mailto:nigelsjayne@gmail.com\">e-mail him</a>, and made available under a <a href=\"https://creativecommons.org/licenses/by-nc/4.0\">Creative Commons 4 attribution</a>. I applaud my assistant directors Hanon Ondricek, Andrew Schultz, Daniel Stelzer, and XanMag. All of Nigel&#39;s interactive fiction, along with source codes, are available at the <a href=\"http://www.nigeljayne.ca/njgames.html\">gwelks website</a> or the <a href=\"http://ifdb.tads.org/search?searchfor=author%3ANigel+Jayne\">Interactive Fiction Database</a>. Cookies were coded using jquery.cookie by Klaus Hartl. Buster Hudson designed and created the <a href=\"AFOTWCover.html\" target=\"_blank\">cover art</a>. ({sequence:see description of the cover:A fly with a large head is standing on a wall in a corner of the room. His eyes are big orange orbs with red stripes. The title is printed in white along the top of the image}). The music heard in the ballroom is &quot;Techno Celebration&quot; by Eric Matyas at <a href=\"http://www.soundimage.org\">Sound Image](http://www.soundimage.org</a>. Sound effects were provided by AlucardsBride, g_lowing, Percy Duke, Robinhood76, SoundMunger, and tmc_zach, all downloaded from <a href=\"http://www.freesound.org\">Freesound.org</a>. Your visit was designed using <a href=\"http://www..textadventures.co.uk\">Squiffy</a>,     created by Alex Warren.</p>",
		'passages': {
		},
	},
}
})();