/* print.c
 *
 *  foblub -- a Z-machine for TI calculators
 *  based on pinfocom by InfoTaskForce and Paul Smith
 *  Ported and extended by Nils Gesbert, 2003
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; see the file COPYING.  If not, write to the
 *  Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>

#include "infocom.h"

#define BUFMIN              80
#define LONG_SCORE_WIDTH    60
#define MAX_MOVES           1599
#define STATLEN             24

#define print_mode 0     /* General Printing Mode           */
int     single_mode;    /* Mode for printing the next char */

static word word_bank;  /* There are 3 banks of common     */
                        /* words, each 32 words long.      */

static char score_buf[STATLEN+1];

Bool
print_init()
{
    text.len = 0;
    text.max = BUFMIN * 5;
    text.buf = (byte *)malloc(text.max + 1);
    if (!text.buf) return FALSE;

    if (STANDARD) {
      room.max = BUFMIN;
      room.buf = (byte *)malloc(room.max + 1);
      if (!room.buf) return FALSE;
      room.buf[0] = ' ';
      room.len = 1;

      ti_location = (char *)room.buf;
      ti_status = (char *)score_buf;
    }
    return TRUE;
}

static inline void open_stream3 (word adresse) {
  stream3.buf = base_ptr + adresse + 2;
  stream3.len = 0;
  stream3.max = data_head.resident_bytes - adresse - 3;
  flush_text_buffer();
  pbf_p = &stream3;
  stream3_active = TRUE;
}

static inline void close_stream3 () {
  if (stream3_active) {
#ifdef DEBUG1
    if (tracing) {
      char buffer[50];
      sprintf (buffer, "stream3 : %.20s -- longueur : %hu", stream3.buf, stream3.len);
      ecrit_texte (buffer);
      ngetchx();
      va_a_la_ligne();
    }
#endif
    stream3.buf[-1] = stream3.len & 0xFF;
    stream3.buf[-2] = stream3.len >> 8;
    stream3.max = 0;
    stream3.buf = NULL;
    pbf_p = &text;
    stream3_active = FALSE;
  }
}

static inline void output_stream (word *param) {
#ifdef DEBUG1
  if (tracing) {
    char buf[50]; sprintf (buf, "output_stream change : %hd", param[0]);
    ecrit_texte (buf); ngetchx(); va_a_la_ligne();
  }
#endif
  switch ((signed) param[0])
    {
    case 3 : open_stream3 (param[1]); break;
    case -3 : close_stream3 (); break;
    case 1 : output_disabled = FALSE; break;
    case -1 : if (pos_x_upper_win != 0xFFFF) output_disabled = TRUE;
    }
}

inline void
print_num A1(word, number)
{
    Bool    neg;
    int     num;
    char    buf[15], *cp;

    num = (signed_word)number;
    if ((neg = (num < 0)))
        num = -num;

    cp = buf;
    do
    {
        *(cp++) = '0' + (num % 10);
        num /= 10;
    }
    while (num > 0);

    if (neg)
        *cp = '-';
    else
        --cp;

    for (; cp >= buf; --cp)
        print_char((word)*cp);
}

inline void
print_str A1(const char *, str)
{
    while (*str != '\0')
        print_char((word)*str++);
}

inline void
print2 A1(word, pck_address)
{
  prt_coded (((unsigned long)pck_address) << (STANDARD ? 1 : 2));
}

inline void
print1 A1(word, address)
{
  prt_coded(address);
}

void
p_obj A1(word, obj_num)
{
    word        address;
    
    address = Z_TO_WORD(data(obj_num));

    /*
     * The first byte at the address is the length of the data: if
     * it's 0 then there's nothing to print, so don't.
     */
    if (get_byte(address++) > 0) prt_coded(address);
}

void
wrt()
{
  word data;
  single_mode = 0;
  /* Last word has high bit set */
  do
    {
      data = next_word();
      decode (data);
    }
  while ((data & 0x8000) == 0);
}

inline void
writeln()
{
    wrt();
    new_line();
    rtn(1);
}

void flush_text_buffer () {
  pbf_p->buf[pbf_p->len] = 0;
  ecrit_texte (pbf_p->buf);
  pbf_p->len = 0;
}

void
new_line()
{
  if (stream3_active) print_char (13);
  else if (output_disabled) return; 
  else if (upper_win_active) scr_newline_upper_win();
  else {
    flush_text_buffer();
    va_a_la_ligne();
  }
}

void
print_char A1(word, ch)
{
  if (!stream3_active) {
    if (output_disabled) return;
    if (ch == 13) {new_line(); return;}
    if (upper_win_active) {scr_putchar_upper_win (ch); return;}
  }
  if (pbf_p->len == pbf_p->max) {
    /*
     * If we're at the end of the buffer then flush it up to the last space...
     */
    char temp_buf[31];
    int i = pbf_p->len;
    while ((pbf_p->buf[--i] != ' ') && i > pbf_p->max - 30);
    memcpy (temp_buf, pbf_p->buf + i, pbf_p->max - i);
    pbf_p->len = i;
    flush_text_buffer ();
    pbf_p->len = pbf_p->max - i;
    memcpy (pbf_p->buf, temp_buf, pbf_p->len);
  }
  pbf_p->buf[pbf_p->len++] = (byte)ch;
}

static inline void
letter A1(char, ch)
{
    if (ch == 0)
    {
        print_char((word)' ');
        single_mode = print_mode;
        return;
    }

    if (ch <= 3)
    {
        /* Set single_mode to "Common Word" & set word_bank */

        single_mode |= 0x80;
        word_bank = (ch - 1) << 6;
        return;
    }

    if (ch <= 5)
    {
        /* Switch printing modes */

      single_mode = ch - 3;
      return;
    }

    if ((ch == 6) && (single_mode == 2))
    {
        /* Increment printing mode to 3 - ASCII Letter. */

        ++single_mode;
        return;
    }

    if ((ch == 7) && (single_mode == 2))
    {
        /* Print a Carriage Return */

        new_line();
        single_mode = print_mode;
        return;
    }

    /* None of the above, so this must be a single character */

    print_char((word)table[(single_mode * 26) + ch - 6]);
    single_mode = print_mode;

}

void
decode A1(word, data)
{
    word            code;
    int             i;
    char            ch[3];

    /* Reduce word to 3 characters of 5 bits */

    code = data;
    for (i = 0; i <= 2; i++)
    {
        ch[i] = code & 0x1F;
        code >>= 5;
    }

    /* Print each character */

    for (i = 2; i >= 0; i--)
    {
        if (single_mode & 0x80)
        {
            /* Print a Special Word */

	  unsigned long address;
	  address = Z_TO_WORD (common_word_ptr + word_bank + (int)(ch[i] << 1));
	  address <<= 1;
	  
	  prt_coded(address);
	  single_mode = print_mode;
	  continue;
        }
        if (single_mode < 3)
	{
            /* Print a single character */

            letter (ch[i]);
            continue;
        }
        if (single_mode == 3)
        {
            /*
             * Print ASCII character - store the high 3 bits of
             * char in the low 3 bits of the current printing mode.
             */

            single_mode = 0x40 + ch[i];
            continue;
        }
        if (single_mode & 0x40)
        {
            /*
             * Print an ASCII character - consists of the current
             * character as the low 5 bits and the high 3 bits coming
             * from the low 3 bits of the current printing mode.
             */

            ch[i] += (single_mode & 0x03) << 5;
            print_char((word)ch[i]);
            single_mode = print_mode;
        }
    }
}

void
prt_coded (unsigned long address)
{
    word    data;
    word (*lit_mot) (unsigned long) = (address & 1) ? get_word : get_aligned_word;

    /*
     * Print mode = < 0 :   Common Word;
     *              = 0 :   Lower Case Letter;
     *              = 1 :   Upper Case Letter;
     *              = 2 :   Number or Symbol;
     *              = 3 :   ASCII Letter - first byte;
     *              > 3 :   ASCII Letter - second byte;
     */
    single_mode = 0;

    /* Last word has high bit set */
    do
    {
        data = lit_mot(address);
        decode(data);
	address += 2;
    }
    while ((data & 0x8000) == 0);
}

void
set_score()
{
    /*
     * Set the description
     */
    pbf_p = &room;
    room.len = 1;
    p_obj(load_var(0x10));
    room.buf[room.len] = '\0';
    pbf_p = &text;

    /*
     * Fill in the score or time fields...
     */
    sprintf (score_buf,
	     F1_IS_SET(B_USE_TIME) ? "Time: %hu:%02hu " : "Score: %-3hd  Moves: %-4hd ",
	     load_var(0x11), load_var(0x12));
}
