/*

BabelZroft.cpp: Treaty of Babel metadata handling.

Copyright (C) 2007 Vladimir Korablin

*/

#include <Windows.h>
#include <aygshell.h>
#include <msxml.h>
#include <afx.h>
#include <atlbase.h>
#include "Utils.h"

extern "C" {
#include <stdlib.h>
#include "blorb/blorb.h"
}

#include "BabelZroft.h"

typedef CComPtr<IXMLDOMDocument/*,&__uuidof(IXMLDOMDocument)*/>
IXMLDOMDocumentPtr;

IXMLDOMDocumentPtr LoadXML(BSTR buffer)
{
    IXMLDOMDocumentPtr pXMLDoc;
    VARIANT_BOOL vSuccess;

    // CoInitializeEx has already been called, we don't need to bother
    HRESULT hr = pXMLDoc.CoCreateInstance(__uuidof (DOMDocument));
    if(hr != S_OK) {
        return NULL;
    }
    pXMLDoc->put_validateOnParse(VARIANT_FALSE);
    pXMLDoc->put_resolveExternals(VARIANT_FALSE);
    pXMLDoc->put_preserveWhiteSpace(VARIANT_FALSE);
    if (pXMLDoc->loadXML(buffer, &vSuccess) != S_OK)
        return NULL;
    return pXMLDoc;
}

// Get a string from an XML node
static CString XPathStr(IXMLDOMDocument* doc, LPCWSTR path)
{
    CComPtr<IXMLDOMNode> node;
    if (SUCCEEDED(doc->selectSingleNode(CComBSTR(path),&node)) && (node != NULL))
    {
        CComBSTR text;
        node->get_text(&text);
        return text.m_str;
    }
    return "";
}

int get_ifiction_info(bb_map_t *blorb_map, ifiction_info_t *info)
{
    bb_result_t result;
    unsigned int id_IFmd = bb_make_id('I','F','m','d');
    if (bb_load_chunk_by_type(blorb_map,
        bb_method_Memory,
        &result,
        id_IFmd,
        0) != bb_err_None)
        return 1;

    IXMLDOMDocumentPtr doc;
    CString meta((const char*)result.data.ptr, result.length);

    BSTR bstr = meta.AllocSysString(); // get a temporary BSTR object
    doc = LoadXML(bstr);
    ::SysFreeString(bstr); // release it
    if(!doc) {
        ReportError(L"Could not load IFiction XML");
        return 2;
    }

    // The following code was adapted from Windows Frotz.
    info->ifid = XPathStr(doc, L"/ifindex/story/identification/ifid");
    info->title = XPathStr(doc, L"/ifindex/story/bibliographic/title");
    info->headline = XPathStr(doc, L"/ifindex/story/bibliographic/headline");
    info->author = XPathStr(doc, L"/ifindex/story/bibliographic/author");
    info->year = XPathStr(doc, L"/ifindex/story/bibliographic/firstpublished");
    info->series = XPathStr(doc, L"/ifindex/story/bibliographic/series");
    info->seriesNumber = XPathStr(doc, L"/ifindex/story/bibliographic/seriesnumber");
    info->description = "";
    {
        CComPtr<IXMLDOMNode> node;
        CComBSTR path(L"/ifindex/story/bibliographic/description");
        if (SUCCEEDED(doc->selectSingleNode(path,&node)) && (node != NULL))
        {
            CComPtr<IXMLDOMNodeList> childList;
            if (SUCCEEDED(node->get_childNodes(&childList)))
            {
                CComPtr<IXMLDOMNode> childNode;
                while (SUCCEEDED(childList->nextNode(&childNode)))
                {
                    if (childNode == NULL)
                        break;

                    DOMNodeType type;
                    if (SUCCEEDED(childNode->get_nodeType(&type)))
                    {
                        switch (type)
                        {
                        case NODE_TEXT:
                            {
                                CComBSTR text;
                                childNode->get_text(&text);

                                //CString s(text.m_str);
                                //std::wstring stds(CString(text.m_str).GetString());
                                std::wstring stds(text.m_str);
                                Util_NormalizeString(stds);
                                // Append:
                                info->description += stds.c_str();
                            }
                            break;
                        case NODE_ELEMENT:
                            {
                                CComBSTR name;
                                childNode->get_nodeName(&name);
                                if (name == L"br") {
                                    // Append
                                    info->description += L"\r\r";
                                }
                            }
                            break;
                        }
                    }

                    childNode.Release();
                }
            }
        }
    }

    return 0;
}

HBITMAP load_cover_page(bb_map_t *blorb_map)
{    
    // Picture
    int imageSize;
    void *imageData;
    bb_result_t blorb_result;
    unsigned int id_Fspc = bb_make_id('F','s','p','c');

    if (bb_load_chunk_by_type(blorb_map,bb_method_Memory,&blorb_result,id_Fspc,0) == bb_err_None)
    {
        unsigned char* data = (unsigned char*)blorb_result.data.ptr;
        int coverPictNo = (data[0]<<24)|(data[1]<<16)|(data[2]<<8)|data[3];
        bb_unload_chunk(blorb_map,blorb_result.chunknum);
        if(bb_load_resource_pict(blorb_map, bb_method_Memory, &blorb_result, coverPictNo, NULL)
            == bb_err_None) {
                imageSize = blorb_result.length;
                imageData = blorb_result.data.ptr;
        }
    }
    if(!imageSize) {
        // No cover image
        return FALSE;
    }
    wchar_t tempPath[256];
    int length = GetTempPath(256, tempPath);
    if(!length) {
        // Error
        //MessageBox(Application.GetMainWnd(), "Cannot create file", "", MB_OK | MB_ICONWARNING);
        return NULL;
    }
    wchar_t tempFileName[MAX_PATH];
    if(!GetTempFileName(tempPath, L"zrf", 0, tempFileName)) {
        return NULL;
    }
    HANDLE hFile = Util_OpenFile(tempFileName, GENERIC_WRITE, CREATE_ALWAYS);
    if(!hFile) {
        return NULL;
    }
    Util_WriteBufferToFile(hFile, (void*)imageData, imageSize);
    CloseHandle(hFile);
    HBITMAP hResult = SHLoadImageFile(tempFileName);
    DeleteFile(tempFileName);
    bb_unload_chunk(blorb_map, blorb_result.chunknum);
    return hResult;
}