/*

Name:
DRV_W32.C

Description:
Enhanced MikMod Win32 driver (from the 2.09 Beta drv_w95.c)

Method:
Single buffer, automatic refill using a high-resolution timer callback routine.

Portability:
Microsoft Win32 - LCC-Win32, MS VC++

*/

#include <windows.h>
#include <mmsystem.h>
#include <stdio.h>
#include "mikmod.h"

static WAVEOUTCAPS woc;
static HWAVEOUT hWaveOut;
static LPVOID mydata;
static HGLOBAL hglobal;
static WAVEHDR WaveOutHdr;
static UINT gwID;		/* timer handle */
static UWORD last;    		/* last recorded playing position */
static char *mydma;

extern DRIVER drv_w32;

#define WIN32BUFFERSIZE 30000

CRITICAL_SECTION win32_mikmod_cs;	/* damned crash-proofing */


BOOL W32_IsThere(void)
{
	UINT numdevs = waveOutGetNumDevs();
	return (numdevs > 0);
}


BOOL W32_Init(void)
{
	MMRESULT err;
	WAVEFORMATEX wf;

InitializeCriticalSection(&win32_mikmod_cs);

	hglobal = GlobalAlloc(GMEM_MOVEABLE|GMEM_SHARE,WIN32BUFFERSIZE);
	if (hglobal==NULL)
	{
		myerr="GlobalAlloc() failed";
		return 0;
	}
	mydata = GlobalLock(hglobal);

	/* Get audio device name and put it into the driver structure */
	waveOutGetDevCaps(0, &woc, sizeof(WAVEOUTCAPS));
	drv_w32.Name = woc.szPname;

	if ((md_mode & DMODE_STEREO) && (woc.wChannels<2) )
	{
		/* Switch output to mono if device doesn't support stereo */
		md_mode &= ~DMODE_STEREO;
	}

	if (md_mode & DMODE_16BITS)
	{
		/* switch output to 8 bit if device doesn't support 16 bit */
		if (!(woc.dwFormats &
			 (WAVE_FORMAT_1M16 | WAVE_FORMAT_1S16 |
			  WAVE_FORMAT_2M16 | WAVE_FORMAT_2S16 |
			  WAVE_FORMAT_4M16 | WAVE_FORMAT_4S16)))
		{
			  md_mode&=~DMODE_16BITS;
		}
	}

	wf.wFormatTag = WAVE_FORMAT_PCM;
	wf.nChannels = (md_mode & DMODE_STEREO) ? 2 : 1;
	wf.nSamplesPerSec = md_mixfreq;

	wf.nAvgBytesPerSec = md_mixfreq;
	if (md_mode & DMODE_STEREO) wf.nAvgBytesPerSec<<=1;
	if (md_mode & DMODE_16BITS) wf.nAvgBytesPerSec<<=1;

	wf.nBlockAlign = 1;
	if (md_mode & DMODE_STEREO) wf.nBlockAlign<<=1;
	if (md_mode & DMODE_16BITS) wf.nBlockAlign<<=1;
	wf.wBitsPerSample = (md_mode & DMODE_16BITS)?16:8;

	err = waveOutOpen(&hWaveOut,WAVE_MAPPER, (WAVEFORMATEX *)&wf, 0, 0, 0);

	if (err)
	{
		if (err==WAVERR_BADFORMAT)
			myerr = "Specified output format is not supported";
		else if(err==MMSYSERR_ALLOCATED)
			myerr = "Audio device already in use";
		else
			myerr = "Cannot open audio device";
		GlobalUnlock(hglobal);
		GlobalFree(hglobal);
		return 0;
	}

	if (!VC_Init())
	{
		waveOutClose(hWaveOut);
		GlobalUnlock(hglobal);
		GlobalFree(hglobal);
		return 0;
	}

	return 1;
}


void W32_Exit(void)
{
	while (waveOutClose(hWaveOut)==WAVERR_STILLPLAYING) Sleep(20);
	GlobalUnlock(hglobal);
	GlobalFree(hglobal);
	VC_Exit();
DeleteCriticalSection(&win32_mikmod_cs);
}


static ULONG GetPos(void)
{
	MMTIME mmt;
	mmt.wType=TIME_BYTES;
	waveOutGetPosition(hWaveOut,&mmt,sizeof(MMTIME));
	return(mmt.u.cb&0xfffffff0);
}


static void CALLBACK TimeProc(
	UINT IDEvent,		/* identifies timer event */
	UINT uReserved,		/* not used */
	DWORD dwUser,		/* application-defined instance data */
	DWORD dwReserved1,	/* not used */
	DWORD dwReserved2)	/* not used */
{
	UWORD todo, curr;
	static volatile int timersema = 0;
	
	/* Use a semaphore to prevent entering the mixing routines twice */
	if (++timersema==1)
	{
EnterCriticalSection(&win32_mikmod_cs);
		curr = GetPos()%WIN32BUFFERSIZE;

		todo = (curr>last)?(curr-last):((WIN32BUFFERSIZE-last)+curr);

		if (todo < (WIN32BUFFERSIZE/2))
		{
			/* Only if the number of bytes we have to fill is too
			   large--prevents lockups if the system can't handle
			   the load
			*/
			if (curr > last)
			{
				VC_WriteBytes(&mydma[last],curr-last);
			}
			else
			{
				VC_WriteBytes(&mydma[last], WIN32BUFFERSIZE-last);
				VC_WriteBytes(mydma,curr);
			}
		}
		last = curr;
LeaveCriticalSection(&win32_mikmod_cs);
	}
	timersema--;
}


void W32_PlayStart(void)
{
	VC_PlayStart();
//	waveOutSetVolume(0, 0xffffffff);

	VC_WriteBytes(mydata, WIN32BUFFERSIZE);		/* fill audio buffer with data */

	WaveOutHdr.lpData = mydata;
	WaveOutHdr.dwBufferLength = WIN32BUFFERSIZE;
	WaveOutHdr.dwFlags = WHDR_BEGINLOOP|WHDR_ENDLOOP;
	WaveOutHdr.dwLoops = 0xffffffff;
	WaveOutHdr.dwUser = 0;
	waveOutPrepareHeader(hWaveOut, &WaveOutHdr, sizeof(WAVEHDR));
	waveOutWrite(hWaveOut, &WaveOutHdr, sizeof(WAVEHDR));
	mydma = mydata;
	last = 0;

	timeBeginPeriod(20);      /* set the minimum resolution */

	gwID = timeSetEvent(40,			/* how often */
			  40,			/* timer resolution */
			  TimeProc,  		/* callback function */
			  0,    		/* info to pass to callback */
			  TIME_PERIODIC);	/* oneshot or periodic */
}


void W32_PlayStop(void)
{
	/* Stop the timer */
	timeKillEvent(gwID);
	timeEndPeriod(20);

	/* Stop playing the wave */
	waveOutReset(hWaveOut);
	waveOutUnprepareHeader(hWaveOut, &WaveOutHdr, sizeof(WAVEHDR));
	VC_PlayStop();
}


void W32_Update(void)
{
	/* Not needed--buffers are automatically updated in the background */
}


DRIVER drv_w32 = 
{
	NULL,
	"Win32",
	"MikMod Enhanced Win32 Driver",
	W32_IsThere,
	VC_SampleLoad,
	VC_SampleUnload,
	W32_Init,
	W32_Exit,
	W32_PlayStart,
	W32_PlayStop,
	W32_Update,
	VC_VoiceSetVolume,
	VC_VoiceSetFrequency,
	VC_VoiceSetPanning,
	VC_VoicePlay
};

