/*
	HDNEWWIN.C
	(Win32 drop-in replacement for HDWINDOW.C)

	contains window/display routines:

		DebugMessageBox
		DrawBox
		InputBox
		OpenMenu
		PrintScreenBorders
		SelectBox
		SwitchtoDebugger
		SwitchtoGame

	for the Debugger build of the Hugo Engine

	Copyright (c) 1997-2006 by Kent Tessman
*/


#include <windows.h>
#undef GetProp	// windows.h

#include "heheader.h"
#include "hdheader.h"
#include "hdwin.h"
#include "hdwinext.h"
#include "hewin32.h"


/* from hdwin.c */
char *ViewAllSymbols(HWND hwnd);
extern debug_childwindow_struct debug_childwindow[];

/* from hewin.c */
extern HINSTANCE AppInstance;
extern HWND wndDebug;
extern HFONT fontDebug;

/* from hdupdate.c */
void UpdateOtherWindow(int v);


/* ALLOCATECHOICES

	Allocates memory for n choices (actually title + n choices)
	before calling SelectBox.
*/

void AllocateChoices(int n)
{
	if ((choice = AllocMemory((size_t)(n+1)*sizeof(char *)))==NULL)
		DebuggerFatal(D_MEMORY_ERROR);
}


/* DRAWBOX

	Draws a basic framed rectangle in the given color from (left, top)
	to (right, bottom).  (Note:  the screen area must be saved
	beforehand if it's going to be restored.)
*/

void DrawBox(int top, int bottom, int left, int right, int fcol, int bcol)
{}


/* DEBUGMESSAGEBOX

	Draws a box with the given title and caption, and waits for
	a keypress.
*/

void DebugMessageBox(char *title, char *caption)
{
	if (MessageBox(wndDebug, caption, title, MB_ICONINFORMATION | MB_OKCANCEL)==IDCANCEL)
		event.action = CANCEL;
	else
		event.action = SELECT;

	debugger_interrupt = true;
}


/* INPUTBOX

	Draws a box with the given title and caption, accepts an
	input of maxlen characters, and returns the input string.
	(The def argument gives the default string.)
*/

#define INPUTBOX_MAXLEN 128
char inputbox_buffer[INPUTBOX_MAXLEN];
int inputbox_len;
static char *inputbox_title, *inputbox_caption, *inputbox_default;

BOOL CALLBACK InputBoxDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
			SetDlgItemText(hwndDlg,
				HD_INPUTBOX_EDITBOX,
				inputbox_default);
			SetDlgItemText(hwndDlg,
				HD_INPUTBOX_CAPTION,
				inputbox_caption);
			SetWindowText(hwndDlg, inputbox_title);
			SetFocus(GetDlgItem(hwndDlg, HD_INPUTBOX_EDITBOX));
			SendMessage(GetDlgItem(hwndDlg, HD_INPUTBOX_EDITBOX), EM_SETSEL, 0, -1);
			return 0;       /* since focus has been set */

		case WM_COMMAND:
			switch LOWORD(wParam)
			{
				case IDOK:
					GetDlgItemText(hwndDlg, HD_INPUTBOX_EDITBOX,
						inputbox_buffer, inputbox_len);
					EndDialog(hwndDlg, 1);
					break;

				case IDCANCEL:
					strcpy(inputbox_buffer, "");
					event.action = CANCEL;
					EndDialog(hwndDlg, 0);
					break;

				case HD_INPUTBOX_BROWSE:
				{
					HWND hwnd;
					char *add;

					add = ViewAllSymbols(hwndDlg);
					if (add==NULL) return 0;

					GetDlgItemText(hwndDlg, HD_INPUTBOX_EDITBOX,
						inputbox_buffer, inputbox_len);

					if (strlen(add)+strlen(inputbox_buffer) >= INPUTBOX_MAXLEN)
					{
						MessageBox(hwndDlg, "Input too long", "Input", MB_ICONEXCLAMATION);
						return 0;
					}
					else
						strcat(inputbox_buffer, add);

					hwnd = GetDlgItem(hwndDlg, HD_INPUTBOX_EDITBOX);
					SendMessage(hwnd, EM_REPLACESEL, TRUE, (LPARAM)add);
					SetFocus(hwnd);
					SendMessage(hwnd, WM_KEYDOWN, VK_END, 0);

					return 0;
				}
			}
			return 0;
	}

	return 0;
}

char *InputBox(char *title, char *caption, int maxlen, char *def)
{
	if (maxlen > INPUTBOX_MAXLEN)
		inputbox_len = INPUTBOX_MAXLEN;
	else
		inputbox_len = maxlen;

	inputbox_title = title;
	inputbox_caption = caption;
	inputbox_default = def;

	if (!DialogBox(AppInstance, MAKEINTRESOURCE(HD_INPUTBOX),
		wndDebug, InputBoxDialog))
	{
		strcpy(inputbox_buffer, "");
	}

	return inputbox_buffer;
}


/* OPENMENU

	Takes a MENU_CONSTANT as an argument.  Sets the object of a SELECT
	event.action to MENU_CONSTANT + MENU_SUBHEADING.
*/

void OpenMenu(int m)
{}


/* OPENMENUBAR

	Sets the object of a SELECT event.action to MENU_CONSTANT + 
	MENU_SUBHEADING.
*/

void OpenMenubar(void)
{}


/* PRINTMENUBAR */

void PrintMenubar(void)
{}


/* PRINTSCREENBORDERS */

/* Win32 - Since this gets called but isn't really needed, use it
   instead to make sure all visible windows are updated, not just
   the selected one
*/

void PrintScreenBorders(void)
{
	char erase;
	int i, view;

	/* Same as in hd.c: */
	sprintf(debug_line, " %s", gamefile);
	if (RoutineName(currentroutine)[0]!='<')
		sprintf(debug_line+strlen(debug_line), ": %s", RoutineName(currentroutine));
	debug_windowbottomrow(debug_line);

	/* Update all visible windows */
	for (i=0; i<DEBUG_CHILD_WINDOWS; i++)
	{
		erase = false;
		SetWindowTitle(i);

		/* If the window is visible, repaint it */
		if (IsWindow(debug_childwindow[i].hwnd))
		{
			/* Checking first to see if it needs to be
			   updated content-wise by the debugger
			*/
			view = debug_childwindow[i].view;
			if (view==CODE_WINDOW)
			{
				if (buffered_code_lines >= FORCE_REDRAW)
					erase = true;

				UpdateCodeWindow();
			}
			else if (window[view].changed)
			{
				/* Completely erase and redraw locals and calls */
				if (view==VIEW_LOCALS || view==VIEW_CALLS)
					erase = true;

				UpdateOtherWindow(view);
			}

			if (erase)
				InvalidateRect(debug_childwindow[i].hwnd, NULL, 1);

			RedrawDebuggerChild(i);
			ForceRedraw(debug_childwindow[i].hwnd);
		}
	}
}


/* SELECTBOX

	Returns the selected choice, or 0 if nothing is selected.
	AllocateChoices(n) must be called first for n choices,
	and the choice[] array must hold the choices (with 0 being
	the title).  The default choice is given by def.
*/

/* These two extern definitions, both from hdtools.c, are necessary
   in case the SelectBox is a color-selection box--in which case
   selecting_color is true:
*/
extern char selecting_color;
extern void PrintSampleText(int n, int y);

int selectbox_count, selectbox_default;

BOOL CALLBACK SelectBoxDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	static HWND hwndList = NULL;
	int i;

	switch (uMsg)
	{
		case WM_INITDIALOG:
			hwndList = GetDlgItem(hwndDlg, HD_SELECTBOX_LISTBOX);
			SetDlgItemText(hwndDlg, HD_SELECTBOX_CAPTION,
				choice[0]);
			for (i=1; i<=selectbox_count; i++)
			{
				SendMessage(hwndList,
					LB_ADDSTRING, 0,
					(LPARAM)choice[i]);
				SendMessage(hwndList, LB_SETITEMDATA, i, (LPARAM)i);
			}
			SendMessage(hwndList, LB_SETCURSEL, selectbox_default, 0);
			
			SetFocus(hwndList);
			return 0;       /* since focus has been set */

		case WM_COMMAND:
			switch LOWORD(wParam)
			{
				case HD_SELECTBOX_LISTBOX:
					if (HIWORD(wParam)!=LBN_DBLCLK)
						break;
					/* fall through on doubleclick */
				case IDOK:
					selectbox_default = SendMessage(hwndList,
							LB_GETCURSEL, 0, 0);
					EndDialog(hwndDlg, 1);
					break;

				case IDCANCEL:
					event.action = CANCEL;
					EndDialog(hwndDlg, 0);
					break;
			}
			return 0;
	}

	return 0;
}

extern COLORREF reg_CustomColors[];	// from hewin.c

int SelectBox(int n, int def)
{
	int r;
	CHOOSECOLOR cc;

	/* If selecting a color, use the Win32 API's fancy built-in
	   color selection common dialog : */
	if (selecting_color)
	{
		cc.lStructSize = sizeof(CHOOSECOLOR);
		cc.hwndOwner = wndDebug;
		cc.rgbResult = def-1;
		cc.lpCustColors = (LPDWORD)reg_CustomColors;
		cc.Flags = CC_RGBINIT | CC_ENABLEHOOK;
		// hook function to center the dialog:
		cc.lpfnHook = (LPCCHOOKPROC)CenterHookProc;
 		
		if (ChooseColor(&cc))
			return cc.rgbResult+1;
		else
			return 0;
	}

	selectbox_count = n, selectbox_default = def-1;

	r = DialogBox(AppInstance, MAKEINTRESOURCE(HD_SELECTBOX),
		wndDebug, SelectBoxDialog);

	free(choice);

	if (r==0)
		return 0;
	else
		return selectbox_default+1;
}


int saved_currentline = 0;


/* SWITCHTODEBUGGER */

void SwitchtoDebugger(void)
{
	saved_currentline = currentline;

	post_picture_update = false;
	UpdateClient(true);

	debug_switchscreen(DEBUGGER);
	active_screen = DEBUGGER;

	hugo_settextcolor(color[NORMAL_TEXT]);
	hugo_setbackcolor(color[NORMAL_BACK]);

	menubar_isactive = false;
	active_menu_heading = MENU_INACTIVE;
}


/* SWITCHTOGAME */

void SwitchtoGame(void)
{
	UpdateDebugScreen();

	currentline = saved_currentline;

	HighlightCurrent(0);
	debug_switchscreen(GAME);
	active_screen = GAME;
}
