/*
	HEJPEG.C

	Image file (JPEG format) display routines, including the main
	function called from the engine:

		hugo_displaypicture

	Plus the non-portable Win32 functions:

		hugo_initJPEGimage
		hugo_writescanline

	And adapted from the IJG library:

		read_JPEG_file

	for the Hugo Engine

	Copyright (c) 1995-2006 by Kent Tessman

	The routines in this file are based in part on the work of the
	Independent JPEG Group.  They are taken from release 6a (7-Feb-96).

	Basically, hugo_displaypicture(file) attempts to read and
	display the JPEG-compressed picture stored in <file>.  The file
	is passed hot, i.e., opened and positioned.  The image should fit
	the screen area described by the current text window.  (This
	calculation is performed by hugo_initJPEGimage().)

	This file assumes it will find a compiled IJG library and the
	necessary headers "jpeglib.h", "jconfig.h", and "jmorecfg.h".

	For what it's worth, if you're using not using some faster,
	OS-integrated JPEG-display mechanism, this should be easily
	portable to any operating system mainly by replacing
	hugo_initpalette() and hugo_writescanline().

	bmpGr (a DIB section) is used to plot the image before 	drawing to
	the device-dependent bmpMem.
*/

#define USE_BILINEARSTRETCHBLT

#ifndef NO_GRAPHICS

#ifdef UNDER_CE
#define GetProp WinGetProp
typedef unsigned char libjpeg_boolean;
#else
typedef int libjpeg_boolean;
#endif
#include <windows.h>
#undef GetProp	// from windows.h
/* needed to fix a conflict with MSVC++'s boolean type */
#define boolean	libjpeg_boolean

#include <math.h>
#include <setjmp.h>
#include "heheader.h"
#include "hewin32.h"

#undef FAR	// from windows.h
#define XMD_H	// to avoid redefining INT32
#include "jpeglib.h"

int hugo_displaypicture(FILE *infile, long len);

/* Internal function prototypes: */
int hugo_createDIB(HBITMAP *dib, int width, int height, BYTE **data);
void hugo_writescanline(j_decompress_ptr cinfo, JSAMPLE *jbuffer);
int read_JPEG_file(FILE *infile);

#ifdef USE_BILINEARSTRETCHBLT
BOOL BilinearStretchBlt(HBITMAP dest_bmp, HBITMAP src_bmp);
static int dest_actual_width;
#endif

static HBITMAP bmpGr;
static BYTE *bitmap_data;
static int image_width, image_height;

extern char post_picture_update;


/* hugo_displaypicture

	(This is the only routine that gets called from the engine proper.)

	Assumes that "filename" is hot--i.e., opened and positioned--but
	closes it before returning.

	Loads and displays a JPEG picture, sizing it to the screen area
	described by the current text window.  hugo_displaypicture() could
	call any OS-specific loading mechanism to accomplish this.  If
	smaller than the current text window, the image should be centered,
	not stretched.

	Ultimately, graphic formats other than JPEG may be supported,
	hence the identification of the type flag and the switch
	at the end.

	Returns false if it fails because of an ERROR.
*/

char low_color_warning = 0;

int hugo_displaypicture(FILE *infile, long len)
{
	HDC dcGr = NULL;
	HDC dcMain;
	float ratio;
	int x, y, width, height;

	bmpGr = NULL;
	image_width = image_height = 0;

	/* So that graphics can be dynamically enabled/disabled */
	if (!hugo_hasgraphics())
	{
		fclose(infile);
		return true;		/* not an error */
	}

	dcMain = GetDC(wndMain);
	if (GetDeviceCaps(dcMain, BITSPIXEL)<16 && !low_color_warning)
	{
		MessageBox(wndMain,
_T("Images may not display properly at this color depth.  \
You may want to select a greater number of colors (using the Control Panel)."),
			_T("Hugo Engine - Display"), MB_ICONEXCLAMATION);
		low_color_warning = true;
	}
	ReleaseDC(wndMain, dcMain);

	/* Before doing any drawing, mainly because we need to make sure there
	   is no scroll_offset, or anything like that:
	*/
	if (client_needs_updating && scroll_offset) UpdateClient(false);

	switch (resource_type)
	{
		case JPEG_R:
			if (!read_JPEG_file(infile))
				goto Failed;
			break;

		default:        /* unrecognized */
#if defined (DEBUGGER)
			SwitchtoDebugger();
			DebugMessageBox("Picture Loading Error", "Unrecognized graphic format");
			SwitchtoGame();
#endif
			goto Failed;
	}

	/* If we get here, the image has been successfully plotted
	   to bmpGr, so transfer it to bmpMem (in dcMem)
	*/
	dcMain = GetDC(wndMain);
	dcGr = CreateCompatibleDC(dcMain);
	ReleaseDC(wndMain, dcMain);

	if (!dcGr) goto Failed;

	SelectObject(dcGr, bmpGr);

	width = image_width;
	height = image_height;
	
	ratio = (float)width/(float)height;
	if (width > physical_windowwidth)
	{
		width = physical_windowwidth;
		height = (int)((float)width / ratio);
	}
	if (height > physical_windowheight)
	{
		height = physical_windowheight;
		width = (int)((float)height * ratio);
	}
	x = physical_windowleft + physical_windowwidth/2 - width/2;
	y = physical_windowtop + physical_windowheight/2 - height/2;

#ifdef USE_BILINEARSTRETCHBLT
	if (!reg_GraphicsSmoothing ||
		(image_width<=physical_windowwidth && image_height<=physical_windowheight))
	{
#ifndef UNDER_CE
		SetStretchBltMode(dcMem, HALFTONE);
		SetBrushOrgEx(dcMem, 0, 0, NULL);
#endif
		StretchBlt(dcMem,
			x, y, width-1, height-1, 
			dcGr,
			0, 0, image_width, image_height,
			SRCCOPY);
	}
	else
	{
		HBITMAP bmpTarget;
		BYTE *target_data = NULL;
		if (hugo_createDIB(&bmpTarget, width-1, height-1, &target_data))
		{
			HDC dcTarget;
			dcTarget = CreateCompatibleDC(dcGr);
			dest_actual_width = width-1;
			BilinearStretchBlt(bmpTarget, bmpGr);
			SelectObject(dcTarget, bmpTarget);
#ifdef UNDER_CE
			// There's some sort of blitting issue on at least some
			// Pocket PC devices currently adding garbage to the lefthand
			// side with USE_BILINEARSTRETCHBLIT
			BitBlt(dcMem, x, y, width-2, height-1, dcTarget, 0, 0, SRCCOPY);
#else
			BitBlt(dcMem, x, y, width-1, height-1, dcTarget, 0, 0, SRCCOPY);
#endif
			DeleteDC(dcTarget);
			DeleteObject(bmpTarget);
		}
	}

#else	// !USE_BILINEARSTRETCHBLIT

#ifndef UNDER_CE
	SetStretchBltMode(dcMem, HALFTONE);
	SetBrushOrgEx(dcMem, 0, 0, NULL);
#endif
	StretchBlt(dcMem,
		x, y, width-1, height-1, 
		dcGr,
		0, 0, image_width, image_height,
		SRCCOPY);

#endif	// USE_BILINEARSTRETCHBLT

	fclose(infile);	// always close file

	DeleteDC(dcGr);
	dcGr = NULL;
	DeleteObject(bmpGr);
	bmpGr = NULL;

	post_picture_update = true;

	return 1;	// success

Failed:
	fclose(infile);	// always close file
	if (dcGr)
	{
		DeleteDC(dcGr);
		dcGr = NULL;
	}
	if (bmpGr)
	{
		DeleteObject(bmpGr);
		bmpGr = NULL;
	}
	return 0;	// error
}


/*  hugo_createDIB

	Creates a DIB section based on supplied parameters.
*/

int hugo_createDIB(HBITMAP *dib, int width, int height, BYTE **data)
{
	BITMAPINFO bmi;
	BITMAPINFOHEADER bmih;
	HDC dcMain;

	/* Create a device-independent bitmap (DIB) for drawing on--
	   a DIB is for writing bytes directly in 24-bit format
	*/
	bmih.biSize = sizeof(BITMAPINFOHEADER);
	bmih.biWidth = width;
	bmih.biHeight = -height;
	bmih.biPlanes = 1;
	bmih.biBitCount = 24;		// JPEG always decompresses to 24-bit color
	bmih.biCompression = BI_RGB;	// i.e., no compression, straight RGB data
	bmih.biSizeImage = 0;		// not needed since BI_RGB
	bmih.biClrUsed = 0;		// i.e., all colors
	bmih.biClrImportant = 0;	// ditto

	/* Have to end on a double-DWORD boundary, not just DWORD as per docs? */
	while (bmih.biWidth % (sizeof(DWORD)*2)) bmih.biWidth++;

	// BITMAPINFO
	bmi.bmiHeader = bmih;

	/* Select the graphics DIB into a memory DC */
	dcMain = GetDC(wndMain);
	*dib = CreateDIBSection(dcMain, &bmi, DIB_RGB_COLORS, data, NULL, 0);
	ReleaseDC(wndMain, dcMain);

	if (!*dib) return false;

	return true;
}


/* hugo_initJPEGimage

	Fits the JPEG image to the current text window, either scaling
	it down or centering it as necessary.
*/

static int row;				// current row
static int row_stride;			// number bytes/row
static int image_display_width_bytes;	// image width in bytes

int hugo_initJPEGimage(j_decompress_ptr cinfo)
{
	row = 0;

	/* cinfo->output_components is 3 if RGB, 1 if quantizing to a
	   color map or grayscale
	*/
	row_stride = cinfo->output_width * cinfo->output_components;

	image_width = cinfo->image_width;
	image_height = cinfo->image_height;

	/* Do some figuring to make sure we're going to write
	   correctly to the device-independent bitmap (DIB) memory--
	   convert to 24 bits (3 bytes per pixel).
	*/
	image_display_width_bytes = image_width * 3;

	/* Also align it properly */
	while (image_display_width_bytes % 24) image_display_width_bytes++;

	if (!hugo_createDIB(&bmpGr, image_width, image_height, &bitmap_data))
		return false;

	return true;
}


/* hugo_writescanline */

void hugo_writescanline(j_decompress_ptr cinfo, JSAMPLE *jbuffer)
{
	int j, x = 0;
	long mem_offset;	/* from start of bitmap_data  */

	mem_offset = row*image_display_width_bytes;

	for (j=0; j<row_stride; j+=cinfo->output_components)
	{
		*(bitmap_data + mem_offset + j) = *(jbuffer+j+2);
		*(bitmap_data + mem_offset + j+1) = *(jbuffer+j+1);
		*(bitmap_data + mem_offset + j+2) = *(jbuffer+j);
	}

	row++;
}


/*---------------------------------------------------------------------------
   JPEG Decompression Interface (from the IJG library):

   For further elaboration on the how and why of read_JPEG_file(), see
   "libjpeg.doc" with the IJG distribution.  For the sake of brevity
   and relevance to the Hugo implementation, some of the commenting has
   been trimmed or modified.
----------------------------------------------------------------------------*/

/* First of all, since a decompression error (memory or who knows what)
   may occur deep in the bowels of the IJG library, we need a way to
   harmlessly recover.  Hugo's default behavior is simply to abort loading
   the JPEG without reporting an error.

   The IJG library allows the user to build a hook into its normal
   error handler using setjmp() and longjmp(), which will instantly
   switch control to the blissfully ignorant return in read_JPEG_file(),
   below.
*/

struct hejpeg_error_mgr
{
	struct jpeg_error_mgr pub;      /* "public" fields */
	jmp_buf setjmp_buffer;          /* for return to caller */
};

typedef struct hejpeg_error_mgr *hejpeg_error_ptr;

void hejpeg_error_exit(j_common_ptr cinfo)
{
	/* cinfo->err really points to a hejpeg_error_mgr struct, so
	   coerce pointer
	*/
	hejpeg_error_ptr myerr = (hejpeg_error_ptr) cinfo->err;

	/* This is where the IJG library displays the error message--Hugo
	   doesn't, since it returns (even in mid-failed-decompression) to
	   the calling function

		(*cinfo->err->output_message) (cinfo);
	*/

	/* Return control to the setjmp point */
	longjmp(myerr->setjmp_buffer, 1);
}


/* read_JPEG_file

	Assumes the file passed as infile has been opened successfully.
*/


int read_JPEG_file(FILE *infile)
{
	/* This struct contains the JPEG decompression parameters and
	 * pointers to working space (which is allocated as needed by
	 * the JPEG library).
	 */
	struct jpeg_decompress_struct cinfo;

	/* We use our private extension JPEG error handler.
	 * Note that this struct must live as long as the main JPEG parameter
	 * struct, to avoid dangling-pointer problems.
	 */
	struct hejpeg_error_mgr jerr;

	JSAMPARRAY jbuffer;     /* output row buffer */
	int row_stride;         /* physical row width in output buffer */


/* Step 1: Allocate and initialize JPEG decompression object */

	/* We set up the normal JPEG error routines, then override
	   error_exit.
	*/
	cinfo.err = jpeg_std_error(&jerr.pub);
	jerr.pub.error_exit = hejpeg_error_exit;

	/* Establish the setjmp return context for hejpeg_error_exit
	   to use.
	*/
	if (setjmp(jerr.setjmp_buffer))
	{
	    /* If we get here, the JPEG code has signaled an error.
	     * We need to clean up the JPEG object, close the input file,
	     * and return.
	     */
	    jpeg_destroy_decompress(&cinfo);
//	    fclose(infile);
	    return 0;
	}

	/* Now we can initialize the JPEG decompression object. */
	jpeg_create_decompress(&cinfo);

/* Step 2: Specify the data source */

	/* Return values from most library initializations are ignored
	   since suspension is not possible from the stdio data source
	*/
	jpeg_stdio_src(&cinfo, infile);

/* Step 3: Read file parameters with jpeg_read_header() */

	jpeg_read_header(&cinfo, TRUE);

/* Step 4: set parameters for decompression */

	cinfo.quantize_colors = FALSE;

	/* Let the library build a suitable color map instead of supplying
	   it with one
	*/
	cinfo.colormap = NULL;
	cinfo.two_pass_quantize = FALSE;

	/* Choose Floyd-Steinberg dithering and give the library a color
	   count to quantize to
	*/
//	cinfo.dither_mode = JDITHER_FS;
//	cinfo.desired_number_of_colors = SCREENCOLORS;

	/* JDCT_DEFAULT (JDCT_ISLOW) is slower/higher quality; JDCT_FASTEST
	   is JDCT_IFAST by default
	*/
	cinfo.dct_method = JDCT_ISLOW;

	/* Little quality cost to forgo fancy upsampling */
//	cinfo.do_fancy_upsampling = FALSE;
	cinfo.do_fancy_upsampling = TRUE;

/* Step 5: Start decompressor */

	jpeg_start_decompress(&cinfo);

	if (!hugo_initJPEGimage(&cinfo))
	{
		jpeg_destroy_decompress(&cinfo);
		return 0;
	}

	/* JSAMPLEs per row in output buffer */
	row_stride = cinfo.output_width * cinfo.output_components;

	/* Make a one-row-high sample array that will go away when
	   done with the image
	*/
	jbuffer = (*cinfo.mem->alloc_sarray)
		((j_common_ptr)&cinfo, JPOOL_IMAGE, row_stride, 1);

/* Step 6: while (scan lines remain to be read) */

	/* Here we use the library's state variable cinfo.output_scanline
	 * as the loop counter, so that we don't have to keep track
	 * ourselves.
	 */
	while (cinfo.output_scanline < cinfo.output_height)
	{
		/* jpeg_read_scanlines expects an array of pointers
		 * to scanlines.  Here the array is only one element
		 * long, but you could ask for more than one scanline
		 * at a time if that's more convenient.
		 */
		jpeg_read_scanlines(&cinfo, jbuffer, 1);
		hugo_writescanline(&cinfo, jbuffer[0]);
	}

/* Step 7: Finish decompression */

	jpeg_finish_decompress(&cinfo);

/* Step 8: Release JPEG decompression object */

	jpeg_destroy_decompress(&cinfo);

//	fclose(infile);

	return 1;
}

//---------------------------------------------------------------------------
// BilinearStretchBlt
//---------------------------------------------------------------------------

#ifdef USE_BILINEARSTRETCHBLT

__inline COLORREF BitmapPixelAt(BYTE *src, int width, int x, int y)
{
	BYTE r, g, b;
	r = *(src + y*width*3 + x*3+2);
	g = *(src + y*width*3 + x*3+1);
	b = *(src + y*width*3 + x*3);
	return RGB(r, g, b);
}

__inline void SetBitmapPixelAt(BYTE *dest, int width, int x, int y, COLORREF c)
{
	*(dest + y*width*3 + x*3+2) = GetRValue(c);
	*(dest + y*width*3 + x*3+1) = GetGValue(c);
	*(dest + y*width*3 + x*3)   = GetBValue(c);
}

BOOL BilinearStretchBlt(HBITMAP dest_bmp, HBITMAP src_bmp)
{
	BYTE *dest, *src;
	int dest_width, dest_height, src_width, src_height;
	BITMAP bm;
	double nXFactor, nYFactor;
	double fraction_x, fraction_y, one_minus_x, one_minus_y;
	int ceil_x, ceil_y, floor_x, floor_y;
	COLORREF c1, c2, c3, c4;
	BYTE red, green, blue, b1, b2;
	int x, y;

	GetObject(src_bmp, sizeof(BITMAP), &bm);
	src = bm.bmBits;
	src_width = bm.bmWidth;
	src_height = bm.bmHeight;
	GetObject(dest_bmp, sizeof(BITMAP), &bm);
	dest = bm.bmBits;
// Have to use a different-than-technical width here because of padding
//	dest_width = bm.bmWidth;
	dest_width = dest_actual_width;
	dest_height = bm.bmHeight;
	
	nXFactor = (double)src_width/(double)dest_width;
	nYFactor = (double)src_height/(double)dest_height;

	for (x=0; x<dest_width; x++)
	{
		for (y=0; y<dest_height; y++)
		{
			floor_x = (int)floor(x * nXFactor);
			floor_y = (int)floor(y * nYFactor);
			ceil_x = floor_x + 1;
			if (ceil_x >= src_width) ceil_x = floor_x;
			ceil_y = floor_y + 1;
			if (ceil_y >= src_height) ceil_y = floor_y;
			fraction_x = x * nXFactor - floor_x;
			fraction_y = y * nYFactor - floor_y;
			one_minus_x = 1.0 - fraction_x;
			one_minus_y = 1.0 - fraction_y;

			c1 = BitmapPixelAt(src, src_width, floor_x, floor_y);
			c2 = BitmapPixelAt(src, src_width, ceil_x,  floor_y);
			c3 = BitmapPixelAt(src, src_width, floor_x, ceil_y);
			c4 = BitmapPixelAt(src, src_width, ceil_x,  ceil_y);

			// Blue
			b1 = (BYTE)(one_minus_x * GetBValue(c1) + fraction_x * GetBValue(c2));
			b2 = (BYTE)(one_minus_x * GetBValue(c3) + fraction_x * GetBValue(c4));
			blue = (BYTE)(one_minus_y * (double)(b1) + fraction_y * (double)(b2));

			// Green
			b1 = (BYTE)(one_minus_x * GetGValue(c1) + fraction_x * GetGValue(c2));
			b2 = (BYTE)(one_minus_x * GetGValue(c3) + fraction_x * GetGValue(c4));
			green = (BYTE)(one_minus_y * (double)(b1) + fraction_y * (double)(b2));

			// Red
			b1 = (BYTE)(one_minus_x * GetRValue(c1) + fraction_x * GetRValue(c2));
			b2 = (BYTE)(one_minus_x * GetRValue(c3) + fraction_x * GetRValue(c4));
			red = (BYTE)(one_minus_y * (double)(b1) + fraction_y * (double)(b2));

			// bm.bmWidth is the destination bitmap here
			SetBitmapPixelAt(dest, bm.bmWidth, x, y, RGB(red, green, blue));
		}
	}

	return TRUE;
}

#endif	// USE_BILINEARSTRETCHBLT

#else	// NO_GRAPHICS

#include <stdio.h>

int hugo_displaypicture(FILE *infile, int len)
{
	return 0;
}

#endif
