/*
	HEMSVC.C

	Non-portable functions specific to Win32

	for the Hugo Engine

	Copyright (c) 1995-2006 by Kent Tessman

	While HEWIN.C provides Windows API interfacing, everything in this
	file is either called by the engine, or is used by a function called
	by the engine.

	NOTE:  See HEBLANK.C for elaboration/further documentation of
	these interface routines.
*/

#ifdef UNDER_CE
#define GetProp WinGetProp
#endif
#include <windows.h>
#undef GetProp			// from windows.h

#ifdef UNDER_CE
#include <commdlg.h>
#endif

#undef TextOut
#define TextOut(hdc, x, y, str, len)  ExtTextOut(hdc, x, y, 0, NULL, str, len, NULL)

#include "heheader.h"
#include "hewin.h"
#include "hewin32.h"
#if defined (DEBUGGER)
#include "hdheader.h"
#include "hdwinext.h"
#endif
#ifdef SPEECH_ENABLED
#include "hetalk.h"
#endif
#ifdef UNDER_CE
#include "hewince.h"
#endif


/* Function prototypes: */
void hugo_addcommand(void);
int hugo_color(int c);
void hugo_restorecommand(int n);
#if defined (NO_LATIN1_CHARSET)
void hugo_stripaccents(char *a);
#endif

/* Specific to hewin.c/hemsvc.c: */
void PushKeypress(int k);
int PullKeypress(void);
void RedrawInputLine(int index);
void ConstrainCursor(void);


#if defined (DEBUGGER)
void *AllocMemory(size_t size);
#endif

#if defined (FRONT_END)
int he_main(int argc, char **argv);     /* from HE.C */
#endif


/* Defined Hugo colors: */
#define HUGO_BLACK         0
#define HUGO_BLUE          1
#define HUGO_GREEN         2
#define HUGO_CYAN          3
#define HUGO_RED           4
#define HUGO_MAGENTA       5
#define HUGO_BROWN         6
#define HUGO_WHITE         7
#define HUGO_DARK_GRAY     8
#define HUGO_LIGHT_BLUE    9
#define HUGO_LIGHT_GREEN   10
#define HUGO_LIGHT_CYAN    11
#define HUGO_LIGHT_RED     12
#define HUGO_LIGHT_MAGENTA 13
#define HUGO_YELLOW        14
#define HUGO_BRIGHT_WHITE  15

int insert_mode = true;

/* Since hemsvc.c provides its own command history: */
#define HISTORY_SIZE    16              /* for command-line editing */
int hcount = 0;
char *history[HISTORY_SIZE];

/* Now some variables and constants to manage graphics/text display--
   all specific to hemsvc.c
*/
int current_text_color = 0, current_back_color = 0;
int newline_count = 0;	// only used by WinCE
// So SelectNewFont() can talk to hugo_font():
int last_hugo_font = -1;

/* Pseudo "keypresses", in case we need to suspend/resume updating */
char override_update_client = false;

/* Windows API management info from hewin.c: */

/* Registry settings: */
extern int reg_FastScrolling;
extern int reg_DisplayGraphics;
extern int reg_PlaySounds;
extern int reg_DefForeground, reg_DefBackground,
	reg_DefSLForeground, reg_DefSLBackground;


/*
    FILENAME MANAGEMENT:

    Different operating systems will have their own ways of naming
    files.  The following routines are simply required to know and
    be able to dissect/build the components of a particular filename,
    storing/restoring the components via the specified char arrays.

    For example, in MS-DOS (or Windows, in this case):

	hugo_splitpath("C:\HUGO\FILES\HUGOLIB.H", ...)
		becomes:  C:, HUGO\FILES, HUGOLIB, H

    and

	hugo_makepath(..., "C:", "HUGO\FILES", "HUGOLIB", "H")
		becomes:  C:\HUGO\FILES\HUGOLIB.H

    The appropriate equivalent nomenclature should be used for the
    operating system in question.
*/

void hugo_splitpath(char *path, char *drive, char *dir, char *fname, char *ext)
{
	char *file;
	char *extension = NULL;

	strcpy(drive, "");
	strcpy(dir, "");
	strcpy(fname, "");
	strcpy(ext, "");

	if ((file = strrchr(path, '\\'))==0)
	{
		if ((file = strrchr(path, ':'))==0) file = path;
	}

	strncpy(dir, path, strlen(path) - strlen(file));
	*(dir + strlen(path)-strlen(file)) = 0;
	extension = strrchr(path, '.');

	if ((extension!=0) && strlen(extension) < strlen(file))
	{
		strncpy(fname, file, strlen(file)-strlen(extension));
		*(fname + strlen(file)-strlen(extension)) = 0;
		strcpy(ext, extension+1);
	}
	else strcpy(fname, file);

	if (fname[0]=='\\') strcpy(fname, fname+1);
}

void hugo_makepath(char *path, char *drive, char *dir, char *fname, char *ext)
{
	strcpy(path, drive);
	strcat(path, dir);

	if ((strlen(dir) && path[strlen(path)-1]!='\\') && fname[0]!='\\')
		strcat(path, "\\");

	strcat(path,fname);

	if (*ext!='\0')
	{
		if (*ext == '.') ext++;
		strcat(path,".");
		strcat(path,ext);
	}
}


/*
    MEMORY ALLOCATION:
*/

void *hugo_blockalloc(long num)
{
	return malloc(num * sizeof(char));
}

void hugo_blockfree(void *block)
{
	free(block);
}


/* hugo_getfilename

    Loads the name of the filename to save or restore (as specified by
    the argument <a>) into the line[] array.

    (Careful to use <a> before changing <line>, since <a> may _be_
    <line>.)
*/

void hugo_getfilename(char *a, char *b)
{
	TCHAR msg[80];
	TCHAR *filetypes;
	TCHAR whichfile[32];
	TCHAR filename[MAXPATH];
	TCHAR initial_dir[MAXPATH];
	char drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILENAME], ext[MAXEXT];
	OPENFILENAME ofn;
	TCHAR temppath[MAXPATH] = _T("");

	UpdateClient(true);

#ifdef UNICODE
	strcpy_AtoU(whichfile, a);
	strcpy_AtoU(filename, b);
#else
	strcpy(whichfile, a);
	strcpy(filename, b);
#endif

	GetCurrentDirectory(MAXPATH, temppath);

	_stprintf(msg, _T("Select file %s"), whichfile);

	memset(&ofn, 0, sizeof(OPENFILENAME));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = wndMain;
	ofn.lpstrCustomFilter = NULL;
	ofn.nFilterIndex = 0;
	ofn.lpstrFile = filename;
	ofn.nMaxFile = MAXPATH;
	ofn.lpstrFileTitle = NULL;
	ofn.lpstrInitialDir = NULL;
	ofn.lpstrTitle = msg;
	ofn.Flags = OFN_PATHMUSTEXIST | OFN_EXPLORER |
		OFN_HIDEREADONLY | OFN_ENABLEHOOK;
	ofn.lpfnHook = (LPOFNHOOKPROC)CenterHookProc;

	if (b==scriptfile)
	{
		ofn.lpstrDefExt = _T("txt");	/* transcription */
		filetypes = _T("Transcription files (*.txt)\0*.txt\0All files (*.*)\0*.*\0");
		hugo_splitpath(b, drive, dir, fname, ext);
#ifdef UNDER_CE
		SHGetSpecialFolderPath(wndMain, initial_dir, CSIDL_PERSONAL, 0);
#else
		hugo_makepath(initial_dir, drive, dir, "", "");
#endif
		ofn.lpstrInitialDir = initial_dir;
	}
	else if (b==recordfile)
	{
		ofn.lpstrDefExt = _T("rec");	/* record/playback */
		filetypes = _T("Hugo recording files (*.rec)\0*.rec\0All files (*.*)\0*.*\0");
		hugo_splitpath(b, drive, dir, fname, ext);
#ifdef UNDER_CE
		SHGetSpecialFolderPath(wndMain, initial_dir, CSIDL_PERSONAL, 0);
#else
		hugo_makepath(initial_dir, drive, dir, "", "");
#endif
		ofn.lpstrInitialDir = initial_dir;
	}
	else
	{
		ofn.lpstrDefExt = _T("sav");	/* save/restore */
		filetypes = _T("Hugo save files (*.sav)\0*.sav\0All files (*.*)\0*.*\0");
/*
		if (GetEnvironmentVariable(_T("HUGO_SAVE"), initial_dir, MAXPATH))
			ofn.lpstrInitialDir = initial_dir;
*/
#ifdef UNDER_CE
		SHGetSpecialFolderPath(wndMain, initial_dir, CSIDL_PERSONAL, 0);
#else
		strcpy(initial_dir, current_save_dir);
#endif
		ofn.lpstrInitialDir = initial_dir;
	}

	ofn.lpstrFilter = filetypes;

	if (!strcmp(a, "to save"))
	{
		if (!GetSaveFileName(&ofn))
			strcpy(line, "");
		else
		{
#ifdef UNICODE
			strcpy_UtoA(line, ofn.lpstrFile);
#else
			strcpy(line, ofn.lpstrFile);
#endif
			GetCurrentDirectory(MAXPATH, current_save_dir);
		}
	}
	else
	{
		if (!GetOpenFileName(&ofn))
			strcpy(line, "");
		else
		{
#ifdef UNICODE
			strcpy_UtoA(line, ofn.lpstrFile);
#else
			strcpy(line, ofn.lpstrFile);
#endif
			if (!strcmp(a, "to restore"))
				GetCurrentDirectory(MAXPATH, current_save_dir);
		}
	}

	/* Restore the path we had coming in--prevents losing track
	   of where the resource files are when there are no
	   environment variables set.
	*/
	SetCurrentDirectory(temppath);
}


/* hugo_overwrite

    Checks to see if the given filename already exists, and prompts to
    replace it.  Returns true if file may be overwritten.
*/

int hugo_overwrite(char *f)
{
	TCHAR msg[MAXPATH+32];
	FILE *tempfile;

	if (!(tempfile = fopen(f, "rb")))       /* if file doesn't exist */
		return true;

	fclose(tempfile);

	_stprintf(msg, _T("Overwrite existing \"%s\"?"), f);

	if (MessageBox(wndMain, msg, _T("File already exists"),
		MB_ICONEXCLAMATION | MB_YESNO)==IDYES)
			return true;
	return false;
}


/* hugo_closefiles

    Closes all open files.  NOTE:  If the operating system automatically
    closes any open streams upon exit from the program, this function may
    be left empty.
*/

void hugo_closefiles()
{
	fclose(game);
	if (script) fclose(script);
	if (io) fclose(io);
	if (record) fclose(record);
}


/* hugo_getkey

    Returns the next keystroke waiting in the keyboard buffer.  It is
    expected that hugo_getkey() will return the following modified
    keystrokes:

	up-arrow        11 (CTRL-K)
	down-arrow      10 (CTRL-J)
	left-arrow       8 (CTRL-H)
	right-arrow     21 (CTRL-U)
*/

/* In order to simulate a getch() in Windows within the non-event-model
   structure of the Hugo Engine, we rely on message handling to push
   ASCII keypresses onto a keypress stack:  see WndProc() in hewin.c.
*/

int hugo_getkey(void)
{
	int k;
	MSG msg;

	/* Do this here so that the compass rose isn't displayed until
	   the engine is ready to accept input--it's useless until then
	   anyway
	*/
#ifndef NO_COMPASS
	if (during_player_input && show_compass_rose)
	{
		ShowCompassRose();
		client_needs_updating = true;
		show_compass_rose = false;
	}
#endif

	post_picture_update = false;
	if (client_needs_updating) UpdateClient(true);

	ConstrainCursor();
	CreateCaret(wndMain, NULL, 2, caret_height);

	while (!(k = PullKeypress()))
	{
		ShowCaret(wndMain);
		ConstrainCursor();

		/* If an accelerator key (like Ctrl+Z for Undo) is
		   pressed, it is not treated as a normal keypress
		*/
		if (processed_accelerator_key)
		{
			DestroyCaret();
			return 0;
		}
		
		if (GetMessage(&msg, NULL, 0, 0)>0)
		{
			if (!TranslateAccelerator(wndMain, accelMain, &msg))
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}
		else
			exit(msg.wParam);

		SleepEx(10, FALSE);
	}

	/* Remove the caret once we're done with it */
	DestroyCaret();

	if (k < 0) k = (unsigned char)k;

	return k;
}

#define MAX_KEYPRESSES 128
int keypress_count = 0;
int keypress_queue[MAX_KEYPRESSES];

void PushKeypress(int k)
{
	if (keypress_count==MAX_KEYPRESSES) return;
	keypress_queue[keypress_count++] = k;
}

int PullKeypress(void)
{
	int i, k;

	if (keypress_count==0) return 0;

	k = keypress_queue[0];
	for (i=0; i<keypress_count-1; i++)
		keypress_queue[i] = keypress_queue[i+1];
	keypress_count--;
	keypress_queue[keypress_count] = 0;

	if (k==1 && keypress_count>=2)
	{
		display_pointer_x = keypress_queue[0];
		display_pointer_y = keypress_queue[1];
		for (i=0; i<keypress_count-2; i++)
			keypress_queue[i] = keypress_queue[i+2];
		keypress_queue[keypress_count-2] = 0;
		keypress_queue[keypress_count-1] = 0;
		keypress_count-=2;
	}

#ifdef SPEECH_ENABLED
	if (k==13)
		CancelSpeaking();
#endif
	return k;
}


/* hugo_getline

    Gets a line of input from the keyboard, storing it in <buffer>.

    (NOTE:  The function keys used here are MS-DOS specific.
    They have been chosen to somewhat mirror the command-line editing
    keys of MS-DOS.  For other systems, the "switch (b)" lines
    will likely need to be changed.)

    The advantage of using this instead of an available library input
    function is when building the Debugger--pressing the Tab key switches
    to the debug screen; if a command such as Restart, Restore, etc. is
    given, the engine must know to abandon the input.

    Note also that this input function assumes a non-proportional font.
*/

char getline_active = false;

void hugo_getline(char *prmpt)
{
	int a, b, thiscommand;
	int c;                          /* c is the character being added */
	int oldx, oldy;
	int tempfont = currentfont;

#if defined (DEBUGGER)
	PrintScreenBorders();
#endif

	hugo_settextcolor(fcolor);
	hugo_setbackcolor(bgcolor);

	strcpy(buffer, "");
	c = 0;
	thiscommand = hcount;

#ifdef SPEECH_ENABLED
	if (strlen(prmpt)>1 || isalnum(prmpt[0]))
		SpeakString(prmpt);
	else
		SpeakString(" (prompt)");
	FlushSpeechBuffer();
#endif
	hugo_print(prmpt);
	FlushBuffer();

	getline_active = true;

#ifdef UNDER_CE
	EnableInputButtons(getline_active);
#endif

	/* An italic input font won't space/overwrite properly */
	hugo_font(currentfont &= ~ITALIC_FONT);

	/* i.e., the finishing position afer printing the prompt */
	oldx = current_text_x;
	oldy = current_text_y;

GetKey:
	b = hugo_getkey();

	/* If an accelerator key (like Ctrl+Z for Undo) is pressed, it
	   voids the current input
	*/
	if (processed_accelerator_key) goto BlankLine;

	hugo_settextcolor(icolor);
	hugo_setbackcolor(bgcolor);

	switch (b)
	{
#if defined (DEBUGGER)
		case (9):                       /* Tab */
		{
			during_input = true;
			Debugger();
			during_input = false;

			/* If the debugger is stopping execution, quitting,
			   etc., a blank string must be returned in the
			   input buffer.
			*/
			if (debugger_collapsing)
			{
				strcpy(buffer, "");
				if (active_screen==GAME)
					hugo_scrollwindowup();

				/* Restore original font */
				hugo_font(currentfont = tempfont);

				getline_active = false;

				return;
			}

			goto GetKey;
		}
#endif
		case (13):                      /* Enter */
		{
			full = 0;

			/* Copy the input to the script file (if open) */
			if (script) fprintf(script, "%s%s\n", prmpt, buffer);

#ifdef SPEECH_ENABLED
			CancelBufferedSpeech();
			SpeakString("\\Emp\\");
			SpeakString(buffer);
			SPEECH_PAUSE(500);
#endif

#ifdef SCROLLBACK_DEFINED
			/* Copy the input to the scrollback buffer */
			hugo_sendtoscrollback(prmpt);
			hugo_sendtoscrollback(buffer);
			hugo_sendtoscrollback("\n");
#endif
			hugo_print("\r\n");

			strcpy(buffer, Rtrim(buffer));
			hugo_addcommand();
#if defined (NO_LATIN1_CHARSET)
			hugo_stripaccents(buffer);
#endif
			/* Restore original font */
			hugo_font(currentfont = tempfont);

			getline_active = false;
#ifdef UNDER_CE
			EnableInputButtons(getline_active);
#endif
			return;
		}
		case INSERT_KEY:
		{
			insert_mode = !insert_mode;
			goto GetKey;
		}
		case BACKSPACE_KEY:
		case DELETE_KEY:
		{
			if (strlen(buffer)>0)
			{
				if (b==BACKSPACE_KEY)
				{
					if (c==0) goto GetKey;
					c--;

					/* Move backward the width of the
					   deleted character
					*/
					current_text_x-=hugo_charwidth(buffer[c]);
					if (current_text_x < oldx)
						current_text_x = oldx;
				}

				/* Shift the buffer to account for the
				   deleted character
				*/
				for (a=c; a<=(int)strlen(buffer); a++)
					buffer[a] = buffer[a+1];

				RedrawInputLine(c);
			}
			goto GetKey;
		}
		case (8):                       /* left-arrow */
		{
			if (c > 0)
				current_text_x -= hugo_charwidth(buffer[--c]);
			goto GetKey;
		}
		case (21):                      /* right-arrow */
		{
			if (c<(int)strlen(buffer))
				current_text_x += hugo_charwidth(buffer[c++]);
			goto GetKey;
		}
		case CTRL_LEFT_KEY:
		{
			if (c)
			{
				do
				{
					do
					{
						current_text_x -= hugo_charwidth(buffer[--c]);
					}
					while (c && buffer[c-1]!=' ');
				}
				while (c && buffer[c]==' ');
			}
			goto GetKey;
		}
		case CTRL_RIGHT_KEY:
		{
			if (c<(int)strlen(buffer))
			{
				do
				{
					do
					{
						current_text_x += hugo_charwidth(buffer[c++]);
					}
					while (c<(int)strlen(buffer) &&
						buffer[c-1]!=' ');
				}
				while (c<(int)strlen(buffer) && buffer[c]==' ');
			}
			goto GetKey;
		}
		case (27):                      /* Escape */
		case (24):                      /* CTRL-X */
		{
BlankLine:
			strcpy(buffer, "");
			c = 0;
			current_text_x = oldx;
			RedrawInputLine(0);
			processed_accelerator_key = 0;
			goto GetKey;
		}
		case HOME_KEY:
		{
			c = 0;
			current_text_x = oldx;
			goto GetKey;
		}
		case END_KEY:
		{
			c = strlen(buffer);
			current_text_x = oldx + hugo_textwidth(buffer);
			goto GetKey;
		}
		case (11):                      /* up-arrow */
		{
			if (--thiscommand<0)
			{
				thiscommand = 0;
				goto GetKey;
			}
			a = strlen(buffer);
RestoreCommand:
			hugo_restorecommand(thiscommand);
			current_text_x = oldx;
			c = 0;
			RedrawInputLine(0);
			current_text_x = oldx+hugo_textwidth(buffer);
			c = strlen(buffer);
			goto GetKey;
		}
		case (10):                      /* down-arrow */
		{
			a = strlen(buffer);
			if (++thiscommand>=hcount)
			{
				thiscommand = hcount;
				goto BlankLine;
			}
			goto RestoreCommand;
		}
		/* See, for example, the compass rose key-pushing routine in
		   hewin.c to see why it pays to turn off client updating
		   temporarily--we don't want to see commands "spelled out"
		*/
		case (OVERRIDE_UPDATE_CLIENT):
		{
			override_update_client = true;
			goto GetKey;
		}
		case (UPDATE_CLIENT):
		{
			int temp_x = current_text_x;
			int temp_y = current_text_y;
			current_text_x = oldx;
			current_text_y = oldy;
			RedrawInputLine(0);
			current_text_x = temp_x;
			current_text_y = temp_y;

			override_update_client = false;
			goto GetKey;
		}
	}

	/* Disallow invalid keystrokes */
	if (b < 32 || b>=256) goto GetKey;

	/* Hugo circa v2.2 allowed '^' and '~' for '\n' and '\"',
	   respectively
	*/
	if (game_version<=22 && (b=='^' || b=='~')) goto GetKey;

	/* (In Windows, we have to check both the window width AND
	   if we're going to overrun the buffer, just because a
	   ridiculously small-font/large-screen combination may be
	   in use.)
	*/
	if (current_text_x >= physical_windowright-charwidth ||
		c >= MAXBUFFER*2)
	{
		goto GetKey;
	}


	/* Add the new character: */

	/* If inserting, shift the post-insertion-point part of the buffer */
	buffer[strlen(buffer)+1] = '\0';
	if (c<(int)strlen(buffer) && insert_mode)
	{
		for (a=strlen(buffer); a>c; a--)
			buffer[a] = buffer[a-1];
	}
	buffer[c] = (char)b;

	/* Actually display the new character */
	if (!override_update_client)
		RedrawInputLine(c);

	current_text_x += hugo_charwidth((char)b);
	c++;

	goto GetKey;
}


/* RedrawInputLine

	Redraws only the changed portion of the input line, i.e., from the
	current text position to the right edge of the window.  <index>
	gives the current position in the buffer (<c> from hugo_getline()).
*/

void RedrawInputLine(int index)
{
	HGDIOBJ penNew, brushNew, penDefault, brushDefault;
	HDC dcMain;

	penNew = CreatePen(PS_SOLID, 1, current_back_color);
	brushNew = CreateSolidBrush(current_back_color);
	penDefault = SelectObject(dcMem, penNew);
	brushDefault = SelectObject(dcMem, brushNew);

	Rectangle(dcMem, current_text_x, current_text_y,
		physical_windowright, current_text_y+lineheight);

	SelectObject(dcMem, penDefault);
	SelectObject(dcMem, brushDefault);

	DeleteObject(penNew);
	DeleteObject(brushNew);

	/* Redraw the input line from the current position to the end
	   (TRANSPARENT so that we don't clip the font--we aren't
	   drawing over anything, so it doesn't matter)
	*/
	SetBkMode(dcMem, TRANSPARENT);
#ifdef UNICODE
	strcpy_AtoU(unicode_buffer, buffer);
	if (index)
		TextOut(dcMem,
			current_text_x-hugo_charwidth(buffer[index-1]),
			current_text_y,
			&unicode_buffer[index-1], _tcslen(unicode_buffer)-index+1);
	else
		TextOut(dcMem, current_text_x, current_text_y,
			&unicode_buffer[index], _tcslen(unicode_buffer)-index);
#else
	if (index)
		TextOut(dcMem,
			current_text_x-hugo_charwidth(buffer[index-1]),
			current_text_y,
			buffer+index-1, strlen(buffer)-index+1);
	else
		TextOut(dcMem, current_text_x, current_text_y,
			buffer+index, strlen(buffer)-index);
#endif
	SetBkMode(dcMem, OPAQUE);

	dcMain = GetDC(wndMain);
	BitBlt(dcMain,
		current_text_x,
		current_text_y,
		physical_windowwidth-current_text_x,
		lineheight,
		dcMem,
		current_text_x,
		current_text_y,
		SRCCOPY);
	ReleaseDC(wndMain, dcMain);
}


/* hugo_waitforkey

    Provided to be replaced by multitasking systems where cycling while
    waiting for a keystroke may not be such a hot idea.
*/

int hugo_waitforkey(void)
{
	int key;

#ifdef SPEECH_ENABLED
	FlushSpeechBuffer();
#endif
	FlushBuffer();
	post_picture_update = false;
	UpdateClient(true);	/* sometimes the screen isn't updated */

	waiting_for_key = true;	
	key = hugo_getkey();
	waiting_for_key = false;

	return key;
}


/* hugo_iskeywaiting

    Returns true if a keypress is waiting to be retrieved.
*/

int hugo_iskeywaiting(void)
{
#ifdef SPEECH_ENABLED
	FlushSpeechBuffer();
#endif
	FlushBuffer();
	return keypress_count?1:0;
}


/* hugo_timewait

    Waits for 1/n seconds.  Returns false if waiting is unsupported.
*/

int hugo_timewait(int n)
{
	MSG msg;
	static int repaint_interval = 0;
	
	if (post_picture_update)
		client_needs_updating = true;
	post_picture_update = false;

	SleepEx(1000/n, 0);

	if (client_needs_updating)
	{
		/* So that we don't bog things down repainting the
		   screen repeatedly, only do it every 1/10th second
		*/
		if ((repaint_interval+=1000/n) > 100)
		{
			UpdateClient(true);
			repaint_interval = 0;
		}	
	}

	/* Check Windows messages */
	if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
	{
		if (msg.message==WM_QUIT)
		{
			exit(msg.wParam);
		}
			
		if (!TranslateAccelerator(wndMain, accelMain, &msg))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}

	return true;
}


/*
    COMMAND HISTORY:

    To store/retrieve player inputs for editing.
*/

void hugo_addcommand(void)
{
	int i;

	if (!strcmp(buffer, "")) return;

	if (hcount>=HISTORY_SIZE)
	{
		hugo_blockfree(history[0]);
		for (i=0; i<HISTORY_SIZE-1; i++)
			history[i] = history[i+1];
		hcount = HISTORY_SIZE-1;
	}

	/* Because the debugger might use (up to) all available memory for
	   code line storage, a different means of memory allocation is
	   needed (at least in MS-DOS due to limited available memory to
	   begin with).
	*/
#if !defined (DEBUGGER)
	if ((history[hcount] = (char *)hugo_blockalloc((long)((strlen(buffer)+1)*sizeof(char))))==NULL)
#else
	if ((history[hcount] = (char *)AllocMemory((size_t)((strlen(buffer)+1)*sizeof(char))))==NULL)
#endif
	{
		hugo_blockfree(history[0]);
		if (hcount)
		{
			for (i=0; i<hcount; i++)
				history[i] = history[i+1];
			hcount--;
		}
		return;
	}

	for (i=0; i<=(int)strlen(buffer); i++)
		history[hcount][i] = buffer[i];
	hcount++;
}

void hugo_restorecommand(int n)
{
	int i;

	if (n < 0 || (n>=hcount && hcount!=HISTORY_SIZE-1)) return;

	i = 0;
	do
		buffer[i] = history[n][i];
	while (history[n][i++]!='\0');
}


/* hugo_stripaccents */

#if defined (NO_LATIN1_CHARSET)

void hugo_stripaccents(char *a)
{
	int i;

	for (i=0; i<(int)strlen(a); i++)
	{
		if ((unsigned char)a[i]>=0x80)
		{
			switch ((unsigned char)a[i])
			{

#if !defined (CONVERT_OEM_CHARSET)

		/*
		 * Latin-1 character set values (default):
		 *
		 */

				case '':	/* 'A' accents */
				case '':
				case '':
				case '':
				case '':
					a[i] = 'a';
					break;
				case '':
				case '':
				case '':
				case '':
				case '':
					a[i] = 'A';
					break;

				case '':               /* 'C' cedilla */
					a[i] = 'c';
					break;
				case '':
					a[i] = 'C';
					break;

				case '':               /* 'E' accents */
				case '':
				case '':
				case '':
					a[i] = 'e';
					break;
				case '':
				case '':
				case '':
				case '':
					a[i] = 'E';
					break;

				case '':               /* 'I' accents */
				case '':
				case '':
				case '':
					a[i] = 'i';
					break;
				case '':
				case '':
				case '':
				case '':
					a[i] = 'I';
					break;

				case '':               /* 'N' tilde */
					a[i] = 'n';
					break;
				case '':
					a[i] = 'N';
					break;

				case '':               /* 'O' accents */
				case '':
				case '':
				case '':
				case '':
					a[i] = 'o';
					break;
				case '':
				case '':
				case '':
				case '':
				case '':
					a[i] = 'O';
					break;

				case '':               /* 'U' accents */
				case '':
				case '':
				case '':
					a[i] = 'u';
					break;
				case '':
				case '':
				case '':
				case '':
					a[i] = 'U';
					break;

				case (char)(253-256):	/* 'Y' accents */
				case '':
					a[i] = 'y';
					break;
/*
				case 'Y':
					a[i] = 'Y';
					break;
*/

				case '':               /* '<<' */
				case '':               /* '>>' */
					a[i] = '\"';
					break;

				default:
				{
					strcpy(a+i, a+i+1);
				}
#else
		/*
		 * OEM character set:
		 *
		 */

				case 131:               /* 'A' accents */
				case 132:
				case 133:
				case 134:
				case 160:
					a[i] = 'a';
					break;
				case 142:
				case 143:
					a[i] = 'A';
					break;

				case 135:               /* 'C' cedilla */
					a[i] = 'c';
					break;
				case 128:
					a[i] = 'C';
					break;

				case 130:               /* 'E' accents */
				case 136:
				case 137:
				case 138:
					a[i] = 'e';
					break;
				case 144:
					a[i] = 'E';
					break;

				case 139:               /* 'I' accents */
				case 140:
				case 141:
				case 161:
					a[i] = 'i';
					break;

				case 164:               /* 'N' tilde */
					a[i] = 'n';
					break;
				case 165:
					a[i] = 'N';
					break;

				case 147:               /* 'O' accents */
				case 148:
				case 149:
				case 162:
					a[i] = 'o';
					break;
				case 153:
					a[i] = 'O';
					break;

				case 129:               /* 'U' accents */
				case 150:
				case 151:
				case 163:
					a[i] = 'u';
					break;
				case 154:
					a[i] = 'U';
					break;

				case 152:               /* 'Y' accents */
					a[i] = 'y';
					break;

				case 174:               /* '<<' */
				case 175:               /* '>>' */
					a[i] = '\"';
					break;

				default:
				{
					strcpy(a+i, a+i+1);
				}

#endif	/* if !defined (CONVERT_OEM_CHARSET) */

			}
		}
	}
}

#endif	/* if defined (NO_LATIN1_CHARSET) */


/*
    DISPLAY CONTROL:

    Note that the Win32 port uses pixel coordinates for all text
    screen/font measurements.
*/

#if !defined (DEBUGGER)
void hugo_init_screen(void)
{
	/* WinMain() takes care of this */
}
#endif


void hugo_cleanup_screen(void)
{}


int hugo_hasgraphics(void)
{
	return reg_DisplayGraphics;
}


void hugo_setgametitle(char *t)
{
#if defined (DEBUGGER)
	/* The array must be larger than MAX_GAME_TITLE in heset.c */
	static char debug_game_title[96];
	sprintf(debug_game_title, "Hugo Debugger - %s", t);
#ifdef UNICODE
	strcpy_AtoU(unicode_buffer, debug_game_title);
	SetWindowText(wndMain, unicode_buffer);
#else
	SetWindowText(wndMain, debug_game_title);
#endif

#else

#ifdef UNICODE
	strcpy_AtoU(unicode_buffer, t);
	SetWindowText(wndMain, unicode_buffer);
#else
	SetWindowText(wndMain, t);
#endif

#endif
}

void hugo_clearfullscreen(void)
{
/* Clears everything on the screen, moving the cursor to the top-left
   corner of the screen */

	HGDIOBJ penNew, brushNew, penDefault, brushDefault;

#if defined (DEBUGGER)
	if (active_screen != GAME)
	{
		debug_clearview(active_view);
		return;
	}
#endif

	FlushBuffer();
#ifdef SPEECH_ENABLED
	FlushSpeechBuffer();
#endif

	penNew = CreatePen(PS_SOLID, 1, current_back_color);
	brushNew = CreateSolidBrush(current_back_color);
	penDefault = SelectObject(dcMem, penNew);
	brushDefault = SelectObject(dcMem, brushNew);

	Rectangle(dcMem, 0, 0,
		GetSystemMetrics(SM_CXSCREEN),
		GetSystemMetrics(SM_CYSCREEN)*2);       /* including unseen area */

	SelectObject(dcMem, penDefault);
	SelectObject(dcMem, brushDefault);
	DeleteObject(penNew);
	DeleteObject(brushNew);

	client_needs_updating = true;

#ifdef USE_SMARTFORMATTING
	leftquote = true;
#endif

#ifdef USE_TEXTBUFFER
	TB_Clear(0, 0, GetSystemMetrics(SM_CXSCREEN),GetSystemMetrics(SM_CYSCREEN));
#endif
	
	/* Must be set: */
	currentpos = 0;
	currentline = 1;
}


void hugo_clearwindow(void)
{
/* Clears the currently defined window, moving the cursor to the top-left
   corner of the window */

	HGDIOBJ penNew, brushNew, penDefault, brushDefault;

	FlushBuffer();
#ifdef SPEECH_ENABLED
	FlushSpeechBuffer();
#endif

	if (client_needs_updating) UpdateClient(false);

#ifdef UNDER_CE
   	if (reg_MinimalWindows && illegal_window) return;
#endif
   	
	penNew = CreatePen(PS_SOLID, 1, current_back_color);
	brushNew = CreateSolidBrush(current_back_color);
	penDefault = SelectObject(dcMem, penNew);
	brushDefault = SelectObject(dcMem, brushNew);

	Rectangle(dcMem, physical_windowleft, physical_windowtop,
		physical_windowright, physical_windowbottom);

	/* Make sure that all the invisible scroll area is cleared to the
	   same color if we're not in a window
	*/
	if (!inwindow) ClipVisibleClient();

	SelectObject(dcMem, penDefault);
	SelectObject(dcMem, brushDefault);
	DeleteObject(penNew);
	DeleteObject(brushNew);

	/* Only forcing updating when not in a window will make for less
	   flickering
	*/
	if (!inwindow) client_needs_updating = true;

#ifdef SCROLLBACK_DEFINED
	/* Send a solid line to the scrollback buffer (unless the buffer is empty)... */
	if (!inwindow && scrollback_pos!=0 &&
		/* ...preventing duplicate linebreaks */
		((scrollback_pos>4) && scrollback_buffer[scrollback_pos-5]!='_'))
	{
		int _count = 160;

		if (scrollback_buffer[scrollback_pos-1]!='\n')
			hugo_sendtoscrollback("\n");
#ifdef UNDER_CE
		_count = 25;	// 160 underscores will wrap several times
#endif
		memset(line, '_', _count);
		sprintf(line+_count, "\n\n");
		hugo_sendtoscrollback(line);
	}
#endif

#ifdef USE_TEXTBUFFER
	TB_Clear(physical_windowleft, physical_windowtop,
		physical_windowright, physical_windowbottom);
#endif

#ifdef USE_SMARTFORMATTING
	leftquote = true;
#endif

	/* Must be set: */
	currentpos = 0;
	currentline = 1;
}


void hugo_settextmode(void)
{
	RECT rect;

/* This function does whatever is necessary to set the system up for
   a standard text display */

	/* charwidth and lineheight are set up by hugo_font();
	   otherwise they would be set up here
	*/

	/* Must be set (as character or pixel coordinates, as
	   applicable):
	*/
	GetClientRect(wndMain, &rect);
	SCREENWIDTH = rect.right;
	SCREENHEIGHT = rect.bottom;
#ifdef UNDER_CE
	SCREENHEIGHT -= MENU_HEIGHT;
#endif

	/* Must be set: */
	hugo_settextwindow(1, 1, SCREENWIDTH/FIXEDCHARWIDTH, SCREENHEIGHT/FIXEDLINEHEIGHT);
}


// For FlushBuffer()
static char flushing_window = false;

void hugo_settextwindow(int left, int top, int right, int bottom)
{
/* Again, coords. are passed as text coordinates with the top corner (1, 1) */

	if (client_needs_updating) UpdateClient(false);

	if (physical_windowleft || physical_windowtop ||
		physical_windowright != client_width-1 ||
		physical_windowbottom != client_height-1)
	{
		flushing_window = true;
	}
	FlushBuffer();
	flushing_window = false;
#ifdef SPEECH_ENABLED
	FlushSpeechBuffer();
#endif

	/* Must be set (as pixel coordinates): */
	physical_windowleft = (left-1)*FIXEDCHARWIDTH;
	physical_windowtop = (top-1)*FIXEDLINEHEIGHT;
	// These right/bottom calculations (i.e., no -1) are due
	// to the way the coordinate system works in Windows (as
	// opposed to something like BeOS, where a coordinate is
	// in the center of the pixel):
	physical_windowright = right*FIXEDCHARWIDTH;//-1;
	physical_windowbottom = bottom*FIXEDLINEHEIGHT;//-1;

	/* Correct for full-width windows where the right border would
	   otherwise be clipped to a multiple of charwidth, leaving a sliver
	   of the former window at the righthand side.
	*/
	if (right>=SCREENWIDTH/FIXEDCHARWIDTH)
		physical_windowright = client_width-1;
	if (bottom>=SCREENHEIGHT/FIXEDLINEHEIGHT)
		physical_windowbottom = client_height-1;

	physical_windowwidth = physical_windowright-physical_windowleft+1;
	physical_windowheight = physical_windowbottom-physical_windowtop+1;

#ifdef UNDER_CE
	// Arbitrary boundaries for what are illegal or not-very-useful windows on the
	// PocketPC display:
	minimal_windowing = reg_MinimalWindows;
	if ((inwindow) &&
		(physical_windowheight > client_height/2 || physical_windowwidth < client_width/4))
	{
		illegal_window = true;
	}
	else
	{
		illegal_window = false;
	}
#endif
	ConstrainCursor();
}


void hugo_settextpos(int x, int y)
{
/* The top-left corner of the current active window is (1, 1).

   (In other words, if the screen is being windowed so that the top row
   of the window is row 4 on the screen, the (1, 1) refers to the 4th
   row on the screen, and (2, 1) refers to the 5th.)

   This function must also properly set currentline and currentpos (where
   currentline is a the current character line, and currentpos may be
   either in pixels or characters, depending on the measure being used).
*/

#ifdef SPEECH_ENABLED
	FlushSpeechBuffer();
#endif
	FlushBuffer();
	if (!reg_FastScrolling && scroll_offset) UpdateClient(false);

	/* Must be set: */
	currentline = y;
	currentpos = (x-1)*FIXEDCHARWIDTH;      /* Note:  zero-based */

	/* current_text_x/row are calculated assuming that the
	   character position (1, 1) is the pixel position (0, 0)
	*/
	current_text_x = physical_windowleft + currentpos;
	current_text_y = physical_windowtop + (y-1)*lineheight;

	ConstrainCursor();

	/* In case we're circumventing a linefeed */
	client_needs_updating = true;
}


#define MAX_FLUSH_BUFFER 512
static TCHAR flush_buffer[MAX_FLUSH_BUFFER] = _T("");
static int flush_x, flush_y, flush_len;
static char last_was_italic = false;

/* We use FlushBuffer() so that we can buffer multiple calls to hugo_print()
 * in order to save on more expensive TextOut() GUI calls.
 */
 
void FlushBuffer(void)
{
	if (flush_len)
	{
		flush_buffer[flush_len] = 0;

		if (last_was_italic)
			SetBkMode(dcMem, TRANSPARENT);
		else
			SetBkMode(dcMem, OPAQUE);

		TextOut(dcMem, flush_x, flush_y+scroll_offset, flush_buffer, flush_len);

#ifdef SPEECH_ENABLED
		// Option to speak status window text can be turned off
		if (!flushing_window || speech_speak_status || flush_y!=0)
		{
			SpeakString(flush_buffer);
		}
#endif

		/* This is all rather finicky stuff.  At present, any
		   PROP_FONT is being handled the same, to make sure that
		   we don't overwrite anything.

		   (Change:  PROP_FONT is commented out, because it's
		   not guaranteed that a given fixed-width font won't
		   lean into any adjacent cell if it's, say, italic.)
		*/
		if (//currentfont & PROP_FONT &&
			(currentfont & BOLD_FONT ||
			currentfont & ITALIC_FONT ||
			currentfont & UNDERLINE_FONT))
		{
			last_was_italic = true;
		}
		else
			last_was_italic = false;

		client_needs_updating = true;
		flush_len = 0;
	}
}


#ifdef SPEECH_ENABLED
// A little bit of hocus pocus because this layer of the engine isn't
// really aware what constitutes a paragraph, and multiple short lines
// can run together when spoken.
static int last_current_text_x = 0;
#endif
static char tweaking_lineheight = 0;

void hugo_print(char *a)
{
	int i, len;

	/* Check Windows messages */
	MSG msg;
	if (PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
	{
		if (msg.message==WM_QUIT)
		{
			exit(msg.wParam);
		}
#ifndef NO_VIDEO
		// Check this here to avoid the one-letter-printing effect
		// immediately after playing video
		GetMessage(&msg, NULL, 0, 0);	// due to PM_NOREMOVE
		if (msg.message==WM_PAINT && just_stopped_video)
		{
			just_stopped_video = false;
		}
		else if (!TranslateAccelerator(wndMain, accelMain, &msg))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
#endif
	}

#ifdef UNDER_CE
	if (reg_MinimalWindows && illegal_window)
	{
		// If we're in an illegal window, trick the engine into
		// thinking it hasn't printed anything
		currentpos = 0;
		currentline = 1;
		full = 0;
		return;
	}
#endif

	len = strlen(a);

	for (i=0; i<len; i++)
	{
		/* If we've passed the bottom of the window, align to the bottom edge */
		if (current_text_y > physical_windowbottom-lineheight)
		{
			int temp_lh = lineheight;
			FlushBuffer();
			lineheight = current_text_y - physical_windowbottom+lineheight;
			current_text_y -= lineheight;
			override_update_client = true;	// don't want to see mini-scroll
			tweaking_lineheight = true;
			hugo_scrollwindowup();
			override_update_client = false;
			tweaking_lineheight = false;
			lineheight = temp_lh;
		}

		switch (a[i])
		{
			case '\n':
				FlushBuffer();
				current_text_y += lineheight;
				last_was_italic = false;
				newline_count++;
				break;
			case '\r':
				FlushBuffer();
				current_text_x = physical_windowleft;
				last_was_italic = false;
				break;
			default:
			{
#ifdef SPEECH_ENABLED
				// Non-wrapping lines (i.e., new paragraphs) get a pause
				last_current_text_x = current_text_x;
#endif
		                if (!inwindow) newline_count = 0;

				if (flush_len==0)
				{
					flush_x = current_text_x;
					flush_y = current_text_y;
				}

				/* First of all, convert Latin-1 characters if we're
				   using an OEM/DOS charset
				*/
				if (((currentfont&PROP_FONT)?prop_charset:fixed_charset==OEM_CHARSET)
					&& reg_AutoTranslateLatin1)
				{
					CharToOemBuff(a+i, (char *)(&flush_buffer[flush_len++]), 1);
				}
				else
				{
#ifdef UNICODE
					if ((unsigned)a[i] > 127)
					{
						char ch[2];
						WCHAR wc[2];

						ch[0] = a[i];
						ch[1] = 0;
						MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
							ch, 1, wc, 1);

						flush_buffer[flush_len++] = wc[0];

					}
					else
#endif
						flush_buffer[flush_len++] = a[i];
				}
				if (flush_len>=MAX_FLUSH_BUFFER-2) FlushBuffer();

				/* Increment the horizontal screen position by
				   the character width
				*/
				current_text_x += hugo_charwidth((char)flush_buffer[flush_len-1]);
			}
		}

		/* If we've passed the bottom of the window, align to the bottom edge */
		if (current_text_y > physical_windowbottom-lineheight)
		{
			int temp_lh = lineheight;
			lineheight = current_text_y - physical_windowbottom+lineheight;
			current_text_y -= lineheight;
			override_update_client = true;	// don't want to see mini-scroll
			tweaking_lineheight = true;
			// hugo_scrollwindowup() will increment this:
			newline_count--;
			hugo_scrollwindowup();
			override_update_client = false;
			tweaking_lineheight = false;
			lineheight = temp_lh;
		}
	}
}


void hugo_scrollwindowup()      /* one "text" line */
{
	static int old_client_width, old_client_height;
	int source_x, source_y, dest_x, dest_y, width, height;
	HGDIOBJ penNew, brushNew, penDefault, brushDefault;
#if defined (SUPER_SMOOTH_SCROLLING)
	HDC dcMain;
#endif

	newline_count++;
#ifdef UNDER_CE
	if (newline_count > 1 && reg_NoBlankLines)
		return;
#endif

#ifdef SPEECH_ENABLED
	if (last_current_text_x < physical_windowwidth-prop_charwidth*2)
	{
		SPEECH_PAUSE(500);
		last_current_text_x = false;
	}
#endif

#ifdef USE_TEXTBUFFER
	if (lineheight >= ((currentfont&PROP_FONT)?prop_lineheight:FIXEDLINEHEIGHT)/2)
		TB_Scroll();
#endif

	penNew = CreatePen(PS_SOLID, 1, current_back_color);
	brushNew = CreateSolidBrush(current_back_color);
	penDefault = SelectObject(dcMem, penNew);
	brushDefault = SelectObject(dcMem, brushNew);

	/* If not in a window, and fast scrolling is active, we
	   can, instead of physically scrolling the screen, simply
	   shift the vertical position by which the current "frame"
	   moves
	*/
	if (!inwindow && reg_FastScrolling)
	{
		scroll_offset_fill = hugo_color(default_bgcolor);
		scroll_offset+=lineheight;

	/* Erase the leading/next line-plus in the scroll area */
		Rectangle(dcMem, 0, client_height+scroll_offset-1,
			client_width, client_height+scroll_offset+lineheight*3);

		goto RestoreGDIObjects;
	}

	/* Basically just copies a hunk of screen to an upward-
	   shifted position and fills in the screen behind/below it
	*/
	source_x = physical_windowleft;
	source_y = physical_windowtop+lineheight;
	dest_x = physical_windowleft;
	dest_y = physical_windowtop;
	width = physical_windowwidth;
	height = physical_windowheight;

#if defined (SUPER_SMOOTH_SCROLLING)
	// Scrolling a single pixel upwards is a good idea, but unless you've
	// got a fast computer, it doesn't seem to work
	dcMain = GetDC(wndMain);
	source_y = dest_y + SUPER_SMOOTH_SCROLLING;
	while (source_y <= physical_windowtop+lineheight)
	{
		BitBlt( dcMain,			/* dest. device context */
			dest_x, dest_y, width, height,
			dcMem,			/* source DC */
			source_x, source_y,
			SRCCOPY );
		source_y += SUPER_SMOOTH_SCROLLING;
	}
#endif
	BitBlt( dcMem,                          /* dest. device context */
		dest_x, dest_y, width, height,
		dcMem,                          /* source DC */
		source_x, source_y,
		SRCCOPY );

	Rectangle(dcMem, physical_windowleft,
			 physical_windowbottom-lineheight,
			 physical_windowright,
			 physical_windowbottom);

	/* To make sure we don't leave traces of unscrolled text/pixels below
	   a window, e.g., the statusline, if the line height changes
	*/
	if (!inwindow && physical_windowbottom < dest_y)
		Rectangle(dcMem, 0, physical_windowbottom, client_width, dest_y-1);

	/* If fast scrolling is not in effect, then update the client window
	   after every scrolled line
	*/
#if !defined (SUPER_SMOOTH_SCROLLING)
	post_picture_update = false;
	UpdateClient(true);
#else
	ReleaseDC(wndMain, dcMain);
#endif

RestoreGDIObjects:

	SelectObject(dcMem, penDefault);
	SelectObject(dcMem, brushDefault);
	DeleteObject(penNew);
	DeleteObject(brushNew);

	/* Clip the off-screen (unscrolled) portion of the bitmap in the
	   event that we've resized it offscreen:  we don't want old text
	   to show up later when we enlarge the window again
	*/
	if ((!inwindow) &&
		old_client_width!=client_width ||
		old_client_height!=client_height)
	{
		ClipVisibleClient();
	}
	old_client_width = client_width;
	old_client_height = client_height;
}


void hugo_font(int f)
{
	HFONT fontOld;

	if (f==last_hugo_font) return;
	last_hugo_font = f;

	FlushBuffer();

	/* Match the new font */
	MatchFont(f);
	/* Select it */
	fontOld = SelectObject(dcMem, fontCurrent);
	/* Delete the replaced font */
	DeleteObject(fontOld);

	/* Reset charwidth and lineheight */
	if (f & PROP_FONT)
	{
		charwidth = prop_charwidth;
		lineheight = prop_lineheight;
	}
	else
	{
		ResetCharDimensions(f);  // FIXEDCHARWIDTH may change for bold/italic

		charwidth = FIXEDCHARWIDTH;
		lineheight = FIXEDLINEHEIGHT;
	}

	// May cause display problems with [MORE] prompt if the proportional
	// font is shorter than the fixed font
	if (lineheight < FIXEDLINEHEIGHT)
		lineheight = FIXEDLINEHEIGHT;

	/* Resize the caret to the (new) line height */
	caret_height = lineheight;
}


void hugo_settextcolor(int c)           /* foreground (print) color */
{
	FlushBuffer();

#ifdef UNDER_CE
	if (!reg_UseColors)
	{
		if (c!=18)		// DEF_SLFCOLOR
			c = 16;		// DEF_FCOLOR
	}
#endif
	current_text_color = hugo_color(c);
	SetTextColor(dcMem, current_text_color);
}

void hugo_setbackcolor(int c)           /* background color */
{
	FlushBuffer();

#ifdef UNDER_CE
	if (!reg_UseColors)
	{
		if (current_text_color==hugo_color(18))  // DEF_SLFCOLOR
			c = 19;		// DEF_SLBGCOLOR
		else
			c = 17;		// DEF_BGCOLOR
	}
#endif
	current_back_color = hugo_color(c);
	SetBkColor(dcMem, current_back_color);
}

int hugo_color(int c)
{
	static char resetting_colors = false;

	/* Sanity check; older versions of the color system had
	   reg_* color values representing "Hugo colors", i.e., all
	   less than 16, which when read from the registry post-
	   upgrade would result in all (nearly) black colors.
	*/
	if (reg_DefForeground <= 20 && reg_DefBackground <= 20 &&
		reg_DefSLForeground <= 20 && reg_DefSLBackground <= 20
		&& !resetting_colors)
	{
		resetting_colors = true;
		reg_DefForeground = hugo_color(DEF_FCOLOR);
		reg_DefBackground = hugo_color(DEF_BGCOLOR);
		reg_DefSLForeground = hugo_color(DEF_SLFCOLOR);
		reg_DefSLBackground = hugo_color(DEF_SLBGCOLOR);
		resetting_colors = false;
	}

	if (c==16)      c = reg_DefForeground;	 /* default foreground */
	else if (c==17) c = reg_DefBackground;	 /* default background */
	else if (c==18) c = reg_DefSLForeground; /* statusline foreground */
	else if (c==19) c = reg_DefSLBackground; /* statusline background */
	else if (c==20) c = hugo_color(fcolor);	 /* match foreground */
	else

	switch (c)        /* 0x00bbggrr */
	{
		case 0:  c = 0x00000000; break; /* black         */
		case 1:  c = 0x007f0000; break; /* blue          */
		case 2:  c = 0x00007f00; break; /* green         */
		case 3:  c = 0x007f7f00; break; /* cyan          */
		case 4:  c = 0x0000007f; break; /* red           */
		case 5:  c = 0x007f007f; break; /* magenta       */
		case 6:  c = 0x00005f7f; break; /* brown         */
		case 7:  c = 0x00cfcfcf; break; /* white         */
		case 8:  c = 0x003f3f3f; break; /* dark gray     */
		case 9:  c = 0x00ff0000; break; /* light blue    */
		case 10: c = 0x0000ff00; break; /* light green   */
		case 11: c = 0x00ffff00; break; /* light cyan    */
		case 12: c = 0x000000ff; break; /* light red     */
		case 13: c = 0x00ff00ff; break; /* light magenta */
		case 14: c = 0x0000ffff; break; /* light yellow  */
		case 15: c = 0x00ffffff; break; /* bright white  */
	}

	return c;
}


/* CHARACTER AND TEXT MEASUREMENT */

int hugo_charwidth(char a)
{
	if (a==FORCED_SPACE) a = ' ';

	if ((unsigned char)a >= ' ') /* alphanumeric characters */
	{
		/* proportional */
		if (currentfont & PROP_FONT)
		{
			TCHAR b[2];
			SIZE size;
#ifdef UNICODE
			if ((unsigned)a > 127)
			{
				char ch[2];

				ch[0] = a;
				ch[1] = 0;
				MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
					ch, 1, b, 1);
			}
			else
#endif
				b[0] = a;

			GetTextExtentPoint32(dcMem, b, 1, &size);
			return size.cx;
		}

		/* fixed-width */
		else
			return FIXEDCHARWIDTH;
	}

	return 0;
}

int hugo_textwidth(char *a)
{
	int i, slen, len = 0;

	slen = (int)strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else
			len += hugo_charwidth(a[i]);
	}

	return len;
}

int hugo_strlen(char *a)
{
	int i, slen, len = 0;

	slen = (int)strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else len++;
	}

	return len;
}
