/* ex:set ts=4 sw=4:
 *
 * footnote.t -- simple footnote allocation library
 * written by david etherton (etherton@sdd.hp.com)
 * do whatever you want to with this (provided you respect the later copyright)
 * requires TADS 2.1 or later because of the use of the 'modify' feature.
 * 
 * footnotes are really easy to overuse, so be careful out there!
 * 
 * common sense says that anything in a footnote should be able
 * to be safely ignored by the player. 
 *
 * typical usage:
 * ...
 * #include <footnote.t>
 * ...
 * desk: ....
 *		noun = 'desk'
 *		adjective = 'fine' 'oak'
 *		sdesc = "fine oak desk"
 *		ldesc = {
 *			"The desk is stained a deep brown color, and has that wonderful
 *			rustic appeal of a Norman Rockwell painting. ";
 * 			footnote.new(	'In fact, this entire room looks like an explosion
 *							at a Normal Rockwell factory.'
 *						);
 * 		}
 * ...
 * >examine the desk
 * The desk is stained a deep brown color, and has that wonderful
 * rustic appeal of a Norman Rockwell painting.  [3]
 * >footnote 3
 * [3]. In fact, this entire room looks like an explosion at a
 * Norman Rockwell factory.
 *
 *	3/8/94	- Jeff Laing changed the following
 *	1.	Added the following copyright message
 *	2.	Rework object names to reduce possibility of collisions (and just
 *		because I didn't like the originals.
 *	3.	Make the function a method call to allow for easier overrides
 *	4.	Add standard introduction text (ala Adventions games)
 *	5.	Correct known bug with early 2.1 TADS with modifying classes without
 *		using "class" keyword.
 *	6.	Issue informative message if attempted with non-number
 *	7.	Fix the off-by-one bug that made the highest footnote duplicate itself 
 *	8.	Reading a footnote shouldn't cost a turn.
 *
 * This module was originally written by David Etherton (etherton@sdd.hp.com)
 * who placed it into the public domain with the phrase "do whatever you want
 * to with this."  All my changes are Copyright (c) 1994 Jeff Laing.
 * Permission to use any or all of this code in other TADS games is granted
 * provided credit is given in your "CREDITS" command for this code and that
 * you leave this copyright message in the source whenever distributed.
 * Please make all changes in a backward compatible manner (where possible)
 * and make them available for free (like David and I did ;-)
 */
footnoteVersion: versionTag
	id="$Id: footnote.t_v 1.6 1994/04/26 06:23:30 jeffl Exp jeffl $\n"
	author='David Etherton (etherton@sdd.hp.com)'
	func='footnote support'
;

/*
 * This object holds all information for the footnote library
 */
footnote: object
	highest = 0		/* highest footnote allocated so far */
	badTries = 0	/* number of times bad footnote number tried */
	text = []		/* accumulated footnote text */
	helpmsg = {
		"\bRead footnotes ";
		self.new('Footnote markers are numbers enclosed in square brackets.');
		" with the note command; e.g. type \"note 1.\"\b";
	}
	new(tx) = {
		local idx;

		/*
		 * see if the footnote has already been allocated.
		 */
		for (idx:=1; idx<=self.highest; idx++)
			if (tx = self.text[idx])	/* already seen? */
				break;					/* yes, leave loop */
	 
		/*
		 * allocate a new footnote if this is the first time it's been seen
		 */
		if (idx > self.highest) {
			++self.highest;			/* get next note number */
			self.text += [tx];		/* save text in a list */
		}
	 
		/*
		 * display the reference number in brackets.
		 */
		"["; say(idx); "]";
	}
;
 
/*
 * footnote is a new verb we're declaring.  The verb and sdesc fields are
 * pretty standard; the former supplies the actual words the players can
 * type to get this verb, and the latter is used by the parser itself
 * occasionally.  action(actor) defines what happens when the verb is
 * supplied with no direct or indirect objects; in this case, we want
 * to inform the player of the proper usage.  Finally, doAction defines
 * what method in the direct object (in this case, the generic number
 * object) gets called to handle this verb; in this case, 'doFootnote'
 * will be the method which is called.
 */
footnoteVerb: deepverb
	verb = 'footnote' 'note'
	sdesc = "footnote"
	doAction = 'Footnote'
	action(actor) = (footnote.helpmsg)
;

/*
 * basicNumObj is an object supplied by TADS in adv.t which we are
 * modifying to include support for new footnotes.  The
 * verDoFootnote method indicates that this object knows how to
 * handle the footnote verb; therefore, 'footnote lamp' would
 * produce a parser error since a lamp would probably not inherit
 * from basicNumObj and as such would not define a verDoFootnote
 * method.  On the other hand, 'footnote 4' would cause
 * basicNumObj (actually, numObj, which directly inherits from it)
 * to receive a verDoFootnote message--which it knows how to handle!
 * Since verDoFootnote produces no output, the parser will automatically
 * call the doFootnote method to complete processing.  The doFootnote
 * method checks to see if the footnote is in range; if it is not
 * in range, it prints an error message.  If the user has obviously
 * been trying a lot of bad footnote numbers, it also offers a
 * suggestion to cease and desist.  On the other hand, if the
 * reference is valid, we display the footnote number along with
 * its associated text.  In either case, we do nothing special
 * besides return to the parser, so the footnote is counted as
 * a normal turn.  
 */
modify class basicNumObj
	verDoFootnote(actor) = {}	/* always allowed */
	doFootnote(actor) = {
		/*
		 * footnote out of range?
		 */
		if (self.value < 1 or self.value > footnote.highest) {

			/*
			 * tried too often? - barf out loud
			 */
			if (++footnote.badTries >= 5)
				"Footnotes aren't assigned until they are encountered in the
				game.  Repeatedly guessing footnotes will accomplish nothing.";
			else
				"No such footnote.";
			}
		else {	/* display the footnote reference & text */
			"["; say(self.value); "].\ ";
			say(footnote.text[self.value]);
			"\n";
		}

		abort;
	}
;

/*
 * setup a more meaningful message if they don't specify a number
 */
modify class thing
	verDoFootnote(a) = (footnote.helpmsg)
;
