#pragma C-

/************************/
/*  The Standard Verbs  */
/************************/

/*
 *  deepverb: object
 *
 *  A "verb object" that is referenced by the parser when the player
 *  uses an associated vocabulary word.  A deepverb contains both
 *  the vocabulary of the verb and a description of available syntax.
 *  The verb property lists the verb vocabulary words;
 *  one word (such as 'take') or a pair (such as 'pick up')
 *  can be used.  In the latter case, the second word must be a
 *  preposition, and may move to the end of the sentence in a player's
 *  command, as in "pick it up."  The action(actor)
 *  method specifies what happens when the verb is used without any
 *  objects; its absence specifies that the verb cannot be used without
 *  an object.  The doAction specifies the root of the message
 *  names (in single quotes) sent to the direct object when the verb
 *  is used with a direct object; its absence means that a single object
 *  is not allowed.  Likewise, the ioAction(preposition)
 *  specifies the root of the message name sent to the direct and
 *  indirect objects when the verb is used with both a direct and
 *  indirect object; its absence means that this syntax is illegal.
 *  Several ioAction properties may be present:  one for each
 *  preposition that can be used with an indirect object with the verb.
 *  
 *  The validDo(actor, object, seqno) method returns true
 *  if the indicated object is valid as a direct object for this actor.
 *  The validIo(actor, object, seqno) method does likewise
 *  for indirect objects.  The seqno parameter is a "sequence
 *  number," starting with 1 for the first object tried for a given
 *  verb, 2 for the second, and so forth; this parameter is normally
 *  ignored, but can be used for some special purposes.  For example,
 *  askVerb does not distinguish between objects matching vocabulary
 *  words, and therefore accepts only the first from a set of ambiguous
 *  objects.  These methods do not normally need to be changed; by
 *  default, they return true if the object is accessible to the
 *  actor.
 *  
 *  The doDefault(actor, prep, indirectObject) and
 *  ioDefault(actor, prep) methods return a list of the
 *  default direct and indirect objects, respectively.  These lists
 *  are used for determining which objects are meant by "all" and which
 *  should be used when the player command is missing an object.  These
 *  normally return a list of all objects that are applicable to the
 *  current command.
 *  
 *  The validDoList(actor, prep, indirectObject) and
 *  validIoList(actor, prep, directObject) methods return
 *  a list of all of the objects for which validDo would be true.
 *  Remember to include floating objects, which are generally
 *  accessible.  Note that the objects returned by this list will
 *  still be submitted by the parser to validDo, so it's okay for
 *  this routine to return too many objects.  In fact, this
 *  routine is entirely unnecessary; if you omit it altogether, or
 *  make it return nil, the parser will simply submit every
 *  object matching the player's vocabulary words to validDo.
 *  The reason to provide this method is that it can significantly
 *  improve parsing speed when the game has lots of objects that
 *  all have the same vocabulary word, because it cuts down on the
 *  number of times that validDo has to be called (each call
 *  to validDo is fairly time-consuming).
 */

class deepverb: object                // A deep-structure verb.
    validDo( actor, obj, seqno ) =
    {
    	return( obj.isReachable( actor ));
    }
    validDoList(actor, prep, iobj) =
    {
	local ret;
	local loc;

	loc := actor.location;
	while (loc.location) loc := loc.location;
	ret := visibleList(actor) + visibleList(loc)
	       + global.floatingList;
	return(ret);
    }
    validIo( actor, obj, seqno ) =
    {
    	return( obj.isReachable( actor ));
    }
    validIoList(actor, prep, dobj) = (self.validDoList(actor, prep, dobj))
    doDefault( actor, prep, io ) =
    {
        return( actor.contents + actor.location.contents );
    }
    ioDefault( actor, prep ) =
    {
        return( actor.contents + actor.location.contents );
    }
;

/*
   darkVerb - a verb that can be used in the dark.  travelVerb's
   are all dark verbs, as are sysverb's (quit, save, etc.).
   In addition, certain special verbs are usable in the dark:  for
   example, you can drop objects you are carrying, and you can turn
   on light sources you are carrying. 
*/

class darkVerb: deepverb
   isDarkVerb = true
;

//ASKME
askVerb: deepverb
    verb = 'ask'
    sdesc = "ask"
    prepDefault = aboutPrep
    ioAction( aboutPrep ) = 'AskAbout'
    validIo( actor, obj, seqno ) = { return( seqno = 1 ); }
    validIoList(actor, prep, dobj) = (nil)
;

//ATTME
attachVerb: deepverb
    verb = 'attach' 'connect'
    sdesc = "attach"
    prepDefault = toPrep
    ioAction( toPrep ) = 'AttachTo'
;

//ATKME
attackVerb: deepverb
    verb = 'attack' 'kill' 'hit'
    sdesc = "attack"
    prepDefault = withPrep
    ioAction( withPrep ) = 'AttackWith'
;

//BOAME
boardVerb: deepverb
    verb = 'get in' 'get into' 'board' 'get on'
    sdesc = "get on"
    doAction = 'Board'
;

//BREME
breakVerb: deepverb
    verb = 'break' 'ruin' 'destroy'
    sdesc = "break"
    doAction = 'Break'
;

//CENME
centerVerb: deepverb
    verb = 'center'
    sdesc = "center"
    doAction = 'Center'
;

//CLEME
cleanVerb: deepverb
    verb = 'clean'
    sdesc = "clean"
    ioAction( withPrep ) = 'CleanWith'
    doAction = 'Clean'
;

//CLIME
climbVerb: deepverb
    verb = 'climb'
    sdesc = "climb"
    doAction = 'Climb'
;

//CLOME
closeVerb: deepverb
    verb = 'close'
    sdesc = "close"
    doAction = 'Close'
;

//DETME
detachVerb: deepverb
    verb = 'detach' 'disconnect'
    prepDefault = fromPrep
    ioAction( fromPrep ) = 'DetachFrom'
    doAction = 'Detach'
    sdesc = "detach"
;

//DIGME
digVerb: deepverb
    verb = 'dig' 'dig in'
    sdesc = "dig in"
    prepDefault = withPrep
    ioAction( withPrep ) = 'DigWith'
;

//DRIME
drinkVerb: deepverb
    verb = 'drink'
    sdesc = "drink"
    doAction = 'Drink'
;

//DROME
dropVerb: deepverb, darkVerb
    verb = 'drop' 'put down'
    sdesc = "drop"
    ioAction( onPrep ) = 'PutOn'
    doAction = 'Drop'
    doDefault( actor, prep, io ) =
    {
        return( actor.contents );
    }
;

//EATME
eatVerb: deepverb
    verb = 'eat' 'consume'
    sdesc = "eat"
    doAction = 'Eat'
;

//FASME
fastenVerb: deepverb
    verb = 'fasten' 'buckle' 'buckle up'
    sdesc = "fasten"
    doAction = 'Fasten'
;

//FLIME
flipVerb: deepverb
    verb = 'flip'
    sdesc = "flip"
    doAction = 'Flip'
;

//FOLME
followVerb: deepverb
    sdesc = "follow"
    verb = 'follow'
    doAction = 'Follow'
;

//GOTME
getOutVerb: deepverb
    verb = 'get out' 'get outof' 'get off' 'get offof'
    sdesc = "get out of"
    doAction = 'Unboard'
    action(actor) = { askdo; }
    doDefault( actor, prep, io ) =
    {
        if ( actor.location and actor.location.location )
            return( [] + actor.location );
        else return( [] );
    }
;

//GIVME
giveVerb: deepverb
    verb = 'give' 'offer'
    sdesc = "give"
    prepDefault = toPrep
    ioAction( toPrep ) = 'GiveTo'
    doDefault( actor, prep, io ) =
    {
        return( actor.contents );
    }
;

//HELME
helloVerb: deepverb
    verb = 'hello' 'hi' 'greetings'
    action( actor ) =
    {
        "Nice weather we've been having.\n";
    }
;

//INSME
inspectVerb: deepverb
    verb = 'inspect' 'examine' 'look at' 'l at' 'x'
    sdesc = "inspect"
    doAction = 'Inspect'
    validDo( actor, obj, seqno ) =
    {
        return( obj.isVisible( actor ));
    }
;

//INVME
iVerb: deepverb
    verb = 'inventory' 'i'
    action( actor ) =
    {
        if (length( actor.contents ))
        {
            "%You% %have% "; listcont( actor ); ". ";
            listcontcont( actor );
        }
	else
            "%You% %are% empty-handed.\n";
    }
;
//JUMME
jumpVerb: deepverb
    verb = 'jump' 'jump over' 'jump off'
    sdesc = "jump"
    doAction = 'Jump'
    action(actor) = { "Wheeee!"; }
;

//LIEME
lieVerb: deepverb
    verb = 'lie' 'lie on' 'lie in' 'lie down' 'lie downon' 'lie downin'
    sdesc = "lie on"
    doAction = 'Lieon'
;

//LOCME
lockVerb: deepverb
    verb = 'lock'
    sdesc = "lock"
    ioAction( withPrep ) = 'LockWith'
    doAction = 'Lock'
    prepDefault = withPrep
;

//LOOME
lookVerb: deepverb
    verb = 'look' 'l' 'look around' 'l around'
    sdesc = "look"
    action( actor ) =
    {
        actor.location.lookAround( true );
    }
;

//LBEME
lookBehindVerb: deepverb
    verb = 'look behind' 'l behind'
    sdesc = "look behind"
    doAction = 'Lookbehind'
;

//LINME
lookInVerb: deepverb
    verb = 'look in' 'look on' 'l in' 'l on'
    sdesc = "look in"
    doAction = 'Lookin'
;

//LTHME
lookThruVerb: deepverb
    verb = 'look through' 'look thru' 'l through' 'l thru'
    sdesc = "look through"
    doAction = 'Lookthru'
;

//LUNME
lookUnderVerb: deepverb
    verb = 'look under' 'look beneath' 'l under' 'l beneath'
    sdesc = "look under"
    doAction = 'Lookunder'
;

//MOVME
moveVerb: deepverb
    verb = 'move'
    sdesc = "move"
    ioAction( withPrep ) = 'MoveWith'
    ioAction( toPrep ) = 'MoveTo'
    doAction = 'Move'
;

//MEAME
moveEVerb: deepverb
    verb = 'move east' 'move e' 'push east' 'push e'
    sdesc = "move east"
    doAction = 'MoveE'
;

//MNOME
moveNVerb: deepverb
    verb = 'move north' 'move n' 'push north' 'push n'
    sdesc = "move north"
    doAction = 'MoveN'
;
//MNEME
moveNEVerb: deepverb
    verb = 'move northeast' 'move ne' 'push northeast' 'push ne'
    sdesc = "move northeast"
    doAction = 'MoveNE'
;
//MNWME
moveNWVerb: deepverb
    verb = 'move northwest' 'move nw' 'push northwest' 'push nw'
    sdesc = "move northwest"
    doAction = 'MoveNW'
;
//MSOME
moveSVerb: deepverb
    verb = 'move south' 'move s' 'push south' 'push s'
    sdesc = "move south"
    doAction = 'MoveS'
;
//MSEME
moveSEVerb: deepverb
    verb = 'move southeast' 'move se' 'push southeast' 'push se'
    sdesc = "move southeast"
    doAction = 'MoveSE'
;
//MSWME
moveSWVerb: deepverb
    verb = 'move southwest' 'move sw' 'push southwest' 'push sw'
    sdesc = "move southwest"
    doAction = 'MoveSW'
;
//MWEME
moveWVerb: deepverb
    verb = 'move west' 'move w' 'push west' 'push w'
    sdesc = "move west"
    doAction = 'MoveW'
;

//OPEME
openVerb: deepverb
    verb = 'open'
    sdesc = "open"
    doAction = 'Open'
;

//PLUME
plugVerb: deepverb
    verb = 'plug'
    sdesc = "plug"
    prepDefault = inPrep
    ioAction( inPrep ) = 'PlugIn'
;

//POKME
pokeVerb: deepverb
    verb = 'poke' 'jab'
    sdesc = "poke"
    doAction = 'Poke'
;

//PULME
pullVerb: deepverb
    verb = 'pull'
    sdesc = "pull"
    doAction = 'Pull'
;

//PUSME
pushVerb: deepverb
    verb = 'push' 'press'
    sdesc = "push"
    doAction = 'Push'
;

//PUTME
putVerb: deepverb
    verb = 'put' 'place'
    sdesc = "put"
    prepDefault = inPrep
    ioAction( inPrep ) = 'PutIn'
    ioAction( onPrep ) = 'PutOn'
    doDefault( actor, prep, io ) =
    {
        return( takeVerb.doDefault( actor, prep, io ) + actor.contents );
    }
;

//REAME
readVerb: deepverb
    verb = 'read'
    sdesc = "read"
    doAction = 'Read'
;

//REMME
removeVerb: deepverb
    verb = 'take off'
    sdesc = "take off"
    doAction = 'Unwear'
    ioAction( fromPrep ) = 'RemoveFrom'
;

//SAYME
sayVerb: deepverb
    verb = 'say'
    sdesc = "say"
    doAction = 'Say'
;

//SCRME
screwVerb: deepverb
    verb = 'screw'
    sdesc = "screw"
    ioAction( withPrep ) = 'ScrewWith'
    doAction = 'Screw'
;

//SEAME
searchVerb: deepverb
    verb = 'search'
    sdesc = "search"
    doAction = 'Search'
;

//SHOME
showVerb: deepverb
    verb = 'show'
    sdesc = "show"
    prepDefault = toPrep
    ioAction( toPrep ) = 'ShowTo'
    doDefault( actor, prep, io ) =
    {
        return( actor.contents );
    }
;

//SITME
sitVerb: deepverb
    verb = 'sit on' 'sit in' 'sit' 'sit down' 'sit downin' 'sit downon'
    sdesc = "sit on"
    doAction = 'Siton'
;

//STAME
standVerb: deepverb
    verb = 'stand' 'stand up' 'get up'
    sdesc = "stand"
    action( actor ) =
    {
        if ( actor.location=nil or actor.location.location = nil )
            "%You're% already standing! ";
        else
        {
	    actor.location.doUnboard( actor );
        }
    }
;

//SONME
standOnVerb: deepverb
    verb = 'stand on'
    sdesc = "stand on"
    doAction = 'Standon'
;

//SWIME
switchVerb: deepverb
    verb = 'switch'
    sdesc = "switch"
    doAction = 'Switch'
;

//TAKME
takeVerb: deepverb                   // This object defines how to take things
    verb = 'take' 'pick up' 'get' 'remove'
    sdesc = "take"
    ioAction( offPrep ) = 'TakeOff'
    ioAction( outPrep ) = 'TakeOut'
    ioAction( fromPrep ) = 'TakeOut'
    ioAction( inPrep ) = 'TakeOut'
    ioAction( onPrep ) = 'TakeOff'
    doAction = 'Take'
    doDefault( actor, prep, io ) =
    {
        local ret, rem, cur, rem2, cur2, tot, i, tot2, j;
	
	ret := [];
        
	/*
	 *   For "take all out/off of <iobj>", return the (non-fixed)
	 *   contents of the indirect object.  Same goes for "take all in
	 *   <iobj>", "take all on <iobj>", and "take all from <iobj>".
	 */
	if (( prep=outPrep or prep=offPrep or prep=inPrep or prep=onPrep
	 or prep=fromPrep ) and io<>nil )
	{
	    rem := io.contents;
	    i := 1;
	    tot := length( rem );
	    while ( i <= tot )
	    {
	        cur := rem[i];
	        if (not cur.isfixed and self.validDo(actor, cur, i))
		    ret += cur;
		++i;
	    }
            return( ret );
	}

        /*
	 *   In the general case, return everything that's not fixed
	 *   in the actor's location, or everything inside fixed containers
	 *   that isn't itself fixed.
	 */
        rem := actor.location.contents;
	tot := length( rem );
	i := 1;
        while ( i <= tot )
        {
	    cur := rem[i];
            if ( cur.isfixed )
            {
                if ((( cur.isopenable and cur.isopen ) or
                  ( not cur.isopenable )) and ( not cur.isactor ))
                {
                    rem2 := cur.contents;
		    tot2 := length( rem2 );
		    j := 1;
                    while ( j <= tot2 )
                    {
		        cur2 := rem2[j];
                        if ( not cur2.isfixed and not cur2.notakeall )
			{
			    ret := ret + cur2;
			}
                        j := j + 1;
                    }
                }
            }
            else if ( not cur.notakeall )
	    {
	        ret := ret + cur;
	    }

	    i := i + 1;            
        }
        return( ret );
    }
;

//TELME
tellVerb: deepverb
    verb = 'tell'
    sdesc = "tell"
    prepDefault = aboutPrep
    ioAction( aboutPrep ) = 'TellAbout'
    validIo( actor, obj, seqno ) = { return( seqno = 1 ); }
    validIoList(actor, prep, dobj) = (nil)
    ioDefault( actor, prep ) =
    {
        if (prep = aboutPrep)
	    return([]);
	else
	    return(actor.contents + actor.location.contents);
    }
;

//THRME
throwVerb: deepverb
    verb = 'throw' 'toss'
    sdesc = "throw"
    prepDefault = atPrep
    ioAction( atPrep ) = 'ThrowAt'
    ioAction( toPrep ) = 'ThrowTo'
;

//TOUME
touchVerb: deepverb
    verb = 'touch'
    sdesc = "touch"
    doAction = 'Touch'
;

//TOFME
turnOffVerb: deepverb
    verb = 'turn off' 'deactiv' 'switch off'
    sdesc = "turn off"
    doAction = 'Turnoff'
;

//TONME
turnOnVerb: deepverb, darkVerb
    verb = 'activate' 'turn on' 'switch on'
    sdesc = "turn on"
    doAction = 'Turnon'
;

//TURME
turnVerb: deepverb
    verb = 'turn' 'rotate' 'twist'
    sdesc = "turn"
    ioAction( toPrep ) = 'TurnTo'
    ioAction( withPrep ) = 'TurnWith'
    doAction = 'Turn'
;

//TYPME
typeVerb: deepverb
    verb = 'type'
    sdesc = "type"
    prepDefault = onPrep
    ioAction( onPrep ) = 'TypeOn'
;

//UNFME
unfastenVerb: deepverb
    verb = 'unfasten' 'unbuckle'
    sdesc = "unfasten"
    doAction = 'Unfasten'
;

//UNLME
unlockVerb: deepverb
    verb = 'unlock'
    sdesc = "unlock"
    ioAction( withPrep ) = 'UnlockWith'
    doAction = 'Unlock'
    prepDefault = withPrep
;

//UNPME
unplugVerb: deepverb
    verb = 'unplug'
    sdesc = "unplug"
    ioAction( fromPrep ) = 'UnplugFrom'
    doAction = 'Unplug'
;

//UNSME
unscrewVerb: deepverb
    verb = 'unscrew'
    sdesc = "unscrew"
    ioAction( withPrep ) = 'UnscrewWith'
    doAction = 'Unscrew'
;

//WEAME
wearVerb: deepverb
    verb = 'wear' 'put on'
    sdesc = "wear"
    doAction = 'Wear'
;

//YELME
yellVerb: deepverb
    verb = 'yell' 'shout' 'yell at' 'shout at'
    action( actor ) =
    {
        "%Your% throat is a bit sore now. ";
    }
;

/*****************/
/*  System Verbs */
/*****************/

/*
 *   sysverb:  A system verb.  Verbs of this class are special verbs that
 *   can be executed without certain normal validations.  For example,
 *   a system verb can be executed in a dark room.  System verbs are
 *   for operations such as saving, restoring, and quitting, which are
 *   not really part of the game.
 */
class sysverb: deepverb, darkVerb
    issysverb = true
;

//AGAME
againVerb: darkVerb         // Required verb:  repeats last command.  No
                            // action routines are necessary; this one's
                            // handled internally by the parser.
    verb = 'again' 'g'
;

//DEBME
debugVerb: sysverb
    verb = 'debug'
    enterDebugger(actor) =
    {
	if (debugTrace())
	    "You can't think this game has any bugs left in it... ";
    }
    action( actor ) =
    {
	self.enterDebugger(actor);
	abort;
    }
;

//QUIME
quitVerb: sysverb
    verb = 'quit' 'q'
    quitGame(actor) =
    {
        scoreRank();
        "\bDo you really want to quit? (YES or NO) > ";
        if ( yorn() )
        {
	    "\b";
            terminate();    // allow user good-bye message
	    quit();
        }
        else
        {
            "Okay. ";
        }
    }
    action( actor ) =
    {
	self.quitGame(actor);
	abort;
    }
;

//RSTME
restartVerb: sysverb
    verb = 'restart'
    restartGame(actor) =
    {
        local yesno;
        while ( true )
        {
            "Are you sure you want to start over? (YES or NO) > ";
            yesno := yorn();
            if ( yesno = 1 )
            {
                "\n";
		scoreStatus();
                restart(initRestart, global.initRestartParam);
                break;
            }
            else if ( yesno = 0 )
            {
                "\nOkay.\n";
		break;
            }
        }
    }
    action( actor ) =
    {
	self.restartGame(actor);
	abort;
    }
;

//RTOME
restoreVerb: sysverb
    verb = 'restore'
    sdesc = "restore"
    doAction = 'Restore'
    restoreGame(actor) =
    {
        local savefile;
	
	savefile := askfile( 'File to restore game from' );
	if ( savefile = nil or savefile = '' )
        {
	    "Failed. ";
            return nil;
	}
	else if (restore( savefile ))
        {
	    "Restore failed. ";
            return nil;
	}
	else
	{
	    scoreStatus();
	    "Restored.\b";
	    Me.location.lookAround(true);
	    return true;
	}
    }
    action( actor ) =
    {
	self.restoreGame(actor);
	abort;
    }
;
//SAVME
saveVerb: sysverb
    verb = 'save'
    sdesc = "save"
    doAction = 'Save'
    saveGame(actor) =
    {
        local savefile;
	
	savefile := askfile( 'File to save game in' );
	if ( savefile = nil or savefile = '' )
        {
	    "Failed. ";
            return nil;
	}
	else if (save( savefile ))
        {
	    "Saved failed. ";
            return nil;
	}
	else
	{
	    "Saved. ";
            return true;
	}
    }
    action( actor ) =
    {
	self.saveGame(actor);
	abort;
    }
;
//SCOME
scoreVerb: sysverb
    verb = 'score' 'status'
    showScore(actor) =
    {
        scoreRank();
    }
    action( actor ) =
    {
	self.showScore(actor);
	abort;
    }
;
//SCRME
scriptVerb: sysverb
    verb = 'script'
    doAction = 'Script'
    startScripting(actor) =
    {
        local scriptfile;
	
	scriptfile := askfile( 'File to write transcript to' );
	if ( scriptfile = nil or scriptfile = '' )
	    "Failed. ";
	else
	{
	    logging( scriptfile );
	    "All text will now be saved to the script file.
            Type UNSCRIPT at any time to discontinue scripting.";
	}
    }
    action( actor ) =
    {
	self.startScripting(actor);
	abort;
    }
;

//TERME
terseVerb: sysverb
    verb = 'brief' 'terse'
    terseMode(actor) =
    {
        "Okay, now in TERSE mode.\n";
        global.verbose := nil;
    }
    action( actor ) =
    {
	self.terseMode(actor);
	abort;
    }
;

//UNDME
undoVerb: sysverb
    verb = 'undo'
    undoMove(actor) =
    {
	/* do TWO undo's - one for this 'undo', one for previous command */
	if (undo() and undo())
	{
	    "(Undoing one command)\b";
	    Me.location.lookAround(true);
	    scoreStatus();
	}
	else
	    "No more undo information is available. ";
    }
    action(actor) =
    {
	self.undoMove(actor);
	abort;
    }
;
//UNSME
unscriptVerb: sysverb
    verb = 'unscript'
    stopScripting(actor) =
    {
        logging( nil );
        "Script closed.\n";
    }
    action( actor ) =
    {
	self.stopScripting(actor);
        abort;
    }
;
//VERME
verboseVerb: sysverb
    verb = 'verbose'
    verboseMode(actor) =
    {
        "Okay, now in VERBOSE mode.\n";
        global.verbose := true;
	Me.location.lookAround( true );
    }
    action( actor ) =
    {
	self.verboseMode(actor);
	abort;
    }
;
//VRSME
versionVerb: sysverb
    verb = 'version'
    showVersion(actor) =
    {
        version.sdesc;
    }
    action( actor ) =
    {
	self.showVersion(actor);
        abort;
    }
;

//WAIME
waitVerb: darkVerb
    verb = 'wait' 'z'
    action( actor ) =
    {
        "Time passes...\n";
    }
;

/****************/
/* Travel Verbs */
/****************/

/*
 *   Travel verbs  - these verbs allow the player to move about.
 *   All travel verbs have the property isTravelVerb set true.
 */
class travelVerb: deepverb, darkVerb
    isTravelVerb = true
;

//DOWME
dVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'd' 'down' 'go down'
    travelDir( actor ) = { return( actor.location.down ); }
;

//EASME
eVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'e' 'east' 'go east'
    travelDir( actor ) = { return( actor.location.east ); }
;

//INNME
inVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'in' 'go in' 'enter'
    sdesc = "enter"
    doAction = 'Enter'
    travelDir( actor ) = { return( actor.location.in ); }
    ioAction(onPrep) = 'EnterOn'
    ioAction(inPrep) = 'EnterIn'
    ioAction(withPrep) = 'EnterWith'
;

//NORME
nVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'n' 'north' 'go north'
    travelDir( actor ) = { return( actor.location.north ); }
;

//NEAME
neVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'ne' 'northeast' 'go ne' 'go northeast'
    travelDir( actor ) = { return( actor.location.ne ); }
;

//NWEME
nwVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'nw' 'northwest' 'go nw' 'go northwest'
    travelDir( actor ) = { return( actor.location.nw ); }
;

//OUTME
outVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'out' 'go out' 'exit' 'leave'
    travelDir( actor ) = { return( actor.location.out ); }
;

//SOUME
sVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 's' 'south' 'go south'
    travelDir( actor ) = { return( actor.location.south ); }
;

//SEAME
seVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'se' 'southeast' 'go se' 'go southeast'
    travelDir( actor ) = { return( actor.location.se ); }
;

//SWEME
swVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'sw' 'southwest' 'go sw' 'go southwest'
    travelDir( actor ) = { return( actor.location.sw ); }
;

//UPPME
uVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'u' 'up' 'go up'
    travelDir( actor ) = { return( actor.location.up ); }
;

//WESME
wVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'w' 'west' 'go west'
    travelDir( actor ) = { return( actor.location.west ); }
;
