// epoc_console.cpp
//
// Copyright (c) 1997-1999 Symbian Ltd.  All rights reserved.
//
// Originally taken from the NetHack for Psion
// code, written by Duncan Booth
//
// Code adapted by Simon Quinn, 14/12/00
//
// Description: This is a wrapper for the Eikon environment to make
// use of the Eikon Console code, instead of the deficient E32 Console.
// This is run as an Active Object, separate from the main STDLIB code.
// Messages for printing, input etc are passed via the EPOC message passing
// mechanism.
// Code is written in a generic way so any STDLIB application can easily
// make use of it.
// To make use of this code just use the following statement:
//		iConsole = NewAdvConsole();		{Creates the console}
//
// Then just call as usual with iConsole->Write(), iConsole->Getch() etc.
//////////////////////////////////////////////////////////////////////////

#include "epoc_console.h"

#ifdef _DO_DEBUG_
	#define DO_LOGL8(arg) Log(_L8(arg));// iLogFile.Write(_L8(arg));
	#define DO_LOG(arg)  Log(arg);//iLogFile.Write(arg);
#else
	#define DO_LOG(arg) 
	#define DO_LOGL8(arg) 
#endif

enum TEikConClPanic
	{
	EEikConClPanicReadAlreadyOutstanding
	};

GLDEF_C void Panic(TEikConClPanic aPanic)
{
    User::Panic(_L("HGCONS"),aPanic);
}

enum
{
    EExit,
    ERead,
    EReadCancel,
    EWrite,
    ENewLine,
    ESetCursorPosAbs,
    ESetCursorPosRel,
    ESetCursorHeight,
    EClearScreen,
    EClearToEndOfLine,
	EClearChars,
	EScrollChars,
    ESetAttr,
    EKeyHit,
    ESetFont,
    EGetPointerPos,
	EScreenZoom,
	ESetTitle
};

//
// class CAdvConsMessager
//
CAdvConsMessager::CAdvConsMessager(CEikConsoleScreen* aScreen,RThread aParentThread)
: CActive(EActivePriorityIpcEventsHigh),
iParentThread(aParentThread)
{
    iScreen=aScreen;
}

CAdvConsMessager::~CAdvConsMessager()
{
    iParentThread.Close();
    delete iKeyQ;
}

void CAdvConsMessager::ConstructL(CAdvConsoleClient* aClient)
{
    iClient = aClient;
    iKeyQ=new(ELeave) CCirBuf<TKeyEvent>;
    iKeyQ->SetLengthL(40);	// buffer length, too high? too low?
    iKeyEvent=(&aClient->iKeyEvent);
    aClient->iThreadStatus=(&iStatus);
    aClient->iMessage=(&iMessage);
    aClient->iParam=(&iParam);
    aClient->iReplyStatus=(&iReplyStatus);
    aClient->iScreen=iScreen;
    CActiveScheduler::Add(this);
    iStatus=KRequestPending;
    SetActive();
}

void CAdvConsMessager::DoCancel()
{
}

void CAdvConsMessager::RunL()
{
    switch (iMessage)
    {
		case EExit:
			CBaActiveScheduler::Exit();
			break;
		case ENewLine:
			iScreen->Lf();
			break;
		case ERead:
			if (iReadStatus)
				Panic(EEikConClPanicReadAlreadyOutstanding);

			// Bring the cursor onto the visible part of the screen whenever we read some input.
			iScreen->DrawCursorInSight();
			iReadStatus=(TRequestStatus*)iParam;
			if (iKeyQ->Count()>0) // already a buffered event
				CompleteReadRequest();
			break;
		case EReadCancel:
			if (iReadStatus)
				iParentThread.RequestComplete(iReadStatus,KErrCancel);
			break;
		case EWrite:
			{
				//Before printing check if we are dealing with INVERSE characters.
				//INVERSE characters must be cleared before overwriting with other
				//INVERSE characters otherwise they become non-INVERSE!
				TInt qatt = iScreen->Att();
				if(qatt & ATT_INVERSE)
				{
					TInt x,y, templen;
					x = iScreen->WhereX();
					y = iScreen->WhereY();
					templen = (*(const TDesC*)iParam).Length();
          			iScreen->ClearChars(TRect(x, y, (x+templen), 1), ATT_INVERSE);
				}

				iScreen->Write(*(const TDesC*)iParam);
				break;
			}
		case ESetCursorPosAbs:
			iScreen->SetCursorPosAbs(*(const TPoint*)iParam);
			break;
		case ESetCursorPosRel:
			iScreen->SetCursorPosRel(*(const TPoint*)iParam);
			break;
		case ESetCursorHeight:
			iScreen->SetCursorHeight((TInt)iParam);
			break;
		case EClearScreen:
			iScreen->ClearScreen();
			break;
		case EClearToEndOfLine:
			iScreen->ClearToEndOfLine();
			break;
		case EClearChars:
			iScreen->ClearChars(*(const TRect*)iParam, 0);
			break;
		case EScrollChars:
		{
			TRect tr = *(const TRect*)iParam;
			iScreen->ScrollChars(tr, TPoint(0,-1));
			//Scrolling doesn't remove the bottom line, so clear the last line of the block
			iScreen->ClearChars(TRect(tr.iTl.iX, tr.iBr.iY - 1, tr.iBr.iX, tr.iBr.iY), 0);
			break;
		}
		case ESetAttr:
			iScreen->SetAtt((TInt)iParam);
			break;
		case EKeyHit:
		{
			// Returns the number of keys waiting in the buffer.
			TInt res = iKeyQ->Count();
			iParentThread.WriteL(iParam, TPckg<TInt>(res), 0);
			break;
		}
		case EGetPointerPos:
			iParentThread.WriteL(iParam, TPckgC<TPoint>(iPointer), 0);
			break;
		case EScreenZoom:
		{
			iZoomFactor = (iZoomFactor ? 0 : 1);

			iScreen->SetFontL(TFontSpec(_L("Courier"), (iZoomFactor ? 160:140)));
	
			TBuf<64> buffer;
			buffer.Format(_L("Screen size %dx%d"),
					  iScreen->ConsoleControl()->Size().iWidth/iScreen->ConsoleControl()->CharSize().iWidth,
					  iScreen->ConsoleControl()->Size().iHeight/iScreen->ConsoleControl()->CharSize().iHeight);
			CEikonEnv::Static()->InfoMsg(buffer);
			break;
		}
	}

    iStatus=KRequestPending;
    SetActive();
    iParentThread.RequestComplete(iReplyStatus,0);
}

void CAdvConsMessager::CompleteReadRequest()
{
    if (iReadStatus)
    { 
        iKeyQ->Remove(iKeyEvent);;
        iParentThread.RequestComplete(iReadStatus,0);
    }
}

void CAdvConsMessager::HandleKeyEvent(const TKeyEvent& aKeyEvent)
{
    TInt ret=iKeyQ->Add(&aKeyEvent);
    if (ret==0)
        CEikonEnv::Beep();
    if (iKeyQ->Count()==1) // client may be waiting on this key event
        CompleteReadRequest();
}

//
// class CAdvConsAppUi
//

CAdvConsAppUi::~CAdvConsAppUi()
{
    delete(iScreen);
    delete(iMessager);
}

void CAdvConsAppUi::ConstructL(const SCommandLine* aComLine)
{
    CEikAppUi::BaseConstructL(ENoAppResourceFile);

    iScreen=new(ELeave) CEikConsoleScreen;
    iScreen->ConstructL(*(aComLine->iTitle), 0);

	iScreen->SetFontL(TFontSpec(_L("Courier"), 140));

    iScreen->ConstructL(*(aComLine->iTitle), 0);

    iScreen->SetKeepCursorInSight(EFalse);

	iScreen->SetPureCRLF(ETrue);

    iControl=iScreen->ConsoleControl();
    iControl->SetFocus(ETrue,EDrawNow);
	iMessager=new(ELeave) CAdvConsMessager(iScreen,aComLine->iParentThread);
    iMessager->ConstructL(aComLine->iClient);
    RThread().SetPriority(EPriorityMore);
}

void CAdvConsAppUi::HandleForegroundEventL(TBool aForeground)
{
    if (aForeground)
        RThread().SetPriority(EPriorityMore);
}

void CAdvConsAppUi::SetAndDrawFocus(TBool aFocus)
{
    if (iControl)
        iControl->SetFocus(aFocus,EDrawNow);
}

TKeyResponse CAdvConsAppUi::HandleKeyEventL(const TKeyEvent& aKeyEvent,TEventCode aType)
{
#ifdef _DO_DEBUG_
//   	iMessager->Log(_L8("os_get_event called\r\n"));
#endif
	if (aType==EEventKey) 
	switch(aKeyEvent.iCode) {
		case NULL:
			break;
//		case EEikSidebarZoomInKey:
//			ChangeZoomL();
//			break;
//		case EEikSidebarZoomOutKey:
//			ChangeZoomL();
//			break;
		default:
			iMessager->HandleKeyEvent(aKeyEvent);
	}
	return EKeyWasConsumed;
}

void CConsEikonEnv::ConstructConsoleEnvironmentL(const SCommandLine* aComLine)
{
    ConstructL();
    CAdvConsAppUi* appUi=new(ELeave) CAdvConsAppUi;
    appUi->ConstructL(aComLine);
    CApaWindowGroupName* wgName=CApaWindowGroupName::NewLC(iWsSession);
    TPtrC caption=*(aComLine->iTitle);
    wgName->SetCaptionL(caption);
    wgName->SetRespondsToShutdownEvent(EFalse);
    wgName->SetRespondsToSwitchFilesEvent(EFalse);
    wgName->SetWindowGroupName(iRootWin);
    CleanupStack::PopAndDestroy(); // wgName
#if defined(__EPOC32__)
    RProcess().Rename(caption);
#endif
    RThread().Rename(caption);
}

TInt ConsoleClientStartFunction(TAny* aParam)
{
    const SCommandLine* comLine=(const SCommandLine*)aParam;
    TInt err=KErrNoMemory;
    CConsEikonEnv* coe=new CConsEikonEnv;
    if (coe)
        TRAP(err,coe->ConstructConsoleEnvironmentL(comLine));
    TRequestStatus* pS=(comLine->iStatus);
    comLine->iParentThread.RequestComplete(pS,err);
    if (!err)
        coe->ExecuteD();
    return(0);
}

//
// class CAdvConsoleClient
//

CAdvConsoleClient::~CAdvConsoleClient()
{
    if (iLogonStatus.Int()==KRequestPending && iReplyStatus)
		SendReceive(EExit,NULL);
// new addition
	if (iConverter)
		delete iConverter;

#ifdef _DO_DEBUG_
	iLogFile.Close();
#endif
	iFs.Close();
//

    iThread.Close();
}

CAdvConsoleClient::CAdvConsoleClient() : iCharSetWin1251(0)
{
}

const TInt KMaxHeapSize=0x1000*254; // chunks are a megabyte anyway

TInt CAdvConsoleClient::Create(const TDesC& aTitle,TSize aSize)
{ 
	TInt err;
    TRequestStatus status=KRequestPending;
    SCommandLine comLine;
    comLine.iStatus=(&status);
    comLine.iClient=this;
    comLine.iSize=aSize;
    comLine.iTitle=&aTitle;
    comLine.iFaceName = &iFaceName;
    comLine.iTwipSize = iTwipSize;
    TBuf<20> threadName;
    TInt num=0;
    do
    {
        threadName.Format(_L("UI%02d"),num++); // !! review the title
        err=iThread.Create(threadName,ConsoleClientStartFunction,KDefaultStackSize,KMinHeapSize,KMaxHeapSize,&comLine,EOwnerThread);
    } while(err==KErrAlreadyExists);
    if (!err)
    {
        iThread.Logon(iLogonStatus);
        comLine.iParentThread.Duplicate(iThread);
        iThread.Resume();
        User::WaitForRequest(status,iLogonStatus);
        err=status.Int();
    }
// new addition
	if (iFs.Connect() == KErrNone)
	{
#ifdef _DO_DEBUG_
		iLogFile.Replace(iFs, _L("tadslog.txt"), EFileWrite);
#endif

		TRAP(err, iConverter = CCnvCharacterSetConverter::NewL());
		if (err == KErrNone)
		{
			DO_LOGL8("Converter created\r\n");
			CArrayFix<CCnvCharacterSetConverter::SCharacterSet>	*theCharSets = NULL;

			TRAP(err, theCharSets = iConverter->CreateArrayOfCharacterSetsAvailableL(iFs));
			if (err == KErrNone)
			{
				DO_LOGL8("Array of charsets obtained\r\n");
				TInt	count = theCharSets->Count();
				
				for (TInt i = 0;i < count;i++)
				{
					if (theCharSets->At(i).Name().Find(_L("1251")) != KErrNotFound)
					{
						DO_LOGL8("1251 charset found\r\n");
						iCharSetWin1251 = theCharSets->At(i).Identifier();
						break;
					}
				}

				delete theCharSets;
			}
		}
	}
//
    return(err);
}

void CAdvConsoleClient::SendReceive(TInt aMessage,const TAny* aParam)
{
    if (iLogonStatus.Int()!=KRequestPending)
        User::Exit(KErrCancel);
    *iMessage=aMessage;
    *iParam=aParam;
    TRequestStatus replyStatus=KRequestPending;
    *iReplyStatus=(&replyStatus);
    TRequestStatus* pS=iThreadStatus;
    iThread.RequestComplete(pS,0);
    User::WaitForRequest(replyStatus,iLogonStatus);
}

void CAdvConsoleClient::Read(TRequestStatus& aStatus)
{
    aStatus=KRequestPending;
    SendReceive(ERead,&aStatus);
}

void CAdvConsoleClient::ReadCancel()
{
    SendReceive(EReadCancel,NULL);
}


void CAdvConsoleClient::Newline()
{
	SendReceive(ENewLine,NULL);
}

void CAdvConsoleClient::Write(const TDesC& aDes)
{
// new addition
	DO_LOGL8("CAdvConsoleClient::Write\r\n");
	TBuf8<512>	theBuf8;
	TBuf16<256>	theBuf16;

	theBuf8.Copy(aDes);
	ConvertTo1251(theBuf8, theBuf16);

	DO_LOGL8("Try to send\r\n");
	DO_LOG(theBuf8);
	DO_LOGL8("\r\n");
#ifdef _DO_DEBUG_
	theBuf8.Copy(theBuf16);
#endif
	DO_LOG(theBuf8);
	DO_LOGL8("\r\n\r\n");

	SendReceive(EWrite, &theBuf16);
//
//    SendReceive(EWrite,&aDes);
}

// new addition
TInt	CAdvConsoleClient::ConvertTo1251(const TDesC8& aSource, TDes& aDest)
{
	DO_LOGL8("CAdvConsoleClient::ConvertTo\r\n");
	TInt	rez;
	TBool	canConvert = EFalse;


	if (iCharSetWin1251)
	{
		DO_LOGL8("Ready to convert\r\n");
		CCnvCharacterSetConverter::TAvailability	theFlag = CCnvCharacterSetConverter::ENotAvailable;

		TRAPD(err, theFlag = iConverter->PrepareToConvertToOrFromL(iCharSetWin1251, iFs));
		if (err == KErrNone)
		{
			if (theFlag == CCnvCharacterSetConverter::EAvailable)
			{
				DO_LOGL8("Converter available\r\n");
				canConvert = ETrue;
			}
		}
	}
	
	if (canConvert)
	{
		TInt	theState = CCnvCharacterSetConverter::KStateDefault;

		rez = iConverter->ConvertToUnicode(aDest, aSource, theState);
		DO_LOGL8("Converted\r\n");
	}
	else
	{
		aDest.Copy(aSource);
		rez = aDest.Length();
	}

	
	return rez;
}

TInt	CAdvConsoleClient::ConvertFrom1251(const TDesC& aSource, TDes8& aDest)
{
	DO_LOGL8("CAdvConsoleClient::ConvertFrom\r\n");
	TInt	rez;
	TBool	canConvert = EFalse;


	if (iCharSetWin1251)
	{
		DO_LOGL8("Ready to convert\r\n");
		CCnvCharacterSetConverter::TAvailability	theFlag = CCnvCharacterSetConverter::ENotAvailable;

		TRAPD(err, theFlag = iConverter->PrepareToConvertToOrFromL(iCharSetWin1251, iFs));
		if (err == KErrNone)
		{
			if (theFlag == CCnvCharacterSetConverter::EAvailable)
			{
			    DO_LOGL8("Converter available\r\n");
				canConvert = ETrue;
			}
		}
	}
	
	if (canConvert)
	{
		rez = iConverter->ConvertFromUnicode(aDest, aSource);
		DO_LOGL8("Converted\r\n");
	}
	else
	{
		aDest.Copy(aSource);
		rez = aDest.Length();
	}

	
	return rez;
}
//

TPoint CAdvConsoleClient::CursorPos() const
{
    return(iScreen->CursorPos());
}

TPoint CAdvConsoleClient::PointerPos()
{
    TPoint position;
    TPckg<TPoint> p(position);
    SendReceive(EGetPointerPos, &p);
    return position;
}

void CAdvConsoleClient::SetCursorPosAbs(const TPoint& aPosition)
{
    SendReceive(ESetCursorPosAbs,&aPosition);
}

void CAdvConsoleClient::SetCursorPosRel(const TPoint &aVector)
{
    SendReceive(ESetCursorPosRel,&aVector);
}

void CAdvConsoleClient::SetCursorHeight(TInt aPercentage)
{
    SendReceive(ESetCursorHeight,aPercentage);
}

void CAdvConsoleClient::SetTitle(const TDesC& aTitle)
{
    SendReceive(ESetTitle,&aTitle);
}

void CAdvConsoleClient::ClearScreen()
{
    SendReceive(EClearScreen,NULL);
}

void CAdvConsoleClient::ClearToEndOfLine()
{
    SendReceive(EClearToEndOfLine,NULL);
}

void CAdvConsoleClient::ClearChars(const TRect &aRect , TUint aCharacterAttributes )	
{
	SendReceive(EClearChars, &aRect);
}

void CAdvConsoleClient::ScrollChars(const TRect &anArea,const TPoint &aVector )
{
	SendReceive(EScrollChars, &anArea);
}

TSize CAdvConsoleClient::ScreenSize() const
{
	TSize tempsize;

	//Do it this way as to allow dynamic screen size changing
	tempsize.iWidth = iScreen->ConsoleControl()->Size().iWidth/iScreen->ConsoleControl()->CharSize().iWidth;
	tempsize.iHeight = iScreen->ConsoleControl()->Size().iHeight/iScreen->ConsoleControl()->CharSize().iHeight;
  
	return(tempsize);
}

void CAdvConsoleClient::ScreenZoom()
{
	SendReceive(EScreenZoom, NULL);
}

TKeyCode CAdvConsoleClient::KeyCode() const
{
    return((TKeyCode)iKeyEvent.iCode);
}

TUint CAdvConsoleClient::KeyModifiers() const
{
    return(iKeyEvent.iModifiers);
}

TInt CAdvConsoleClient::KeyHit() 
{
    TInt Status = 0;
    TPckg<TInt> p(Status);
    SendReceive(EKeyHit, &p);
    return Status;
}

void CAdvConsoleClient::SetAttr(int attrib)
{
    SendReceive(ESetAttr,attrib);
}

CAdvConsoleClient* NewAdvConsole()
{
#if defined(__WINS__)
    // return null if the graphical window server thread is not running
    TFindThread findT(_L("Wserv"));
    TFullName name;
    if (findT.Next(name)!=KErrNone)
        return(NULL);
#endif
    return(new CAdvConsoleClient);
}
