/* bwindow.c - buffered window routines */

#include <stdlib.h>
#include "bwindow.h"

/* manifest constants */
#define TEXTMARGIN	2	/* text margins */
#define TIMEON		40	/* cursor on time */
#define TIMEOFF		20	/* cursor off time */

/* cursor variables */
static BWindowPtr cursorbw;
static int cursorstate=-1;
static long cursortime;

void BWInit(BWindowPtr bw,WindowPtr w,Rect *bounds)
{
    GrafPtr savePort;
    FontInfo info;

    /* initialize */
    bw->bw_window = w;
    bw->bw_bounds = *bounds;
    bw->bw_linelen = 0;
    bw->bw_input = 0;
    
    /* make this the current port */
    GetPort(&savePort);
    SetPort(w);

    /* get font information */
    GetFontInfo(&info);

    /* compute the x and y increments */
    bw->bw_xinc = info.widMax;
    bw->bw_yinc = info.ascent + info.descent + info.leading;
    
    /* save the font ascent */
    bw->bw_ascent = info.ascent;
        
    /* compute the character dimensions of the window */
    bw->bw_h = ((bounds->bottom - bounds->top) - (2 * TEXTMARGIN)) / bw->bw_yinc;
    bw->bw_w = ((bounds->right - bounds->left) - (2 * TEXTMARGIN)) / bw->bw_xinc;

    /* allocate the buffers */
    bw->bw_buffer = malloc(bw->bw_h * bw->bw_w);
    bw->bw_linebuf = malloc(bw->bw_w + 2);
    bw->bw_linepos = (int *)malloc(bw->bw_w * sizeof(int));
    
    /* clear the buffer */
    BWClear(bw);

    /* restore the port */
    SetPort(savePort);
}

void BWRedraw(BWindowPtr bw)
{
    char *line;
    int y;
    line = bw->bw_topline;
    for (y = 0; y < bw->bw_h; ++y) {
	BWPosition(bw,0,y);
	DrawText(line,0,bw->bw_w);
	BWNextLine(bw,&line);
    }
}

void BWClear(BWindowPtr bw)
{
    char *p; int x,y;
    Rect r;

    /* clear the buffer */
    p = bw->bw_buffer;
    for (y = 0; y < bw->bw_h; ++y)
	for (x = 0; x < bw->bw_w; ++x)
	    *p++ = ' ';

    /* initialize the cursor */
    bw->bw_topline = bw->bw_curline = bw->bw_buffer;
    bw->bw_x = bw->bw_y = 0;

    /* erase the screen */
    r = bw->bw_bounds;
    r.top += bw->bw_window->portRect.top;
    r.left += bw->bw_window->portRect.left;
    r.bottom += bw->bw_window->portRect.top;
    r.right += bw->bw_window->portRect.left;
    EraseRect(&r);
}

void BWPutC(BWindowPtr bw,int ch)
{
    GrafPtr savePort;
    GetPort(&savePort);
    SetPort(bw->bw_window);
    TextFont(monaco);
    TextSize(9);
    switch (ch) {
    case '\r':
	bw->bw_x = 0;
        break;
    case '\n':
	BWNextLine(bw,&bw->bw_curline);
	if (++bw->bw_y >= bw->bw_h) {
	    bw->bw_y = bw->bw_h - 1;
	    BWScrollUp(bw);
	}
        break;
    case '\t':
	do { BWPutC(bw,' '); } while (bw->bw_x & 7);
        break;
    case '\010':
	if (bw->bw_x) --bw->bw_x;
        break;
    default:
	if (ch >= 0x20 && ch < 0x7F) {
/*
	    if (ch == ' ')
	        BWEraseChar(bw);
	    else {
	        BWPosition(bw,bw->bw_x,bw->bw_y);
	        DrawChar(ch);
	    }
*/
	    bw->bw_curline[bw->bw_x] = ch;
	    BWPosition(bw,0,bw->bw_y);
	    DrawText(bw->bw_curline,0,bw->bw_w);
	    if (++bw->bw_x >= bw->bw_w) {
	        BWNextLine(bw,&bw->bw_curline);
	        if (++bw->bw_y >= bw->bw_h) {
		    bw->bw_y = bw->bw_h - 1;
	            BWScrollUp(bw);
	        }
	        bw->bw_x = 0;
	    }
	}
        break;
    }
    SetPort(savePort);
}

void BWKey(BWindowPtr bw,int ch)
{
    GrafPtr savePort;
    GetPort(&savePort);
    SetPort(bw->bw_window);

    /* check for an input handler */
    if (bw->bw_input == 0) {
	SysBeep(1);
	return;
    }
    
    /* handle the input character */
    switch (ch) {
    case '\r':
	bw->bw_linebuf[bw->bw_linelen++] = '\r';
	bw->bw_linebuf[bw->bw_linelen] = '\0';
	(*bw->bw_input)(bw->bw_linebuf);
	bw->bw_linelen = 0;
	break;
    case '\010':
	if (bw->bw_linelen > 0) {
	    --bw->bw_linelen;
	    BWEraseChar(bw); /* this one erases the cursor */
	    do {
		--bw->bw_x;
		BWEraseChar(bw);
	    } while (bw->bw_x > bw->bw_linepos[bw->bw_linelen]);
	}
	break;
    default:
	if (bw->bw_linelen < bw->bw_w-1) {
	    bw->bw_linebuf[bw->bw_linelen] = ch;
	    bw->bw_linepos[bw->bw_linelen] = bw->bw_x;
	    ++bw->bw_linelen;
	    BWPutC(bw,ch);
	}
	else
	    SysBeep(1);
	break;
    }
    SetPort(savePort);
}

void BWPosition(BWindowPtr bw,int x,int y)
{
    MoveTo((x * bw->bw_xinc)
            + bw->bw_bounds.left
            + TEXTMARGIN,
           (y * bw->bw_yinc)
            + bw->bw_bounds.top
            + TEXTMARGIN
            + bw->bw_ascent);
}

void BWEraseChar(BWindowPtr bw)
{
    int x,y;
    Rect r;
    x = (bw->bw_x * bw->bw_xinc) + bw->bw_bounds.left + TEXTMARGIN;
    y = (bw->bw_y * bw->bw_yinc) + bw->bw_bounds.top + TEXTMARGIN;
    SetRect(&r,x,y,x+bw->bw_xinc,y+bw->bw_yinc);
    EraseRect(&r);
}

void BWNextLine(BWindowPtr bw,char **pline)
{
    if ((*pline += bw->bw_w) >= &bw->bw_buffer[bw->bw_h * bw->bw_w])
	*pline = bw->bw_buffer;
}

void BWScrollUp(BWindowPtr bw)
{
    RgnHandle updateRgn;
    Rect r;
    int x;

    updateRgn = NewRgn();
    r = bw->bw_bounds;
    r.top += bw->bw_window->portRect.top;
    r.left += bw->bw_window->portRect.left;
    r.bottom += bw->bw_window->portRect.top;
    r.right += bw->bw_window->portRect.left;
    InsetRect(&r,TEXTMARGIN,TEXTMARGIN);
    ScrollRect(&r,0,-bw->bw_yinc,updateRgn);
    DisposeRgn(updateRgn);
    for (x = 0; x < bw->bw_w; ++x)
	bw->bw_topline[x] = ' ';
    BWNextLine(bw,&bw->bw_topline);
}

void BWIdle(void)
{
    GrafPtr savePort;
    if (cursorstate != -1)
	if (cursortime < TickCount()) {
	    GetPort(&savePort);
	    SetPort(cursorbw->bw_window);
	    BWPosition(cursorbw,cursorbw->bw_x,cursorbw->bw_y);
	    if (cursorstate) {
		DrawChar(' ');
		cursortime = TickCount() + TIMEOFF;
		cursorstate = 0;
	    }
	    else {
		DrawChar('_');
		cursortime = TickCount() + TIMEON;
		cursorstate = 1;
	    }
	    SetPort(savePort);
	}
}

void BWCursorOn(BWindowPtr bw)
{
    cursorbw = bw;
    cursortime = TickCount();
    cursorstate = 0;
}

void BWCursorOff(void)
{
    if (cursorstate == 1) {
	BWPosition(cursorbw,cursorbw->bw_x,cursorbw->bw_y);
	DrawChar(' ');
    }
    cursorstate = -1;
}
