#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "pickle.h"

/* This file is the source code for the PICKLE Editor Library. 
	Version 1. */

/* This library includes everything in the PICKLE Reader Library,
	plus a bunch of extra functions which are useful for analyzing
	and writing out PICKLE files. */

#ifdef BIG_ENDIAN
static char contentmessage[] = "\nPICKLE Editor Library 1.0.0 (big-endian)\n";
#endif

#ifdef LITTLE_ENDIAN
static char contentmessage[] = "\nPICKLE Editor Library 1.0.0 (little-endian)\n";
#endif

#ifdef __STDC__
static char contentmessage2[] = "\nLibrary compiled with __STDC__\n";
#else
static char contentmessage2[] = "\nLibrary compiled without __STDC__\n";
#endif

#define pik_InitedMagic (0x18328EEB)

#ifdef BIG_ENDIAN
#define pikNative(v) (v)
#endif

#ifdef LITTLE_ENDIAN
#define pikNative(v) (   \
	  ((((long)(v)) >> 24) & 0x000000ff)   \
	| ((((long)(v)) >> 8)  & 0x0000ff00)   \
	| ((((long)(v)) << 8)  & 0x00ff0000)   \
	| ((((long)(v)) << 24) & 0xff000000)   \
)
#endif

struct pikDescriptorData {
	pikType use;
	pikLong number;
	pikType format;
	pikLong formatversion;
	pikLong startpos;
	pikLong length;
	
	void *data; /* (non-Mac systems) */
	/*Handle data;*/ /* (Mac only) */
};
typedef struct pikDescriptorData pikDescriptor;

struct pikMapData {
	pikLong inited; /* contains pik_InitedMagic while the map is in existence */
	FILE *file; /* (non-Mac systems) */
	/*short file;*/ /* (Mac only) */
	
	pikLong numchunks;
	pikLong filelength;
	
	pikDescriptor *chunks; /* pointer to an array of descriptors */
};
typedef struct pikMapData pikMap;

/* The first set of functions is identical to those in the 
	reader library. */

#ifdef __STDC__
short pikIsPickleHeader(char *header)
#else
short pikIsPickleHeader(header)
char *header;
#endif
{
	if (header[0] == 0x70
		&& header[1] == 0x69
		&& header[2] == 0x6b
		&& header[3] == 0x6c) {
		return TRUE;	
	}
	else {
		return FALSE;
	}
}

#ifdef __STDC__
pikErr pikCreateMap(FILE *file, pikMapPtr *newmap)
#else
pikErr pikCreateMap(file, newmap)
FILE *file;
pikMapPtr *newmap;
#endif
{
	pikMapPtr map;
	pikLong buffer[6];
	int err;
	long count;
	long lx;
	pikLong numchunks;
	pikLong filelength;
	pikDescriptor *chunks;
	
	if (sizeof(pikLong) != 4) {
		return pikerr_WrongSizeInts;
	}
	{
		pikLong testval;
		unsigned char *cx;
		cx = (unsigned char *)(&testval);
		cx[0] = 0x12;
		cx[1] = 0x34;
		cx[2] = 0x56;
		cx[3] = 0x78;
		if (pikNative(testval) != 0x12345678) {
			return pikerr_WrongEndian;
		}
	}
	
	if (!file) {
		return pikerr_BadOption;
	}
	
	err = fseek(file, 0, 0);
	if (err) {
		return pikerr_CantRead;
	}
	
	count = fread(buffer, 1, 16, file);
	if (count != 16) {
		return pikerr_CantRead;
	}
	
	if (pikNative(buffer[0]) != pikMakeType('p', 'i', 'k', 'l')
		|| pikNative(buffer[1]) != 1) {
		return pikerr_NotAMap;
	}
	
	numchunks = pikNative(buffer[2]);
	filelength = pikNative(buffer[3]);
	
	err = fseek(file, 0, 2);
	if (err) {
		return pikerr_CantRead;
	}
	count = ftell(file);
	if (count == (-1)) {
		return pikerr_CantRead;
	}
	if (count != filelength) {
		return pikerr_NotAMap;
	}
	err = fseek(file, 16, 0);
	if (err) {
		return pikerr_CantRead;
	}
		
	map = (pikMapPtr)malloc(sizeof(pikMap));
	if (!map) {
		return pikerr_Memory;
	}
	chunks = (pikDescriptor *)malloc(sizeof(pikDescriptor) * numchunks);
	if (!chunks) {
		free(map);
		return pikerr_Memory;
	}
	
	for (lx=0; lx<numchunks; lx++) {
	
		count = fread(buffer, 1, 24, file);
		if (count != 24) {
			free(chunks);
			free(map);
			return pikerr_CantRead;
		}
		
		chunks[lx].use = pikNative(buffer[0]);
		chunks[lx].number = pikNative(buffer[1]);
		chunks[lx].format = pikNative(buffer[2]);
		chunks[lx].formatversion = pikNative(buffer[3]);
		chunks[lx].startpos = pikNative(buffer[4]);
		chunks[lx].length = pikNative(buffer[5]);
		chunks[lx].data = NULL;
	}
	
	map->chunks = chunks;
	map->numchunks = numchunks;
	map->filelength = filelength;
	map->file = file;
	map->inited = pik_InitedMagic;
	
	*newmap = map;
	return pikerr_None;
}

#ifdef __STDC__
pikErr pikDestroyMap(pikMapPtr map)
#else
pikErr pikDestroyMap(map)
pikMapPtr map;
#endif
{
	pikLong lx;
	pikDescriptor *desc;
	
	if (!map || !map->chunks || map->inited != pik_InitedMagic) {
		return pikerr_NotAMap;
	}
	
	for (lx=0; lx<map->numchunks; lx++) {
		desc = (&(map->chunks[lx]));
		if (desc->data) {
			free(desc->data);
			desc->data = NULL;
		}
	}

	map->inited = 0;
	free(map->chunks);
	free(map);

	return pikerr_None;
}

#ifdef __STDC__
pikErr pikFindChunk(pikMapPtr map, pikType use, pikLong number, 
	short numformats, pikFormat *formatlist, pikChunkID *idfound)
#else
pikErr pikFindChunk(map, use, number, numformats, formatlist, idfound)
pikMapPtr map;
pikType use;
pikLong number;
short numformats;
pikFormat *formatlist;
pikChunkID *idfound;
#endif
{
	pikDescriptor *desc;
	pikLong id, bestid;
	short fx, sofar;
	
	if (!map || !map->chunks || map->inited != pik_InitedMagic) {
		return pikerr_NotAMap;
	}
	
	if (numformats < 0 || (numformats > 0 && !formatlist)) {
		return pikerr_BadOption;
	}
	
	sofar = (-1);
	for (id=0, desc=map->chunks; id<map->numchunks; id++, desc++) {
		if (desc->use == use && desc->number == number) {
			if (numformats == 0) {
				if (idfound)
					*idfound = id;
				return pikerr_None;
			}
			for (fx=0; fx<numformats; fx++) {
				if (desc->format == formatlist[fx].name && 
					desc->formatversion == formatlist[fx].version) {
					if (sofar < 0 || fx < sofar) {
						sofar = fx;
						bestid = id;
					}
				}
			}
		}
	}
	
	if (sofar >= 0) {
		if (idfound)
			*idfound = bestid;
		return pikerr_None;
	}
	
	if (idfound)
		*idfound = pik_NoChunk;
	return pikerr_NotFound;
}

#ifdef __STDC__
pikErr pikLoadChunk(pikMapPtr map, pikChunkID id, short method, pikChunk *found)
#else
pikErr pikLoadChunk(map, id, method, found)
pikMapPtr map;
pikChunkID id;
short method;
pikChunk *found;
#endif
{
	pikDescriptor *desc;
	
	if (!map || !map->chunks || map->inited != pik_InitedMagic) {
		return pikerr_NotAMap;
	}
	
	if (id < 0 || id >= map->numchunks) {
		return pikerr_BadOption;
	}	
	if (!found) {
		return pikerr_BadOption;
	}
	
	desc = (&(map->chunks[id]));
	switch (method) {
		case pikmethod_DontRead:
			break;
		case pikmethod_FilePos:
			found->data.startpos = desc->startpos;
			break;
		case pikmethod_Memory:
			if (!desc->data) {
				long count;
				int err;
				desc->data = malloc(desc->length);
				if (!desc->data) {
					return pikerr_Memory;
				}
				err = fseek(map->file, desc->startpos, 0);
				if (err) {
					return pikerr_CantRead;
				}
				count = fread(desc->data, 1, desc->length, map->file);
				if (count != desc->length) {
					return pikerr_CantRead;
				}
			}
			found->data.ptr = desc->data;
			break;
		default:
			return pikerr_BadOption;
	}

	found->length = desc->length;
	found->format.name = desc->format;
	found->format.version = desc->formatversion;
	
	return pikerr_None;
}

#ifdef __STDC__
pikErr pikUnloadChunk(pikMapPtr map, pikChunkID id)
#else
pikErr pikUnloadChunk(map, id)
pikMapPtr map;
pikChunkID id;
#endif
{
	pikDescriptor *desc;
	
	if (!map || !map->chunks || map->inited != pik_InitedMagic) {
		return pikerr_NotAMap;
	}
	
	if (id < 0 || id >= map->numchunks) {
		return pikerr_BadOption;
	}	
	
	desc = (&(map->chunks[id]));
	
	if (desc->data) {
		free(desc->data);
		desc->data = NULL;
		return pikerr_None;
	}
	else {
		return pikerr_NotFound;
	}
}

/* The rest of the functions are specific to the editor library. */

#include "picklewr.h"

#ifdef __STDC__
void pikxTypeToString(pikType type, char *result)
#else
void pikxTypeToString(type, result)
pikType type;
char *result;
#endif
{
	result[0] = (type >> 24) & 0xff;
	result[1] = (type >> 16) & 0xff;
	result[2] = (type >> 8) & 0xff;
	result[3] = (type) & 0xff;
	result[4] = '\0';
}

#ifdef __STDC__
pikErr pikxGetChunkCount(pikMapPtr map, pikLong *num)
#else
pikErr pikxGetChunkCount(map, num)
pikMapPtr map;
pikLong *num;
#endif
{
	if (!map || !map->chunks || map->inited != pik_InitedMagic) {
		return pikerr_NotAMap;
	}

	*num = map->numchunks;
	return pikerr_None;
}

#ifdef __STDC__
pikErr pikxFindIndChunk(pikMapPtr map, pikLong num, pikIndChunk *found)
#else
pikErr pikxFindIndChunk(map, num, found)
pikMapPtr map;
pikLong num;
pikIndChunk *found;
#endif
{
	pikDescriptor *desc;

	if (!map || !map->chunks || map->inited != pik_InitedMagic) {
		return pikerr_NotAMap;
	}

	if (num < 0 || num >= map->numchunks) {
		return pikerr_BadOption;
	}	

	if (!found) {
		return pikerr_BadOption;
	}

	desc = (&(map->chunks[num]));
	
	found->length = desc->length;
	found->format.name = desc->format;
	found->format.version = desc->formatversion;
	found->use = desc->use;
	found->number = desc->number;
	found->startpos = desc->startpos;
	found->id = num;
	
	return pikerr_None;
}

struct pikBuilderChunkData {
	short used;
	pikLong startpos; /* in input file */
	pikLong outputpos; /* in output file */
	pikLong length;
	pikType use;
	pikLong number;
	pikFormat format;
	char *filename;
};

struct pikBuilderData {
	pikLong numchunks;
	pikLong listsize;
	pikBuilderChunk *chunks;
};
typedef struct pikBuilderData pikBuilder;

#ifdef __STDC__
pikErr pikxCreateBuilder(pikBuilderPtr *newmap)
#else
pikErr pikxCreateBuilder(newmap)
pikBuilderPtr *newmap;
#endif
{
	pikBuilderPtr map;
	pikLong lx;
	
	map = (pikBuilderPtr)malloc(sizeof(pikBuilder));
	if (!map) {
		return pikerr_Memory;
	}
	
	map->numchunks = 0;
	map->listsize = 4;
	map->chunks = (pikBuilderChunk *)malloc(sizeof(pikBuilderChunk) * map->listsize);
	if (!map->chunks) {
		free(map);
		return pikerr_Memory;
	}
	
	for (lx=0; lx<map->listsize; lx++) {
		map->chunks[lx].used = FALSE;
	}
	
	*newmap = map;
	return pikerr_None;
}

#ifdef __STDC__
pikErr pikxDestroyBuilder(pikBuilderPtr map)
#else
pikErr pikxDestroyBuilder(map)
pikBuilderPtr map;
#endif
{
	pikLong lx;
	
	if (map->chunks) {
		for (lx=0; lx<map->listsize; lx++) {
			if (map->chunks[lx].used && map->chunks[lx].filename) {
				free(map->chunks[lx].filename);
			}
		}
		free(map->chunks);
	}
	free(map);
	return pikerr_None;
}

#ifdef __STDC__
pikErr pikxBuilderBuild(pikBuilderPtr map, FILE *file)
#else
pikErr pikxBuilderBuild(map, file)
pikBuilderPtr map;
FILE *file;
#endif
{
	pikLong lx;
	pikBuilderChunk *chunk;
	pikLong buffer[6];
	FILE *infile;
	long count;
	pikLong totallength;
	int err, let;

	totallength = 16 + 24 * map->numchunks;
	
	for (lx=0; lx<map->listsize; lx++) {
		if (map->chunks[lx].used) {
			chunk = (&map->chunks[lx]);
			
			if (chunk->length < 0) {
				infile = fopen(chunk->filename, "rb");
				if (!infile) {
					fprintf(stderr, "Unable to open %s to measure its length.\n", chunk->filename);
					return pikerr_CantRead;
				}
				err = fseek(infile, 0, 2);
				if (err) {
					fprintf(stderr, "Unable to read %s to measure its length.\n", chunk->filename);
					fclose(infile);
					return pikerr_CantRead;
				}
				chunk->length = ftell(infile);
				if (chunk->length == (-1)) {
					fprintf(stderr, "Unable to read %s to measure its length.\n", chunk->filename);
					fclose(infile);
					return pikerr_CantRead;
				}
				chunk->length -= chunk->startpos;
				fclose(infile);
			}
			
			chunk->outputpos = totallength;
			totallength += chunk->length;
		}
	}
	
	rewind(file);
	buffer[0] = pikNative(0x70696b6c);
	buffer[1] = pikNative(0x0001);
	buffer[2] = pikNative(map->numchunks);
	buffer[3] = pikNative(totallength);
	
	count = fwrite(buffer, 1, 16, file);
	if (count != 16) {
		return pikerr_CantWrite;
	}
	
	for (lx=0; lx<map->listsize; lx++) {
		if (map->chunks[lx].used) {
			chunk = (&map->chunks[lx]);
			buffer[0] = pikNative(chunk->use);
			buffer[1] = pikNative(chunk->number);
			buffer[2] = pikNative(chunk->format.name);
			buffer[3] = pikNative(chunk->format.version);
			buffer[4] = pikNative(chunk->outputpos);
			buffer[5] = pikNative(chunk->length);
			count = fwrite(buffer, 1, 24, file);
			if (count != 24) {
				return pikerr_CantWrite;
			}
		}
	}
	
	for (lx=0; lx<map->listsize; lx++) {
		if (map->chunks[lx].used) {
			chunk = (&map->chunks[lx]);
			
			infile = fopen(chunk->filename, "rb");
			if (!infile) {
				fprintf(stderr, "Unable to open %s for copying.\n", chunk->filename);
				return pikerr_CantRead;
			}
			
			err = fseek(infile, chunk->startpos, 0);
			if (err) {
				fprintf(stderr, "Unable to position %s to begin copy.\n", chunk->filename);
				fclose(infile);
				return pikerr_CantRead;
			}
			
			for (count=0; count<chunk->length; count++) {
				let = getc(infile);
				if (let == EOF)
					break;
				err = putc(let, file);
				if (err == EOF)
					break;
			}
			
			fclose(infile);
			
			if (count != chunk->length) {
				fprintf(stderr, "Length of %s has changed.\n", chunk->filename);
				return pikerr_CantRead;
			}
		}
	}

	return pikerr_None;
}

#ifdef __STDC__
pikErr pikxBuilderAdd(pikBuilderPtr map, pikChunkID *newid, 
	pikType use, pikLong number, pikFormat format, 
	pikLong startpos, pikLong length, char *filename)
#else
pikErr pikxBuilderAdd(map, newid, use, number, format, 
	startpos, length, filename)
pikBuilderPtr map;
pikChunkID *newid;
pikType use;
pikLong number;
pikFormat format;
pikLong startpos;
pikLong length;
char *filename;
#endif
{
	pikLong id, lx;
	pikBuilderChunk *chunk;
	
	if (map->numchunks == map->listsize) {
		map->listsize *= 2;
		map->chunks = (pikBuilderChunk *)realloc(map->chunks, 
			sizeof(pikBuilderChunk) * map->listsize);
		if (!map->chunks) {
			return pikerr_Memory;
		}
		for (lx=map->numchunks; lx<map->listsize; lx++) {
			map->chunks[lx].used = FALSE;
		}
		id = map->numchunks;
	}
	else {
		for (id=0; id<map->listsize; id++) {
			if (!map->chunks[id].used)
				break;
		}
		if (id == map->listsize)
			return pikerr_Memory;
	}
	
	chunk = (&map->chunks[id]);
	
	chunk->filename = malloc(strlen(filename)+1);
	if (!chunk->filename) {
		return pikerr_Memory;
	}
	
	strcpy(chunk->filename, filename);
	
	chunk->use = use;
	chunk->number = number;
	chunk->format = format;
	chunk->startpos = startpos;
	chunk->length = length;
	
	chunk->used = TRUE;
	map->numchunks++;
	if (newid)
		*newid = id;
		
	return pikerr_None;
}

