/* cscr.h: CSCR text library definititions
    see individual implementations for details
    By L. Ross Raszewski <lraszewski@justice.loyola.edu>
*/

// Declare the 'chtype' if it's not already.
#ifndef __PDCURSES__
typedef unsigned long chtype;
#endif

// Module magic constants
#define CSCR_ALLEGRO 1
#define CSCR_CURSES  0
#define CSCR_UNIX    2
#define CSCR_DOS     3

// The character representing an inline graphic
#define GRX_CHAR 1

// Error keycode
#ifndef ERR
#define ERR 0
#endif

// The key representing a mouse click
#define MOUSE_CLICK_KEY 0xffff

// The mouse info block
struct mouse_info_s {
        int x;
        int y;
        int text_x;
        int text_y;
        };
extern struct mouse_info_s mouse_info;

extern int VHORIZ_REZ;
extern int VVERT_REZ;
        /* These are used only in environments supporting
           virtual screens, and give the dimentions of the screen
           in a version-dependant format.
           (pixels in calleg, characters in a character-based module.)
         */

// Group 1: System control
int             cscr_version();
                /* Returns one of the CSCR_* constants above
                   to determine the library version
                */
void            cscr_start();
                /* Start cscr text interface; set up the output
                   device and associated underlying library
                */
void            cscr_exit();
                /* End cscr, unload associated library
                */
void            cscr_resume();
void            cscr_suspend();
                /* temporarily block cscr's screen management
                */

// Group 2: Metrics
void            cscr_charpos_to_pxl(int *, int *);
                /* Converts a character-position to screen-units
                   (pixels or characters) -- Clobbers input vars.
                */
unsigned int    cscr_getcols();
unsigned int    cscr_getlines();
                /* Return the dimentions of the (virtual) screen in
                   character-sized units
                */

// Group 3: Mouse
void            cscr_wake_mouse();
void            cscr_sleep_mouse();
                /* Controlls access to the mouse;
                   cscr_wake_mouse makes the mouse visible and
                   responsive. cscr_sleep_mouse cloaks and disables it
                */
int             cscr_query_mouse();
                /* returns true if a mouse is available -- only meaningful
                   AFTER cscr_start
                */
// Group 4: Output
// 4a - Display maintenance
void            cscr_clear();
                /* Clear the cscr screen
                */
void            cscr_clrtoeol();
                /* clear the current line from the text-point onward.
                */
void            cscr_refresh();
                /* refresh the cscr screen. Changes made may not be
                   visible until cscr_refresh is called
                   (some modules might update the screen at other points
                   as well)
                */
void            cscr_moveyx(int y, int x);
                /* Locate the text output point as specified
                */
void            cscr_screencap();
                /* Perform a screen capture, if the facility is available
                */

// 4b - Text output

int             cscr_addstr(char *s);
void            cscr_addch(chtype c);
int             cscr_printw(char *fmt , ...);
                /* Print to the current output location.
                */

// 4c - Text appearance
chtype          cscr_color(int f, int b);
                /* return the specified 4-bit color pair as a cscr-internal
                   attribute
                */
unsigned char   cscr_color_to_8bit(chtype c);
                /* the compliment of cscr_color;
                   cscr_color_to_8bit(cscr_color(f,b)) == b <<4 | f;
                */
void            cscr_attron(chtype a);
void            cscr_attrset(chtype x);
                /* Set the attributes for text output
                */
chtype          cscr_bkgd(chtype i);
                /* Set the attribute for unprinted screen region to i
                */


// Group 5: Input
int             cscr_kbhit();
                /* Returns nonzero if there is input waiting
                   (ie. cscr_getch will not block)
                */
void            cscr_blinkey();
                /* Display a visible cursor at the current
                   text point. Some versions of the library
                   maintain the blinkey automatically, and
                   will ignore this.
                */
chtype          cscr_getch();
                /* Return the next character from the input stream,
                   converted to a curses-compatable character code.
                   If no characters are waiting, block until one is recieved.
                */
                


// Curses-compatabile keybindings.
#ifndef KEY_DOWN
#ifndef ALLEGRO_H
#define KEY_DOWN        0x102   
#define KEY_UP          0x103   
#define KEY_LEFT        0x104   
#define KEY_RIGHT       0x105   
#define KEY_HOME        0x106   
#define KEY_DELETE      0x14a   
#define KEY_INSERT      0x14b
#define KEY_PAGEDN      0x152 
#define KEY_PAGEUP      0x153 
#define KEY_ENTER       10 
#define KEY_END         0x166

#define ALT_0           0x197
#define ALT_1           0x198
#define ALT_2           0x199
#define ALT_3           0x19a
#define ALT_4           0x19b
#define ALT_5           0x19c
#define ALT_6           0x19d
#define ALT_7           0x19e
#define ALT_8           0x19f
#define ALT_9           0x1a0
#define ALT_A           0x1a1
#define ALT_B           0x1a2
#define ALT_C           0x1a3
#define ALT_D           0x1a4
#define ALT_E           0x1a5
#define ALT_F           0x1a6
#define ALT_G           0x1a7
#define ALT_H           0x1a8
#define ALT_I           0x1a9
#define ALT_J           0x1aa
#define ALT_K           0x1ab
#define ALT_L           0x1ac
#define ALT_M           0x1ad
#define ALT_N           0x1ae
#define ALT_O           0x1af
#define ALT_P           0x1b0
#define ALT_Q           0x1b1
#define ALT_R           0x1b2
#define ALT_S           0x1b3
#define ALT_T           0x1b4
#define ALT_U           0x1b5
#define ALT_V           0x1b6
#define ALT_W           0x1b7
#define ALT_X           0x1b8
#define ALT_Y           0x1b9
#define ALT_Z           0x1ba
#define CTL_LEFT        0x1bb
#define CTL_RIGHT       0x1bc
#define CTL_PGUP        0x1bd
#define CTL_PGDN        0x1be
#define CTL_HOME        0x1bf
#define CTL_END         0x1c0
#define CTL_INS         0x1dd
#define ALT_DEL         0x1de
#define ALT_INS         0x1df
#define CTL_UP          0x1e0
#define CTL_DOWN        0x1e1
#define CTL_TAB         0x1e2
#define ALT_TAB         0x1e3
#define ALT_MINUS       0x1e4
#define ALT_EQUAL       0x1e5
#define ALT_HOME        0x1e6
#define ALT_PGUP        0x1e7
#define ALT_PGDN        0x1e8
#define ALT_END         0x1e9
#define ALT_UP          0x1ea
#define ALT_DOWN        0x1eb
#define ALT_RIGHT       0x1ec
#define ALT_LEFT        0x1ed
#define ALT_ENTER       0x1ee
#define ALT_ESC         0x1ef
#define ALT_BQUOTE      0x1f0
#define ALT_LBRACKET    0x1f1
#define ALT_RBRACKET    0x1f2
#define ALT_SEMICOLON   0x1f3
#define ALT_FQUOTE      0x1f4
#define ALT_COMMA       0x1f5
#define ALT_FSLASH      0x1f7
#define ALT_BKSP        0x1f8
#define CTL_BKSP        0x1f9
#define CTL_PAD0        0x1fa
#define CTL_PAD1        0x1fb
#define CTL_PAD2        0x1fc
#define CTL_PAD3        0x1fd
#define CTL_PAD4        0x1fe
#define CTL_PAD5        0x1ff
#define CTL_PAD6        0x200
#define CTL_PAD7        0x201
#define CTL_PAD8        0x202
#define CTL_PAD9        0x203
#define CTL_DEL         0x204
#define ALT_BSLASH      0x205
#define CTL_ENTER       0x206

#endif
#endif

#ifndef A_NORMAL
#define A_NORMAL              0x00000000L
#define A_BOLD		      0x00800000L
#define A_UNDERLINE           0x00100000L
#define A_REVERSE             0x00200000L
#define A_DIM                 0x00020000L
#define A_CHARTEXT            0x0000FFFFL
#endif
#define A_BGBRITE             0x00400000L
