
package org.ifarchive.glk;
import java.util.Vector;

/** This class deals with passing persistent arrays across the C / Java 
 * boundary.
 * <p>For functions where the arrays are known not to be persistent,
 * the C code converts between Java arrays and C arrays. For arrays
 * which <em>are</em> persistent, this class maintains a list of 
 * current arrays and whether Glk is using them.
 * <p>Because, with the Dispatch layer, we can't tell whether an array
 * is going to be retained or not until the call has passed to Glk, 
 * we behave as if it is and use this class.
 * <p>The four main operations are all static members, and they are:
 * <ul>
 * <li>create: Create an array in the C layer containing the data
 *    from the passed array. 
 * <li>reg: Record an array as being in use by Glk.
 * <li>unreg: Record that Glk has finished with this array. The matching
 *           C array is destroyed and the original Java array now contains
 *           the correct data. 
 * <li>destroy: Free the C array and move its data back to the Java array.
 *             If this returns false, Glk has registered the array as 
 *             persistent, and <em>it</em> will destroy it when it releases
 *             it.
 * </ul>
 * <p>The normal calling pattern, then, is: 
 * <ul>
 * <li>Call create().
 * <li>Call the C layer, passing the integer ID of the array.
 * <li>Call destroy(). If destroy() returned false, Glk has retained the
 *    array, which will be destroyed later.
 * </ul>
 * @version 1.0
 * @author John Elliott
 */

class GlkArray
{
// A list of arrays that currently exist in the C layer	
	static Vector st_arrays = new Vector();
// The retained array registry
	static ArrayRegistry st_registry;
// Note that I don't support floats, doubles or booleans yet, though it 
// wouldn't be hard to.
	int     m_bufid;	// ID of the buffer in the C layer
	Object  m_array;	// This is actually byte[], long[] or whatever.
	boolean m_retained = false;
	boolean m_locked = false;
	Object  m_dispRock = null;
	String  m_typecode = null;

	static void setArrayRegistry(ArrayRegistry r)
	{
		// Note that if we're unsetting a registry, I don't bother
		// to unregister all the objects in it. The Glk spec doesn't
		// seem to demand this.
		st_registry = r;
		if (r == null) return;
		// Register anything that's already been retained.
		for (int n = 0; n < st_arrays.size(); n++)
		{
			GlkArray a = (GlkArray)st_arrays.elementAt(n);
			if (a.m_retained) 
			{
				a.m_dispRock = st_registry.reg(a.m_array, a.m_typecode);
			}
		}
				
	}

	// Constructor. "array" can be any type of object that
	// lock() and unlock() know how to deal with. 
	GlkArray(int bufid, Object array)   
	{ 
		m_bufid = bufid; 
		m_array = array;
	}

	void retain(String typecode)
	{
		if (m_retained) return;

		m_retained = true;
		m_typecode = typecode;
		if (st_registry != null) 
		{
			m_dispRock = st_registry.reg(m_array, typecode);
		}
	}

	void unretain(String typecode)
	{
		if (!m_retained) return;
		
		m_retained = false;
		if (st_registry != null) st_registry.unreg(m_array, typecode, m_dispRock);
	}

	// To support floats, add extra lines here... 
	void unlock()
	{
		if (m_array instanceof byte[])  Glk.jniglk_unlock_byte_array ((byte[])m_array,  m_bufid);
		if (m_array instanceof short[]) Glk.jniglk_unlock_short_array((short[])m_array, m_bufid);
		if (m_array instanceof char[])  Glk.jniglk_unlock_char_array ((char[])m_array,  m_bufid);
		if (m_array instanceof int[])   Glk.jniglk_unlock_int_array  ((int[])m_array,   m_bufid);
		if (m_array instanceof long[])  Glk.jniglk_unlock_long_array ((long[])m_array,  m_bufid);
	}

	static int lock(Object array)
	{
		if (array instanceof byte[])  return Glk.jniglk_lock_byte_array ((byte[])array);
		if (array instanceof short[]) return Glk.jniglk_lock_short_array((short[])array);
		if (array instanceof char[])  return Glk.jniglk_lock_char_array ((char[])array);
		if (array instanceof int[])   return Glk.jniglk_lock_int_array  ((int[])array);
		if (array instanceof long[])  return Glk.jniglk_lock_long_array ((long[])array);
		return 0;
	}

	/* Retained array helpers
	 * The situation here is complicated by the fact that we have to
	 * lock the array in C memory before passing it to Glk. Then, 
	 * /after/ we've done that, Glk decides whether it wants to retain
	 * the array.
	 * Therefore our state vector contains the following elements for
	 * each array:
	 *   1. The actual Java array.
	 *   2. The ID of the C array (the C glue code transforms this
	 *     into a pointer and back).
	 *   3. Whether Glk wants to retain this array.
	 *   4. Whether we have locked the array in memory.
	 *
	 * The usual cycle of use is:
	 *   1. In Java code, we lock the array by calling create().
	 *   2. Call the function that uses the array.
	 *   3. If Glk wants to retain the array, it asks to. We set a 
	 *     flag on the array but otherwise do nothing.
	 *   4. When the function returns, we call destroy(). If Glk
	 *     has retained the array, this returns false. If it hasn't,
	 *     it returns true, and the Java array now holds the correct data.
	 *   5. When Glk has finished with the array, it unregisters it.
	 *     This also releases the array and ensures that it holds correct
	 *     data.
	 */

	 	
	/* Create a C-layer array, corresponding to the passed Java array */
	static int create(Object array)
	{	
		int bufid = exists(array); 
		if (bufid != 0) return bufid;
		bufid = lock(array);
		GlkArray a = new GlkArray(bufid, array);	
		a.m_locked = true;
		st_arrays.addElement(a);
		return bufid;
	}

	/* Glk has asked for this array to be persistent */
	static void reg(int bufid, int len, String typecode)
	{
		for (int n = 0; n < st_arrays.size(); n++)
		{
			GlkArray a = (GlkArray)st_arrays.elementAt(n);
			if (a.m_bufid == bufid) a.retain(typecode);
		}
	} 

	// Once Glk releases a retained array, unlock it and remove it 
	// from the list of retained arrays.
	static void unreg(int bufid, String typecode)
	{
		for (int n = 0; n < st_arrays.size(); n++)
		{
			GlkArray a = (GlkArray)st_arrays.elementAt(n);
			if (a.m_bufid == bufid)
			{
				a.unretain(typecode);
				destroy(bufid);
			}
		}
			
	}

	static boolean destroy(int bufid)
	{
		for (int n = 0; n < st_arrays.size(); n++)
		{
			GlkArray a = (GlkArray)st_arrays.elementAt(n);
			if (a.m_bufid == bufid)
			{
				// Array has been retained & can't be deleted.
				if (a.m_retained) return false;
				// Array is still locked (probably never retained)
				if (a.m_locked) a.unlock();
				st_arrays.removeElement(a);
				return true;
			}
		}
		/* Array not found. Glk must have released it. */
		return true;
	}


	static int exists(Object array)
	{
		if (array == null) return 0;
		for (int n = 0; n < st_arrays.size(); n++)
		{
			GlkArray a = (GlkArray)st_arrays.elementAt(n);
			if (a.m_array == array) return a.m_bufid;
		}
		return 0;
	}
}
