
package org.ifarchive.glk;
import java.util.Vector;

/** Base class for wrapping opaque Glk objects.
 * <p>This class contains general methods for handling any type of Glk 
 * object. Objects supported by the library have particular subclasses
 * such as Window or Stream. You will only need to deal with OpaqueObject
 * directly if the underlying Glk library contains object types not supported
 * in this wrapper library. </p>
 *
 * @version 1.0
 * @author John Elliott
 */


public class OpaqueObject
{
	/** Reference to the underlying Glk object. 
	 * This is a unique integer; the C layer translates it to a pointer.
	 * If m_dispid is 0, the Glk object has been destroyed by the Glk
	 * library. */
	protected int    m_dispid   = 0;
	/** Rock given to this object if it was created within Java.
	 * If this is null, the object was created by Glk, and its
	 * rock must be found by a glk_get_*_rock() call. */
	protected Object m_jrock    = null;
	/** The number of the Glk class to which this object belongs.
	 * @see clWindow
	 * @see clStream
	 * @see clFileRef
	 * @see clSoundChannel
	 */
	protected int    m_glkClass = -1;

	// Dispatch rock
	private Object   m_dispRock = null;

	// User object registry, if present.
	static ObjectRegistry st_registry = null;
//
// I'm using one Vector for each opaque Glk class. Each Vector contains a 
// list of all objects of that class. Thus this array of vectors forms the
// object registry.
//
	static Vector st_mapping[];
	static int st_count = 1;	// Initial dispatch ID. It is 
					// incremented every time an object
					// is registered. 

	// Glk object classes (as used by getClassCount() )
	// that the library knows about.
	/** Glk class number for windows */
	public static final int clWindow       = 0;
	/** Glk class number for streams */
	public static final int clStream       = 1;
	/** Glk class number for file references */
	public static final int clFileRef      = 2;
	/** Glk class number for sound channels */
	public static final int clSoundChannel = 3;

	// Called by the runtime system when the library is 
	// first being hooked up. It initialises the opaque object
	// registry.
	static void init()
	{
		int max = getClassCount();

		st_mapping = new Vector[max];
		for (int n = 0; n < max; n++) st_mapping[n] = new Vector();
	}


	/** Constructor. 
	 * @param glkClass The Glk class this object is in.
	 * @param dispid The dispatch ID assigned to this object.
	 */
	protected OpaqueObject(int glkClass, int dispid)
	{
		m_dispid   = dispid;
		m_glkClass = glkClass;
		st_mapping[glkClass].addElement(this);
	}

	/** Text representation of this object.
	 * @return A text string containing this object's Glk class, 
	 *        identity, and rock. */
        public String toString()
        {
                String s = "Glk opaque object: Class = ";
		s += Integer.toString(m_glkClass);
	 	s += " ID = " + Integer.toHexString(m_dispid);
        	if (m_jrock != null) s += " Rock (Java) = " + m_jrock.toString();
		else		     s += " Rock (Glk ) = " + Long.toString(getGlkRock());
                return s;
        }


	/* Find an object in the registry, based on its class 
	 * and Glk ID. If the object doesn't exist, create a new
	 * one. */
	static OpaqueObject find(int glkClass, int dispid)
	{
		Vector v = st_mapping[glkClass];

		// The null pointer -> null reference
		if (dispid == 0) return null;	

		// Existing object
		for (int n = 0; n < v.size(); n++)
		{
			OpaqueObject o = (OpaqueObject)(v.elementAt(n));
			if (o.m_dispid == dispid)
			{
				return o;
			}
		}
		// New object. The library knows about three types of Glk
		// objects and creates special wrapper classes for them; 
		// others just get wrapped by OpaqueObject.
		switch (glkClass)
		{
			case clWindow:  return new Window (dispid);
			case clStream:  return new Stream (dispid);
			case clFileRef: return new FileRef(dispid);
			case clSoundChannel:
					return new SoundChannel(dispid);
			default:        return new OpaqueObject(glkClass, dispid);
		}
	}


	/** Generalised iterator. For a given object, find the next 
	 * object of that class (ordering is arbitrary, but this will
	 * find all objects of the chosen class). If the parameter "o"
	 * is null, the first object of that class is returned.
	 * 
	 * @param glkClass Glk class number of objects to iterate over
	 * @param o null, or an object of the correct class.
	 * @param r If not null, r[0] will be set to the returned object's rock.
	 * @return The next object, or null if there is no next object
	 */
        public static OpaqueObject iterate(int glkClass, OpaqueObject o, Object r[])
        {
                OpaqueObject next = null;
		int index;
		Vector v = st_mapping[glkClass];

		// Find index to start search.
                if (o == null) index = 0;
		else	       index = v.lastIndexOf(o) + 1;

		// Find next object 
		if (v.size() <= index) next = null;
		else next = (OpaqueObject)(v.elementAt(index));
                if (next != null && r != null && r.length > 0) r[0] = next.getRock();
                return next;
        }


	/** If the Glk object is still in existence when this object is
	 * garbage-collected, destroy it. */	
	protected void finalize()
	{
		if (m_dispid != 0) destroy();
	}

	/** Destroy this object and the underlying Glk object. This method
	 * must be overridden in subclasses since Glk has no way of
	 * destroying an object of unknown type. */
	protected void destroy()
	{
		m_dispid = 0;
	}

	/** Return this object's rock. If the object was created by Glk,
	 * the rock will be of class Long. Otherwise it can be any class.
	 * @return The object's rock.
	 */
	public Object getRock()
	{
		if (m_jrock != null) return m_jrock;
		return new Long(getGlkRock());
	}

	
	/* Since Glk has no generic rock-getting function, this has to
	 * be handled by subclasses. I haven't made this function abstract
	 * because OpaqueObject has to be instantiated to support Glk 
	 * objects of unknown types. */
	long getGlkRock() { return 0; }
	

	/* Opaque object registry: An object has been created. Create a
	 * Java wrapper for it and return the ID of this new wrapper. */
	static int reg(int glkClass)
	{
		// If object is not found, find() adds it to the list. 
		// If it *is* found, no harm done.
		OpaqueObject o = find(glkClass, st_count++);
		if (st_count == 0) st_count++;
		if (o == null) return 0; // Shouldn't happen...
		o.reg();
	//	System.out.println("Register object: dispid="
	//		+ Integer.toString(o.m_dispid) 
	//		+ " glkClass = " 
	//		+ Integer.toString(glkClass));
		return o.m_dispid;
	}

	/* Opaque object registry: Unregister an object. The "objid" is 
	 * a C pointer */
	static void unreg(int glkClass, int dispid)
	{
	//	System.out.println("Unregister object: dispid="
	//		+ Integer.toString(dispid) 
	//		+ " glkClass = " 
	//		+ Integer.toString(glkClass));
		OpaqueObject o = find(glkClass, dispid);

		// Remove the reference to this object		
		if (o != null) 
		{
			o.unreg();
			st_mapping[glkClass].removeElement(o);	
		// Flag that this object isn't a Glk object any more
			o.m_dispid = 0;
		}
	}


	/** Convert a Java rock to a Glk rock.
	 * <p>Glk for Java allows a rock to be an object of any type, 
	 *  by storing the rock in the Java layer rather than the underlying
	 *  Glk layer. However, if the object really is a number, then 
	 *  there's no harm it setting it at the Glk level as well as
	 *  the Java level.</p>
	 *  @param object Any object.
	 *  @return If object is a number, its integer value. Otherwise 0.
	 */
	static protected int castRock(Object object)
	{
		if (object == null) return 0;
        	if (object instanceof Number)
		{
			return ((Number)object).intValue();
		}
		return 0;
	}


	/** Count Glk classes.
	 * @return The number of Glk object classes in the underlying
	 * Glk library.
	 */
	public static int getClassCount()
	{
		return Glk.jnigidispatch_count_classes();
	}

	/** Get this object's dispatch rock. This will only have
	 * been set if an opaque object registry is in use.
	 * @return The dispatch rock.
	 */
	public Object getDispatchRock()
	{
		return m_dispRock;
	}

	/** Set an opaque object registry. It will be called when
	 * opaque objects are created or destroyed. 
	 * @param r An instance of your object registry.
	 */
        public static void setRegistry(ObjectRegistry r)
        {
		int c, n, max;
                // Note that if we're unsetting a registry, I don't bother
                // to unregister all the objects in it. The Glk spec doesn't
                // seem to demand this.
                st_registry = r;
                if (r == null) return;
                // Register anything that's already been retained.
		max = getClassCount();

		for (c = 0; c < max; c++)
			for (n = 0; n < st_mapping[c].size(); n++)
			{
				OpaqueObject o = (OpaqueObject)st_mapping[c].elementAt(n);
				o.reg();
			}
        }

	// Register this object.
	void reg()
	{
		if (st_registry != null) m_dispRock = st_registry.reg(this, m_glkClass);
	}
	
	void unreg()
	{
		if (st_registry != null) st_registry.unreg(this, m_glkClass, m_dispRock);
	}
	
}
