
package org.ifarchive.glk;

/** All character output in Glk is done through streams. Every window has an
output stream associated with it. You can also write to files on disk;
every open file is represented by an output stream as well.
<p>There are also input streams; these are used for reading from files
on disk. It is possible for a stream to be both an input and an output
stream.</p>
<p>It is also possible to create a stream that reads or writes to a buffer
in memory.</p>

@author John Elliott
@version 1.0
*/

public class Stream extends OpaqueObject 
{
	/**  The style of normal or body text. A new window or
	    stream always starts with style_Normal as the current style. */
	public static final int style_Normal       = 0;
	/** Text which is emphasized. */
	public static final int style_Emphasized   = 1;
	/**  Text which has a particular arrangement of characters.
	 * It will always be rendered in a fixed-width font. */
	public static final int style_Preformatted = 2;
	/**  Text which introduces a large section. This is
	 *  suitable for the title of an entire game, or a major division such
	 *  as a chapter. */
	public static final int style_Header       = 3;
	/** Text which introduces a smaller section within a large section. */
	public static final int style_Subheader    = 4;
	/** Text which warns of a dangerous condition, or one which
	    the player should pay attention to. */
	public static final int style_Alert        = 5;
	/** Text which notifies of an interesting condition. */
	public static final int style_Note         = 6;
	/** Text which forms a quotation or otherwise abstracted text. */
	public static final int style_BlockQuote   = 7;
	/** Text which the player has entered. You should
	    generally not use this style at all; the library uses it for text
	    which is typed during a line-input request. One case when it <em>is</em>
	    appropriate for you to use style_Input is when you are simulating
	    player input by reading commands from a text file.
	 */
	public static final int style_Input        = 8;
	/** This style has no particular semantic meaning. You may
    define a meaning relevant to your own work, and use it as you see fit. */
	public static final int style_User1        = 9;
	/** This style has no particular semantic meaning. You may
    define a meaning relevant to your own work, and use it as you see fit. */
	public static final int style_User2        = 10;
	/** The number of styles this library supports. */
	public static final int style_NUMSTYLES    = 11;

	/** How much to indent lines of text in the given style. 
	May be a negative number, to shift the text out (left)
	instead of in (right). The exact metric isn't precisely specified;
	you can assume that +1 is the smallest indentation possible which
	is clearly visible to the player. */
	public static final int stylehint_Indentation  = 0;
	/** How much to indent the first line of each paragraph. 
	This is in addition to the indentation specified by 
	stylehint_Indentation. This too may be negative, and is measured in
	the same units as stylehint_Indentation. */
	public static final int stylehint_ParaIndentation  = 1;
	/** How to justify text.
	 * @see stylehint_just_LeftFlush
	 * @see stylehint_just_LeftRight
	 * @see stylehint_just_RightFlush
	 * @see stylehint_just_Centred
	 */
	public static final int stylehint_Justification  = 2; 
	/**  How much to increase or decrease the font size. This
	is relative; 0 means the interpreter's default font size will be used,
	positive numbers increase it, and negative numbers decrease it. Again,
	+1 is the smallest size increase which is easily visible. */
	public static final int stylehint_Size  = 3;
	/** The value of this hint must be 1 for heavy-weight
	fonts (boldface), 0 for normal weight, and -1 for light-weight fonts. */
	public static final int stylehint_Weight  = 4;
	/**  The value of this hint must be 1 for oblique fonts (italic), 
	 or 0 for normal angle. */
	public static final int stylehint_Oblique  = 5;
	/** The value of this hint must be 1 for proportional-width fonts, 
	 or 0 for fixed-width.   */
	public static final int stylehint_Proportional  = 6;
	/** The foreground color of the text. This is
	encoded in the 32-bit hint value: the top 8 bits must be zero, the
	next 8 bits are the red value, the next 8 bits are the green value,
	and the bottom 8 bits are the blue value. Color values range from 0
	to 255. */
	public static final int stylehint_TextColor  = 7;
	/** The background color behind the text. This
	    is encoded the same way as stylehint_TextColor.
	    @see stylehint_TextColor */
	public static final int stylehint_BackColor  = 8;
	/** The value of this hint must be 0 for normal printing, 1 for reverse.  */
	public static final int stylehint_ReverseColor  = 9;
	/** The number of stylehints this library knows about. */
	public static final int stylehint_NUMHINTS  = 10;
	
	/** Text will be left aligned. */	
	public static final int stylehint_just_LeftFlush = 0;
	/** Text will be fully justified. */	
	public static final int stylehint_just_LeftRight = 1;
	/** Text will be centred. */	
	public static final int stylehint_just_Centred = 2;
	/** Text will be right aligned. */	
	public static final int stylehint_just_RightFlush = 3;

	/** Seek to pos characters after the beginning of the file. */
	public static final int seekmode_Start     = 0;
	/** Seek to pos characters after the current position.
	 * (pos can be negative to move backwards). */
	public static final int seekmode_Current   = 1;
	/** Seek to pos characters after the beginning of the file.
	 * (pos should always be 0 or negative). */
	public static final int seekmode_End       = 2;


	int     m_bufid = 0;

	Stream(int id)
	{
		super(clStream, id);
	}

	static Stream findStream(int id)
	{
		return (Stream)(find(clStream, id));
	}

        /** Iterate over streams. 
         * @see OpaqueObject#iterate
         */
        public static Stream iterate(Stream s, Object r[])
        {
                return (Stream)iterate(clStream,s,r);
        }

	/** Open a stream which reads from or writes to an array in memory.
	 * @param buf The array to use.
	 * @param fmode The mode to open in (FileRef.filemode_*) 
	 * @see #openMemory(byte[],int,java.lang.Object)
	 * @return A stream that uses this array, or null if there isn't one.
	 */
	public static Stream openMemory(byte buf[], int fmode)
	{
		return openMemory(buf, fmode, null);
	}

	/** Open a stream which reads from or writes to an array in memory.
	 * <p>When outputting, if more than buflen characters are written to the stream,
	 * all of them beyond the buffer length will be thrown away, so as not to
	 * overwrite the buffer. (The character count of the stream will still be
	 * maintained correctly. That is, it will count the number of characters
	 * written into the stream, not the number that fit in the buffer.)</p>
	 * 
	 * <p>If buf is null, then <em>everything</em>
	 * written to the stream is thrown away. This may be useful if you are
	 * interested in the character count.</p>
	 * 
	 * <p>When inputting, if more than buflen characters are read from the stream,
	 * the stream will start returning -1 (signalling end-of-file.) If buf is
	 * NULL, the stream will always return end-of-file.</p>
	 * 
	 * <p>The data are written to the buffer exactly as they were passed to the
	 * printing functions (glk_put_char(), etc); input functions will read the
	 * data exactly as it exists in memory. No platform-dependent cookery will
	 * be done on it.</p>
	 * 
	 * <p>Whether reading or writing, the contents of the buffer are undefined
	 * until the stream is closed. The library may store the data there as it
	 * is written, or deposit it all in a lump when the stream is closed. It
	 * is illegal to change the contents of the buffer while the stream is open.</p>
	 * 
	 * @param buf The array to use.
	 * @param fmode The mode to open in (FileRef.filemode_*) 
	 * @param jrock The rock to give the new stream.
	 * @return A stream that uses this array, or null if there isn't one.
	 */
	public static Stream openMemory(byte buf[], int fmode, Object jrock)
	{ 
		int irock = castRock(jrock);
		int bufid;

		bufid = GlkArray.create(buf);
		Stream s = findStream(Glk.jniglk_stream_open_memory(bufid, buf.length, fmode, irock));
		if (GlkArray.destroy(bufid))
		{
			System.err.println("jniglk warning: Glk did not retain buffer in openMemory()");
			bufid = 0;
		}
		if (s != null)
		{
			s.m_bufid = bufid;
			s.m_jrock = jrock;
		}
		return s;
	}

	/** Read a character. 
	 * @return The character read (0-255) or -1 for end of file. */
	public int getChar()
	{
		return Glk.jniglk_get_char_stream(m_dispid);
	}

	/** Read from the stream until the buffer passed is full or 
	 * a newline character is read.
	 * @param line The byte array to populate.
	 * @return The number of bytes read, including the newline. */
	public int getLine(byte[] line)
	{
		return Glk.jniglk_get_line_stream(m_dispid, line);	
	}

	/** Read from the stream until the buffer passed is full or 
	 * a newline character is read.
	 * @param line The character array to populate.
	 * @return The number of bytes read, including the newline. */
	public int getLine(char [] line)
	{
		return getLine(line, 0, line.length);
	}
	
	/** Read from the stream until the buffer passed is full or 
	 * a newline character is read.
	 * @param line The character array to populate.
	 * @param first The first character to be replaced.
	 * @param len The maximum number of characters to read.
	 * @return The number of bytes read, including the newline. */
	public int getLine(char[] line, int first, int len)
	{
		byte bytes[] = new byte[len];
		int res      = getLine(bytes);
		for (int n = 0; n < len; n++) line[n + first] = (char)(bytes[n]);
		return res;	
	}

	/** Read from the stream until the buffer passed is full or 
	 * end of file.
	 * @param buf The byte array to populate.
	 * @return The number of bytes read. */
	public int getBuffer(byte[] buf)
	{
		return Glk.jniglk_get_buffer_stream(m_dispid, buf);	
	}

	/** Read from the stream until the buffer passed is full or 
	 * end of file.
	 * @param buf The byte array to populate.
	 * @param first The first byte to be replaced.
	 * @param len The maximum number of bytes to read.
	 * @return The number of bytes read. */
	public int getBuffer(byte[] buf, int first, int len)
	{
		byte bytes[] = new byte[len];
		int res = getBuffer(bytes);
		for (int n = 0; n < len; n++) buf[n + first] = bytes[n];
		return res;
	}


	/** Print one character.
	 * Only characters that can be represented in the ISO Latin-1 encoding will be 
	 * printed. Others will appear as ? marks. 
	 * 
	 * @param c the character to write */
	public void putChar(char c)
	{
		Glk.jniglk_put_char_stream(m_dispid, Glk.charToByte(c));	
	}

	/** Print a string.
	 * Only characters that can be represented in the ISO Latin-1 encoding will be 
	 * printed. Others will appear as ? marks. 
	 * 
	 * @param s the string to write */
	public void putString(String s)
	{
		Glk.jniglk_put_buffer_stream(m_dispid, Glk.stringToBuffer(s, false));
	}

	/** Print a block of characters.
	 * Only characters that can be represented in the ISO Latin-1 encoding will be 
	 * printed. Others will appear as ? marks. 
	 * 
	 * @param buf the characters to write */
	public void putBuffer(char buf[])
	{
		Glk.jniglk_put_buffer_stream(m_dispid, Glk.charsToBuffer(buf, false));
	}
	
	/** Print a block of characters.
	 * Only characters that can be represented in the ISO Latin-1 encoding will be 
	 * printed. Others will appear as ? marks. 
	 * 
	 * @param buf the characters to write
	 * @param first the index of the first character to be written
	 * @param last  the index of the last character to be written  */
	public void putBuffer(char buf[], int first, int len)
	{
		Glk.jniglk_put_buffer_stream(m_dispid, Glk.charsToBuffer(buf, first, len, false));
	}

	/** Change the style of this stream.
	 * <p>If the style is not recognised, style_Normal is assumed.
	 *
	 * @param style the new style to use */
	public void setStyle(int style)
	{
		Glk.jniglk_set_style_stream(m_dispid, style);
	}


	protected void finalize() { if (m_dispid != 0) close(); }

	/** Close the stream, not bothering to read the character counts. 
	 * 
	 * @see #close(StreamResult)
	 */
        public void close() { close(null); }
	/** Close this stream.
	 * If this is the current output stream, the current output stream is set to null.
	 *
	 * @param r Will be populated with character counts if not null.
	 */
        public void close(StreamResult r)
        {
		if (Glk.getStream() == this) Glk.setStream(null);
                Glk.jniglk_stream_close(m_dispid, r);
		if (m_bufid != 0) if (GlkArray.destroy(m_bufid)) m_bufid = 0;
		super.destroy();
	}


        long getGlkRock()       
        {
                return Glk.jniglk_stream_get_rock(m_dispid);
        }

	/** Set the position of the read/write mark.
	 * <p>It is illegal to specify a position before the beginning 
	 * or after the end of the file.</p>
	 *
	 * <p> In binary files, the mark position is exact -- it corresponds with
	 * the number of characters you have read or written. In text files, this
	 * mapping can vary, because of linefeed conversions or other character-set
	 * approximations. setPosition() and getPosition() measure positions 
	 * in the platform's native encoding -- after character cookery. 
	 * Therefore, in a text stream, it is safest to use setPosition() only 
	 * to move to the beginning or end of a file, or to a position 
	 * determined by getPosition().</p>
	 *
	 * @param pos The meaning of this depends on seekMode. 
	 * @param seekMode How to calculate the new position. 
	 * @see seekmode_Start
	 * @see seekmode_Current
	 * @see seekmode_End
	 */
	public void setPosition(int pos, int seekMode)
	{
		Glk.jniglk_stream_set_position(m_dispid, pos, seekMode);
	}

	/** Get the position of the read/write mark.
	 * <p>For memory streams and binary
	 * file streams, this is exactly the number of bytes read or written
	 * from the beginning of the stream (unless you have moved the mark with
	 * setPosition().) For text file streams, matters are more
	 * ambiguous, since (for example) writing one byte to a text file may store
	 * more than one character in the platform's native encoding. You can only
	 * be sure that the position increases as you read or write to the file.</p>
	 * @see #setPosition(int,int)
	 * @return The position of the mark. 	
	 */
	public long getPosition()
	{
		return Glk.jniglk_stream_get_position(m_dispid);
	}

        /** Set the current hyperlink value in this stream.
         * A link value is any non-zero * integer; zero indicates no link. 
         * Subsequent text output is considered to make up the body of the 
         * link, which continues until the link value is changed 
         * (or set to zero).
         * @param linkval The link value, 0 for no hyperlink.
         */
	public void setHyperlink(int linkval)
	{
		Glk.jniglk_set_hyperlink_stream(m_dispid, linkval);
	}

	/** Tell the library to load Blorb resources from this stream.
	 * Whenever your program calls an image or sound
	 * function, such as Window.imageDraw(), the library will 
	 * search this file for the resource you request.
	 * <p>The stream must be open for reading in binary mode. </p>
	 * <p>Do <em>not</em> close the stream after calling this 
	 * function. The library is responsible for closing the stream 
	 * at shutdown time. </p> */
	public void setResourceMap() throws BlorbException
	{
		int err = Glk.jnigiblorb_set_resource_map(m_dispid);
		if (err != 0) BlorbException.doThrow(err);
	}
}
