
package org.ifarchive.glk;

//
// How to support buffers? If we're passing a buffer to Glk, then we need
// to lock the buffer before Glk starts using it, and unlock it after Glk
// has finished with it. 
//
// Since a window can only request one line event at a time, here's what
// we do: The window contains an "m_buffer" byte array. When a line event
// is requested, call GlkArray.create() on it to create a persistent array.
//   When Glk returns a line event (either by cancel_line_event() or 
// select() ) then the window's endLineEvent() is called. This calls 
// GlkArray.destroy() [though Glk will almost certainly have released the 
// array first and saved us the trouble] and transforms the resulting 
// data back to a java.lang.String.
//   If you use the dispatch layer to request a line event, the Window
// won't know you did this and you'll have to unpack the buffer yourself.
//
/** A wrapper for all types of Glk window.
 * 
 * It might be an idea in the future to have separate subclasses for the 
 * separate window types. However, the way that window capabilities overlap
 * (eg: both text buffer and text grid can request line events; text grid
 * and graphics windows can request mouse events) makes it rather nasty to
 * do without full multiple inheritance.
 * 
 * @author John Elliott
 * @version 1.0
 */

public class Window extends OpaqueObject
{
	byte[] m_buffer;
	String[] m_callBuffer;
	int m_bufid;

	Window(int id)
	{
		super(clWindow, id);
	}	

	// Convenient wrapper for OpaqueObject.find
	static Window findWindow(int id)
	{
		return (Window)find(clWindow, id);
	}

	/** Get the root window. 
	 * The root window is the one which covers the entire screen area,
	 * with all the other windows inside it. 
	 * @return The root window. */
	public static Window getRoot()
	{
		int id = Glk.jniglk_window_get_root();
		return findWindow(id);
	}

	/** Used to set / clear a style hint for all window types. */
	public static final int wintype_AllTypes   = 0;
	/** Pair windows contain exactly two child windows. */
	public static final int wintype_Pair       = 1;	
	/** Blank windows support neither input nor output. */
	public static final int wintype_Blank      = 2;
	/** Text buffer windows: A stream of text.
	 *   You can only print at the end of the stream, and input a line 
	 * of text at the end of the stream.  */
	public static final int wintype_TextBuffer = 3;
	/** Text grid windows: A grid of characters in a fixed-width font. 
	 * You can print anywhere in the grid.  */
	public static final int wintype_TextGrid   = 4;
	/** A grid of colored pixels. Graphics windows do not
    	 * support text input or output, but there are image commands 
	 * to draw in them. */
	public static final int wintype_Graphics   = 5;

	/** When splitting a window, the new window will be to the left of the old one. */
	public static final int winmethod_Left          = 0;
	/** When splitting a window, the new window will be to the right of the old one. */
	public static final int winmethod_Right         = 1;
	/** When splitting a window, the new window will be above the old one. */
	public static final int winmethod_Above         = 2;
	/** When splitting a window, the new window will be below the old one. */
	public static final int winmethod_Below         = 3;
	/** Mask for the left/right/above/below constants. */
	public static final int winmethod_DirMask       = 0x0F; 
	/** When splitting a window, the new window will be a fixed size. */
	public static final int winmethod_Fixed         = 0x10;
	/** When splitting a window, the new window will be a proportion of the old window's size. */
	public static final int winmethod_Proportional  = 0x20;
	/** Mask for the fixed/proportional constants. */
	public static final int winmethod_DivisionMask  = 0xF0;
	
	/** The image appears at the current point in the text, sticking up. 
	 * That is, the bottom edge of the image is aligned with the baseline 
	 * of the line of text. */
	public static final int imagealign_InlineUp = 0x01;
	/** The image appears at the current point, and the top edge is 
	 * aligned with the top of the line of text. */
	public static final int imagealign_InlineDown = 0x02;
	/** The image appears at the current point,
	    and it is centered between the top and baseline of the line of
	    text. If the image is taller than the line of text, it will stick
	    up and down equally. */
	public static final int imagealign_InlineCenter = 0x03;
	/** The image appears in the left margin. Subsequent text will be 
	displayed to the right of the image, and will flow around it -- 
	that is, it will be left-indented for as many lines as it takes to 
	pass the image. */
	public static final int imagealign_MarginLeft = 0x04;
	/**  The image appears in the right margin,
	    and subsequent text will flow around it on the left. */
	public static final int imagealign_MarginRight = 0x05;


	/** Open a new window.
	 * 
	 * @see #open(org.ifarchive.glk.Window,int,long,int,java.lang.Object)
 	 * @param split The window to split, null if you are creating the root window.
	 * @param method The method used to split the existing window.
	 * @param size The size of the split.
	 * @param wintype The type of window to create.
	 * @return The newly-created Window, or null if one could not be
	 *        created.
	 */
	public static Window open(Window split, int method, long size,
				int wintype)
	{
		return open(split, method, size, wintype, null);
	}

	/** Open a new window.
	 * <p>If there are no windows, the first three arguments are meaningless. 
	 * split <em>must</em> be null, and method and size are ignored.</p>
	 * <p>If any windows exist, new windows must be created by splitting
	 * existing ones. split is the window you want to split; this 
	 * <em>must not</em> be null. method is a mask of constants to specify 
	 * the direction and the split method. size is the size of the split.</p>
	 * <p>The method constants are:</p>
	 * <ul>
	 * <li>winmethod_Above, winmethod_Below, winmethod_Left, winmethod_Right:
	 * The new window will be above, below, to the left, or to the right
	 * of the old one which was split.
	 * <li>winmethod_Fixed, winmethod_Proportional: The new window is a 
	 * fixed size, or a given proportion of the old window's size. 
	 * </ul>
	 * 
 	 * @param split The window to split, null if you are creating the root window.
	 * @param method The method used to split the existing window.
	 * @param size The size of the split.
	 * @param wintype The type of window to create.
	 * @param rock The rock for this window.
	 * @return The newly-created Window, or null if one could not be
	 *        created.
	 */
	public static Window open(Window split, int method, long size,
				int wintype, Object jrock)
	{
		int  splitid = 0;
		int  irock   = castRock(jrock);

		if (split != null) splitid = split.m_dispid;

		int id = Glk.jniglk_window_open(splitid, method, size, wintype, irock);
		Window w = findWindow(id);
		if (w != null) w.m_jrock = jrock;	
		return w;
	}

	protected void finalize() { if (m_dispid != 0) close(); }

	/** Close a window, ignoring the count of characters written.
	 * @see #close(org.ifarchive.glk.close(StreamResult) */
	public void close() { close(null); }
	/** This closes a window.
	 * <p>It is legal to close all your windows, or to close
	 * the root window (which does the same thing.)</p>
	 * <p>The result argument is filled with the output character 
	 * count of the window stream.</p>
	 * @param r will be populated with the number of characters output to the window stream. 
	 * @see StreamResult
	 */
	public void close(StreamResult r)
	{
		if (Glk.getStream() == getStream()) Glk.setStream(null);
		Glk.jniglk_window_close(m_dispid, r);
		// Closing the window has removed the matching Glk object.
		m_dispid = 0;
		super.destroy();
	}

	
	/** Get the width of this window.
	 * @return The width of this window, using its measurement system. */
	public long getWidth()
	{
		return Glk.jniglk_window_get_width(m_dispid);
	}

	/** Get the height of this window.
	 * @return The height of this window, using its measurement system. */
	public long getHeight()
	{
		return Glk.jniglk_window_get_height(m_dispid);
	}

	/* Get what Glk thinks this window's rock is */
	long getGlkRock() 
	{
		return Glk.jniglk_window_get_rock(m_dispid);
	}

	/** Change the constraints of an existing pair window.
	 * If you do change the key window of a pair window, the new key 
	 * window <strong>must</strong> be a descendant of that pair window.
	 * @param key The key window for the pair, or null not to change it. 
	 * @param method The split method to be used
	 * @param size The size for the key window.
	 * @see #open */
	public void setArrangement(int method, long size, Window key)
	{
		long keyid = 0;
		if (key != null) keyid = key.m_dispid;
		Glk.jniglk_window_set_arrangement(m_dispid, method, size, keyid);
	}

	/** Returns the arrangement method of a pair window. */
	public int getArrangementMethod()
	{
		return Glk.jniglk_window_get_arrangement_method(m_dispid);
	}

	/** Returns the arrangement size of a pair window. */
	public long getArrangementSize()
	{
		return Glk.jniglk_window_get_arrangement_size(m_dispid);
	}

	/** Returns the key window of a pair window. */
	public Window getArrangementKey()
	{
		return findWindow(Glk.jniglk_window_get_arrangement_key(m_dispid));
	}

	/** Iterate over windows. 
	 * @see OpaqueObject#iterate
	 */
	public static Window iterate(Window w, Object r[])
	{
		return (Window)iterate(clWindow,w,r);
	}

	/** This returns the window's type.
	 *
	 * @see #wintype_Pair
	 * @see #wintype_Blank
	 * @see #wintype_TextBuffer
	 * @see #wintype_TextGrid
	 * @see #wintype_Graphics
	 */
	public int    getType()
	{
		return Glk.jniglk_window_get_type(m_dispid);
	}

	/** This returns the window which is the parent of this window. 
	 * If this is the root window, this returns null, since the root 
	 * window has no parent. The parent of every window is a pair window;
	 * other window types are always childless.
	 * @return This window's parent
	 */
	public Window getParent()
	{
		return findWindow(Glk.jniglk_window_get_parent(m_dispid));
	}

	/** This returns the other child of this window's parent. 
	 * If this is the root window, getSibling() returns NULL.
	 * @return This window's sibling 
	 */
	public Window getSibling()
	{
		return findWindow(Glk.jniglk_window_get_sibling(m_dispid));
	}

	/** Erase the window. The meaning of this depends on the window type.
	* <ul>
	* <li>Text buffer: This may do any number of things, such as 
	*    delete all text in the window, or print enough blank lines to 
	*    scroll all text beyond visibility, or insert a page-break marker 
	*    which is treated specially by the display part of the library.</li>
	* <li>Text grid: This will clear the window, filling all positions
	*    with blanks. The window cursor is moved to the top left corner
	*    (position 0,0).</li>
	* <li>Graphics: Clears the entire window to its current background
	*    color.
	* <li>Other window types: No effect.
	* </ul>
	* <p>It is illegal to erase a window which has line input pending.
	*/
	public void clear()
	{
		Glk.jniglk_window_clear(m_dispid);
	}

	/** Set the text cursor position. This is only effective in
	 * text grid windows.
	 * 
	 * <p>If you move the cursor right past the end of a line, it wraps; 
	 * the next character which is printed will appear at the beginning
	 * of the next line.</p>
	 * 
	 * If you move the cursor below the last line, or when the 
	 * cursor reaches the end of the last line, it goes "off the screen" 
	 * and further output has no effect. You must call moveCursor() or 
	 * clear() to move the cursor back into the visible region.
	 *
	 * @param x The column to move to
	 * @param y The row to move to
	 * @see #clear()
	 */
	public void moveCursor(long x, long y)
	{
		Glk.jniglk_window_move_cursor(m_dispid, x, y);
	}

	/** Return this window's output stream.
	 *
	 * <p>Every window has an output stream associated with it. 
	 * This is created automatically, with FileRef.filemode_Write, when 
	 * you open the window. </p>
	 * 
	 * <p>A window stream cannot be closed with Stream.close(). It is closed
	 * automatically when you close its window with Window.close().</p>
	 *
	 * <p>Only printable characters (including newline) may be printed 
	 * to a window stream.</p>
	 * @return This window's output stream.
	 */
	public Stream getStream()
	{
		int sid = Glk.jniglk_window_get_stream(m_dispid);
		return Stream.findStream(sid);	
	}

	/** Return this window's echo stream.
	 * @see #setEchoStream(org.ifarchive.glk.Stream)
	 * @return This window's echo stream, or null if there is none set. */
	public Stream getEchoStream()
	{
		int sid = Glk.jniglk_window_get_echo_stream(m_dispid);
		return Stream.findStream(sid);	
	}


	/** Set the echo stream for this window.
	 *
	 * <p>Every window has an associated window stream; you print to the window
	 * by printing to this stream. However, it is possible to attach a second
	 * stream to a window. Any text printed to the window is also echoed to
	 * this second stream, which is called the window's "echo stream."</p>
	 * 
	 * <p>Effectively, any call to putChar() (or the other output commands)
	 * which is directed to the window's window stream, is replicated to the
	 * window's echo stream. This also goes for the style commands such as
	 * setStyle().</p>
	 * 
	 * <p>Note that the echoing is one-way. You can still print text directly to
	 * the echo stream, and it will go wherever the stream is bound, but it
	 * does not back up and appear in the window.</p>
	 * 
	 * <p>It is illegal to set a window's echo stream to be its 
	 * <strong>own</strong> window stream. That would create an infinite loop,
	 * and is nearly certain to crash the Glk library. It is similarly 
	 * illegal to create a longer loop (two or more windows echoing to 
	 * each other.)</p>
	 * 
	 * @param s The echo stream. Pass null to stop this window echoing. 
	 */
	public void setEchoStream(Stream s)
	{
		Glk.jniglk_window_set_echo_stream(m_dispid, s.m_dispid);
	}

        /** Request line input from this window.
         * <p>A window cannot have requests for both character and 
         * line input at the same time. It is illegal to call 
         * requestLineEvent() if the window already has a pending request 
         * for either character or line input.</p> 
	 *
	 * <p>If a window has a pending request for line input, and the player
	 * hits enter in that window (or whatever action is appropriate to
	 * enter his input), Glk.select() will return an event whose type is
	 * Event.evtype_LineInput. Once this happens, the request is complete; it is
	 * no longer pending. You must call requestLineEvent() if you want
	 * another line of text from that window.</p>
	 *
	 * <p>In the event structure, win tells what window the event came
	 * from. val1 tells how many characters were entered. val2 will be 0. The
	 * characters themselves are stored in the string passed to the 
	 * original requestLineEvent() call.</p>
	 *
	 * <p>On entry, any text in input[0] will be displayed as if the
	 *  player had typed it. When the input is completed (either by
	 *  a line buffer input event, or by cancelLineEvent(), input[0]
	 *  will contain the text typed.</p>
	 *
	 * <p>It is illegal to print anything to a window which has line input
	 * pending.</p> 
	 * @param input Initial text to display / result string.
	 * @param maxLen The maximum length of input that will be accepted.
	 */
	public void requestLineEvent(String input[], int maxLen)
	{
		String init  = input[0];
		m_buffer     = new byte[maxLen];
		m_callBuffer = input;
		// Initialise buffer from initial string
		for (int n = 0; n < init.length(); n++)
		{
			char c = init.charAt(n);
			if ((c & 0xFF00) != 0) c = '?';
			m_buffer[n] = (byte)(c & 0xFF);
		}
		m_bufid = GlkArray.create(m_buffer);
		Glk.jniglk_request_line_event(m_dispid, m_bufid, maxLen, init.length());
		if (GlkArray.destroy(m_bufid))
			System.err.println("jniglk warning: Glk did not retain the input array in requestLineEvent()");
	}

	/** Cancel a pending line event and ignore anything composed
	 * so far. 
	 * @see #cancelLineEvent(org.ifarchive.glk.Event)	
  	 */
	public void cancelLineEvent()
	{
		cancelLineEvent(null);
	}

	/** Cancel a pending line event. 
	 * 
	 * <p>This cancels a pending request for line input. The event pointed to by
	 * the event argument will be filled in as if the player had hit enter, and
	 * the input composed so far will be stored in the string passed to
	 * requestLineEvent().</p>
	 * 
	 * <p>For convenience, it is legal to call cancelLineEvent() even if
	 * there is no line input request on that window. The event type will be
	 * set to Event.evtype_None in this case.
	 */
	public void cancelLineEvent(Event e)
	{
		if (e == null) e = new Event();
		Glk.jniglk_cancel_line_event(m_dispid, e);
		endLineEvent(e);	
	}

	/* Whether line input was ended by cancelLineEvent() or 
	 * by Glk.select(), this deals with converting the data
	 * back from a raw ISO-8859-1 buffer to a Java string. */
	void endLineEvent(Event e)
	{
		int n;
		if (m_bufid == 0) return;	// Glk must have been asked
						// for this event through the
						// dispatch layer. Don't 
						// convert the buffer.
		// Release the buffer
		GlkArray.destroy(m_bufid);
		m_bufid = 0;

		// Now convert the raw buffer into a java.lang.string
		StringBuffer str = new StringBuffer(m_buffer.length);

		for (n = 0; n < (int)e.val1; n++)
		{
			str.append((char)m_buffer[n]);			
		}
		m_callBuffer[0] = new String(str);
	}

	/** Request character input from this window.
	 * <p>A window cannot have requests for both character and 
	 * line input at the same time. It is illegal to call 
	 * requestCharEvent() if the window already has a pending request 
	 * for either character or line input.</p> 
	 * <p>If a window has a pending request for character input, and the 
	 * player hits a key in that window, Glk.select() will return an 
	 * event whose type is Event.evtype_CharInput. Once this happens, 
	 * the request is complete; it is no longer pending. You must call 
	 * requestCharEvent() if you want another character from that window.</p>
	 * <p>In the event structure, win tells what window the event came from. val1
	 * tells what character was entered; this will be a code from 0 to 255, or
	 * a special keycode. val2 will be 0.</p>
	 */
	public void requestCharEvent()
	{
		Glk.jniglk_request_char_event(m_dispid);
	}

	/** Request mouse input from this window.
	 *
	 * <p>If the player clicks in a window which has a mouse input event 
	 * pending, Glk.select() will return an event whose type is 
	 * Event.evtype_MouseInput. Again, once this happens, the request 
	 * is complete, and you must request another if you want further 
	 * mouse input.</p>
	 * <p>In the event structure, win tells what window the event came 
	 * from.</p>
	 * <p>In a text grid window, the val1 and val2 fields are the x and y
	 * coordinates of the character that was clicked on. The top leftmost 
	 * character is considered to be (0,0).</p>
	 * <p>In a graphics window, they are the x and y coordinates of the 
	 * pixel that was clicked on. Again, the top left corner of the window 
	 * is (0,0).</p>
	 * <p>You can test whether mouse input is supported with the 
	 * gestalt_MouseInput selector.
	 * <p><code>res = Glk.gestalt(Glk.gestalt_MouseInput, windowtype);</code></p>
	 * <p>This will return TRUE (1) if windows of the given type support
	 * mouse input. If this returns FALSE (0), it is still legal to call
	 * requestMouseEvent(), but it will have no effect, and you will
	 * never get mouse events. */
	public void requestMouseEvent()
	{
		Glk.jniglk_request_mouse_event(m_dispid);
	}
	
	/** This cancels a pending request for character input. For convenience, it
	 * is legal to call cancelCharEvent() even if there is no character
	 * input request on that window. Glk will ignore the call in this case.
	 */
	public void cancelCharEvent()
	{
		Glk.jniglk_cancel_char_event(m_dispid);
	}

	/** This cancels a pending event for mouse input. */	
	public void cancelMouseEvent()
	{
		Glk.jniglk_cancel_mouse_event(m_dispid);
	}

	// Images
	/** This draws the given image resource.
	 * <p>In a text buffer window, val1 should be the alignment 
	 * (imagealign_*) and val2 should be 0. </p>
	 * <p>In a graphics window, val1 and val2 are the coordinates 
	 * at which to draw the image.</p>
	 * @param image The number of the image resource to draw.
	 * @param val1 The X-coordinate or alignment of the image.
	 * @param val2 The Y-coordinate of the image, or 0. 
	 * @return true if the image was drawn, otherwise false. */
	public boolean imageDraw(int image, int val1, int val2)
	{
		return Glk.jniglk_image_draw(m_dispid, image, val1, val2);
	}

	/** This draws the given image resource, scaling it to a different size.
	 * <p>In a text buffer window, val1 should be the alignment 
	 * (imagealign_*) and val2 should be 0. </p>
	 * <p>In a graphics window, val1 and val2 are the coordinates 
	 * at which to draw the image.</p>
	 * @param image The number of the image resource to draw.
	 * @param val1 The X-coordinate or alignment of the image.
	 * @param val2 The Y-coordinate of the image, or 0. 
	 * @param w The width that the image should appear.
	 * @param h The width that the image should appear.
	 * @return true if the image was drawn, otherwise false. */
	public boolean imageDrawScaled(int image, int val1, int val2, long w, long h)
	{
		return Glk.jniglk_image_draw_scaled(m_dispid, image, val1, val2, w, h);
	}

	/** Check if image exists and get its dimensions. 
	 * @param image The number of the image resource to check.
	 * @param res If not null, an array of two integers. On return
	 *           these will hold the width and height of the image in pixels.
	 * @return true if the image exists, else false. */
	public static boolean imageGetInfo(int image, int[] res)
	{
		return Glk.jniglk_image_get_info(image, res);
	}

	/** Break the stream of text around an image.
	If the current point in the text is indented around a margin-aligned
	image, this acts like the correct number of newlines to start a new
	line below the image. (If there are several margin-aligned images,
	it goes below all of them.) If the current point is <em>not</em> 
	beside a margin-aligned image, this call has no effect.
	*/
	public void flowBreak()
	{
		Glk.jniglk_window_flow_break(m_dispid);
	}

	/** Fill a rectangle with the window's background colour.
	 * It is legitimate for part of the rectangle to fall outside 
	 * the window. If width or height is zero, nothing is drawn.
	 * <p>This function can only be used in graphics windows. 
	 * @param left The left-hand edge of the rectangle.
	 * @param top The top row of the rectangle.
	 * @param w The width of the rectangle.
	 * @param h The height of the rectangle.
	 */
	public void eraseRect(int left, int top, long w, long h)
	{
        	Glk.jniglk_window_erase_rect(m_dispid, left, top, w, h);
	}

	/** Fill a rectangle with given colour.
	 * It is legitimate for part of the rectangle to fall outside 
	 * the window. If width or height is zero, nothing is drawn.
	 * <p>This function can only be used in graphics windows. 
	 * <p>The colour is defined as described in setBackgroundColor().
	 * @see #setBackgroundColor(int)
	 * @param col The colour to use.
	 * @param left The left-hand edge of the rectangle.
	 * @param top The top row of the rectangle.
	 * @param w The width of the rectangle.
	 * @param h The height of the rectangle.
	 */
	public void fillRect(int col, int left, int top, long w, long h)
	{
        	Glk.jniglk_window_fill_rect(m_dispid, col, left, top, w, h);
	}
	
	/** Set the background colour for this window.
	 * <p>This function may only be used with graphics windows.
	 * <p>Colors are encoded in a 32-bit value: the top 8 bits must be zero,
	 * the next 8 bits are the red value, the next 8 bits are the green value,
	 * and the bottom 8 bits are the blue value. Color values range from 0 to
	 * 255. 
	 * @param col The colour to use as background.
	 */
	public void setBackgroundColor(int col)
	{
        	Glk.jniglk_window_set_background_color(m_dispid, col);
	}
	

	// Hyperlinks
	public void requestHyperlinkEvent()
	{
		Glk.jniglk_request_hyperlink_event(m_dispid);
	}


	public void cancelHyperlinkEvent()
	{
		Glk.jniglk_cancel_hyperlink_event(m_dispid);
	}
}
