# multiwin.c: Sample program for Glk API, version 0.5.
#	Designed by Andrew Plotkin <erkyrath@netcom.com>
#	http://www.eblong.com/zarf/glk/index.html
#	This program is in the public domain.

#* This example demonstrates multiple windows and timed input in the
#	Glk API.

# This Python port of the original C program is by Joe Mason, 2002.  It's
# still in the public domain.

import glk

TRUE = 1
FALSE = 0

KEYWINROCK = 97

inputpending1 = FALSE
inputpending2 = FALSE
already1 = 0
already2 = 0 

timer_on = FALSE

commandbuf1 = glk.buffer_t(None, 255)
commandbuf2 = glk.buffer_t(None, 255)

mainwin1 = glk.window_open(None, 0, 0, glk.wintype_TextBuffer, 1)
if mainwin1 == None:
	glk.exit()

statuswin = glk.window_open(mainwin1, glk.winmethod_Above | glk.winmethod_Fixed, 
	5, glk.wintype_TextGrid, 0)

mainwin2 = glk.window_open(mainwin1, glk.winmethod_Below | glk.winmethod_Proportional, 
	50, glk.wintype_TextBuffer, 0)

def draw_statuswin():

	global statuswin

	if statuswin == None:
		return

	glk.set_window(statuswin)
	glk.window_clear(statuswin)

	(width, height) = glk.window_get_size(statuswin)

	width = width / 2
	if width > 0:
		width -= 1
	height = height / 2
	if height > 0:
		height -= 1

	glk.window_move_cursor(statuswin, width, height+0)
	glk.put_string("\\|/")
	glk.window_move_cursor(statuswin, width, height+1)
	glk.put_string("-*-")
	glk.window_move_cursor(statuswin, width, height+2)
	glk.put_string("/|\\")

def draw_keywins():

	(win, rock) = glk.window_iterate(None)
	while win != None:
		if rock == KEYWINROCK:
			glk.set_window(win)
			glk.window_clear(win)
			(width, height) = glk.window_get_size(win)
			glk.window_move_cursor(win, 0, 0)
			glk.put_char('O')
			glk.window_move_cursor(win, width-1, 0)
			glk.put_char('O')
			glk.window_move_cursor(win, 0, height-1)
			glk.put_char('O')
			glk.window_move_cursor(win, width-1, height-1)
			glk.put_char('O')
		(win, rock) = glk.window_iterate(win)

def perform_key(win, key):
	
	if key == ord('h') or key == ord('v'):
		if key == ord('h'):
			loc = glk.winmethod_Right | glk.winmethod_Proportional
		else:
			loc = glk.winmethod_Below | glk.winmethod_Proportional
		newwin = glk.window_open(win, 
			loc, 50, glk.wintype_TextGrid, KEYWINROCK)
		if newwin != None:
			glk.request_char_event(newwin)
			draw_keywins()

		glk.request_char_event(win)
		return
	elif key == ord('c'):
		glk.window_close(win)
		draw_keywins()
		draw_statuswin()
		return

	if key == ord(' '):
		keyname = "space"
	elif key == glk.keycode_Left:
		keyname = "left"
	elif key == glk.keycode_Right:
		keyname = "right"
	elif key == glk.keycode_Up:
		keyname = "up"
	elif key == glk.keycode_Down:
		keyname = "down"
	elif key == glk.keycode_Return:
		keyname = "return"
	elif key == glk.keycode_Delete:
		keyname = "delete"
	elif key == glk.keycode_Escape:
		keyname = "escape"
	elif key == glk.keycode_Tab:
		keyname = "tab"
	elif key == glk.keycode_PageUp:
		keyname = "page up"
	elif key == glk.keycode_PageDown:
		keyname = "page down"
	elif key == glk.keycode_Home:
		keyname = "home"
	elif key == glk.keycode_End:
		keyname = "end"
	else:
		if key >= glk.keycode_Func1 and key < glk.keycode_Func12:
			keyname = "function key"
		elif key < 32:
			keyname = "ctrl-" + chr(key)
		elif key <= 255:
			keyname = chr(key)
		else:
			keyname = "unknown key"

	buf = "Key: " + keyname
	length = len(buf)

	glk.set_window(win)
	(width, height) = glk.window_get_size(win)
	glk.window_move_cursor(win, 0, height/2)
	for ix in range(0, width):
		glk.put_char(' ')

	width = width/2
	length = len/2

	if width > length:
		width = width-length
	else:
		width = 0

	glk.window_move_cursor(win, width, height/2)
	glk.put_string(buf)

	glk.request_char_event(win)

def perform_timer():

	global mainwin1
	global inputpending1
	global already1

	if mainwin1 == None:
		return
	
	if inputpending1:
		ev = glk.cancel_line_event(mainwin1)
		if ev.type == glk.evtype_LineInput:
			already1 = ev.val1
		inputpending1 = FALSE

	glk.set_window(mainwin1)
	glk.put_string("Tick.\n")

def print_to_otherwin(win):

	global mainwin1
	global mainwin2
	global already1
	global already2
	global inputpending1
	global inputpending2

	otherwin = None

	if win == mainwin1:
		if mainwin2 != None:
			otherwin = mainwin2
			ev = glk.cancel_line_event(mainwin2)
			if ev.type == glk.evtype_LineInput:
				already2 = ev.val1
			inputpending2 = FALSE
	elif win == mainwin2:
		if mainwin1 != None:
			otherwin = mainwin1
			ev = glk.cancel_line_event(mainwin1)
			if ev.type == glk.evtype_LineInput:
				already1 = ev.val1
			inputpending1 = FALSE

	return otherwin

def verb_help(win):
	glk.set_window(win)
	glk.put_string("This model only understands the following commands:\n")
	glk.put_string("HELP: Display this list.\n")
	glk.put_string("JUMP: Print a short message.\n")
	glk.put_string("YADA: Print a long paragraph.\n")
	glk.put_string("BOTH: Print a short message in both main windows.\n")
	glk.put_string("CLEAR: Clear one window.\n")
	glk.put_string("PAGE: Print thirty lines, demonstrating paging.\n")
	glk.put_string("PAGEBOTH: Print thirty lines in each window.\n")
	glk.put_string("TIMER: Turn on a timer, which ticks in the upper ")
	glk.put_string("main window every three seconds.\n")
	glk.put_string("UNTIMER: Turns off the timer.\n")
	glk.put_string("CHARS: Prints the entire Latin-1 character set.\n")
	glk.put_string("QUIT: Quit and exit.\n")

def verb_jump(win):
	glk.set_window(win)
	glk.put_string("You jump on the fruit, spotlessly.\n")

def verb_both(win):
	glk.set_window(win)
	glk.put_string("Something happens in this window.\n")

	otherwin = print_to_otherwin(win)

	if otherwin != None:
		glk.set_window(otherwin)
		glk.put_string("Something happens in the other window.\n")

def verb_clear(win):
	glk.window_clear(win)

def verb_page(win):
	glk.set_window(win)
	for ix in range(0, 30):
		buf = "%d" % ix
		glk.put_string(buf)
		glk.put_char('\n')

def verb_pageboth(win):
	str = glk.window_get_stream(win)
	otherwin = print_to_otherwin(win)
	if otherwin:
		otherstr = glk.window_get_stream(otherwin)
	else:
		otherstr = None

	for ix in range(0, 30):
		buf = "%d\n" % ix
		glk.put_string_stream(str, buf)
		if otherstr != None:
			glk.put_string_stream(otherstr, buf)

def verb_timer(win):

	global timer_on

	glk.set_window(win)

	if timer_on:
		glk.put_string("The timer is already running.\n")
		return

	if glk.gestalt(glk.gestalt_Timer, 0) == 0:
		glk.put_string("Your Glk library does not support timer events.\n")
		return

	glk.put_string("A timer starts running in the upper window.\n")
	glk.request_timer_events(3000)
	timer_on = TRUE

def verb_untimer(win):

	global timer_on

	glk.set_window(win)
	
	if not timer_on:
		glk.put_string("The timer is not currently running.\n")
		return

	glk.put_string("The timer stops running.\n")
	glk.request_timer_events(0)
	timer_on = FALSE

def verb_chars(win):

	glk.set_window(win)

	for ix in range(0,256):
		buf = "%d" % ix
		glk.put_string(buf)
		glk.put_string(": ")
		glk.put_char(chr(ix))
		glk.put_char('\n')

def verb_yada(win):

	NUMWORDS  = 13

	wordcaplist = [ 
		"Ga", "Bo", "Wa", "Mu", "Bi", "Fo", "Za", "Mo", "Ra", "Po",
			"Ha", "Ni", "Na" ]
	wordlist = [
		"figgle", "wob", "shim", "fleb", "moobosh", "fonk", "wabble",
			"gazoon", "ting", "floo", "zonk", "loof", "lob" ]
	wcount1 = 0
	wcount2 = 0
	wstep = 1
	jx = 0
	first = TRUE

	glk.set_window(win)

	for ix in range(0, 85):
		if ix > 0:
			glk.put_string(" ")

		if first:
			glk.put_string(wordcaplist[(ix / 17) % NUMWORDS])
			first = FALSE

		glk.put_string(wordlist[jx])
		jx = (jx + wstep) % NUMWORDS
		wcount1 += 1
		if wcount1 >= NUMWORDS:
			wcount1 = 0
			wstep += 1
			wcount2 += 1
			if wcount2 >= NUMWORDS-2:
				wcount2 = 0
				wstep = 1

		if (ix % 17) == 16:
			glk.put_string(".")
			first = TRUE

	glk.put_char('\n')

def verb_quit(win):
	glk.set_window(win)

	glk.put_string("Thanks for playing.\n")
	glk.exit()

glk.set_window(mainwin1)
glk.put_string("Multiwin\nAn Interactive Sample Glk Program\n")
glk.put_string("By Andrew Plotkin.\nRelease 3.\n")
glk.put_string("Type \"help\" for a list of commands.\n")
	
glk.set_window(mainwin2)
glk.put_string("Note that the upper left-hand window accepts character")
glk.put_string(" input. Hit 'h' to split the window horizontally, 'v' to")
glk.put_string(" split the window vertically, 'c' to close a window,")
glk.put_string(" and any other key (including special keys) to display")
glk.put_string(" key codes. All new windows accept these same keys as")
glk.put_string(" well.\n\n")
glk.put_string("This bottom window accepts normal line input.\n")

if statuswin != None:
	keywin = glk.window_open(statuswin, glk.winmethod_Left | glk.winmethod_Proportional, 
		66, glk.wintype_TextGrid, KEYWINROCK)
	if keywin != None:
		glk.request_char_event(keywin)
	
draw_keywins()

while 1:

	draw_statuswin()

	if mainwin1 and not inputpending1:
		glk.set_window(mainwin1)
		glk.put_string("\n>")
		glk.request_line_event(mainwin1, commandbuf1, already1)
		inputpending1 = TRUE
	
	if mainwin2 and not inputpending2:
		glk.set_window(mainwin2)
		glk.put_string("\n>")
		glk.request_line_event(mainwin2, commandbuf2, already2)
		inputpending2 = TRUE

	doneloop = FALSE
	while not doneloop:

		ev = glk.select()

		if ev.type == glk.evtype_LineInput:
			if mainwin1 and ev.win == mainwin1:
				whichwin = mainwin1
				inputpending1 = FALSE
				commandbuf = commandbuf1
				doneloop = TRUE
			elif mainwin2 and ev.win == mainwin2:
				whichwin = mainwin2
				inputpending2 = FALSE
				commandbuf = commandbuf2
				doneloop = TRUE

		elif ev.type == glk.evtype_CharInput:
			perform_key(ev.win, ev.val1)

		elif ev.type == glk.evtype_Timer:
			whichwin = None
			commandbuf = None
			doneloop = TRUE

		elif ev.type == glk.evtype_Arrange:
			draw_statuswin()
			draw_keywins()

	if commandbuf == None:
		perform_timer()
		continue

	length = ev.val1

	cmd = ""
	for cx in commandbuf.snapshot(length).rstrip().lstrip():
		cmd += glk.char_to_lower(cx)
	
	if cmd == "":
		glk.set_window(whichwin)
		glk.put_string("Excuse me?\n")
	elif cmd == "help":
		verb_help(whichwin)
	elif cmd == "yada":
		verb_yada(whichwin)
	elif cmd == "both":
		verb_both(whichwin)
	elif cmd == "clear":
		verb_clear(whichwin)
	elif cmd == "page":
		verb_page(whichwin)
	elif cmd == "pageboth":
		verb_pageboth(whichwin)
	elif cmd == "timer":
		verb_timer(whichwin)
	elif cmd == "untimer":
		verb_untimer(whichwin)
	elif cmd == "chars":
		verb_chars(whichwin)
	elif cmd == "quit":
		verb_quit(whichwin)
	else:
		glk.set_window(whichwin)
		glk.put_string("I don't understand the command \"")
		glk.put_string(cmd)
		glk.put_string("\".\n")

	if whichwin == mainwin1:
		already1 = 0
	elif whichwin == mainwin2:
		already2 = 0
